/** 
* @file ColorTable64.cpp
* Implementation of class ColorTable64.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

#include "ColorTable64.h"
#include "Platform/GTAssert.h"

void ColorTable64::generateColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage) const
{
  colorClassImage.width = image.cameraInfo.resolutionWidth;
  colorClassImage.height = image.cameraInfo.resolutionHeight;

  for (register int y = 0; y < image.cameraInfo.resolutionHeight; y++) {
    for (register int x = 0; x < image.cameraInfo.resolutionWidth; x++) {
      colorClassImage.image[y][x] 
        = colorClasses[image.image[y][0][x]/4][image.image[y][1][x]/4][image.image[y][2][x]/4];
    }
  }
}

void ColorTable64::generateHighResColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage
    ) const
{
  colorClassImage.width = image.cameraInfo.resolutionWidth * 2;
  colorClassImage.height = image.cameraInfo.resolutionHeight * 2;

  for (register int y = 0; y < image.cameraInfo.resolutionHeight * 2; y++) {
    for (register int x = 0; x < image.cameraInfo.resolutionWidth * 2; x++) {
      colorClassImage.image[y][x] = colorClasses
        [image.getHighResY(x, y)/4]
        [image.image[y / 2][1][x / 2]/4]
        [image.image[y / 2][2][x / 2]/4];
    }
  }
}

void ColorTable64::generateColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage,
    colorClass colorClass
    ) const
{
  colorClassImage.width = image.cameraInfo.resolutionWidth;
  colorClassImage.height = image.cameraInfo.resolutionHeight;

  for (register int y = 0; y < image.cameraInfo.resolutionHeight; y++) {
    for (register int x = 0; x < image.cameraInfo.resolutionWidth; x++) {
      if(colorClasses[image.image[y][0][x]/4][image.image[y][1][x]/4][image.image[y][2][x]/4] == colorClass)
      {
        colorClassImage.image[y][x] = colorClass;
      }
      else
      {
        colorClassImage.image[y][x] = noColor;
      }
    }
  }
}

void ColorTable64::generateHighResColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage,
    colorClass colorClass
    ) const
{
  colorClassImage.width = image.cameraInfo.resolutionWidth * 2;
  colorClassImage.height = image.cameraInfo.resolutionHeight * 2;

  for (register int y = 0; y < image.cameraInfo.resolutionHeight * 2; y++) {
    for (register int x = 0; x < image.cameraInfo.resolutionWidth * 2; x++) {
      if(colorClasses
        [image.getHighResY(x,y)/4]
        [image.image[y / 2][1][x / 2]/4]
        [image.image[y / 2][2][x / 2]/4] 
        == colorClass)
      {
        colorClassImage.image[y][x] = colorClass;
      }
      else
      {
        colorClassImage.image[y][x] = noColor;
      }
    }
  }
}

ColorTable64::~ColorTable64()
{
}

ColorTable64::ColorTable64()
{
  clear();
}

void ColorTable64::addColorClass
(
 colorClass colorClass,
 unsigned char y,
 unsigned char u,
 unsigned char v
 )
{
  this->colorClasses[y/4][u/4][v/4] = colorClass;
}

void ColorTable64::addColorClass
(
 colorClass colorClass,
 unsigned char y,
 unsigned char u,
 unsigned char v,
 unsigned char range
 )
{
  y /= 4;
  u /= 4;
  v /= 4;

  if (y < range / 2) y = range / 2;
  if (u < range / 2) u = range / 2;
  if (v < range / 2) v = range / 2;

  for(unsigned char currentY = y - range / 2; currentY < y - range / 2 + range && currentY < 64; currentY++)
  {
    for(unsigned char currentU = u - range / 2; currentU < u - range / 2 + range && currentU < 64; currentU++)
    {
      for(unsigned char currentV = v - range / 2; currentV < v - range / 2 + range && currentV < 64; currentV++)
      {
        ASSERT(currentY < 64);
        ASSERT(currentU < 64);
        ASSERT(currentV < 64);
        this->colorClasses[currentY][currentU][currentV] = colorClass;
      }
    }
  }
}

void ColorTable64::addCuboidToColorClass
(
 colorClass colorClass,
 unsigned char yMin, 
 unsigned char uMin, 
 unsigned char vMin, 
 unsigned char yMax, 
 unsigned char uMax, 
 unsigned char vMax
 )
{
  yMin /= 4; uMin /= 4; vMin /= 4;
  yMax /= 4; uMax /= 4; vMax /= 4;

  for(unsigned char currentY = yMin; currentY <= yMax; currentY++)
  {
    for(unsigned char currentU = uMin; currentU <= uMax; currentU++)
    {
      for(unsigned char currentV = vMin; currentV <= vMax; currentV++)
      {
        this->colorClasses[currentY][currentU][currentV] = colorClass;
        ASSERT(currentY < 64);
        ASSERT(currentU < 64);
        ASSERT(currentV < 64);
      }
    }
  }
}

void ColorTable64::removeColorClass
(
 unsigned char y,
 unsigned char u,
 unsigned char v,
 unsigned char range
 )
{
  y /= 4;
  u /= 4;
  v /= 4;

  if (y < range / 2) y = range / 2;
  if (u < range / 2) u = range / 2;
  if (v < range / 2) v = range / 2;

  for(unsigned char currentY = y - range / 2; currentY < y - range / 2 + range + 1 && currentY < 64; currentY++)
  {
    for(unsigned char currentU = u - range / 2; currentU < u - range / 2 + range + 1 && currentU < 64; currentU++)
    {
      for(unsigned char currentV = v - range / 2; currentV < v - range / 2 + range + 1 && currentV < 64; currentV++)
      {
        this->colorClasses[currentY][currentU][currentV] = noColor;
      }
    }
  }
}


void ColorTable64::clearChannel(colorClass colorClass)
{
  for (unsigned char y = 0; y < 64; y++)
  {
    for (unsigned char u = 0; u < 64; u++)
    {
      for (unsigned char v = 0; v < 64; v++)
      {
        if (colorClasses[y][u][v] == colorClass) colorClasses[y][u][v] = noColor;
      }
    }
  }
}

void ColorTable64::clear()
{
  for (unsigned char y = 0; y < 64; y++)
  {
    for (unsigned char u = 0; u < 64; u++)
    {
      for (unsigned char v = 0; v < 64; v++)
      {
        this->colorClasses[y][u][v] = noColor;
      }
    }
  }
}

void ColorTable64::getBoxAroundColorClass(colorClass color, 
                                          Vector3<int>& pNear, Vector3<int>& pFar)
{
  pNear = Vector3<int>(0,0,0); 
  pFar = Vector3<int>(255,255,255); 
  unsigned char y(0),u(0),v(0);
  while(y<64)
  {
    u = 0;
    while(u<64)
    {
      v = 0;
      while(v<64)
      {
        if(colorClasses[y][u][v] == color)
        {
          Vector3<int> first(y,u,v);
          pNear = first;
          pFar = first;
          v++;
          goto expandBox;
        }
        v++;
      }
      u++;
    }
    y++;
  }
  return; //Return if the color has not been found
  expandBox:
  while(y<64)
  {
    u = 0;
    while(u<64)
    {
      v = 0;
      while(v<64)
      {
        if(colorClasses[y][u][v] == color)
        {
          Vector3<int> expand(y,u,v);
          if(expand.x < pNear.x)
          {
            pNear.x = expand.x;
          }
          else if(expand.x > pFar.x)
          {
            pFar.x = expand.x;
          }
          if(expand.y < pNear.y)
          {
            pNear.y = expand.y;
          }
          else if(expand.y > pFar.y)
          {
            pFar.y = expand.y;
          }
          if(expand.z < pNear.z)
          {
            pNear.z = expand.z;
          }
          else if(expand.z > pFar.z)
          {
            pFar.y = expand.y;
          }
        }
        v++;
      }
      u++;
    }
    y++;
  }
  pNear *= 4;
  pFar *= 4;
}

Out& operator << (Out& stream, const ColorTable64& colorTable64)
{
  // the old version of the streaming operator, can be used for converting
  // old colortables to new ones (with the VC debugger)
//  stream.write(&colorTable64.colorClasses,sizeof(colorTable64.colorClasses));
//  return stream;


  // the current color class in the color table
  unsigned char currentColorClass = (unsigned char)(colorTable64.colorClasses[0][0][0]);

  // the next color class to be compared with the last one 
  unsigned char nextColorClass;

  // a pointer to the begin of the color table
  const unsigned char* colorTable = (const unsigned char*)&colorTable64.colorClasses;

  // the number of bytes in one sequence that have the same color class
  int currentLength = 1;

  for (unsigned int i=1;i<sizeof(colorTable64.colorClasses);i++)
  {
    nextColorClass = colorTable[i];
    
    if (nextColorClass != currentColorClass)
    {
      stream << currentLength << currentColorClass;
      currentColorClass = nextColorClass;
      currentLength = 1;
    }
    else
    {
      currentLength++;
    }
  }
  stream << currentLength << currentColorClass;

  // write a 0 at the end of the stream as a marker for the In streaming operator
  stream << (int)0;

  return stream;
}

In& operator>>(In& stream,ColorTable64& colorTable64)
{
  // the old version of the streaming operator, can be used for converting
  // old colortables to new ones (with the VC debugger)
//  stream.read(&colorTable64.colorClasses,sizeof(colorTable64.colorClasses));
//  return stream;

  // a pointer to the begin of the color table
  unsigned char* colorTable = (unsigned char*)&colorTable64.colorClasses;

  // the current color class in the color table
  unsigned char currentColorClass;

  // the number of bytes in one sequence that have the same color class
  unsigned int currentLength;

  // the position in the color table
  unsigned int pos=0;

  while(pos < sizeof(colorTable64.colorClasses))
  {
    stream >> currentLength;
    if (currentLength == 0)
      if(!pos) // support for old format that contains table size
        stream >> currentLength >> currentLength; // skip length
      else
        break;
    stream >> currentColorClass;
    
    for (unsigned int i=0;i<=currentLength && i + pos < sizeof(colorTable64.colorClasses);i++)
    {
      colorTable[i+pos] = currentColorClass;
      if (i+pos > sizeof(colorTable64.colorClasses) - 1)
        break;;
    }

    pos += currentLength;
  }

  return stream;
}


/*
 * Change log :
 * 
 * $Log: ColorTable64.cpp,v $
 * Revision 1.12  2004/04/07 13:00:44  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.4  2004/04/06 13:19:34  risler
 * cleaned up and improved high resolution image support
 *
 * Revision 1.3  2004/04/02 14:46:18  risler
 * added segmented high resolution image
 * colortable dialog supports high resolution images
 *
 * Revision 1.2  2004/04/01 11:01:26  risler
 * bugfix in addColorClass range check
 *
 * Revision 1.1.1.1  2004/03/29 08:28:45  Administrator
 * initial transfer from tamara
 *
 * Revision 1.11  2004/03/05 17:57:57  tim
 * fixed bug in box computation
 *
 * Revision 1.10  2004/03/01 11:41:32  juengel
 * bug fixed
 *
 * Revision 1.9  2004/02/16 20:38:47  tim
 * fixed warning
 *
 * Revision 1.8  2004/02/16 18:23:47  tim
 * Added getBoxAroundColorClass
 *
 * Revision 1.7  2004/01/13 09:24:38  thomas
 * fix: increment-limit in for
 *
 * Revision 1.6  2003/12/30 23:55:01  roefer
 * Image size is now 208 x 160. Smaller images are placed in the upper left corner
 *
 * Revision 1.5  2003/12/15 11:46:13  juengel
 * Introduced CameraInfo
 *
 * Revision 1.4  2003/12/09 16:25:28  jhoffman
 * - addColorClass( , , , , ) now works like addColorClass( , , , , , range)
 * - fixed x fixed --> cameraResolutionHeight x ...Width
 *
 * Revision 1.3  2003/10/31 11:48:26  juengel
 * Replaced fixed image height and width.
 *
 * Revision 1.2  2003/10/29 13:24:42  juengel
 * Added addCuboidToColorClass to ColorTable64
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/07/21 13:43:51  juengel
 * generateColorClassImage returns void
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/04/14 16:12:09  loetzsch
 * ATH after GermanOpen CVS merge
 * improved color table dialog bar
 *
 * Revision 1.2  2003/04/10 08:29:19  juengel
 * no message
 *
 * Revision 1.5  2003/03/10 18:18:40  dueffert
 * const cast warning removed
 *
 * Revision 1.4  2003/03/06 11:09:00  dueffert
 * signed comparison warnings removed
 *
 * Revision 1.3  2003/02/18 21:29:17  osterhues
 * Changed all instances of ColorTable64 to new base class ColorTable
 *
 * Revision 1.2  2003/01/09 14:11:31  jhoffman
 * added some stuff needed for ISL
 *
 * Revision 1.1  2002/09/10 15:26:40  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.5  2002/07/23 12:55:58  loetzsch
 * removed base class ColorTable
 *
 * new streaming classes
 *
 * Revision 1.4  2002/07/09 20:08:26  roefer
 * Extract simulation colors from color table
 *
 * Revision 1.3  2002/06/08 17:02:15  juengel
 * Replaced image.cameraInfo.resolutionHeight by 144.
 *
 * Revision 1.2  2002/05/14 12:39:04  dueffert
 * corrected some documentation mistakes
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.12  2002/04/23 15:19:42  roefer
 * Buffered file i/o
 *
 * Revision 1.11  2002/04/08 20:23:07  roefer
 * Fast loading format for color tables
 *
 * Revision 1.10  2002/02/10 12:44:27  loetzsch
 * fixed a bug in the >> operator
 *
 * Revision 1.9  2002/01/22 22:04:06  loetzsch
 * new streaming operator
 *
 * Revision 1.8  2001/12/22 20:35:35  loetzsch
 * removed ColorTable::segmentImage, replaced by
 * ColorTable64::generateColorClassImage
 *
 * Revision 1.7  2001/12/21 14:22:41  roefer
 * warnings fixed: inline/eof
 *
 * Revision 1.6  2001/12/21 00:49:48  loetzsch
 * segmentImage in die Basisklasse verschoben
 *
 * Revision 1.5  2001/12/20 21:20:43  tim
 * Umbenennung im Konstructor (wegen Warnung)
 *
 * Revision 1.4  2001/12/20 19:24:31  juengel
 * Add Methoden gendert.
 *
 * Revision 1.3  2001/12/19 14:04:47  loetzsch
 * In ColorTables Pointer gegen Referenzen ausgetauscht,
 * Funktion segmentImage hinzugefgt und implementiert
 *
 * Revision 1.2  2001/12/17 19:24:42  juengel
 * no message
 *
 */
