/** 
* @file MotionRequest.h
*
* This file contains the MotionRequest class.
*
* @author <A href=mailto:brunn@sim.informatik.tu-darmstadt.de>Ronnie Brunn</A>
* @author <A href=mailto:kuntze@sim.informatik.tu-darmstadt.de>Nicolai Kuntze</A>
* @author <A href=mailto:kunz@sim.informatik.tu-darmstadt.de>Michael Kunz</A>
* @author Max Risler
*/

#ifndef __MOTIONREQUEST_H__
#define __MOTIONREQUEST_H__

#include "Tools/Streams/InOut.h"
#include "Tools/Math/Pose2D.h"


/**@name walking max speeds
*@{
* @attention These values are implementation independent maximum speeds.
* They dont reflect speeds reached with current walking engine implementations.
*
* They should be used e.g. for the Motion Tester and for the behavior to walk with
* maximum speed.
*/
/** the maximum forward speed for walk in mm/s */
static const double walkMaxForwardSpeed = 450.0;
/** the maximum backward speed for walk in mm/s */
static const double walkMaxBackwardSpeed = 350.0;
/** the maximum sideward left/right speed for walk in mm/s */
static const double walkMaxLeftRightSpeed = 400.0;
/** the maximum rotation speed for walk in rad/s */
static const double walkMaxRotationSpeed = fromDegrees(250);
/** 
*@} */

/**
* This describes the MotionRequest
*/
class MotionRequest
{
public:
  
  unsigned long frameNumber;
  
  void setFrameNumber(unsigned long frameNumber) {this->frameNumber = frameNumber;}
  
  
  /** ids for all motion types */
  enum MotionID {
    stand,
      walk,
      specialAction,
      getup,
      numOfMotion
  };
  
  /** returns names for motion ids */
  static const char* getMotionName(MotionID id)
  {
    switch (id)
    {
    case stand: return "stand";
    case walk: return "walk";
    case specialAction: return "specialAction";
    case getup: return "getup";
      
    default: return "Unknown motion name, please edit MotionRequest::getMotionName";
    }
  }
  
  /** ids for all special actions */
  enum SpecialActionID {
	  ballFunEnd,
	  ballFunStart,
    //please sort alphabetically, but only these IDs here:
    
      athErs7ChestKick2,
      athErs7fastDiveKick,
      athErs7ForwardLeapingKick,
      athErs7HeadKickLeft,
      athErs7HeadKickRight,
      athErs7KickForwardWithLeftPaw,
      athErs7KickForwardWithRightPaw,
      athErs7SideHeadKickL,
      athErs7SideHeadKickR,
      athErs7ExactForwardKick,
      athErs7GrabBall,
      athErs7GOAssShacker,
      athErs7GOJaul1,
      athErs7GOJaul2,
      athErs7GOJaul3,
      athErs7GOJaul4,
	  athErs210GOJaul5,
	  athErs210GOJaul6,
      athErs7GOLost,
      athErs7GOScored1,
      athErs7GOScored2,
      athErs7GOScored3,
      athErs7GOScored4,
	  athErs210GOScored5,
	  athErs210GOScored6,
      athErs7GOWon,
      athErs7GOShortDance,
      athGreet,
      athHeadStand,
      athLegTrick1,
      athLegTrick2,
      athScratchHead,
      athSit,
      athWaveLeft,
      athWaveRight,
      AA,
      BBloser,
      BBwinner,
      BB2004ArmKickLeft,
      BB2004ArmKickRight,
      BB2004ChestKick,
      BB2004ChestKick2,
      BB2004ChestKick3,
      BB2004ChestFallKick,
      BB2004GoalieDefend,
      BB2004HeadKick,
      BB2004HeadKickLeft,
      BB2004HeadKickRight,
      BB2004PawKick,
      armKickLeft,
      armKickRight,
      ballChallengeForwardBicycleKick,
      ballChallengeForwardPawKick,
      ballChallengeGrabBall,
      ballChallengeGrabWalk,
      ballChallengeKickForward,
      ballChallengeStand,
      ballChallengeTurnLeft,
      ballChallengeTurnRight,
      ballChallengeWalkBackwards,
      bicycleKick,
      bicycleKickFromHold,
      bicycleKickLeft,
      bicycleKickRight,
      bombayGoodbye,
      bombayWarmup,
      bow,
      catchBall,
      catchBall2,
      catchBallAndCheck,
      chestKick1,
      chestKick2,
      darmstadtGOCheckForBall,
      dddErs7bande,
      dddErs7bash,
      dddErs7catch,
      dddErs7chest,
      dddErs7dive,
      dddErs7grab,
      dddErs7headLeft,
      dddErs7headRight,
      dddErs7headStraight,
      dddErs7hold,
      dddErs7kickLeft,
      dddErs7kickRight,
      dddErs7PassLeftUnderChest,
      dddErs7press,
      dddErs7push,
      dddErs7SoccerKickLeft,
      dddErs7SoccerKickRight,
      escape,
      escapePadua,
      fastDiveKick,
      fastHeadKickLeft,
      fastHeadKickRight,
      forwardLeapingKick,
      goalieDefend,
      goalieDefendLeft,
      goalieDefendRight,
      goalieDefendStand,
      goalieJumpLeft,
      goalieJumpRight,
      grabBall,
      hail,
      headKick,
      headKickERS7,
      headStand,
      headStand2,
      heelKickLeft,
      heelKickRight,
      howl,
      kickFromHold,
      kickWithLeftToRight,
      kickWithRightToLeft,
      lanceKickHardLeft,
      lanceKickHardRight,
      lanceKickSoftLeft,
      lanceKickSoftRight,
      leftHeadKick,
      leftHeadKick2,
      leftKick,
      leftKickFromHold,
      legKick,
      legRotate,
      lookForLandmarksWithBallCaught,
      mantisKick,
      mantisKickFast,
      MSH710cm,
      MSH7ABombBehind,
      MSH7ABombFront,
      MSH7AnyLeft,
      MSH7AnyRight,
      MSH7CatchBall,
    MSH7CatchFarBall,
    MSH7CatchFarBallGrypho,
    MSH7CheckCaughtBall,
    MSH7KoppInNacken,
    MSH7KlappeZu,
      MSH7FakeKickLeft,
      MSH7FakeKickRight,
      MSH7FastChestKick,
      MSH7FastHeadKickToLeft,
      MSH7FastHeadKickToRight,
      MSH7FastLeftArm,
      MSH7FastRightArm,
      MSH7ForwardLeftPaw,
      MSH7ForwardRightPaw,
      MSH7HeadKickToLeft,
      MSH7HeadKickToRight,
      MSH7LeftHook,
      MSH7RightHook,
    MSH7TwoPawsOnce,
      MSH7TwoPawsTwice,
    MSH7TurnRight30,
    MSH7TurnRight60,
    MSH7TurnRight90,
    MSH7TurnRight120,
    MSH7TurnRight180,
    MSH7TurnLeft30,
    MSH7TurnLeft60,
    MSH7TurnLeft90,
    MSH7TurnLeft120,
    MSH7TurnLeft180,
      MSH7unswBash,
    MSH7unswBashFromHold,
    MSH7Warmup,
    MSH7VanGogh,
    MSH7StrangeBackSlow,
    MSH7PutRight90,
    MSH7PutLeft90,
    MSH7OnNeckLeft,
    MSH7OnNeckRight,
    MSH7NewGrabLeft,
    MSH7NewGrabRight,
    MSH7NewBicycleRight,
    MSH7NewBicycleFromLeft,
    MSH7NewBicycleFromRight,
    MSH7LeftArmGrabBackup,
    MSH7GrabKickLeft,
    MSH7ForwardRight,
    MSH7ForwardLeft,
    MSH7ComplicatedKick,
    MSH7BackKickSideward,
      nodHeadLR,
      nodHeadRL,
      openArm,
      pack,
      packHeadStand,
      pawKickWithLeftForward,
      pawKickWithRightForward,
      playDead,
      rightHeadKick,
      rightHeadKick2,
      rightHeadKickFast,
      rightKick,
      rightKickFromHold,
      rollLeft,
      rollRight,
      scratchHead,
      sit,
      sitAndWaveLeft,
      sitAndWaveRight,
      standStraight,
      standUpsideDown,
      swimKickHard,
      swimKickSoft,
      swing,
      unswBash,
      unswBashFromHold,
      unswBashOptimizedERS7,
      unswBashOptimized,
      unswChestKick,
      wakeUp,
      wakeUp2,
      wakeUp3,
      wakeUp4,
      wakeUp5,
      wakeUp6,
      wakeUp7,
      wakeUp8,
      waveLeft,
      waveRight,
      
      numOfSpecialAction
  };
  
  /** returns names for special action ids */
  static const char* getSpecialActionName(SpecialActionID id)
  {
    switch (id)
    {
		case ballFunEnd: return "ballFunEnd";
		case ballFunStart: return "ballFunStart";
      //kicks
    case headKickERS7: return "headKickERS7";
    case headKick: return "headKick";
    case chestKick1: return "chestKick1";
    case chestKick2: return "chestKick2";
    case leftKick: return "leftKick";
    case rightKick: return "rightKick";
    case leftKickFromHold: return "leftKickFromHold";
    case rightKickFromHold: return "rightKickFromHold";
    case leftHeadKick: return "leftHeadKick";
    case leftHeadKick2: return "leftHeadKick2";
    case kickFromHold: return "kickFromHold";
    case rightHeadKick: return "rightHeadKick";
    case rightHeadKick2: return "rightHeadKick2";
    case rightHeadKickFast: return "rightHeadKickFast";
    case legKick: return "legKick";
    case mantisKick: return "mantisKick";
    case mantisKickFast: return "mantisKickFast";
    case armKickLeft: return "armKickLeft";
    case armKickRight: return "armKickRight";
    case bicycleKick: return "bicycleKick";
    case bicycleKickFromHold: return "bicycleKickFromHold";
    case bicycleKickRight: return "bicycleKickRight";
    case bicycleKickLeft: return "bicycleKickLeft";
    case pawKickWithLeftForward: return "pawKickWithLeftForward";
    case pawKickWithRightForward: return "pawKickWithRightForward";
    case kickWithLeftToRight: return "kickWithLeftToRight";
    case kickWithRightToLeft: return "kickWithRightToLeft";
    case heelKickRight: return "heelKickRight";
    case heelKickLeft: return "heelKickLeft";
    case unswBashOptimized: return "unswBashOptimized";
    case catchBall2: return "catchBall2";
    case catchBallAndCheck: return "catchBallAndCheck";
    case grabBall: return "grabBall";
    case goalieDefendStand: return "goalieDefendStand";
    case goalieDefendLeft: return "goalieDefendLeft";
    case goalieDefendRight: return "goalieDefendRight";
    case goalieJumpLeft: return "goalieJumpLeft";
    case goalieJumpRight: return "goalieJumpRight";
    case forwardLeapingKick: return "forwardLeapingKick";
    case swimKickHard: return "swimKickHard";
    case swimKickSoft: return "swimKickSoft";
    case fastDiveKick: return "fastDiveKick";
    case fastHeadKickLeft: return "fastHeadKickLeft";
    case fastHeadKickRight: return "fastHeadKickRight";
    case lanceKickHardLeft: return "lanceKickHardLeft";
    case lanceKickHardRight: return "lanceKickHardRight";
    case lanceKickSoftLeft: return "lanceKickSoftLeft";
    case lanceKickSoftRight: return "lanceKickSoftRight";
      
      //special darmstadt go
    case legRotate: return "legRotate";
    case darmstadtGOCheckForBall: return "darmstadtGOCheckForBall";
    case dddErs7bande: return "dddErs7bande";
    case dddErs7bash: return "dddErs7bash";
    case dddErs7catch: return "dddErs7catch";
    case dddErs7chest: return "dddErs7chest";
    case dddErs7dive: return "dddErs7dive";
    case dddErs7headLeft: return "dddErs7headLeft";
    case dddErs7headRight: return "dddErs7headRight";
    case dddErs7headStraight: return "dddErs7headStraight";
    case dddErs7hold: return "dddErs7hold";
    case dddErs7grab: return "dddErs7grab";
    case dddErs7kickLeft: return "dddErs7kickLeft";
    case dddErs7kickRight: return "dddErs7kickRight";
    case dddErs7PassLeftUnderChest: return "dddErs7PassLeftUnderChest";
    case dddErs7press: return "dddErs7press";
    case dddErs7push: return "dddErs7push";
    case dddErs7SoccerKickLeft: return "dddErs7SoccerKickLeft";
    case dddErs7SoccerKickRight: return "dddErs7SoccerKickRight";

    //nonsense
    case sit: return "sit";
    case scratchHead: return "scratchHead";
    case headStand: return "headStand";
    case headStand2: return "headStand2";
    case swing: return "swing";
    case standStraight: return "standStraight";
    case playDead: return "playDead";
    case wakeUp: return "wakeUp";
    case wakeUp2: return "wakeUp2";
    case wakeUp3: return "wakeUp3";
    case wakeUp4: return "wakeUp4";
    case wakeUp5: return "wakeUp5";
    case wakeUp6: return "wakeUp6";
    case wakeUp7: return "wakeUp7";
    case wakeUp8: return "wakeUp8";
    case rollRight: return "rollRight";
    case rollLeft: return "rollLeft";
    case standUpsideDown: return "standUpsideDown";
      
    case pack: return "pack";
    case escape: return "escape";
    case escapePadua: return "escapePadua";
    case packHeadStand: return "packHeadStand";
    case bombayWarmup: return "bombayWarmup";
    case bombayGoodbye: return "bombayGoodbye";
    case howl: return "howl";
    case AA: return "AA";
      
      // Ball Challenge
    case unswBash: return "unswBash";
    case unswChestKick: return "unswChestKick";
    case goalieDefend: return "goalieDefend";      
    case catchBall: return "catchBall";
    case lookForLandmarksWithBallCaught: return "lookForLandmarksWithBallCaught";
    case openArm: return "openArm";
    case ballChallengeForwardBicycleKick: return "ballChallengeForwardBicycleKick";
    case ballChallengeForwardPawKick: return "ballChallengeForwardPawKick";
    case ballChallengeGrabBall: return "ballChallengeGrabBall";
    case ballChallengeGrabWalk: return "ballChallengeGrabWalk";
    case ballChallengeKickForward: return "ballChallengeKickForward";
    case ballChallengeWalkBackwards: return "ballChallengeWalkBackwards";
    case ballChallengeTurnLeft: return "ballChallengeTurnLeft";
    case ballChallengeTurnRight: return "ballChallengeTurnRight";
    case ballChallengeStand: return "ballChallengeStand";
      
      
      
    case BBloser: return "BBloser";
    case BBwinner: return "BBwinner";
    case unswBashFromHold: return "unswBashFromHold";
    case bow: return "bow";
    case hail: return "hail";
    case nodHeadLR: return "nodHeadLR";
    case nodHeadRL: return "nodHeadRL";
    case sitAndWaveLeft: return "sitAndWaveLeft";
    case sitAndWaveRight: return "sitAndWaveRight";
    case waveLeft: return "waveLeft";
    case waveRight: return "waveRight";
      
      // ATH 2004 German Open
    case athErs7ChestKick2: return "athErs7ChestKick2";
    case athErs7fastDiveKick: return "athErs7fastDiveKick";
    case athErs7SideHeadKickR: return "athErs7SideHeadKickR";
    case athErs7SideHeadKickL: return "athErs7SideHeadKickL";
    case athErs7KickForwardWithRightPaw: return "athErs7KickForwardWithRightPaw";
    case athErs7KickForwardWithLeftPaw: return "athErs7KickForwardWithLeftPaw";
    case athErs7ForwardLeapingKick: return "athErs7ForwardLeapingKick";
    case athErs7HeadKickLeft: return "athErs7HeadKickLeft";
    case athErs7HeadKickRight: return "athErs7HeadKickRight";
    case athErs7ExactForwardKick: return "athErs7ExactForwardKick";
    case athErs7GrabBall: return "athErs7GrabBall";
    case athGreet: return "athGreet";
    case athHeadStand: return "athHeadStand";
    case athLegTrick1: return "athLegTrick1";
    case athLegTrick2: return "athLegTrick2";
    case athScratchHead: return "athScratchHead";
    case athSit: return "athSit";
    case athWaveLeft: return "athWaveLeft";
    case athWaveRight: return "athWaveRight";
    case athErs7GOShortDance: return "athErs7GOShortDance";
      //Cheering
    case athErs7GOJaul1: return "athErs7GOJaul1";
    case athErs7GOJaul2: return "athErs7GOJaul2";
    case athErs7GOJaul3: return "athErs7GOJaul3";
    case athErs7GOJaul4: return "athErs7GOJaul4";
	case athErs210GOJaul5: return "athErs210GOJaul5";
    case athErs210GOJaul6: return "athErs210GOJaul6";
    case athErs7GOScored1: return "athErs7GOScored1";
    case athErs7GOScored2: return "athErs7GOScored2";
    case athErs7GOScored3: return "athErs7GOScored3";
    case athErs7GOScored4: return "athErs7GOScored4";
	case athErs210GOScored5: return "athErs210GOScored5";
	case athErs210GOScored6: return "athErs210GOScored6";
    case athErs7GOLost: return "athErs7GOLost";
    case athErs7GOWon: return "athErs7GOWon";
      
    case athErs7GOAssShacker: return "athErs7GOAssShacker";
      
     // BB2004 German Open
    case BB2004ChestFallKick: return "BB2004ChestFallKick";
    case BB2004ChestKick: return "BB2004ChestKick";
    case BB2004ArmKickLeft: return "BB2004ArmKickLeft";
    case BB2004ArmKickRight: return "BB2004ArmKickRight";
    case BB2004HeadKickLeft: return "BB2004HeadKickLeft";
    case BB2004HeadKickRight: return "BB2004HeadKickRight";
    case BB2004HeadKick: return "BB2004HeadKick";
    case BB2004ChestKick2: return "BB2004ChestKick2";
    case BB2004ChestKick3: return "BB2004ChestKick3";
    case BB2004GoalieDefend: return "BB2004GoalieDefend";
    case BB2004PawKick: return "BB2004PawKick";
    case unswBashOptimizedERS7: return "unswBashOptimizedERS7";
      
      // MSH 2004 German Open
    case MSH710cm: return "MSH710cm";
    case MSH7ABombBehind: return "MSH7ABombBehind";
    case MSH7ABombFront: return "MSH7ABombFront";
    case MSH7AnyRight: return "MSH7AnyRight";
    case MSH7AnyLeft: return "MSH7AnyLeft";
    case MSH7CatchBall: return "MSH7CatchBall";
  case MSH7CatchFarBall: return "MSH7CatchFarBall";
  case MSH7CatchFarBallGrypho: return "MSH7CatchFarBallGrypho";
  case MSH7CheckCaughtBall: return "MSH7CheckCaughtBall";
    case MSH7FakeKickLeft: return "MSH7FakeKickLeft";
    case MSH7FakeKickRight: return "MSH7FakeKickRight";
    case MSH7FastChestKick: return "MSH7FastChestKick";
    case MSH7FastHeadKickToLeft: return "MSH7FastHeadKickToLeft";
    case MSH7FastHeadKickToRight: return "MSH7FastHeadKickToRight";
    case MSH7FastRightArm: return "MSH7FastRightArm";
    case MSH7FastLeftArm: return "MSH7FastLeftArm";
    case MSH7ForwardLeftPaw: return "MSH7ForwardLeftPaw";
    case MSH7ForwardRightPaw: return "MSH7ForwardRightPaw";
    case MSH7HeadKickToLeft: return "MSH7HeadKickToLeft";
    case MSH7HeadKickToRight: return "MSH7HeadKickToRight";
    case MSH7LeftHook: return "MSH7LeftHook";
    case MSH7RightHook: return "MSH7RightHook";
  case MSH7TwoPawsOnce: return "MSH7TwoPawsOnce";
    case MSH7TwoPawsTwice: return "MSH7TwoPawsTwice";
    case MSH7unswBash: return "MSH7unswBash";
  case MSH7TurnRight30: return "MSH7TurnRight30";
  case MSH7TurnRight60: return "MSH7TurnRight60";
  case MSH7TurnRight90: return "MSH7TurnRight90";
  case MSH7TurnRight120: return "MSH7TurnRight120";
  case MSH7TurnRight180: return "MSH7TurnRight180";
  case MSH7TurnLeft30: return "MSH7TurnLeft30";
  case MSH7TurnLeft60: return "MSH7TurnLeft60";
  case MSH7TurnLeft90: return "MSH7TurnLeft90";
  case MSH7TurnLeft120: return "MSH7TurnLeft120";
  case MSH7TurnLeft180: return "MSH7TurnLeft180";
  case MSH7unswBashFromHold: return "MSH7unswBashFromHold";
  case MSH7Warmup: return "MSH7Warmup";
  case MSH7VanGogh: return "MSH7VanGogh";
  case MSH7StrangeBackSlow: return "MSH7StrangeBackSlow";
  case MSH7PutRight90: return "MSH7PutRight90";
  case MSH7PutLeft90: return "MSH7PutLeft90";
  case MSH7OnNeckLeft: return "MSH7OnNeckLeft";
  case MSH7OnNeckRight: return "MSH7OnNeckRight";
  case MSH7NewGrabLeft: return "MSH7NewGrabLeft";
  case MSH7NewGrabRight: return "MSH7NewGrabRight";
  case MSH7NewBicycleRight: return "MSH7NewBicycleRight";
  case MSH7NewBicycleFromLeft: return "MSH7NewBicycleFromLeft";
  case MSH7NewBicycleFromRight: return "MSH7NewBicycleFromRight";
  case MSH7LeftArmGrabBackup: return "MSH7LeftArmGrabBackup";
  case MSH7GrabKickLeft: return "MSH7GrabKickLeft";
  case MSH7ForwardRight: return "MSH7ForwardRight";
  case MSH7ForwardLeft: return "MSH7ForwardLeft";
  case MSH7ComplicatedKick: return "MSH7ComplicatedKick";
  case MSH7BackKickSideward: return "MSH7BackKickSideward";
  case MSH7KoppInNacken: return "MSH7KoppInNacken";
  case MSH7KlappeZu: return "MSH7KlappeZu";


      
      
    default: return "please edit MotionRequest::getSpecialActionName";
    }
  }
  
  
  /** ids for all walking types */
  enum WalkType {
    normal,
    upsideDown,
    turnWithBall,
    turnWithBall2,
    turnWithBall3,
    turnWithBall4,
    turnWithBall5,
    turnWithBallRearOnly,
    dash,
    debug,
      numOfWalkType
  };
  
  /** returns names for walking types */
  static const char* getWalkTypeName(WalkType id)
  {
    switch (id)
    {
    case normal: return "normal";
    case upsideDown: return "upsideDown";
    case turnWithBall: return "turnWithBall";
    case turnWithBall2: return "turnWithBall2";
    case turnWithBall3: return "turnWithBall3";
    case turnWithBall4: return "turnWithBall4";
    case turnWithBall5: return "turnWithBall5";
    case turnWithBallRearOnly:  return "turnWithBallRearOnly";
    case dash: return "dash";
    case debug: return "debug";
    default: return "Unknown motion name, please edit MotionRequest::getWalkTypeName";
    }
  }
  
  /** ids for tail request */
  enum TailRequest
  {
    wagHorizontal,
      wagVertical,
      wagLeftRightHorizontal,
      wagLeftRightVertical,
      wagUpDownLeft,
      wagUpDownRight,
      stayAsForced,
      stayAsForcedPan,
      stayAsForcedTilt,
      tailFollowsHead,
      tailParallelToGround,
      noTailWag,
      wagHorizontalFast,
      wagVerticalFast,
      twoPositionSwitchVertical,
      twoPositionSwitchHorizontal,
      threePositionSwitchVertical,
      threePositionSwitchHorizontal,
      fourPositionSwitchUpDownLeftRight,
      fourPositionSwitchCorners,
      fivePositionSwitch,
      eightPositionSwitch,
      tailLeftTop,    tailCenterTop,    tailRightTop,
      tailLeftCenter, tailCenterCenter, tailRightCenter,
      tailLeftBottom, tailCenterBottom, tailRightBottom,
      numOfTailRequests
  };
  
  /** returns names for tail requests */
  static const char* getTailRequestName(TailRequest id)
  {
    switch (id)
    {
    case wagHorizontal: return "wagHorizontal";
    case wagVertical: return "wagVertical";
    case wagLeftRightHorizontal: return "wagLeftRightHorizontal";
    case wagLeftRightVertical: return "wagLeftRightVertical";
    case wagUpDownLeft: return "wagUpDownLeft";
    case wagUpDownRight: return "wagUpDownRight";
    case stayAsForced: return "stayAsForced";
    case stayAsForcedPan: return "stayAsForcedPan";
    case stayAsForcedTilt: return "stayAsForcedTilt";
    case tailFollowsHead: return "tailFollowsHead";
    case tailParallelToGround: return "tailParallelToGround";
    case noTailWag: return "noTailWag";
    case wagHorizontalFast: return "wagHorizontalFast";
    case wagVerticalFast: return "wagVerticalFast";
    case twoPositionSwitchVertical: return "twoPositionSwitchVertical";
    case twoPositionSwitchHorizontal: return "twoPositionSwitchHorizontal";
    case threePositionSwitchVertical: return "threePositionSwitchVertical";
    case threePositionSwitchHorizontal: return "threePositionSwitchHorizontal";
    case fourPositionSwitchUpDownLeftRight: return "fourPositionSwitchUpDownLeftRight";
    case fourPositionSwitchCorners: return "fourPositionSwitchCorners";
    case fivePositionSwitch: return "fivePositionSwitch";
    case eightPositionSwitch: return "eightPositionSwitch";
    case tailLeftTop: return "tailLeftTop";
    case tailCenterTop: return "tailCenterTop";
    case tailRightTop: return "tailRightTop";
    case tailLeftCenter: return "tailLeftCenter";
    case tailCenterCenter: return "tailCenterCenter";
    case tailRightCenter: return "tailRightCenter";
    case tailLeftBottom: return "tailLeftBottom";
    case tailCenterBottom: return "tailCenterBottom";
    case tailRightBottom: return "tailRightBottom";
      
    default: return "Unknown tail request, please edit MotionRequest::getTailRequestName";
    }
  }
  
  /** motion type */
  MotionID motionType;
  
  /** walking type */
  WalkType walkType;
  
  /** walk parameters, in mm/s / rad/s */
  Pose2D walkParams;
  
  SpecialActionID specialActionType;
  
  /** tail request */
  TailRequest tailRequest;

  /** 
  only for the executedMotionRequest 
  slides from 0 to 1 during one step
  */
  double positionInWalkCycle;
  
  /** constructor, startup motion defined here */
#ifdef _WIN32
  MotionRequest():motionType(getup),walkType(normal),specialActionType(playDead),tailRequest(wagHorizontal),stabilize(false),updateRP(true),breathe(false),frameNumber(0) {}
#else
  MotionRequest():motionType(specialAction),walkType(normal),specialActionType(playDead),tailRequest(wagHorizontal),stabilize(false),breathe(false),updateRP(true),frameNumber(0) {}
#endif
  /** == operator */
  bool operator==(const MotionRequest& other) const 
  {
    switch (motionType)
    {
    case stand:
    case getup:
    default:
      return (
        motionType == other.motionType &&
        stabilize == other.stabilize);
    case walk:
      return (walkParams == other.walkParams &&
        stabilize == other.stabilize &&
        motionType == other.motionType &&
        walkType == other.walkType);
    case specialAction:
      return (
        stabilize == other.stabilize &&
        motionType == other.motionType &&
        specialActionType == other.specialActionType);
    }
  }
  
  /** = operator */
  void operator=(const MotionRequest& other);
  
  /** determines if the motion is being stabilized */
  bool stabilize;
  
  /** turns on or off the propagation (by odometry) of the "robot pose"
  in the "motion" process */
  bool updateRP;
  
  /** switches robot breathing motion on/off */
  bool breathe;
  
  /** returns name of current motion */
  const char* getMotionName() const {return getMotionName(motionType);};
  
  /** returns name of current special action if any */
  const char* getSpecialActionName() const
  {
    if (motionType == specialAction)
      return getSpecialActionName(specialActionType);
    else
      return "";
  };
  
  /** return name of current walking type if any */
  const char* getWalkTypeName() const
  {
    if (motionType == walk)
      return getWalkTypeName(walkType);
    else
      return "";
  }
  
  /**
  * Prints the motion request to a readable string. (E.g. "NormalWalk 100,0,0")
  * @param destination The string to fill
  */
  void printOut(char* destination) const;
  
  /**
  * Returns the special action id for a name
  */
  static SpecialActionID getSpecialActionIDFromName(const char* name);
  
  int padding; /**< Without this padding, the gcc-version crashes. */
};

/**
* Streaming operator that reads a MotionRequest from a stream.
* @param stream The stream from which is read.
* @param motionRequest The MotionRequest object.
* @return The stream.
*/ 
In& operator>>(In& stream,MotionRequest& motionRequest);

/**
* Streaming operator that writes a MotionRequest to a stream.
* @param stream The stream to write on.
* @param motionRequest The MotionRequest object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const MotionRequest& motionRequest);

#endif // __MOTIONREQUEST_H__

/*
 * Change log :
 * 
 * $Log: MotionRequest.h,v $
 * Revision 1.49  2004/06/11 20:34:56  altmeyer
 * added 4 new special actions for Lange Nacht der Wissenschaften Berlin
 *
 * Revision 1.48  2004/05/17 13:56:51  rossdeutscher
 * Added mofs for open challenge
 *
 * Revision 1.47  2004/05/10 10:29:17  juengel
 * Added positionInWalkCycle.
 *
 * Revision 1.46  2004/05/06 12:24:35  hamerla
 * add some mofs
 *
 * Revision 1.45  2004/04/22 14:38:48  dueffert
 * speeds adapted to new abilities
 *
 * Revision 1.44  2004/04/09 12:49:59  tim
 * added kicks fom BB2004
 *
 * Revision 1.43  2004/04/08 15:33:07  wachter
 * GT04 checkin of Microsoft-Hellounds
 *
 * Revision 1.42  2004/04/07 13:00:47  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.10  2004/04/04 01:18:48  risler
 * grab mof added
 *
 * Revision 1.9  2004/04/03 21:35:58  risler
 * added dddErs7hold mof
 * head kick from grab improved
 *
 * Revision 1.8  2004/04/03 21:21:05  dthomas
 * added new mof dddErs7bande
 *
 * Revision 1.7  2004/04/02 17:01:25  Schmitt
 * Added DDD SoccerKickLeft
 *
 * Revision 1.6  2004/04/01 14:32:09  Schmitt
 * Changes removed
 *
 * Revision 1.5  2004/04/01 13:59:09  Schmitt
 * Unified SpecialAction Names
 *
 * Revision 1.4  2004/03/31 18:28:26  dthomas
 * updated kick-engine with tester, catch and catch-and-turn
 *
 * Revision 1.3  2004/03/31 10:50:58  risler
 * walk type dash added
 * adjusted max speed
 *
 * Revision 1.2  2004/03/30 16:26:23  risler
 * ddd ers7 special actions renamed
 *
 * Revision 1.1.1.1  2004/03/29 08:28:45  Administrator
 * initial transfer from tamara
 *
 * Revision 1.41  2004/04/05 17:56:50  loetzsch
 * merged the local German Open CVS of the aibo team humboldt with the tamara CVS
 *
 * Revision 1.40  2004/04/01 12:15:37  dueffert
 * missing string added
 * added athErs7GOShortDance
 *
 * Revision 1.4  2004/04/03 02:32:36  aldi
 * added new athErs7 Cheers
 *
 * Revision 1.3  2004/04/01 21:59:10  aldi
 * new athErs7 cheering added
 *
 * Revision 1.2  2004/04/01 20:49:44  loetzsch
 * renamed athErs7UnswBash to athErs7ExactForwardKick
 *
 * Revision 1.1.1.1  2004/03/31 11:16:53  loetzsch
 * created ATH repository for german open 2004
 *
 * Revision 1.39  2004/03/29 17:50:29  loetzsch
 * added athers7unswbashfromhold
 *
 * Revision 1.38  2004/03/29 15:30:22  dueffert
 * new turn parameterset
 *
 * Revision 1.37  2004/03/29 15:14:42  loetzsch
 * added athers7GrabBall
 *
 * Revision 1.36  2004/03/28 14:09:51  jhoffman
 * - added error member variables to ball model
 * - minor improvement in handling of updateRP in motiontesterdialog
 *
 * Revision 1.35  2004/03/28 11:07:35  thomas
 * added more ddd-special-actions
 *
 * Revision 1.34  2004/03/19 21:18:00  altmeyer
 * SideHeadKick and SideHeadKick2 changed and renamed to SideHeadKick L and SideHeadKickR
 * a few ATH ERS7-kicks removed
 *
 * Revision 1.33  2004/03/19 18:58:23  jhoffman
 * "update robot pose in motion" is now turned on by default
 *
 * Revision 1.32  2004/03/19 14:58:12  jess4279
 * BB2004 Kicks  hinzugefgt
 *
 * Revision 1.31  2004/03/19 13:47:00  jess4279
 * BB2004HeadKick hinzugefgt
 *
 * Revision 1.30  2004/03/18 11:59:59  thomas
 * added: first ers7 special actions from ddd
 *
 * Revision 1.29  2004/03/17 21:08:39  cesarz
 * Added turnWithBall3, which points to MSH2004TurnWithBallWalkingParameters.
 *
 * Revision 1.28  2004/03/17 19:52:09  juengel
 * Added walkType turnWithBall2
 *
 * Revision 1.27  2004/03/17 15:38:51  thomas
 * added walkType debug for debugging WalkingEngineParameterSets
 * paramSets-messages send by debug-message are stored in the new debug-walkType only
 *
 * Revision 1.26  2004/03/16 14:00:21  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.2  2004/03/15 17:11:40  hoffmann
 * - added ATH2004HeadControl
 * - added ATH2004LEDControl
 * - headmotiontester shows "tilt2"
 * - motion process updates odometry while no new robotPose is received, added to motion request
 * - some ui adjustments
 * - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
 *
 * Revision 1.25  2004/03/16 13:49:25  altmeyer
 * added new ATHErs7 Special Actions
 *
 * Revision 1.24  2004/03/16 13:46:07  altmeyer
 * added
 *
 * Revision 1.23  2004/03/15 17:47:11  rossbacher
 * Fixed naming problem with MSH7 and added another .mof
 *
 * Revision 1.22  2004/03/12 15:34:37  rossbacher
 * Added new MSH7 Special Actions
 *
 * Revision 1.21  2004/03/11 11:23:04  schmitt
 * Added ERS-7 headKick
 *
 * Revision 1.20  2004/03/10 15:10:10  rossdeutscher
 * Adden some new MSH7 .mofs
 *
 * Revision 1.19  2004/03/10 11:16:17  rossbacher
 * Added MSH7 mofs
 *
 * Revision 1.18  2004/03/04 18:32:38  rempe
 * new goalie defend motions added
 *
 * Revision 1.17  2004/02/27 18:20:17  brueckne
 * mofs added
 *
 * Revision 1.16  2004/02/27 17:26:06  heinze
 * added 3 mofs
 *
 * Revision 1.15  2004/02/27 15:24:14  juengel
 * Added some wake-up motions.
 *
 * Revision 1.14  2004/02/26 10:54:02  jhoffman
 * - hopefully fixed ath-mof name problems
 *
 * Revision 1.13  2004/02/26 09:35:08  schumann
 * removed compiler errors caused by misspelling of ath kick
 *
 * Revision 1.12  2004/02/25 15:28:01  jhoffman
 * kick renamed
 *
 * Revision 1.11  2004/02/25 13:43:28  jhoffman
 * ATH kick renamed
 *
 * Revision 1.10  2004/02/06 20:24:39  roefer
 * Maximum rotation speed increased
 *
 * Revision 1.9  2004/01/19 14:53:46  dueffert
 * all frameNumbers (and not only some of them) are unsigned long now
 *
 * Revision 1.8  2004/01/08 13:52:35  loetzsch
 * added new special action ATH_kickForwardWithRightPaw
 *
 * Revision 1.7  2004/01/05 17:56:27  juengel
 * Added HeadControlModes stayAsForcedPan and stayAsForcedTilt.
 *
 * Revision 1.6  2004/01/05 11:37:35  juengel
 * Added new TailModes: wagUpDown* and wagLeftRight*.
 *
 * Revision 1.5  2003/12/21 15:43:12  kerdels
 * einrckung..
 *
 * Revision 1.4  2003/12/16 18:52:55  loetzsch
 * added a function for retrieving the id for as string.
 *
 * Revision 1.3  2003/11/14 19:02:26  goehring
 * frameNumber added
 *
 * Revision 1.2  2003/10/29 19:38:41  lohmann
 * New Headstand and new behavior for game-state-final (test-final.xml) added.
 *
 * Revision 1.1  2003/10/07 10:07:01  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.13  2003/09/25 11:23:52  juengel
 * Removed BlobCollection.
 *
 * Revision 1.12  2003/09/25 10:16:18  juengel
 * Removed some walk-types.
 *
 * Revision 1.11  2003/09/24 16:39:03  juengel
 * comments removed
 *
 * Revision 1.10  2003/07/07 14:11:02  thomas
 * remove old diveKick
 *
 * Revision 1.9  2003/07/07 08:56:02  brunn
 * lanceKicks
 *
 * Revision 1.8  2003/07/05 22:14:56  dueffert
 * kicks renamed and properly linked
 *
 * Revision 1.7  2003/07/05 15:25:20  dueffert
 * escapePadua added
 *
 * Revision 1.6  2003/07/05 12:09:25  dueffert
 * most important mof ever added
 *
 * Revision 1.5  2003/07/05 11:53:11  schumann
 * challenge kicks added and grabbing behavior prepared
 *
 * Revision 1.4  2003/07/03 15:09:46  thomas
 * fix: missing entries for new kicks
 *
 * Revision 1.3  2003/07/03 14:54:23  risler
 * new kicks
 *
 * Revision 1.2  2003/07/02 10:21:11  dueffert
 * missing kick added
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.42  2003/06/27 13:39:08  reiss
 * New Kicks added (von Andreas Rei, Carsten Schuhmann, Michael Wachter, Wenchuan Ding)
 *
 * Revision 1.41  2003/06/23 12:56:07  schumann
 * added special bicycle kick for challenge
 *
 * Revision 1.40  2003/06/19 20:19:28  risler
 * diveKickFromHold added
 *
 * Revision 1.39  2003/06/19 18:26:24  thomas
 * added: diveKick
 *
 * Revision 1.38  2003/06/18 18:08:50  jhoffman
 * forward leaping kick reordered
 *
 * Revision 1.37  2003/06/13 17:06:17  jhoffman
 * added new kick motion
 *
 * Revision 1.36  2003/05/06 16:50:51  juengel
 * Added tail-mode eightPositionSwitch.
 *
 * Revision 1.35  2003/05/06 07:57:30  dueffert
 * alphabetically reordered
 *
 * Revision 1.34  2003/04/15 15:52:12  risler
 * DDD GO 2003 code integrated
 *
 * Revision 1.25  2003/04/11 23:12:33  max
 * added howl mof
 *
 * Revision 1.33  2003/04/14 16:37:47  dueffert
 * several cheer motions added
 *
 * Revision 1.4  2003/04/10 19:12:34  dueffert
 * context sensitive reset in MotionTester
 *
 * Revision 1.3  2003/04/10 17:35:44  Daniel
 * cheer motions implemented
 *
 * Revision 1.2  2003/04/10 08:25:05  dueffert
 * fun stuff added
 *
 * Revision 1.32  2003/04/08 19:27:50  hebbel
 * added bicycleKickFromHold
 *
 * Revision 1.31  2003/04/08 16:07:39  dueffert
 * mofs added
 *
 * Revision 1.30  2003/04/05 15:08:46  loetzsch
 * removed chestKick3 again
 *
 * Revision 1.29  2003/04/04 17:20:05  juengel
 * Added chestKick3
 *
 * Revision 1.28  2003/04/04 16:06:08  cesarz
 * kicks added
 *
 * Revision 1.27  2003/04/03 14:38:46  hebbel
 * kickFromHold added
 *
 * Revision 1.26  2003/04/02 14:55:21  timrie
 * Added BBwinner
 *
 * Revision 1.25  2003/04/02 14:10:39  dueffert
 * walkType turnWithBall added, sorted alphabetically
 *
 * Revision 1.24  2003/04/02 13:54:36  loetzsch
 * added lookForLandmarksWithBallCaught
 *
 * Revision 1.23  2003/04/01 22:45:34  cesarz
 * added catchBallAndCheck special action
 *
 * Revision 1.22  2003/03/25 14:19:30  timrie
 * Added Specialaction BBloser
 *
 * Revision 1.21  2003/03/24 14:45:49  juengel
 * Added several tail modes
 *
 * Revision 1.20  2003/03/07 17:08:41  juengel
 * initialized breathe with false
 *
 * Revision 1.19  2003/03/06 11:55:50  dueffert
 * re-order warning removed
 *
 * Revision 1.18  2003/02/25 09:02:11  roefer
 * breathe initialized under _WIN32
 *
 * Revision 1.17  2003/02/05 17:03:38  dueffert
 * packs added, getUpsideDown renamed
 *
 * Revision 1.16  2003/02/03 16:40:41  risler
 * increased maximum walking speeds
 *
 * Revision 1.15  2003/02/01 17:06:42  risler
 * added rollRight, RollLeft
 *
 * Revision 1.14  2003/02/01 17:04:18  risler
 * added rollRight, RollLeft
 *
 * Revision 1.13  2003/01/27 19:27:08  juengel
 * added walkType upsideDown
 *
 * Revision 1.12  2003/01/21 23:33:32  juengel
 * Added getUpsideDown.mof
 *
 * Revision 1.11  2003/01/15 16:20:25  loetzsch
 * reanamed boundary kick to bicycle kick
 *
 * Revision 1.10  2002/12/01 18:52:28  roefer
 * No playDead in simulation
 *
 * Revision 1.9  2002/11/26 19:09:19  loetzsch
 * MotionRequest is initialized with play-dead
 *
 * Revision 1.8  2002/11/26 17:19:22  risler
 * added wakeUp specialAction for smooth startup
 *
 * Revision 1.7  2002/11/25 14:49:08  jhoffman
 * added "breathing" motion
 *
 * Revision 1.6  2002/11/20 18:02:29  risler
 * added PID values to GT2001MotionNetSpecialActions
 * added playDead motion
 *
 * Revision 1.5  2002/11/19 17:12:24  juengel
 * "switch" tail modes added
 *
 * Revision 1.4  2002/10/11 16:59:15  roefer
 * Fast tail wags added
 *
 * Revision 1.3  2002/09/22 18:40:54  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.2  2002/09/17 23:55:19  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.1  2002/09/10 15:26:40  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.26  2002/09/03 16:00:50  juengel
 * HeadControlMode follwTail, TailMode stayAsForced.
 *
 * Revision 1.25  2002/08/22 14:41:03  risler
 * added some doxygen comments
 *
 * Revision 1.24  2002/08/04 17:53:18  roefer
 * SimGT2002 connection to physical robots added
 *
 * Revision 1.23  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.22  2002/07/14 14:03:07  roefer
 * First working gcc-version
 *
 * Revision 1.21  2002/06/27 11:18:59  risler
 * added additional walk type for bar challenge
 *
 * Revision 1.20  2002/06/21 04:52:04  risler
 * added walktype walkWithBar
 *
 * Revision 1.19  2002/06/22 07:57:27  RoboCup
 * boundaryKickRight and -Left added.
 *
 * Revision 1.18  2002/06/19 11:30:25  jhoffman
 * added heel kicks
 *
 * Revision 1.17  2002/06/13 21:51:21  loetzsch
 * removed the comparisons for tail modes in the == operator
 *
 * Revision 1.16  2002/06/13 13:08:06  jhoffman
 * added some kicks
 *
 * Revision 1.15  2002/06/10 11:05:17  risler
 * added timestamp to motion request
 * motioncontrol executes swing when no request was received
 *
 * Revision 1.14  2002/06/08 14:33:24  kosen
 * openArm added
 *
 * Revision 1.13  2002/06/08 11:24:12  Cesarz
 * Added GoalieJumps
 *
 * Revision 1.12  2002/06/06 16:15:14  jhoffman
 * added unsw-bash-optimized (kick mof)
 *
 * Revision 1.11  2002/06/06 13:20:44  risler
 * removed bug with MotionRequest == operator
 *
 * Revision 1.10  2002/06/06 12:51:44  risler
 * removed bug with MotionRequest == operator / executed motion request
 *
 * Revision 1.9  2002/06/06 10:24:53  AndySHB
 * Neue mof hinzugefuegt: grabBall.
 *
 * Revision 1.8  2002/06/04 16:44:55  risler
 * tailRequest added
 *
 * Revision 1.7  2002/06/03 14:38:24  dueffert
 * Jan's new paw kick
 *
 * Revision 1.6  2002/06/01 10:17:14  juengel
 * Added catchBall2.
 *
 * Revision 1.5  2002/05/27 14:40:01  cesarz
 * boundaryKick added
 *
 * Revision 1.4  2002/05/24 15:36:53  kosen
 * Ball Challenge mofs added
 *
 * Revision 1.3  2002/05/16 15:05:49  risler
 * added walk type walkWithBall
 *
 * Revision 1.2  2002/05/13 07:22:34  cesarz
 * mantisKick, armKickleft and armKickRight added
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.41  2002/05/06 17:15:20  risler
 * increased max walk speeds
 *
 * Revision 1.40  2002/05/05 15:13:36  risler
 * changed stand implementation, removed not working special actions
 *
 * Revision 1.39  2002/05/04 18:18:06  risler
 * made printOut const
 *
 * Revision 1.38  2002/05/04 12:43:38  loetzsch
 * The currently executed MotionRequest is now sent from the MotionControl
 * to the BehaviorControl via the OdometryData structure
 *
 * Revision 1.37  2002/05/03 16:34:36  loetzsch
 * added == operator
 *
 * Revision 1.36  2002/05/03 11:49:56  juengel
 * Removed secretSpecialActionID.
 *
 * Revision 1.35  2002/04/29 13:46:42  risler
 * no message
 *
 * Revision 1.34  2002/04/24 07:49:14  risler
 * changed normalWalkMaxSpeed to walkMaxSpeed
 *
 * Revision 1.33  2002/04/23 15:08:45  risler
 * changed MotionRequest: walk instead of normalWalk,... and walkType added
 *
 * Revision 1.32  2002/04/18 10:25:31  roefer
 * Bremen GO
 *
 * Revision 1.31  2002/04/17 17:04:40  risler
 * Darmstadt GO
 *
 * Revision 1.30  2002/04/16 16:23:04  dueffert
 * no message
 *
 * Revision 1.8  2002/04/08 18:30:23  dueffert
 * no message
 *
 * Revision 1.29  2002/04/15 13:39:24  rentmeister
 * DoWalkingEngine (Dortmunder WalkingEngine)
 *
 * Revision 1.28  2002/04/08 16:27:26  mierswa
 * New input for FuzzyBeahviourControl.
 *
 * Revision 1.27  2002/04/04 15:07:59  rentmeister
 * Walk with Ball hinzugefgt
 *
 * Revision 1.26  2002/04/04 10:21:33  risler
 * changed normalWalkMaxRotationSpeed to angle
 *
 * Revision 1.25  2002/04/03 16:48:53  jhoffman
 * added "stabilizeRobot" to motionControl (which is turned off as a default)
 *
 * Revision 1.24  2002/04/03 15:20:24  risler
 * changed normalWalkMaxRotationSpeed to angle
 *
 * Revision 1.23  2002/04/02 16:06:27  loetzsch
 * added printOut function
 */
