/** 
* @file JointDataSequence.h
*
* Definition of JointDataSequence class.
* The JointDataSequence is a sequence of joint data sets 
* with timing and interpolation information.
* This is sent from PC to robot and executed by DebugMotionControl.
*
* @author Max Risler
*/
#ifndef __JOINTDATASEQUENCE_H__
#define __JOINTDATASEQUENCE_H__

#include "JointData.h"

#include "Tools/Streams/InOut.h"

/** Maximum number of entries per sequence. */
static const int jointDataSequenceMaxNumOfEntries = 400;

/**
* The JointDataSequence is a sequence of joint data sets 
* with timing and interpolation information.
*
* This is sent from PC to robot and executed by DebugMotionControl.
*/
class JointDataSequence
{
public:
  /** Constructor */
  JointDataSequence();
  
 /** Assignment operator */
  JointDataSequence& operator=(const JointDataSequence& other);

  /** Number of Entries */
  int numOfEntries;

  /** If true sequence is executed repeatedly */
  bool loop;

  /** adds an entry at end of the sequence */
  void addEntry(const JointData& jointData, bool interp, int rep);


  /** The joint data entries */
  JointData entry[jointDataSequenceMaxNumOfEntries];

  /** Number of times an entry gets repeated */
  int repeats[jointDataSequenceMaxNumOfEntries];

  /** If true linear interpolation will be done to get to this entry from last */ 
  bool interpolate[jointDataSequenceMaxNumOfEntries];
};

/**
 * Streaming operator that reads a JointDataSequence from a stream.
 * @param stream The stream from which is read.
 * @param jointDataSequence The JointDataSequence object.
 * @return The stream.
 */ 
In& operator>>(In& stream,JointDataSequence& jointDataSequence);
 
/**
 * Streaming operator that writes a JointDataSequence to a stream.
 * @param stream The stream to write on.
 * @param jointDataSequence The JointDataSequence object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const JointDataSequence& jointDataSequence);

#endif //__JOINTDATASEQUENCE_H__

/*
 * Change log :
 * 
 * $Log: JointDataSequence.h,v $
 * Revision 1.1  2003/10/07 10:07:01  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.3  2003/02/05 12:07:30  dueffert
 * more mof lines allowed
 *
 * Revision 1.2  2002/11/28 16:50:43  risler
 * added addEntry
 *
 * Revision 1.1  2002/09/10 15:26:40  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/08/22 14:41:03  risler
 * added some doxygen comments
 *
 * Revision 1.2  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.2  2002/02/21 18:02:51  risler
 * added assignment operator
 *
 * Revision 1.1  2002/02/08 22:31:45  risler
 * added JointDataSequence, finished DebugMotionControl
 *
 */
