/** 
* @file JointData.h
*
* Definition of JointData struct.
* The jointdata struct is a non NDA violating description of join vector data
*
* @author Max Risler
*/
#ifndef __JOINTDATA_H__
#define __JOINTDATA_H__

/** Invalid motor data value. 
 * If this value was returned by motion module for the head, 
 * other head movement is possible */
static const long jointDataInvalidValue = 10000000;

#include "Tools/Streams/InOut.h"

/**
* One joint data set. Containing values for all used joints.
* @author Max Risler
*/
struct JointData
{
public:
  /**
   * enum describing indexes of joints 
   */
  enum JointID {
    headTilt,
    headTilt1 = headTilt,
    headPan,
    headRoll, 
    headTilt2 = headRoll,
    mouth,
    earL, earR,
    legFR1, legFR2, legFR3,
    legFL1, legFL2, legFL3,
    legHR1, legHR2, legHR3,
    legHL1, legHL2, legHL3,
    tailPan, tailTilt,
      numOfJoint
  };

  /** The joint data values */
  long data[numOfJoint];
  
  /** Default constructor */
  JointData(long headTiltValue = jointDataInvalidValue,
    long headPanValue = jointDataInvalidValue,
    long headRollValue = jointDataInvalidValue,
    long mouthValue = jointDataInvalidValue,
    long earLValue = jointDataInvalidValue, 
    long earRValue = jointDataInvalidValue,
    long legFR1Value = jointDataInvalidValue, 
    long legFR2Value = jointDataInvalidValue, 
    long legFR3Value = jointDataInvalidValue,
    long legFL1Value = jointDataInvalidValue, 
    long legFL2Value = jointDataInvalidValue, 
    long legFL3Value = jointDataInvalidValue,
    long legHR1Value = jointDataInvalidValue, 
    long legHR2Value = jointDataInvalidValue, 
    long legHR3Value = jointDataInvalidValue,
    long legHL1Value = jointDataInvalidValue, 
    long legHL2Value = jointDataInvalidValue, 
    long legHL3Value = jointDataInvalidValue,
    long tailPanValue = jointDataInvalidValue, 
    long tailTiltValue = jointDataInvalidValue
  );

  /** Assignment operator */
  JointData& operator=(const JointData& other);

  /** Copy constructor */
  JointData(const JointData& other);

  /** returns the name of a joint */
  static const char* getJointName(JointID joint)
  {
    switch (joint)
    {
    case headTilt: return "headTilt(1)";
    case headPan: return "headPan"; 
    case headRoll: return "headRoll/Tilt2"; 
    case mouth: return "mouth"; 
    case earL: return "earL"; 
    case earR: return "earR"; 
    case legFR1: return "legFR1"; 
    case legFR2: return "legFR2"; 
    case legFR3: return "legFR3"; 
    case legFL1: return "legFL1"; 
    case legFL2: return "legFL2"; 
    case legFL3: return "legFL3"; 
    case legHR1: return "legHR1"; 
    case legHR2: return "legHR2"; 
    case legHR3: return "legHR3"; 
    case legHL1: return "legHL1"; 
    case legHL2: return "legHL2"; 
    case legHL3: return "legHL3"; 
    case tailPan: return "tailPan"; 
    case tailTilt: return "tailTilt"; 
    default: return "";
    }
  }

  static JointID getIDFromName(const char* jointName);
};


/**
 * Streaming operator that reads a JointData from a stream.
 * @param stream The stream from which is read.
 * @param jointData The JointData object.
 * @return The stream.
 */ 
In& operator>>(In& stream,JointData& jointData);
 
/**
 * Streaming operator that writes a JointData to a stream.
 * @param stream The stream to write on.
 * @param jointData The JointData object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const JointData& jointData);

#endif //__JOINTDATA_H__

/*
 * Change log :
 * 
 * $Log: JointData.h,v $
 * Revision 1.4  2003/12/31 16:58:44  roefer
 * Joints and LEDs for ERS-7
 *
 * Revision 1.3  2003/12/31 14:29:20  roefer
 * Joints and LEDs for ERS-7
 *
 * Revision 1.2  2003/12/16 18:52:55  loetzsch
 * added a function for retrieving the id for as string.
 *
 * Revision 1.1  2003/10/07 10:07:01  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/11 15:55:13  dueffert
 * doxygen bug fixed
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.2  2002/11/19 17:14:14  risler
 * coding conventions: renamed JointData::joint to JointID, GetName to getName
 *
 * Revision 1.1  2002/09/10 15:26:40  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.14  2002/02/21 18:02:13  risler
 * now thats a constructor
 *
 * Revision 1.13  2002/01/30 11:10:37  risler
 * still no need to include GTMath.h here
 *
 * Revision 1.12  2002/01/30 01:08:56  dueffert
 * fixed endless-compile-bug caused by up/downcase difference MathLib vs. Mathlib, makefile should be caseUNsensitive (in such cases) now
 *
 * Revision 1.11  2002/01/29 17:05:39  risler
 * for some reason its a bad idea to include GTMath.h here
 *
 * Revision 1.10  2002/01/28 16:24:23  juengel
 * Tippfehler in GetJointName beseitigt
 *
 * Revision 1.9  2002/01/28 14:57:27  jhoffman
 * no message
 *
 * Revision 1.8  2002/01/26 03:11:27  loetzsch
 * JointViewer continued
 *
 * Revision 1.7  2002/01/25 16:52:40  loetzsch
 * function for enum names added
 *
 * Revision 1.6  2001/12/10 17:47:05  risler
 * change log added
 *
 */
