/**
* @file BallModel.cpp
*
* Implementation of class BallModel.
*
* @author <A href=mailto:xiang@informatik.uni-bremen.de>Hong Xiang</A>
* @author <A href=mailto:rlx@informatik.uni-bremen.de>Lang Lang</A>
* @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
* @author <A href=mailto:jan@aiboteamhumboldt.com>Jan Hoffmann</A>
*/

#include "BallModel.h"
#include "Platform/SystemCall.h"

SeenBallPosition::SeenBallPosition()
: timeWhenLastSeen(0),
timeWhenFirstSeenConsecutively(SystemCall::getCurrentSystemTime()),
timeUntilSeenConsecutively(SystemCall::getCurrentSystemTime()),
angleError(0), distanceError(0), ballInFrontOfOpponentGoal(false)
{
}

long SeenBallPosition::getConsecutivelySeenTime() const
{
  long notSeenTime = (long)SystemCall::getTimeSince(timeUntilSeenConsecutively);
  if (notSeenTime<200)
  {
    return ((long)timeUntilSeenConsecutively - (long)timeWhenFirstSeenConsecutively);
  }
  else
  {
    return -notSeenTime;
  }
}

PropagatedBallPosition::PropagatedBallPosition()
{
  positionWhenLastObserved.x = 0;
  positionWhenLastObserved.y = 0;
  observedSpeed.x = 0;
  observedSpeed.y = 0;
  timeOfObservation = 0;
  x = 0;
  y = 0;
  speed.x = 0;
  speed.y = 0;
}


Vector2<double> PropagatedBallPosition::getPropagatedPosition(long time) const
{
  //double timeDiff = (time - timeOfObservation)/1000.0;
  return positionWhenLastObserved;// + observedSpeed * (1-exp(-timeDiff/2));
}

Vector2<double> PropagatedBallPosition::getPropagatedSpeed(long time) const
{
  return Vector2<double>(0,0);
  
  //	double timeDiff = (time - timeOfObservation)/1000.0;
  //	return observedSpeed * exp(-timeDiff/2);
}

CommunicatedBallPosition::CommunicatedBallPosition()
: timeWhenLastObserved(0)
{
}

const Vector2<double>& BallModel::getKnownPosition(unsigned long timeAfterWhichCommunicatedBallsAreAccepted) const
{
  if (SystemCall::getTimeSince(seen.timeWhenLastSeen) < timeAfterWhichCommunicatedBallsAreAccepted)
  {
    // use the seen position
    return seen;
  }
  else
  {
    if (SystemCall::getTimeSince(seen.timeWhenLastSeen)
      < SystemCall::getTimeSince(communicated.timeWhenLastObserved))
    {
      return seen;
    }
    else
    {
      return communicated;
    }
  }
}

unsigned long BallModel::getTimeSinceLastKnown(unsigned long timeAfterWhichCommunicatedBallsAreAccepted) const
{
  if (SystemCall::getTimeSince(seen.timeWhenLastSeen) < timeAfterWhichCommunicatedBallsAreAccepted)
  {
    // use the seen position
    return SystemCall::getTimeSince(seen.timeWhenLastSeen);
  }
  else
  {
    return min(SystemCall::getTimeSince(seen.timeWhenLastSeen),
      SystemCall::getTimeSince(communicated.timeWhenLastObserved));
  }
}


BallModel::BallModel()
: validity(1), motionValidity(1)
{
  
}

In& operator>>(In& stream,BallModel& ballPosition)
{
  stream >> ballPosition.frameNumber;
  stream.read(&ballPosition,sizeof(BallModel));
  return stream;
}

Out& operator<<(Out& stream, const BallModel& ballPosition)
{
  stream << ballPosition.frameNumber;
  stream.write(&ballPosition,sizeof(BallModel));
  return stream;
}

/*
* Change log :
*
* $Log: BallModel.cpp,v $
* Revision 1.5  2004/05/18 21:14:33  goehring
* vector2 Propagated speed added
*
* Revision 1.4  2004/04/05 17:56:48  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.2  2004/04/02 10:02:10  jumped
* added ballIFrontOfOpponentGoal member
*
* Revision 1.1.1.1  2004/03/31 11:16:53  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.3  2004/03/28 14:09:51  jhoffman
* - added error member variables to ball model
* - minor improvement in handling of updateRP in motiontesterdialog
*
* Revision 1.2  2004/02/29 13:37:15  dueffert
* doxygen bugs fixed and beautified
*
* Revision 1.1  2004/02/03 12:56:41  spranger
* renamed BallPosition class to BallModel (including the files)
*
* Revision 1.6  2004/02/02 19:10:30  kerdels
* adjusted some options for the use with reflexes, created a simple version of turn-for-ball and modified some option ratings...
*
* Revision 1.5  2004/01/21 14:28:06  loetzsch
* some initializations
*
* Revision 1.4  2003/11/15 17:58:10  juengel
* fixed bug in streaming operator
*
* Revision 1.3  2003/11/14 19:02:25  goehring
* frameNumber added
*
* Revision 1.2  2003/10/21 13:11:15  goehring
* review
*
* Revision 1.1  2003/10/07 10:07:01  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/07/08 03:27:53  wachter
* - added challengeBallBetweenFeetd
* - removed propagetedBallDistance imput symbol
* - fixed cvs ;)
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.16  2003/06/22 15:25:02  dueffert
* getConsecutivelySeenTime improved
*
* Revision 1.15  2003/06/17 20:01:25  dueffert
* getConsecutivelySeenTime() added
*
* Revision 1.14  2003/06/17 19:26:46  thomas
* modified: auskommentiert von PropagatedBallPosition::getPropagatedPosition
*
* Revision 1.13  2003/06/17 18:26:54  thomas
* modified: auskommentiert von PropagatedBallPosition::getPropagatedSpeed und range abgefangen in DefaultObstaclesLocator::addObstaclePoint
*
* Revision 1.12  2003/06/17 11:28:36  jhoffman
* bug fixed (variable not initialized)
*
* Revision 1.11  2003/06/15 16:43:56  jhoffman
* propagated position is calculated by a function rather then in iteratively in cognition
*
* Revision 1.10  2003/06/15 14:26:44  jhoffman
* + moved "relative2FieldCoord" to Geometry
* + added member function to ballposition to calculate the propagated position and speed for a given time
* + propagated speed and time calculation using exponential decay instead of using an iterative calculation
* + in motion you can now use propageted ball pos at 125 Hz rather then the framerate determined by cognition
*
* Revision 1.9  2003/05/02 14:06:36  wachter
* Moved distanceInMayDirection and distanceInMinDirection from BallPosition to CommunicatedBallPosition
*
* Revision 1.8  2003/05/01 18:10:03  loetzsch
* renamed CommunicatedBallPosition::timeWhenLastReceived to CommunicatedBallPosition::timeWhenLastObserved
*
* Revision 1.7  2003/05/01 17:09:05  loetzsch
* Redesign of ball modeling:
* - Modularized class BallPosition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*
* Revision 1.6  2003/04/14 16:00:42  loetzsch
* ATH after GermanOpen CVS merge:
* added timeUntilSeenConsecutively
*
* Revision 1.2  2003/04/11 21:47:15  Jan Hoffmann
* timeWhenFirstSeenConsecutively and timeUntilSeenConsecutively are initialized with CurrentSystemTime.
*
* Revision 1.1.1.1  2003/04/09 14:22:16  loetzsch
* started Aibo Team Humboldt's GermanOpen CVS
*
* Revision 1.5  2003/04/08 08:50:03  schmidt
* Initializing validity with 0
*
* Revision 1.4  2003/02/14 14:34:02  wachter
* Added SensorFusionBallLocator
*
* Revision 1.3  2003/01/30 13:16:48  loetzsch
* Redesign of class BallPosition
*
* Revision 1.2  2002/09/22 18:40:50  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/10 15:26:39  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.12  2002/04/25 20:29:57  roefer
* New BallPercept and BallPosition, GTMath errors in SimGT2002 fixed
*
* Revision 1.11  2002/04/06 02:28:45  loetzsch
* added time when ball was seen last
*
* Revision 1.10  2002/02/11 11:19:57  roefer
* no message
*
* Revision 1.9  2002/02/11 11:13:06  roefer
* BallPerceptor and BallLocator inserted
*
* Revision 1.8  2002/02/05 03:30:52  loetzsch
* replaced direct member access by
* inline const VALUE& get...() const   and
* inline void set...(const Value&) methods.
*
* Revision 1.7  2002/01/27 22:14:54  roefer
* BallPosition is now a pair of two vectors
*
* Revision 1.6  2002/01/25 15:40:13  roefer
* The Oracle
*
* Revision 1.5  2002/01/22 00:06:55  loetzsch
* In den Get...() Funktionen wurden die Parameter nicht als Referenz bergeben,
* gendert
*
* Revision 1.4  2002/01/11 23:50:24  xiang
* BallPosition von Hong & Lang
*
* Revision 1.3  2001/12/10 17:47:05  risler
* change log added
*
*/
