/**
* @file Motion.h
* Declaration of Motion
* @author <A href=mailto:brunn@sim.informatik.tu-darmstadt.de>Ronnie Brunn</A>, <A href=mailto:kuntze@sim.informatik.tu-darmstadt.de>Nicolai Kuntze</A>, <A href=mailto:kunz@sim.informatik.tu-darmstadt.de>Michael Kunz</A>
* @author Max Risler
*/

#ifndef __Motion_h_
#define __Motion_h_

#include "Tools/Process.h"

#include "Modules/SensorDataProcessor/SensorDataProcessorSelector.h"
#include "Modules/HeadControl/HeadControlSelector.h"
#include "Modules/LEDControl/LEDControlSelector.h"
#include "Modules/SoundControl/SoundControlSelector.h"
#include "Modules/MotionControl/MotionControlSelector.h"
#include "Tools/RingBuffer.h"

#include "PackageMotionCognition.h"
#include "PackageCognitionMotion.h"

/**
* @class Motion 
*
* A process for actorics. 
*/
class Motion : public Process
{
private:
  DEBUGGING;
  RECEIVER(SensorDataBuffer);
  RECEIVER(PackageCognitionMotion);

  SENDER_MOTORCOMMANDS;
  SENDER_SOUNDDATA;
  SENDER(PackageMotionCognition);
  SENDER(OdometryData);
public:
  
  Motion();
  
  ~Motion();
  
  /** is called from the framework once in every frame */
  virtual int main();
  
  /** 
  * The function handles incoming debug messages.
  * @param message the message to handle
  * @return Has the message been handled?
  */
  virtual bool handleMessage(InMessage& message);
  
protected:
  //!@name modules
  //!@{
  
  /** a reference to the sensor data processor */
  SensorDataProcessorSelector* pSensorDataProcessor;

  /** a pointer to the HeadControl module */
  HeadControlSelector *pHeadControl;
  
  /** a pointer to the LEDControl module */
  LEDControlSelector *pLEDControl;
  
  /** a pointer to the MotionControl module */
  MotionControlSelector *pMotionControl;

  /** a pointer to the SoundControl module */
  SoundControlSelector *pSoundControl;
  
  //!@}
  
  unsigned long frameNumber, frameNumberImage;

  //!@name representations
  //!@{
  
  /** the head motion request */
  HeadMotionRequest headMotionRequest;

  /** The position and rotation of the camera relative to the robot */
  CameraMatrix cameraMatrix;

  /** the motion request that was multiplied with the game speed factor */
  MotionRequest motionRequest;

  /** the executed motion request that becomes devided through the game speed factor */
  MotionRequest executedMotionRequest;

  /** The psd percept calculated by the sensor data processor */
  PSDPercept psdPercept;

  /** The body percept calculated by the sensor data processor */
  BodyPercept bodyPercept;

  OdometryData lastOdometry;
  unsigned long lastFrameFromCognition;
  RobotPose lastRobotPoseUpdateFrame;
  RingBuffer<OdometryData,40> odometryHistory;
  RingBuffer<RobotVertices,4> robotVerticesBuffer;
  //!@}

  /** 
  * Can be used to slow down the game for debugging purposes (slow motion). 
  * Is multiplied to the incoming MotionRequest and to the outgoing executedMotionRequest.
  * 1: Normal speed -- 0.1: almost dont move
  */
  double gameSpeed;

  /** Specifies if the head is blocked by a special action or walk.*/
  bool headIsBlockedBySpecialActionOrWalk;
};

#endif // __Motion_h_

/*
* Change log :
* 
* $Log: Motion.h,v $
* Revision 1.8  2004/05/18 11:38:16  loetzsch
* the Motion process now also executes a SensorDataProcessor for the calculation of the cameraMatrix
*
* Revision 1.7  2004/03/20 17:20:01  juengel
* Added selector for LEDControl
*
* Revision 1.6  2004/03/20 09:55:27  roefer
* Preparation for improved odometry
*
* Revision 1.5  2004/03/16 14:00:21  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.2  2004/03/15 17:11:40  hoffmann
* - added ATH2004HeadControl
* - added ATH2004LEDControl
* - headmotiontester shows "tilt2"
* - motion process updates odometry while no new robotPose is received, added to motion request
* - some ui adjustments
* - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
*
* Revision 1.4  2004/03/04 10:05:27  jhoffman
* - motion process now uses odometry to propagate the robot pose while no new robot pose is being sent (this makes headcontrol a little better)
* - removed headcontrol member variable "propagatedPose" from headcontrol and cognition->motion-sender
*
* Revision 1.3  2004/01/19 14:56:22  dueffert
* all frameNumbers (and not only some of them) are unsigned long now
*
* Revision 1.2  2003/11/14 19:02:26  goehring
* frameNumber added
*
* Revision 1.1  2003/10/07 10:07:01  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/08/09 14:53:10  dueffert
* files and docu beautified
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.13  2003/05/27 16:16:10  juengel
* Added headIsBlockedBySpecialActionOrWalk.
*
* Revision 1.12  2003/05/13 11:42:05  goehring
* CollisionDetector removed
*
* Revision 1.11  2003/05/02 18:26:17  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.10  2003/04/25 19:49:44  goehring
* Added new module CollisionDetector
*
* Revision 1.9  2003/03/20 20:34:21  loetzsch
* Game Toolbar now can adjust the game speed
*
* Revision 1.8  2002/11/19 17:38:31  dueffert
* doxygen bugs corrected
*
* Revision 1.7  2002/09/18 19:52:36  loetzsch
* the head state is now sent from Motion to Cognition using the package.
*
* Revision 1.6  2002/09/17 23:55:22  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.5  2002/09/12 14:20:05  juengel
* Created a package for all data sent from Cognition to Motion.
*
* Revision 1.4  2002/09/11 20:06:19  loetzsch
* continued experiments with modules/solutions
*
* Revision 1.3  2002/09/11 13:43:19  loetzsch
* Created a package for all data that are sent from Motion to Cognition.
* Removed all previous Senders
*
* Revision 1.2  2002/09/11 00:06:59  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:41:25  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
* - Removed all process layouts
* - Added process layout CMD
*
* Revision 1.7  2002/07/23 13:40:52  loetzsch
* - new streaming classes
* - removed many #include statements
* - new design of debugging architecture
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - general clean up
*
* Revision 1.6  2002/07/13 10:54:58  roefer
* New command and sound sender
*
* Revision 1.5  2002/05/27 15:39:13  fischer
* Added SoundState (Sender and Receiver)
*
* Revision 1.4  2002/05/25 22:52:19  roefer
* WLan, first working approach
*
* Revision 1.3  2002/05/16 22:36:12  roefer
* Team communication and GTMath bugs fixed
*
* Revision 1.2  2002/05/15 07:29:56  hebbel
* Uses Soundprotocol
*
* Revision 1.1.1.1  2002/05/10 12:40:19  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.15  2002/05/06 16:03:48  hebbel
* Added buffered Sender for SoundDataOut
*
* Revision 1.14  2002/05/04 17:47:44  loetzsch
* LEDControl now has access to an instance of WLanStatus
* (which shall be displayed then with the green leds. follows next)
*
* Revision 1.13  2002/04/29 17:17:28  hebbel
* Put SoundPlay to Motion Process
*
* Revision 1.12  2002/02/05 20:02:16  risler
* handleDebugMessage now returns bool, added debug message handling to ImageProcessor
*
* Revision 1.11  2002/01/30 17:29:56  loetzsch
* handleDebugMessage um Parameter timestamp erweitert
*
* Revision 1.10  2001/12/21 14:09:39  roefer
* Added several destructors
*
* Revision 1.9  2001/12/15 20:32:08  roefer
* Senders and receivers are now part of the processes
*
* Revision 1.8  2001/12/14 12:34:09  dueffert
* no message
*
* Revision 1.7  2001/12/13 18:49:25  risler
* added odometry updates in GT2001MotionNetSpecialActions
*
* Revision 1.6  2001/12/12 18:08:56  loetzsch
* Streaming- Operatoren fr Bilder eingebaut, DebugKeyTable nicht- statisch gemacht, Debuggin Mechanismen weitergemacht, Bilder aus Logfiles in RobotControl anzeigen, Logfiles in HU1/Debug auf den Stick schreiben
*
* Revision 1.5  2001/12/11 18:14:45  kosen
* SensorData added.
*
* problems with Receiver (multiple definition error) solved.
*
* Revision 1.4  2001/12/11 16:21:55  kosen
* SensorData added
*
* Revision 1.3  2001/12/10 17:47:08  risler
* change log added
*
*/
