/** 
* @file Processes/CMD/Cognition.cpp
* Implementation of Cognition
*
* @author <a href="mailto:dueffert@informatik.hu-berlin.de">Uwe Dffert</a>
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
* @author <a href="mailto:martin@martin-loetzsch.de">Martin Ltzsch</a>
* @author <a href="mailto:risler@sim.informatik.tu-darmstadt.de">Max Risler</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#include "Cognition.h"
#include "Representations/Perception/JPEGImage.h"
#include "Representations/Perception/LowResImage.h"
#include "Tools/Debugging/Stopwatch.h"
#include "Tools/RobotConfiguration.h"
#include "Platform/GTAssert.h"
#include "Tools/Location.h"


Cognition::Cognition() : 
INIT_DEBUGGING,
INIT_RECEIVER_SENSORDATA(true),
INIT_RECEIVER_IMAGE(false),
INIT_RECEIVER(GameControlData,false),
INIT_RECEIVER(PackageMotionCognition,false),
INIT_RECEIVER(OdometryData,false),
INIT_NET_RECEIVER(TeamMessage1,false),
INIT_NET_RECEIVER(TeamMessage2,false),
INIT_NET_RECEIVER(TeamMessage3,false),
INIT_NET_SENDER(TeamMessage1,false),
INIT_NET_SENDER(TeamMessage2,false),
INIT_NET_SENDER(TeamMessage3,false),
#ifdef FIVEDOGS
   INIT_NET_RECEIVER(TeamMessage4,false),
   INIT_NET_SENDER(TeamMessage4,false),
#endif
INIT_SENDER(PackageCognitionMotion,false),
processImage(false),
processSensorData(true),
processPercepts(false),
timeOfImageProcessing(0),
colorTable((ColorTable&) *colorTableBuffer),
specialPercept(thePackageCognitionMotionSender.specialPercept)
{
  //  PRINT("Cognition::Cognition");
  debugOut.setSize(200000);
  debugIn.setSize(400000);
  
#ifdef NO_COLOR_TABLE_SELECTOR
  InBinaryFile stream(getLocation().getFilename("coltable.c64"));
  if (stream.exists())
  {
    ColorTable64 colorTable64;
    stream >> colorTable64;
    memcpy(colorTableBuffer,&colorTable64,sizeof(ColorTable64));
  }
  else{
    InBinaryFile stream(getLocation().getFilename("coltable.c32"));
    if (stream.exists())
    {
      ColorTable32K colorTable32K;
      stream >> colorTable32K;
      memcpy(colorTableBuffer,&colorTable32K,sizeof(ColorTable32K));
    }
    else
    {
      InBinaryFile stream(getLocation().getFilename("coltable.tsl"));
      if (stream.exists())
      {
        ColorTableTSL colorTableTSL;
        stream >> colorTableTSL;
        colorTableTSL.calculateLUT();
        memcpy(colorTableBuffer,&colorTableTSL,sizeof(ColorTableTSL));
      }
      else
      {
        OUTPUT(idText,text,"Cognition : couldn't load color table");
      }
    }
  }
  
#else  
  
  // the new colortable selector module is created here
  ColorTableModInterfaces colorTableModInterfaces(colorTable);
  pColorTableMod = new ColorTableModSelector(moduleHandler, colorTableModInterfaces);
  
#endif
  
  SensorDataProcessorInterfaces sensorDataProcessorInterfaces(    
    thePackageMotionCognitionReceiver.headState,
    theSensorDataBufferReceiver,theImageReceiver.frameNumber,
    bodyPercept, cameraMatrix, psdPercept);
  pSensorDataProcessor = new SensorDataProcessorSelector(moduleHandler, sensorDataProcessorInterfaces);
  
  //  PRINT("Cognition - SensorDataProcessor created");
  
  ImageProcessorInterfaces imageProcessorInterfaces(
    theImageReceiver,cameraMatrix,colorTable,
    thePackageCognitionMotionSender.robotPose,thePackageCognitionMotionSender.ballModel,
    thePackageCognitionMotionSender.playerPoseCollection,thePackageCognitionMotionSender.robotState,
    calibrationRequest,
    landmarksPercept,ballPercept, linesPercept, playersPercept, obstaclesPercept, specialPercept);
  pImageProcessor = new ImageProcessorSelector(moduleHandler, imageProcessorInterfaces);
  
  //  PRINT("Cognition - ImageProcessor created");
  
  
  RobotStateDetectorInterfaces robotStateDetectorInterfaces(
    bodyPercept, collisionPercept, thePackageCognitionMotionSender.robotState);
  pRobotStateDetector = new RobotStateDetectorSelector(moduleHandler, robotStateDetectorInterfaces);
  
  //  PRINT("Cognition - RobotStateDetector created");
  
  BallLocatorInterfaces ballLocatorInterfaces(
    theOdometryDataReceiver,
    cameraMatrix, ballPercept, landmarksPercept, thePackageCognitionMotionSender.robotPose, 
    calibrationRequest, theSensorDataBufferReceiver, 
    thePackageCognitionMotionSender.ballModel,timeOfImageProcessing);
  pBallLocator = new BallLocatorSelector(moduleHandler, ballLocatorInterfaces);
  
  //  PRINT("Cognition - BallLocator created");
  
  TeamBallLocatorInterfaces teamBallLocatorInterfaces(
    thePackageCognitionMotionSender.ballModel.seen,
    theOdometryDataReceiver,thePackageCognitionMotionSender.robotPose, 
    teamMessageCollection,thePackageCognitionMotionSender.ballModel.communicated
    );
  pTeamBallLocator = new TeamBallLocatorSelector(moduleHandler,teamBallLocatorInterfaces);
  
  //  PRINT("Cognition - TeamBallLocator created");
  
  PlayersLocatorInterfaces playersLocatorInterfaces(
    playersPercept, thePackageCognitionMotionSender.robotPose, 
    teamMessageCollection, theOdometryDataReceiver,
    thePackageCognitionMotionSender.playerPoseCollection);
  pPlayersLocator = new PlayersLocatorSelector(moduleHandler, playersLocatorInterfaces);
  
  //  PRINT("Cognition - PlayersLocator created");
  
  ObstaclesLocatorInterfaces obstaclesLocatorInterfaces(
    obstaclesPercept, linesPercept, psdPercept, cameraMatrix, theOdometryDataReceiver, thePackageCognitionMotionSender.playerPoseCollection, thePackageCognitionMotionSender.robotPose, thePackageCognitionMotionSender.ballModel,
    thePackageCognitionMotionSender.obstaclesModel);
  pObstaclesLocator = new ObstaclesLocatorSelector(moduleHandler, obstaclesLocatorInterfaces);
  
  //  PRINT("Cognition - ObstaclesLocator created");
  
  SelfLocatorInterfaces selfLocatorInterfaces(
    landmarksPercept, linesPercept, specialPercept,
    psdPercept, theOdometryDataReceiver, cameraMatrix, 
    thePackageCognitionMotionSender.robotPose, 
    selfLocatorSamples, 
    thePackageCognitionMotionSender.landmarksState);
  pSelfLocator = new SelfLocatorSelector(moduleHandler, selfLocatorInterfaces);
  
  //  PRINT("Cognition - SelfLocator created");
  
  const BehaviorControlInterfaces behaviorControlInterfaces(
    thePackageCognitionMotionSender.robotPose, 
    thePackageCognitionMotionSender.ballModel, 
    thePackageCognitionMotionSender.playerPoseCollection, 
    thePackageCognitionMotionSender.obstaclesModel, 
    thePackageCognitionMotionSender.robotState, 
    thePackageMotionCognitionReceiver.executedMotionRequest, 
    specialPercept,
    teamMessageCollection,
    joystickData,
    theOdometryDataReceiver,
    selfLocatorSamples,
    theSensorDataBufferReceiver,
    psdPercept,
    theGameControlDataReceiver,
    thePackageCognitionMotionSender.motionRequest, 
    thePackageCognitionMotionSender.ledRequest, 
    thePackageCognitionMotionSender.headControlMode, 
    thePackageCognitionMotionSender.soundRequest,
    thePackageCognitionMotionSender.invKinWalkingParameters,
    thePackageCognitionMotionSender.udParameters,
    thePackageCognitionMotionSender.walkParameterTimeStamp,
    specialVisionRequest, calibrationRequest, outgoingBehaviorTeamMessage);
  
  pBehaviorControl = new BehaviorControlSelector(moduleHandler, behaviorControlInterfaces);
  
  //  PRINT("Cognition - BehaviorControl created");
  
  
  SensorBehaviorControlInterfaces sensorBehaviorControlInterfaces(
    theImageReceiver,
    theSensorDataBufferReceiver,
    cameraMatrix,
    colorTable,
    theOdometryDataReceiver,
    thePackageCognitionMotionSender.robotState,
    thePackageCognitionMotionSender.soundRequest, 
    thePackageCognitionMotionSender.motionRequest, 
    thePackageCognitionMotionSender.ledRequest, 
    thePackageCognitionMotionSender.headControlMode
    );
  pSensorBehaviorControl = new SensorBehaviorControlSelector(moduleHandler, sensorBehaviorControlInterfaces);
  
  //  PRINT("Cognition - SensorBehaviorControl created");
  
  SpecialVisionInterfaces specialVisionInterfaces(theImageReceiver,specialVisionRequest,
    colorTable, specialPercept);
  pSpecialVision = new DefaultSpecialVision(specialVisionInterfaces);
  
  //  PRINT("Cognition - SpecialVision created");
  
  CollisionDetectorInterfaces collisionDetectorInterfaces(theSensorDataBufferReceiver, 
    thePackageMotionCognitionReceiver.executedMotionRequest, collisionPercept);
  pCollisionDetector = new CollisionDetectorSelector(moduleHandler, collisionDetectorInterfaces);
  
  //  PRINT("Cognition - CollisionDetection created");

  //just to be sure it is initialized:
  thePackageCognitionMotionSender.walkParameterTimeStamp=0;
  
  /* Init fr bessere "TeamMessageCollection" */
  
  teamMessageCollection.setInTeamMessages(theTeamMessage1Receiver);
  teamMessageCollection.setOutTeamMessages(theTeamMessage1Sender);
  teamMessageCollection.setInTeamMessages(theTeamMessage2Receiver);
  teamMessageCollection.setOutTeamMessages(theTeamMessage2Sender);
  teamMessageCollection.setInTeamMessages(theTeamMessage3Receiver);
  teamMessageCollection.setOutTeamMessages(theTeamMessage3Sender);
#ifdef FIVEDOGS
  teamMessageCollection.setInTeamMessages(theTeamMessage4Receiver);
  teamMessageCollection.setOutTeamMessages(theTeamMessage4Sender);
#endif
  
  
    
}

Cognition::~Cognition()
{
  delete pImageProcessor;
  delete pSensorDataProcessor;
  delete pRobotStateDetector;
  delete pBallLocator;
  delete pTeamBallLocator;
  delete pPlayersLocator;
  delete pObstaclesLocator;
  delete pSelfLocator;
  delete pBehaviorControl;
  delete pSensorBehaviorControl;
  delete pSpecialVision;
  delete pCollisionDetector;
#ifndef NO_COLOR_TABLE_SELECTOR
  delete pColorTableMod;
#endif
}

int Cognition::main() 
{
  
#ifndef _WIN32
  theUDPHandler.doRegularStuff();
#endif
  
  WATCH(sendSensorData,idSensorData,bin,theSensorDataBufferReceiver);
  
  if (theSensorDataBufferReceiver.receivedNew()) processSensorData = true;
  
  if (processSensorData)
  {
    STOP_TIME_ON_REQUEST(sensorDataProcessor, pSensorDataProcessor->execute(); );
    STOP_TIME_ON_REQUEST(collisionDetector, pCollisionDetector->execute(); );
    STOP_TIME_ON_REQUEST(robotStateDetector, pRobotStateDetector->execute(); );
  }
  
  if (theImageReceiver.receivedNew()) 
  {
    processImage = true;
  }
  //  OUTPUT(idText, text, "image frame number" << theImageReceiver.frameNumber << " " << processImage);
  //  OUTPUT(idText, text, "num of frames" << theSensorDataBufferReceiver.numOfFrames << " " << processSensorData);
  
  if (!processImage && !theDebugReceiver.receivedNew())
    return 0; // the process received SensorData but not an image
  
  teamMessageCollection.processMessages();
  
  INFO(sendOdometryData,idOdometryData,bin,theOdometryDataReceiver);
  
  INFO(sendPlayerConfig,idText,text,"player: " << Player::getTeamColorName(getPlayer().getTeamColor())
    << " " << Player::getPlayerNumberName(getPlayer().getPlayerNumber()));
  
  
  // time needed to properly calculate ball speed
  // (sort of a image time stamp)
  timeOfImageProcessing = SystemCall::getCurrentSystemTime();
  
  if (processImage) 
  {
    WATCH(sendImage,idImage,bin,SEND_IMAGE(theImageReceiver,cameraMatrix));
    INFO(sendJPEGImage,idJPEGImage,bin,JPEGImage(theImageReceiver) << cameraMatrix);
    INFO(sendLowResImage,idLowResImage,bin,LowResImage(theImageReceiver) << cameraMatrix);
    STOP_TIME_ON_REQUEST(imageProcessor, pImageProcessor->execute(); );
    STOP_TIME_ON_REQUEST(specialVision,  pSpecialVision->execute();  );
    processPercepts = true;
  }
  
  if (processPercepts)
  {
    STOP_TIME_ON_REQUEST(selfLocator, pSelfLocator->execute();  );
    STOP_TIME_ON_REQUEST(ballLocator, pBallLocator->execute();  );
    STOP_TIME_ON_REQUEST(teamBallLocator, pTeamBallLocator->execute();  );
    STOP_TIME_ON_REQUEST(playersLocator, pPlayersLocator->execute(); );
    STOP_TIME_ON_REQUEST(obstaclesLocator, pObstaclesLocator->execute(); );
  }
  
  WATCH(sendWorldState,idWorldState,bin,SEND_WORLD_STATE(
    thePackageCognitionMotionSender.robotPose,
    thePackageCognitionMotionSender.ballModel,
    thePackageCognitionMotionSender.playerPoseCollection,
    thePackageCognitionMotionSender.obstaclesModel,
    thePackageCognitionMotionSender.robotState,
    cameraMatrix,
    theImageReceiver.cameraInfo
    ));
  WATCH(sendPercepts,idPercepts,bin,SEND_PERCEPTS(cameraMatrix, theImageReceiver.cameraInfo,
    ballPercept,landmarksPercept,linesPercept,playersPercept, obstaclesPercept, psdPercept, collisionPercept));
  
  
  STOP_TIME_ON_REQUEST(behaviorControl,  pBehaviorControl->execute();  );
  STOP_TIME_ON_REQUEST(sensorBehaviorControl, pSensorBehaviorControl->execute();  );
  INFO(sendSpecialPercept, idSpecialPercept, bin, getPlayer() << specialPercept << cameraMatrix);
  
  thePackageCognitionMotionSender.teamColor = getPlayer().getTeamColor();

  thePackageCognitionMotionSender.wLanStatus = 0;
  
  if (theTeamMessage1Receiver.isActual()) thePackageCognitionMotionSender.wLanStatus += 1;
  if (theTeamMessage2Receiver.isActual()) thePackageCognitionMotionSender.wLanStatus += 2;
  if (theTeamMessage3Receiver.isActual()) thePackageCognitionMotionSender.wLanStatus += 4;
  
#ifdef FIVEDOGS
  if (theTeamMessage4Receiver.isActual()) thePackageCognitionMotionSender.wLanStatus += 8;
  #endif
 
  thePackageCognitionMotionSender.timeStamp = SystemCall::getCurrentSystemTime();

  thePackageCognitionMotionSender.send();
  
  INFO(sendGameControlData,idText,text,"GameControlData: state: " << theGameControlDataReceiver.getStateName(theGameControlDataReceiver.data.state)
    << ", kickoff: " << theGameControlDataReceiver.getKickoffName(theGameControlDataReceiver.data.kickoff)
    //    << ", team color: " << theGameControlDataReceiver.getTeamColorName(theGameControlDataReceiver.data.teamColor)
    << ", own score: " << theGameControlDataReceiver.data.ownScore);
  //    << ", opponent score: " << theGameControlDataReceiver.data.opponentScore
  //    << ", penalty: "<< theGameControlDataReceiver.getPenaltyName(theGameControlDataReceiver.data.penalty));
  
  theDebugSender.send();
  
  
  teamMessageCollection.send (thePackageCognitionMotionSender.robotPose);
  teamMessageCollection.send (thePackageCognitionMotionSender.ballModel.seen);
  teamMessageCollection.send (outgoingBehaviorTeamMessage);
  
  
  if ( (teamMessageCollection.processOutMessages())) { 
    theTeamMessage1Sender.send();
    theTeamMessage2Sender.send();
    theTeamMessage3Sender.send();
#ifdef FIVEDOGS
	theTeamMessage4Sender.send();
#endif
  }
  
  
  processImage = false;
  processPercepts = false;
  processSensorData = true;
  
  return 0;
}

void Cognition::init()
{
  INIT_UDP_HANDLER(TeamMessage1);
  INIT_UDP_HANDLER(TeamMessage2);
  INIT_UDP_HANDLER(TeamMessage3);
#ifdef FIVEDOGS
  INIT_UDP_HANDLER(TeamMessage4);
#endif
  START_UDP_HANDLER;
  // !!! don't remove that lines, it is needed to trigger the Debug process to send 
  // debug requests back.
  OUTPUT(idText,text,"player: " << Player::getTeamColorName(getPlayer().getTeamColor())
    << " " << Player::getPlayerNumberName(getPlayer().getPlayerNumber())
    << ", MAC Address: " << getRobotConfiguration().getMacAddressString());
}

bool Cognition::handleMessage(InMessage& message)
{
  switch (message.getMessageID())
  {
  case idSensorData:
    message.bin >> theSensorDataBufferReceiver;
    processSensorData = true;
    return true;
  case idImage:
    message.bin >> RECEIVE_IMAGE(theImageReceiver,cameraMatrix);
    processImage = true;
    processSensorData = false;
    return true;
  case idJPEGImage:
    {
      JPEGImage jpe;
      message.bin >> jpe >> cameraMatrix;
      jpe.toImage(theImageReceiver);
      processImage = true;
      processSensorData = false;
      return true;
    }
  case idLowResImage:
    {
      LowResImage lrImage(theImageReceiver);
      message.bin >> lrImage >> cameraMatrix;
      return true;
    }
  case idPercepts:
    {
      Player pl;
      message.bin >> RECEIVE_PERCEPTS(cameraMatrix, theImageReceiver.cameraInfo, ballPercept,landmarksPercept,
        linesPercept,playersPercept, obstaclesPercept, psdPercept, collisionPercept);
      processPercepts = true;
      return true;
    }
  case idWorldState:
    {
      Player pl;
      message.bin >> RECEIVE_WORLDSTATE(thePackageCognitionMotionSender.robotPose,
        thePackageCognitionMotionSender.ballModel,
        thePackageCognitionMotionSender.playerPoseCollection,
        thePackageCognitionMotionSender.obstaclesModel,
        thePackageCognitionMotionSender.robotState,
        cameraMatrix,
        theImageReceiver.cameraInfo);
      return true;
    }
  case idHeadControlMode:
    message.bin >> thePackageCognitionMotionSender.headControlMode;
    return true;
  case idMotionRequest:
    message.bin >> thePackageCognitionMotionSender.motionRequest;
    return true;
  case idLEDRequest:
    message.bin >> thePackageCognitionMotionSender.ledRequest;
    return true;
  case idGameControlData:
    message.bin >> theGameControlDataReceiver;
    return true;
  case idGlobalGameControlData:
    {
      GlobalGameControlData ggcd;
      message.bin >> ggcd;
      theGameControlDataReceiver.translateFromGlobalGameControlData(ggcd,getPlayer());
    }
    return true;
  case idSoundRequest:
    message.bin >> thePackageCognitionMotionSender.soundRequest;
    return true;
  case idSpecialPercept:
    {
      Player pl;
      CameraMatrix cm;
      message.bin >> pl >> specialPercept >> cm;
      processPercepts = true;
      return true;
    }
  case idOracledWorldState:
    {
      // send an oracled world state immediately back for paint synchronization
      message >> debugOut;
      return true;
    }
  case idCameraParameters:
    {
      message.bin >> cameraParameters;
      this->setCameraParameters(cameraParameters);
      return true;
    }
  case idColorTable64:
    {
      ColorTable64 colorTable64;
      message.bin >> colorTable64;
      memcpy(colorTableBuffer,&colorTable64,sizeof(ColorTable64));
      processImage = true;
      pImageProcessor->handleMessage(message);
      OUTPUT(idText,text,"Using ColorTable64");
      return true;
    }
  case idColorTableTSL:
    {
      ColorTableTSL colorTableTSL;
      message.bin >> colorTableTSL;
      colorTableTSL.calculateLUT();
      memcpy(colorTableBuffer,&colorTableTSL,sizeof(ColorTableTSL));
      processImage = true;
      OUTPUT(idText,text,"Using ColorTableTSL");
      return true;
    }
  case idParametersForGridImageProcessor2:
    {
      pImageProcessor->handleMessage(message);
      processImage = true;
      return true;
    }
  case idGridImageProcessor2DebugParameters:
    {
      pImageProcessor->handleMessage(message);
      processImage = true;
      return true;
    }
  case idLinesImageProcessorParameters:
    {
      pImageProcessor->handleMessage(message);
      processImage = true;
      return true;
    }
  case idLinesSelfLocatorParameters:
    {
      pSelfLocator->handleMessage(message);
      return true;
    }
  case idXabsl2DebugRequest:
  case idGT2003BehaviorConfiguration:
  case idTacticEntryArray:
  case idFreezeRequest:
  case idKickSelectionTable:
  case idMotionRatingBehaviorMessage:
  case idUDEvolutionRequest:
  case idUDParameters:
    pBehaviorControl->handleMessage(message);
    return true;
  case idJoystickData:
    message.bin >> joystickData;
    return true;
  case idBodyOffsets:
    getRobotConfiguration().handleMessage(message);
    return true;
  default:
    {
      return Process::handleMessage(message);
    }
  }
}

MAKE_PROCESS(Cognition);

/*
* Change log :
* 
* $Log: Cognition.cpp,v $
* Revision 1.60  2004/06/03 16:10:53  kerdels
* added new headcontrolmode
*
* Revision 1.59  2004/05/24 19:40:06  goehring
* ballPosition to ballModel renamed
*
* Revision 1.58  2004/05/14 14:12:08  wachter
* - Added communication support for 5 robots
* - rewrote parts of team-communication to be faster and more stable
*
* Revision 1.57  2004/04/16 14:57:39  dueffert
* cleanup for Martins data flow graphics
*
* Revision 1.56  2004/04/14 06:37:16  loetzsch
* removed the hack "kickInProgress" from the BehaviorControlInterfaces
*
* Revision 1.55  2004/04/11 18:49:45  roefer
* Team color is sent again from Cognition to Motion.
* Otherwise, messages sent by Motion may contain the wrong team color.
*
* Revision 1.54  2004/04/09 11:35:53  roefer
* Bremen Byters German Open check-in
*
* Revision 1.53  2004/04/08 15:33:07  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.52  2004/04/07 14:42:56  risler
* moved LandsmarksState to Cognition directory, generated by SelfLocator
*
* Revision 1.51  2004/04/07 13:00:47  risler
* ddd checkin after go04 - second part
*
* Revision 1.3  2004/04/07 11:44:06  risler
* added sending low res images
* added Image::setCameraInfo
*
* Revision 1.2  2004/04/01 19:52:43  Charlie
* added LandmarkState
*
* Revision 1.1.1.1  2004/03/29 08:28:46  Administrator
* initial transfer from tamara
*
* Revision 1.50  2004/04/07 11:40:18  dueffert
* calibration reset bug fixed
*
* Revision 1.49  2004/04/05 17:56:47  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.2  2004/04/02 10:02:40  jumped
* changed balllocatorinterface
*
* Revision 1.1.1.1  2004/03/31 11:16:52  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.48  2004/03/30 11:57:30  dueffert
* first useful evolutionMode implementation (0=none, 1=auto); receiving parameters improved; beautified
*
* Revision 1.47  2004/03/29 15:32:01  dueffert
* evolution mode idea added
*
* Revision 1.46  2004/03/27 14:45:08  loetzsch
* removed team color from PackageCognitionMotion
*
* Revision 1.45  2004/03/21 12:39:30  juengel
* Added CalibrationRequest to BallLocatorInterfaces.
*
* Revision 1.44  2004/03/19 16:41:21  dueffert
* output improved
*
* Revision 1.43  2004/03/19 09:30:18  dueffert
* handling for iDUDParameters added
*
* Revision 1.42  2004/03/17 19:53:36  juengel
* Added idKickSelectionTable.
*
* Revision 1.41  2004/03/17 16:31:20  kerdels
* added boolean input symbol "robot-pose.something-in-front-of-chest" utilizing the chest distance sensor
*
* Revision 1.40  2004/03/17 16:18:49  thomas
* added preversion of motion optimisation with behaviour, selflocator, headcontrol and robotcontrol dialog
*
* Revision 1.39  2004/03/16 16:00:09  tim
* Added SensorDataBuffer to BehaviorControlInterfaces
*
* Revision 1.38  2004/03/16 12:39:43  dueffert
* typo fixed
*
* Revision 1.37  2004/03/16 10:24:36  dueffert
* output improved
*
* Revision 1.36  2004/03/15 12:50:31  tim
* Adaptions to new GameController
*
* Revision 1.35  2004/03/15 12:36:53  dueffert
* measurement of free choosen MotionRequest allowed now
*
* Revision 1.34  2004/03/11 00:01:34  roefer
* OUTPUT removed
*
* Revision 1.33  2004/03/10 12:17:50  roefer
* Player bug fixed
*
* Revision 1.32  2004/03/09 10:41:12  dueffert
* compilability restored; tim: please update GameControlData if you wanna use new features
*
* Revision 1.31  2004/03/09 08:04:43  dueffert
* initialization added, comments corrected
*
* Revision 1.30  2004/03/08 15:29:08  tim
* Sending the Player object to the Motion process
*
* Revision 1.29  2004/03/04 10:05:24  jhoffman
* - motion process now uses odometry to propagate the robot pose while no new robot pose is being sent (this makes headcontrol a little better)
* - removed headcontrol member variable "propagatedPose" from headcontrol and cognition->motion-sender
*
* Revision 1.28  2004/03/03 08:30:18  dueffert
* two UDEvolutionRequests added and bug fixed
*
* Revision 1.27  2004/02/29 13:38:47  dueffert
* symmetries in UDParametersSet handled
*
* Revision 1.26  2004/02/27 16:42:14  dueffert
* UDEvolutionRequest introduced
*
* Revision 1.25  2004/02/23 16:48:50  dueffert
* several improvements for measurement of walking
*
* Revision 1.24  2004/02/18 14:49:59  dueffert
* behavior control can now change walking parameters
*
* Revision 1.23  2004/02/16 18:56:32  tim
* Notification of ImageProcessor in case of new ColorTable64
*
* Revision 1.22  2004/02/16 17:57:43  dueffert
* packageCognitionMotion extended with invkin parameters
*
* Revision 1.21  2004/02/01 12:30:41  nistico
* Fixed compilation problem when NO_COLOR_TABLE_SELECTOR is defined, after the memory leak was fixed
*
* Revision 1.20  2004/01/31 10:30:11  roefer
* Memory leak closed
*
* Revision 1.19  2004/01/30 15:44:33  nistico
* Created a Color Table Selector, which permits to switch color tables at run time using the same image processor.
* If your image processor was designed to use only ColorTable64, leave it as the default choice and you should
* (hopefully) have no problems.
* In case of trouble, i put a conditional compilation switch in SolutionRequest.h, simply uncomment the definition of NO_COLOR_TABLE_SELECTOR; in that case, the code should get back to the original solution, but the coltable which will be looked upon first for loading will be ColorTable64 (as it's the default solution), so there's no reason anymore to remove the other color tables from the CVS.
*
* Revision 1.18  2004/01/26 14:14:50  wachter
* activated direct team-communication
*
* Revision 1.17  2004/01/20 23:09:49  loetzsch
* bug fix
*
* Revision 1.16  2004/01/20 12:40:08  nistico
* - Added support for ColorTable32K (65K elements in packed format)
* - RobotControl can now convert GT *.log files into AIBOVision (external ColorTable32K calibration tool) *.amv file format
*
* Revision 1.15  2004/01/17 14:39:22  kerdels
* modified idTacitcEntryArray Message routing
*
* Revision 1.14  2004/01/10 10:09:14  juengel
* Added CameraInfo to and removed Player from (SEND|RECEIVE)_(PERCEPTS|WORLDSTATE).
*
* Revision 1.13  2003/12/09 16:30:19  jhoffman
* robotState is now being send to sensorbehaviorcontrols
*
* Revision 1.12  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.11  2003/12/06 06:31:19  loetzsch
* no message
*
* Revision 1.10  2003/11/29 07:40:19  roefer
* Doxygen comments corrected
*
* Revision 1.9  2003/11/24 15:31:20  dueffert
* SpecialPercept removed from PackageCognition
*
* Revision 1.8  2003/11/14 19:02:26  goehring
* frameNumber added
*
* Revision 1.7  2003/11/11 13:16:37  loetzsch
* corrected message handling for special percepts
* moved invocation of special vision upward
*
* Revision 1.6  2003/11/11 10:46:21  loetzsch
* removed the
* #ifdef APERIOS1_3_2
* statement
*
* Revision 1.5  2003/11/10 13:37:42  dueffert
* checkerboard localization improved
*
* Revision 1.4  2003/11/05 16:41:39  goehring
* no message
*
* Revision 1.3  2003/11/05 16:34:40  goehring
* FrameNumber added
*
* Revision 1.2  2003/10/12 11:48:14  juengel
* Added CalibrationRequest.
*
* Revision 1.1  2003/10/07 10:07:00  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.7  2003/09/26 21:23:20  loetzsch
* renamed class JointState to CollisionPercept
*
* Revision 1.6  2003/09/26 15:30:38  juengel
* Renamed DataTypes to representations.
*
* Revision 1.5  2003/09/26 11:41:17  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.4  2003/07/30 14:59:36  dueffert
* SpecialPerceptSelfLocator for checkerboard implemented
*
* Revision 1.3  2003/07/05 09:49:05  roefer
* Generic debug message for bodyOffsets improved
*
* Revision 1.2  2003/07/02 20:42:11  loetzsch
* bug fix
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.84  2003/06/21 12:50:18  juengel
* Added STOP_TIME_ON_REQUEST for collision detector and robotState detector.
*
* Revision 1.83  2003/06/20 20:21:17  goehring
* ExecutedMotionReques instead of MotionRequest implemented for CollisionDetector
*
* Revision 1.82  2003/06/18 19:35:32  goehring
* RobotStateDetectorInterfaces extended by JointState
*
* Revision 1.81  2003/06/18 13:57:46  juengel
* Added ballPosition to ObstaclesLocatorInterfaces.
*
* Revision 1.80  2003/06/17 19:55:41  risler
* timeOfImageProcessing initialisation
*
* Revision 1.79  2003/06/15 22:42:21  loetzsch
* no message
*
* Revision 1.78  2003/06/10 15:03:23  goehring
* JointState added
*
* Revision 1.77  2003/06/02 09:39:34  goehring
* Added JointState to SEND_PERCEPTS and RECEIVE_PERCEPTS.
*
* Revision 1.76  2003/05/27 12:23:42  risler
* added GT2003BehaviorConfiguration debug message
*
* Revision 1.75  2003/05/26 08:27:56  juengel
* Added cameraMatrix to worldState.
*
* Revision 1.74  2003/05/23 09:31:39  goehring
* CollisionDetector - Interface extended to MotionRequest
*
* Revision 1.73  2003/05/18 22:36:44  loetzsch
* changes in the ::init() function by Matthias Jngel
*
* Revision 1.72  2003/05/13 11:41:45  goehring
* CollisionDetector added
*
* Revision 1.71  2003/05/12 14:08:39  brunn
* renamed selfLocationSampleSetProxy to selfLocatorSamples
* "Ha, glad am I that no one knew that Rumpelstiltskin I am styled"
*
* Revision 1.70  2003/05/12 12:28:12  brunn
* renamed sampleSetProxy to selfLocationSampleSetProxy
* added selfLocationSampleSetProxy to BehaviourControl-Interfaces
*
* Revision 1.69  2003/05/11 17:03:00  risler
* added location.cfg
*
* Revision 1.68  2003/05/08 23:52:24  roefer
* SampleSet and SampleSetProxy added
*
* Revision 1.67  2003/05/08 13:20:22  loetzsch
* added the execution of the team ball locator to the Cognition process
*
* Revision 1.66  2003/05/06 16:47:22  juengel
* Increased size of debugIn.
*
* Revision 1.65  2003/05/05 14:47:57  risler
* idJointData debug message sends JointDataBuffer
* JointViewerDlg shows reference values
*
* Revision 1.64  2003/05/05 12:44:12  juengel
* no message
*
* Revision 1.63  2003/05/05 12:41:24  juengel
* Removed idGridImageProcessor2Thresholds
*
* Revision 1.62  2003/05/03 16:21:09  roefer
* Destructor call for TeamBallLocator added
*
* Revision 1.61  2003/05/02 18:15:18  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.60  2003/05/02 12:57:13  loetzsch
* TeamMessage now contains a SeenBallPosition instead of a BallPercept
*
* Revision 1.59  2003/05/01 17:09:08  loetzsch
* Redesign of ball modeling:
* - Modularized class BallPosition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*
* Revision 1.58  2003/04/16 14:14:38  loetzsch
* removed Xabsl 1 from GT2003
*
* Revision 1.57  2003/04/14 16:23:27  loetzsch
* ATH after GermanOpen CVS merge
* added new proposal for sending the ball percept to other players
*
* Revision 1.56  2003/04/06 15:05:42  loetzsch
* Matthias added LinesPercept to ObstaclesLocatorInterfaces
*
* Revision 1.55  2003/04/05 16:49:49  juengel
* Added specialPercept to DrawingMethods.
*
* Revision 1.54  2003/04/03 17:35:51  dueffert
* free part of goal kick added
*
* Revision 1.53  2003/03/31 04:04:03  osterhues
* Added "coltable.tsl" loading mechanism
* handleMessage() now notifies user which color table is used
*
* Revision 1.52  2003/03/28 14:31:34  juengel
* Added PSDPercept to SEND_PERCEPTS and RECEIVE_PERCEPTS.
*
* Revision 1.51  2003/03/27 03:21:54  deutsch
* no message
*
* Revision 1.50  2003/03/25 10:50:24  juengel
* WATCH and INFO macros for image and JPEGImage are only reached, when processImage is true.
*
* Revision 1.49  2003/03/22 16:56:34  juengel
* Added PSDPercept to ObstaclesLocatorInterfaces.
*
* Revision 1.48  2003/03/20 20:33:41  loetzsch
* removed some hacks
*
* Revision 1.47  2003/03/19 15:40:41  jhoffman
* GL simulator support improved
*
* Revision 1.46  2003/03/15 13:24:30  juengel
* Added CameraMatrix to ObstaclesLocatorInterfaces
*
* Revision 1.45  2003/03/12 22:27:16  roefer
* Destructor call of PlayersPerceptor added
*
* Revision 1.44  2003/03/12 13:46:04  dasmuli
* PlayersPerceptor added to cognition, modules.cfg, DebugKey-table etc.
*
* Revision 1.43  2003/03/10 13:59:05  juengel
* Added ObstaclesLocator
*
* Revision 1.42  2003/03/05 14:07:10  roefer
* LinesPerceptor2 is now LinesImageProcessor
*
* Revision 1.41  2003/03/05 11:10:05  loetzsch
* added class GlobalGameControlData
*
* Revision 1.40  2003/03/04 15:05:11  juengel
* Added handling for idParametersForGridImageProcessor2
*
* Revision 1.39  2003/02/27 12:02:30  dueffert
* JoystickData added
*
* Revision 1.38  2003/02/21 22:20:13  roefer
* LinesSelfLocator is working
*
* Revision 1.37  2003/02/21 18:32:04  roefer
* pColorTable -> colorTable finished
*
* Revision 1.36  2003/02/19 14:59:55  roefer
* pColorTable -> colorTable
*
* Revision 1.35  2003/02/18 21:29:17  osterhues
* Changed all instances of ColorTable64 to new base class ColorTable
*
* Revision 1.34  2003/02/18 13:24:37  wachter
* added new TeamMessageCollection and TeamMessage
*
* Revision 1.33  2003/02/16 08:29:40  roefer
* sendOdometryData added
*
* Revision 1.32  2003/02/08 18:36:44  juengel
* added idGridImageProcessor2Thresholds
*
* Revision 1.31  2003/02/07 16:43:15  dueffert
* Bombay stuff added after cleanup
*
* Revision 1.30  2003/01/30 22:31:47  juengel
* Added LinesPercept to ImageProcessorInterfaces.
*
* Revision 1.29  2003/01/30 11:28:54  juengel
* Added idGridImageProcessor2Parameters and id LinesPerceptorParameters.
*
* Revision 1.28  2003/01/22 18:09:23  loetzsch
* continued implementation of sending of intermediate code from xabsl2 dialog
* to a xabsl2 behavior control
*
* Revision 1.27  2003/01/21 15:34:36  loetzsch
* idXabslDebugRequest is handled directly by the BehaviorControl
* selector instead of the ModuleHandler
*
* Revision 1.26  2003/01/15 13:46:34  roefer
* SelfLocator has access to CameraMatrix now
*
* Revision 1.25  2003/01/13 18:25:30  juengel
* Added odometry to SensorBehaviorControlInterfaces.
*
* Revision 1.24  2002/12/08 07:11:14  roefer
* Decompress JPEG images in Cognition
*
* Revision 1.23  2002/12/07 16:40:45  roefer
* Blocking for theDebugReceiver changed
*
* Revision 1.22  2002/12/07 12:35:38  loetzsch
* world states and percepts now can be viewed without sending images
*
* Revision 1.21  2002/12/06 16:41:01  goehring
* no message
*
* Revision 1.20  2002/11/28 18:54:53  juengel
* Added SoundRequest to SensorBehaviorControl interfaces.
*
* Revision 1.19  2002/11/28 14:46:51  jhoffman
* added special percept for motion detection
*
* Revision 1.18  2002/11/26 19:22:18  juengel
* added stopwatchID for SensorBehaviorControl
*
* Revision 1.17  2002/11/26 19:19:24  loetzsch
* JPEG images are put into local processes
*
* Revision 1.16  2002/11/20 15:44:49  juengel
* Added cameraMatrix to SensorBehaviorControlInterfaces.
*
* Revision 1.15  2002/11/18 19:24:16  juengel
* Line Follower added.
*
* Revision 1.14  2002/11/18 17:30:31  loetzsch
* Sound requests generated from BehaviorControl are now automatically sent
* to RobotControl and played there.
*
* Revision 1.13  2002/11/07 17:30:10  loetzsch
* renamed Module SensorDataToMotionRequest to SensorBehaviorControl
*
* Revision 1.12  2002/10/11 13:54:44  roefer
* JPEGImage added
*
* Revision 1.11  2002/10/10 13:09:50  loetzsch
* First experiments with the PSD Sensor
* - SensorDataProcessor now calculates PSDPercept
* - Added the PerceptBehaviorControl solution PSDTest
* - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
*
* Revision 1.10  2002/10/04 10:20:47  loetzsch
* renamed bool imageIsNew to processImage,
* images are also processed when a new color table arrives
*
* Revision 1.9  2002/09/25 10:25:13  loetzsch
* removed the "executeVisionModules" variable
* from SolutionRequest and ModuleHandler.
*
* Revision 1.8  2002/09/18 19:52:36  loetzsch
* the head state is now sent from Motion to Cognition using the package.
*
* Revision 1.7  2002/09/18 16:35:12  loetzsch
* made GT2003 compilable,
* rechanged message ids
*
* Revision 1.6  2002/09/17 23:55:22  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.5  2002/09/16 17:34:23  dueffert
* anonymous contructors returns &CLASS with VS, but CLASS with gcc.
*
* Revision 1.4  2002/09/12 14:20:05  juengel
* Created a package for all data sent from Cognition to Motion.
*
* Revision 1.3  2002/09/12 12:40:58  juengel
* continued change of module/solution mechanisms
*
* Revision 1.2  2002/09/10 21:07:30  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:41:25  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
* - Removed all process layouts
* - Added process layout CMD
*
* Revision 1.33  2002/09/07 13:36:58  loetzsch
* unified the vision modules into one module "ImageProcessor"
* - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
*   and PlayersPerceptor were are combined to the new solution
*   "BlobImageProcessor"
* - The GridImageProcessor and the SubPixelGradientCalculator became
*   a solution of "ImageProcessor"
*
* Revision 1.32  2002/08/04 19:42:09  roefer
* SimGT2002 receives player config
*
* Revision 1.31  2002/07/29 17:45:11  jhoffman
* added braitenberg vehicle-style behavior (or actually: a "sensor data to motion request")
*
* Revision 1.30  2002/07/23 13:40:51  loetzsch
* - new streaming classes
* - removed many #include statements
* - new design of debugging architecture
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - general clean up
*
* Revision 1.29  2002/07/09 20:08:27  roefer
* Extract simulation colors from color table
*
* Revision 1.28  2002/07/09 16:26:48  roefer
* OUTPUT without braces fixed
*
* Revision 1.27  2002/06/28 10:30:51  roefer
* OUTPUT is possible in constructors
*
* Revision 1.26  2002/06/28 10:26:21  roefer
* OUTPUT is possible in constructors
*
* Revision 1.25  2002/06/20 00:40:21  Thomas Rfer
* WLan crash removed
*
* Revision 1.24  2002/06/13 12:15:24  roefer
* sim.c64 is default color table under SimGT2002
*
* Revision 1.23  2002/06/12 11:34:29  roefer
* SimpleLinesPerceptor removed, PerceptBehaviorControl added
*
* Revision 1.22  2002/06/10 11:05:17  risler
* added timestamp to motion request
* motioncontrol executes swing when no request was received
*
* Revision 1.21  2002/06/09 15:24:52  loetzsch
* Added TeamMessageCollection and BehaviorTeamMessage to the execute of BehaviorControl
*
* Revision 1.20  2002/06/08 20:41:15  mkunz
* specialPercept queue completed
*
* Revision 1.19  2002/06/08 11:44:00  mkunz
* Special Percept on Field
*
* Revision 1.18  2002/06/08 09:26:32  Thomas Rfer
* Team ball position, first draft
*
* Revision 1.17  2002/06/07 10:24:16  loetzsch
* added teamCollection to the execute of the players locator
*
* Revision 1.16  2002/06/04 23:27:02  loetzsch
* 4-robots-in-RobotControl related bug fixes and improvements
*
* Revision 1.15  2002/06/04 00:15:36  loetzsch
* RobotControl now can simulate four robots.
*
* Revision 1.14  2002/06/03 15:27:30  roefer
* BallLocator gets TeamMessageCollection
*
* Revision 1.13  2002/06/02 23:21:09  roefer
* Single color table and progress in LinesSelfLocator
*
* Revision 1.12  2002/05/29 15:54:29  cesarz
* Removed deadlock in acoustic communication
*
* Revision 1.11  2002/05/27 15:39:13  fischer
* Added SoundState (Sender and Receiver)
*
* Revision 1.10  2002/05/26 14:55:25  roefer
* Team communication is working
*
* Revision 1.9  2002/05/26 14:22:09  juengel
* SpecialVisionRequest for ImageToPerceptCollection modules.
*
* Revision 1.8  2002/05/23 12:16:37  hebbel
* detects acoustic messages only if dog is not sending himself
*
* Revision 1.7  2002/05/22 13:46:42  fischer
* Changed SoundInProcessor interface to accept WorldState
*
* Revision 1.6  2002/05/21 12:23:12  hebbel
* added startSoundAnalyse() in SoundProtocol
*
* Revision 1.5  2002/05/16 22:36:11  roefer
* Team communication and GTMath bugs fixed
*
* Revision 1.4  2002/05/15 15:47:30  risler
* added debug message AcousticMessage, DebugMessageGenerator generates AcousticMessage
*
* Revision 1.3  2002/05/14 21:04:04  hebbel
* processing of SoundDataIn added
*
* Revision 1.2  2002/05/10 17:29:43  juengel
* Added SpecialVision and SpecialPercept.
*
* Revision 1.1.1.1  2002/05/10 12:40:19  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.66  2002/05/07 15:34:46  jhoffman
* world state is being passed to imagetoperceptcollection. if none is passed, it is set to 0
*
* Revision 1.65  2002/05/05 22:12:43  loetzsch
* GameControlData can now be sent from the Game toolbar to BehaviorControl
*
* Revision 1.64  2002/05/05 18:52:03  loetzsch
* added
* - GameControlData,
* - Receivers for GameControlData
* - access by behavior to GameControlData
*
* Revision 1.63  2002/05/04 12:43:38  loetzsch
* The currently executed MotionRequest is now sent from the MotionControl
* to the BehaviorControl via the OdometryData structure
*
* Revision 1.62  2002/04/29 17:17:51  hebbel
* Put SoundPlay to Motion Process
*
* Revision 1.61  2002/04/28 19:19:46  giese
* SoundPlay added...
*
* Revision 1.60  2002/04/23 00:01:37  roefer
* Redesign of Bremen process layout
*
* Revision 1.59  2002/04/20 15:52:21  roefer
* Project simpified, WATCH and WATCH_PART added
*
* Revision 1.58  2002/04/09 11:19:06  loetzsch
* debug message handling for idPlayerConfig
*
* Revision 1.57  2002/04/08 13:34:13  dueffert
* PlayersToolBar added
*
* Revision 1.56  2002/04/06 09:55:53  roefer
* Image and SensorData path through DebugQueues changed
*
* Revision 1.55  2002/04/02 15:04:46  loetzsch
* debug message handling for Xabsl debug requests added
*
* Revision 1.54  2002/04/02 13:10:21  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.53  2002/04/02 10:30:34  juengel
* GridImageProcessor enhanced.
*
* Revision 1.52  2002/03/29 14:53:45  juengel
* ImageToPerceptCollection reactivated.
*
* Revision 1.51  2002/03/28 16:55:58  risler
* RobotStateDetector receives BodyPercept instead of PerceptCollection
* added switch duration in RobotStateDetector
*
* Revision 1.50  2002/03/24 18:15:00  loetzsch
* continued change to blocking sensor data receivers
*
* Revision 1.49  2002/03/24 17:47:02  juengel
* LinesPercept defined and LinesPerceptor added.
*
* Revision 1.48  2002/02/25 16:31:52  loetzsch
* added setting of camera parameters
*
* Revision 1.47  2002/02/24 10:02:21  juengel
* TimeDiagramDlgBar completed.
*
* Revision 1.46  2002/02/21 14:22:47  loetzsch
* added several STOP_WATCH_ON_REQUEST macros
*
* Revision 1.45  2002/02/13 15:07:35  fischer
* copyfiles.bash copies the file "rules.rb" to the stick.
* Minor changes to FuzzyBehaviorControl
*
* Revision 1.44  2002/02/11 00:57:15  loetzsch
* no message
*
* Revision 1.43  2002/02/06 01:15:40  loetzsch
* oracled world states are now first send through the local processes
* before they are painted. (for synchronization with the painting of the
* calculated world states)
*
* Revision 1.42  2002/02/05 20:02:16  risler
* handleDebugMessage now returns bool, added debug message handling to ImageProcessor
*
* Revision 1.41  2002/02/05 04:19:04  loetzsch
* replaced several team color hacks by getPlayer().getTeamColor()
*
* added a few new module selectors
*
* changed distribution of debug messages completely
*
* Revision 1.40  2002/02/04 13:47:09  kspiess
* BremenBerlin2001PlayersLocator in GT2001PlayersLocator umbenannt
* alte Aufrufe in neue gendert
* DebugDrawing fr GT2001PlayersLocator eingebaut
*
* Revision 1.39  2002/02/03 20:42:26  juengel
* Ball perception in improved.
*
* Revision 1.38  2002/02/03 16:11:57  juengel
* wordStateOracle drawing added
*
* Revision 1.37  2002/02/03 14:37:59  juengel
* Drawing of the world state removed from Berlin2001BehaviorControl.
* Drawing method for world states added to PaintMethods.
* Drawing of the world state added to the Processes with BehaviorControl.
*
* Revision 1.36  2002/01/31 19:41:24  risler
* ImageProcessorSelector added
*
* Revision 1.35  2002/01/30 17:29:55  loetzsch
* handleDebugMessage um Parameter timestamp erweitert
*
* Revision 1.34  2002/01/30 02:22:57  tim
* BremenBerlin2001PlayersPerceptor und Bremen2002ImageProcessor umbenannt. DefaultPlayersPerceptor hinzugefgt
*
* Revision 1.33  2002/01/28 14:01:33  loetzsch
* useless OUTPUTs removed
*
* Revision 1.32  2002/01/26 18:10:16  juengel
* DebugDrawingManager umstrukturiert.
*
* Revision 1.31  2002/01/25 16:51:45  loetzsch
* #if defined (_WIN32) etc. changed
*
* Revision 1.30  2002/01/25 14:02:52  brunn
* Defines verbessert, kein WIN32 auf dem Roboter
*
* Revision 1.29  2002/01/24 18:17:36  loetzsch
* Added automatic sending of images under Win32 (not SimRobot
*
* Revision 1.28  2002/01/23 13:51:55  loetzsch
* from Debug Queues odometry data is now sent to Motion and not to Perception
*
* Revision 1.27  2002/01/23 12:35:37  juengel
* Repaint von DebugDrawings neu organisiert.
*
* Revision 1.26  2002/01/23 07:50:31  loetzsch
* Dies & das
*
* Revision 1.25  2002/01/23 02:47:39  loetzsch
* mcf tuned
*
* Revision 1.24  2002/01/22 19:49:06  loetzsch
* stopwatch changed
*
* Revision 1.23  2002/01/22 14:56:30  juengel
* ColorTable verschicken angefangen.
*
* Revision 1.22  2002/01/20 23:34:27  loetzsch
* Sending images and sensor data to processes running in RobotControl now possible
*
* Revision 1.21  2002/01/19 12:43:16  risler
* enabled SolutionRequest, changed HandleDebugMessage calls
*
* Revision 1.20  2002/01/19 08:07:29  juengel
* stopwatch weiterentwickelt
*
* Revision 1.19  2002/01/18 23:30:45  loetzsch
* Distribution of HeadControlModes, HeadMotionRequests and SolutionRequests added
*
* Revision 1.18  2002/01/18 14:09:03  risler
* BehaviorControlSelector added
*
* Revision 1.17  2002/01/18 02:17:37  loetzsch
* MotionTester dialog and distribution of MotionRequests
* to the processes programmed.
*
* Revision 1.16  2002/01/06 13:50:29  juengel
* WalkDemo eingebaut
*
* Revision 1.15  2001/12/21 14:09:39  roefer
* Added several destructors
*
* Revision 1.14  2001/12/20 17:14:35  loetzsch
* Using debug requests from the memory stick.
*
* Revision 1.13  2001/12/19 18:37:32  bach
* Camera.cfg added
* Sensors: CameraParameters are set
*
* Revision 1.12  2001/12/19 16:03:55  bach
* SystemDataTypes replaced by Sensors
*
* Revision 1.11  2001/12/16 17:18:54  loetzsch
* DebugKeyToolBar wieder groe Member Funktionsnamen eingebaut, Technote zu Toolbars
*
* Revision 1.10  2001/12/15 20:32:09  roefer
* Senders and receivers are now part of the processes
*
* Revision 1.9  2001/12/13 12:57:18  loetzsch
* Aufruf des SensorDataProcesssors entfernt, da dieser abstrzt.
*
* Revision 1.8  2001/12/12 18:08:55  loetzsch
* Streaming- Operatoren fr Bilder eingebaut, DebugKeyTable nicht- statisch gemacht, Debuggin Mechanismen weitergemacht, Bilder aus Logfiles in RobotControl anzeigen, Logfiles in HU1/Debug auf den Stick schreiben
*
* Revision 1.7  2001/12/10 17:47:08  risler
* change log added
*
*/
