/** 
* @file Processes/CD/Cognition.h
* Declaration of the process Cognition
*
* @author Martin Ltzsch
*/

#ifndef __Cognition_H__
#define __Cognition_H__

#include "Tools/Process.h"
#include "Platform/Sensors.h"

#include "Modules/SensorDataProcessor/SensorDataProcessorSelector.h"
#include "Modules/RobotStateDetector/RobotStateDetectorSelector.h"
#include "Modules/LEDControl/DefaultLEDControl.h"
#include "Modules/SoundControl/SoundControlSelector.h"

#include "Representations/RoboCup/GameControlData.h"
#include "Representations/WLan/TeamMessageCollection.h"
#include "Representations/Motion/OdometryData.h"
#include "Modules/SensorActorLoop/SensorActorLoopSelector.h"

/**
* @class Cognition
*
* A Process for cognition related modules in the HU1 process layout.
*/
class Cognition: public Process, public Sensors
{
private:
  DEBUGGING;
  RECEIVER(SensorDataBuffer);
  RECEIVER(Image);
  RECEIVER(GameControlData);
  RECEIVER(TeamMessage1);
  RECEIVER(TeamMessage2);
  RECEIVER(TeamMessage3);

  SENDER_MOTORCOMMANDS;
  SENDER_SOUNDDATA;
  SENDER(TeamMessage1);
  SENDER(TeamMessage2);
  SENDER(TeamMessage3);
  SENDER(OdometryData);

public:
  /** constructor */
  Cognition();
  
  /** destructor */
  ~Cognition();
  
  /** is called from the framework once in every frame */
  virtual int main();

  /** is called direct before the first call of main() */
  virtual void init();

  /** 
  * The function handles incoming debug messages.
  * @param message the message to handle
  * @return Has the message been handled?
  */
  virtual bool handleMessage(InMessage& message);

protected:
  //!@name modules
  //!@{

  /** a reference to the sensor data processor */
  SensorDataProcessorSelector* pSensorDataProcessor;

  /** a reference to the robot state detector */
  RobotStateDetectorSelector* pRobotStateDetector;
  
  /** a pointer to the LEDControl module */
  LEDControl *pLEDControl;
  
  /** a pointer to the SoundControl module */
  SoundControlSelector *pSoundControl;
  
  /** a reference to the sensor actor loop */
  SensorActorLoopSelector* pSensorActorLoop;

  //!@}

  //!@name representations
  //!@{

  /** The offset and rotation of the camera */
  CameraMatrix cameraMatrix;

  /** The collision percept */
  CollisionPercept collisionPercept;

  /** Detected switches */
  BodyPercept bodyPercept;
  
  /** A spot relative to the robot that was detected by the PSD sensor */
  PSDPercept psdPercept;

  /** The team message collection */
  TeamMessageCollection teamMessageCollection;

  /** The outgoing behavior team message */
  BehaviorTeamMessage outgoingBehaviorTeamMessage;

  /** The ledRequest*/
  LEDRequest ledRequest;
  
  /** The SoundRequest */
  SoundRequest soundRequest;

  /** The status of the WLan connection 1 to be displayed by the LEDControl */
  bool wLanStatus1;
  
  /** The status of the WLan connection 2 to be displayed by the LEDControl */
  bool wLanStatus2;
  
  /** The status of the WLan connection 3 to be displayed by the LEDControl */
  bool wLanStatus3;

  /** The neck height and the body tilt of the robot */
  HeadState headState;

  /** Pressed switches and fall downs */
  RobotState robotState;
  
  //!@}
};


#endif // __Cognition_H__


/*
 * Change log :
 * 
 * $Log: Cognition.h,v $
 * Revision 1.4  2003/11/29 07:40:19  roefer
 * Doxygen comments corrected
 *
 * Revision 1.3  2003/10/29 19:12:21  juengel
 * Added module SensorActorLoop.
 *
 * Revision 1.2  2003/10/27 19:47:30  loetzsch
 * no message
 *
 * Revision 1.1  2003/10/27 19:02:20  loetzsch
 * no message
 *
 */
