/**
 * @file Platform/Win32Linux/TcpConnection.cpp
 *
 * Implementation of class TcpConnection.
 *
 * @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
 */

#include "Platform/GTAssert.h"
#include "TcpConnection.h"
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>

#include <stdio.h>
#include <iostream>

using namespace std;

#ifdef _WIN32

#define ERRNO WSAGetLastError()
#define RESET_ERRNO WSASetLastError(0)
#define EWOULDBLOCK WSAEWOULDBLOCK
#define NON_BLOCK(socket) ioctlsocket(socket,FIONBIO,(u_long*) "NONE")
#define CLOSE(socket) closesocket(socket)

class _WSAFramework
{
  public:
    _WSAFramework() 
    {
      WORD wVersionRequested = MAKEWORD( 1, 0 );
      WSADATA wsaData;
      WSAStartup(wVersionRequested,&wsaData);
    }
    ~_WSAFramework() {WSACleanup();}
} _wsaFramework;

#else

#include <sys/select.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <unistd.h>

#define ERRNO errno
#define RESET_ERRNO errno = 0
#define NON_BLOCK(socket) fcntl(socket,F_SETFL,O_NONBLOCK)
#define CLOSE(socket) close(socket)

#endif

TcpConnection::TcpConnection()
{
  transferSocket = 0;
  createSocket = 0;
  client = false;
}

TcpConnection::TcpConnection(const char* ip,int port)
{
  connect(ip,port);
}

TcpConnection::~TcpConnection()
{
  if(isConnected())
    closeTransferSocket();
  if(createSocket > 0)
    CLOSE(createSocket);
}

void TcpConnection::connect(const char* ip,int port)
{
  client = false;
  createSocket = 0;
  transferSocket = socket(AF_INET,SOCK_STREAM,0);
  ASSERT(isConnected());
  address.sin_family = AF_INET;
  address.sin_addr.s_addr = inet_addr(ip);
  address.sin_port = htons(port);
  if(::connect(transferSocket,(sockaddr*) &address,sizeof(sockaddr_in)) == 0)
  {
    NON_BLOCK(transferSocket);
    client = true;
  }
  else
  {
    CLOSE(transferSocket);
    transferSocket = 0;
    createSocket = socket(AF_INET,SOCK_STREAM,0);
    ASSERT(createSocket > 0);
    address.sin_addr.s_addr = INADDR_ANY;
    VERIFY(bind(createSocket,(sockaddr*) &address,sizeof(sockaddr_in)) == 0);
    VERIFY(listen(createSocket,SOMAXCONN) == 0);
    NON_BLOCK(createSocket);
  }
  ack = false;
}

bool TcpConnection::sendAndReceive(char* dataToSend,int sendSize,
                                   char*& dataRead,int& readSize)
{
  readSize = 0;

  // if transferSocket is invalid, try to create a new one
  if(!isConnected())
  {
#ifdef LINUX
    unsigned int addrlen = sizeof(sockaddr_in);
#else
    int addrlen = sizeof(sockaddr_in);
#endif

    if(createSocket) 
    {
       transferSocket = accept(createSocket,(sockaddr*) &address,&addrlen);
    }
    else
    {
      ::connect(transferSocket,(sockaddr*) &address,sizeof(sockaddr_in));
    }

    if(isConnected())
    {
      NON_BLOCK(transferSocket); // switch socket to nonblocking
      int heartbeat = 0; // connection establish, send heartbeat now
    /*  if(!send((char*) &heartbeat,sizeof(heartbeat)))
        closeTransferSocket();
      else
        ack = true; */
    }
  }

  // Try to read data
  if(isConnected())
  {
    readSize = receive(dataRead);
    if(readSize < 0) // error, socket is invalid
    {
      closeTransferSocket();
    }
    else
    { 
      if((readSize > 0) && (!sendSize))
      {
        // we have received a package, but we don't want to send one now.
        // so, send heartbeat instead to acknowledge that we are still alive
        /* if(!send((char*) &sendSize,sizeof(sendSize)))
        {
           closeTransferSocket();
        } Try to do this without heartbeat */
      }
    }
  }

  // Try to send data
  if(isConnected() && sendSize>0)
  {
   
    if(!send((char*) &sendSize,sizeof(sendSize)) ||   // sends size of block
       !send(dataToSend,sendSize))                    // sends data
    {
      closeTransferSocket();                          // if this fails close socket
    }
    else
    {
      ack = false;
      return true;
    }
  }
  return false;
}

int TcpConnection::receive(char*& buffer)
{
  RESET_ERRNO;
  int size,
      received = recv(transferSocket,(char*) &size,sizeof(size),0);  
      // looks for data. (non blocking, tries to read size-field)
      // If the received is >0 data arrived
  if(received > 0)
  {
    
    /*char ausgabe[80];
    sprintf(ausgabe,"Received %d bytes \n",received);
    OutputDebugString(ausgabe);*/

    if(received < sizeof(size) &&   // not all data for size received
       !read(((char*) &size) + received,sizeof(size) - received))  // read size-field
      return -1; // cannot read the size

    /*sprintf(ausgabe,"Size is %d bytes \n",size);
    OutputDebugString(ausgabe);*/
    
    if(size == 0)
    {
      ack = false;  // was true
      return 0; // nothing to read (heartbeat sent by RobotControl)
    }
    else
    {
      buffer = new char[size];    // get buffer for complete package
      ASSERT(buffer);
      if(!read(buffer,size))    // read complete package (wait and read size bytes)
      {
        delete [] buffer;
        return -1; // error
      }
      else
        return size; // package received
    }
  }
  else if(received < 0 && ERRNO == EWOULDBLOCK)
    return 0; // nothing read, but ok
  else
    return -1; // error
}

bool TcpConnection::read(char* buffer,int size)
{
  int received = 0;
  while(received < size)
  {
    RESET_ERRNO;

   

    int received2 = recv(transferSocket,buffer + received,
                          size - received,0);

    if(received2 < 0 && ERRNO != EWOULDBLOCK)  // error during reading of package
    {
      return false;
    }
    else 
    {
      if(ERRNO == EWOULDBLOCK) // wait for the rest
      {
        received2 = 0;
        timeval timeout;
        timeout.tv_sec = 0;
        timeout.tv_usec = 100000;
        fd_set rset;
        FD_ZERO(&rset);
        FD_SET(transferSocket,&rset);
        if(select(transferSocket + 1,&rset,0,0,&timeout) == -1)
        {
          return false; // error while waiting
        }
      }
    }
    received += received2;

    /*char ausgabe[80];
    sprintf(ausgabe,"read received %d of %d bytes \n",received,size);
    OutputDebugString(ausgabe);*/
  }
  return true; // ok, data received
}

bool TcpConnection::send(const char* buffer,int size)
{
  RESET_ERRNO;
  int sent = ::send(transferSocket,buffer,size,0);
  while(sent < size && (ERRNO == EWOULDBLOCK || ERRNO == 0))
  {
    timeval timeout;
    timeout.tv_sec = 0;
    timeout.tv_usec = 100000;
    fd_set wset;
    FD_ZERO(&wset);
    FD_SET(transferSocket,&wset);
    RESET_ERRNO;
    if(select(transferSocket + 1,0,&wset,0,&timeout) == -1) 
      break;
    RESET_ERRNO;
    int sent2 = ::send(transferSocket,buffer + sent,size - sent,0);
    if(sent2 >= 0)
      sent += sent2;
  }
  return ERRNO == 0 && sent == size;
}

void TcpConnection::closeTransferSocket()
{
  CLOSE(transferSocket);
  transferSocket = 0;
  ack = false;
}

/*
 * Change log :
 * 
 * $Log: TcpConnection.cpp,v $
 * Revision 1.5  2004/05/28 15:47:56  thomas
 * back-ported wlan-fix from gt2004_wm for better stability of wlan-communication
 *
 * Revision 1.4  2004/04/14 14:47:57  roefer
 * Only wait one second if a robot cannot be connected
 *
 * Revision 1.3  2004/01/04 16:52:18  roefer
 * Anpassung an VC 2003.NET
 *
 * Revision 1.2  2004/01/03 19:15:03  wachter
 * Debug-communication without router working now
 *
 * Revision 1.1  2003/10/07 10:07:00  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/08/17 18:35:37  roefer
 * Communication with router standardized and synchronized
 *
 * Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.4  2003/04/07 13:50:57  timrie
 * Added conditional compile for type of addrlen (signed on cygwin, unsigned on
 * Linux)
 *
 * Revision 1.3  2002/10/11 11:37:24  roefer
 * Router is working now
 *
 * Revision 1.2  2002/10/07 11:21:10  dueffert
 * includes corrected for gcc3.2
 *
 * Revision 1.1  2002/09/10 15:40:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/08/22 14:41:03  risler
 * added some doxygen comments
 *
 * Revision 1.2  2002/08/04 17:53:18  roefer
 * SimGT2002 connection to physical robots added
 *
 * Revision 1.1  2002/08/01 12:52:27  roefer
 * RouterCtrl and TcpConnection added
 *
 */
