/** 
* @file  Platform/Win32/SystemCall.cpp
* @brief static class for system calls, Windows implementation
* @attention this is the Win32 implementation
*
* @author <A href=mailto:brunn@sim.informatik.tu-darmstadt.de>Ronnie Brunn</A>
* @author <A href=mailto:martin@martin-loetzsch.de>Martin Ltzsch</A>
* @author <A href=mailto:risler@sim.informatik.tu-darmstadt.de>Max Risler</A>
* @author <a href=mailto:dueffert@informatik.hu-berlin.de>Uwe Dffert</a>
*/

#include "Platform/SystemCall.h"
#ifndef SIMROBOT
#include "RoboCupCtrl.h"
#else
#ifndef ATHAIBOCONTROL
#include "RoboCupCtrl2.h"
#endif
#endif
#include <sys/timeb.h>
#include <winuser.h>

unsigned long SystemCall::getCurrentSystemTime() 
{
#ifndef ATHAIBOCONTROL
  if(RoboCupCtrl::getController())
    return RoboCupCtrl::getController()->getTime();
  else
#endif
  {
    _timeb sysTime;
    _ftime(&sysTime);
    return sysTime.time * 1000 + sysTime.millitm;
  }
}

unsigned long SystemCall::getFreeMem()
{
	/**
	 * this seems to be the best way of memory usage test:
	 * - testing free (instead of used by own process)
	 *   memory is much simpler, getUsedMemory would
	 *   require recursive calls to VirtualQuery()
	 * - checking physical memory depends on current page
   *   file usage or swapping activity, checking virtual
   *   memory gives exact results about changes of memory
   *   usage but without considering available memory
   *   (2GB-x), so pagefile is the best compromise as long
   *   as the size of the pagefile is not changed
	 */
  MEMORYSTATUS m;
  GlobalMemoryStatus(&m);
  return m.dwAvailPageFile;
  /** @todo check out GetProcessMemoryInfo() */
}

unsigned char SystemCall::getRemainingPower()
{
  SYSTEM_POWER_STATUS stat;
  VERIFY(GetSystemPowerStatus(&stat)!=0);
  unsigned char percent=stat.BatteryLifePercent;
  //we may use stat.BatteryLifeTime in seconds
  if (stat.ACLineStatus==1) percent=100;
  return percent;
}

void SystemCall::reboot()
{
  /** we need the SE_SHUTDOWN_NAME privilege to be able to reboot */
  HANDLE hToken;
  OpenProcessToken(GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, &hToken);

  LUID privValue;
  LookupPrivilegeValue(NULL, SE_SHUTDOWN_NAME, &privValue );

  TOKEN_PRIVILEGES tkp;
  tkp.PrivilegeCount = 1;
  tkp.Privileges[0].Luid = privValue;
  tkp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;

  ASSERT(AdjustTokenPrivileges(hToken, FALSE, &tkp, sizeof(tkp), NULL, NULL)!=0);
  CloseHandle(hToken);

  /** now we have the right privileges to reboot */

  //ASSERT(ExitWindowsEx(EWX_REBOOT | EWX_FORCEIFHUNG,0)!=0);
  //people do not want RobotControl to reboot their computer -> so comment it out :-(
  //ASSERT(ExitWindowsEx(EWX_REBOOT,0)!=0);
}

void SystemCall::shutdown()
{
  /** we need the SE_SHUTDOWN_NAME privilege to be able to shutdown */
  HANDLE hToken;
  OpenProcessToken(GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, &hToken);

  LUID privValue;
  LookupPrivilegeValue(NULL, SE_SHUTDOWN_NAME, &privValue );

  TOKEN_PRIVILEGES tkp;
  tkp.PrivilegeCount = 1;
  tkp.Privileges[0].Luid = privValue;
  tkp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;

  ASSERT(AdjustTokenPrivileges(hToken, FALSE, &tkp, sizeof(tkp), NULL, NULL)!=0);
  CloseHandle(hToken);

  /** now we have the right privileges to reboot */
  //ASSERT(ExitWindowsEx(EWX_SHUTDOWN | EWX_FORCEIFHUNG,0)!=0);

  //people do not want RobotControl to shutdown their computer -> so comment it out :-(
  //ASSERT(ExitWindowsEx(EWX_SHUTDOWN,0)!=0);
}

void SystemCall::getMacAddress(unsigned char address[6])
{
  memset(address,0,6);
#ifdef SIMROBOT
  if(getRobotDesign() == RobotDesign::ERS7)
    address[5] = 2;
  else if(RoboCupCtrl::getController()->getObjectReference("GT2004")->getChildNode(0)->getName() != "simple") 
    address[5] = 1;
#endif
}

RobotDesign::Design SystemCall::getRobotDesign()
{
#ifdef SIMROBOT
  return RoboCupCtrl::getController()->getRobotDesign();
#else
  return RobotDesign::ERS210;
#endif
}

/*
 * Change log :
 * 
 * $Log: SystemCall.cpp,v $
 * Revision 1.13  2004/04/07 13:00:46  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.2  2004/03/29 09:53:42  risler
 * RobotControl getRobotDesign returns ERS7
 *
 * Revision 1.1.1.1  2004/03/29 08:28:46  Administrator
 * initial transfer from tamara
 *
 * Revision 1.12  2004/04/07 12:29:00  risler
 * ddd checkin after go04 - first part
 *
 * Revision 1.11  2004/03/10 23:55:28  roefer
 * ERS7 support for log files
 *
 * Revision 1.10  2004/02/29 14:53:20  roefer
 * Separate MAC adress for simulated ERS-7
 *
 * Revision 1.9  2004/01/24 14:55:29  loetzsch
 * created ATH AiboControl
 *
 * Revision 1.8  2004/01/21 00:52:24  roefer
 * Include files and first version of simulated ERS-7
 *
 * Revision 1.7  2004/01/17 17:22:07  roefer
 * Does not crash anymore when no robot in scene
 *
 * Revision 1.6  2004/01/15 23:20:17  roefer
 * RobotDesign detection in simulator
 *
 * Revision 1.5  2003/12/31 12:21:25  roefer
 * getRobotDesign added
 *
 * Revision 1.4  2003/10/26 08:57:15  roefer
 * Symbol SIMROBXP removed
 *
 * Revision 1.3  2003/10/21 22:34:38  roefer
 * Neck height of complex Aibo is different
 *
 * Revision 1.2  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/05/03 13:05:49  roefer
 * getMacAddress added
 *
 * Revision 1.5  2003/02/14 09:45:07  dueffert
 * no real shutdown for win32 anymore, people didnt like it
 *
 * Revision 1.4  2002/12/18 16:22:55  dueffert
 * doxygen docu corrected
 *
 * Revision 1.3  2002/12/11 15:17:26  dueffert
 * getRemainingPower(), shutdown(), reboot() implemented
 *
 * Revision 1.2  2002/12/04 13:43:42  dueffert
 * getFreeMem() implemented
 *
 * Revision 1.1  2002/09/10 15:40:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/06/28 10:26:21  roefer
 * OUTPUT is possible in constructors
 *
 * Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.4  2002/04/07 13:50:27  roefer
 * Simulation time mode added
 *
 * Revision 1.3  2001/12/10 17:47:08  risler
 * change log added
 *
 */
