/**
 * @file Platform/Win32/RoboCupCtrl.cpp
 *
 * This file implements the class RoboCupCtrl.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#include "RoboCupCtrl.h"
#include "SimRob95/SimRobot/sim3DWorld.h"
#include "Tools/Streams/InStreams.h"
#include "Representations/Perception/ColorTable64.h"
#include <sys/timeb.h>
#include "Platform/GTAssert.h"
#include "Tools/Location.h"

RoboCupCtrl* RoboCupCtrl::controller = 0;
STRING RoboCupCtrl::robotName;

CONNECT_CONTROLLER_TO_SCENE(RoboCupCtrl,"GT2003");

RoboCupCtrl::RoboCupCtrl()
{
  adaptColors();
  simTime = false;
  time = 8000 - getCurrentSystemTime();

  controller = this;
  obj = ((SIM3DGroup*) GetObject("robots"))->FirstSubObject();
  if(obj)
  {
    SIM3DOBJECT objFirst = obj;
    do
    {
      robotName = STRING("robots.") + obj->GetName();
      SIM3DOBJECT objs[5];
      objs[0] = obj;
      VERIFY(objs[1] = GetObject(robotName + ".left arm.y.x.y.touch"));
      VERIFY(objs[2] = GetObject(robotName + ".right arm.y.x.y.touch"));
      VERIFY(objs[3] = GetObject(robotName + ".left leg.y.x.y.touch"));
      VERIFY(objs[4] = GetObject(robotName + ".right leg.y.x.y.touch"));
      robots.insert(new Robot(obj->GetName(),objs));
      obj = obj->Next;
    }
    while(obj != objFirst);
  }
  obj = 0;

  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    connect(robots[i]);
  
  for(i = robots.getFirst(); i; ++i)
    robots[i].start();

  SetThreadPriority(GetCurrentThread(),THREAD_PRIORITY_IDLE);
}

RoboCupCtrl::~RoboCupCtrl()
{
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    robots[i].announceStop();
  Sleep(1000);
  controller = 0;
}

BOOLEAN RoboCupCtrl::SetActors(STRING& m)
{
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    robots[i].update();
  if(simTime)
    time += 8;
  return TRUE;
}

void RoboCupCtrl::OnBeginDrag(const CLICKINFO& ci)
{
  objDrag = 0;
  VECTOR clickPos = CalcClickPosition(ci);
  if(ci.DistanceTo(GetObject("robots")) != HUGE_VAL) // only true in main object view
  {
    const double threshold = 100;
    SIM3DOBJECT obj = ((SIM3DGroup*) GetObject("balls"))->FirstSubObject();
    if(obj)
    {
      SIM3DOBJECT objFirst = obj;
      do
      {
        posOffset = VECTOR(obj->Transform.Offset.x,obj->Transform.Offset.y,0);
        if(abs(posOffset - clickPos) < threshold)
        {
          objDrag = obj;
          drag = true;
        }
        obj = obj->Next;
      }
      while(!objDrag && obj != objFirst);
    }

    if(!objDrag)
    {
      // search active robots
      obj = ((SIM3DGroup*) GetObject("robots"))->FirstSubObject();
      if(obj)
      {
        SIM3DOBJECT objFirst = obj;
        do
        {
          posOffset = VECTOR(obj->Transform.Offset.x,obj->Transform.Offset.y,0);
          if(abs(posOffset - clickPos) < threshold)
          {
            objDrag = obj;
            drag = true;
          }
          else
          {
            posOffset = obj->Transform * VECTOR(-2 * threshold,0,0);
            posOffset.z = 0;
            if(abs(posOffset - clickPos) < threshold)
            {
              objDrag = obj;
              drag = false;
            }
          }
          obj = obj->Next;
        }
        while(!objDrag && obj != objFirst);
      }
      if(!objDrag)
      {
        // search passive robots
        obj = ((SIM3DGroup*) GetObject("extras"))->FirstSubObject();
        if(obj)
        {
          SIM3DOBJECT objFirst = obj;
          do
          {
            posOffset = VECTOR(obj->Transform.Offset.x,obj->Transform.Offset.y,0);
            if(abs(posOffset - clickPos) < threshold)
            {
              objDrag = obj;
              drag = true;
            }
            else
            {
              posOffset = obj->Transform * VECTOR(-2 * threshold,0,0);
              posOffset.z = 0;
              if(abs(posOffset - clickPos) < threshold)
              {
                objDrag = obj;
                drag = false;
              }
            }
            obj = obj->Next;
          }
          while(!objDrag && obj != objFirst);
        }
      }
    }

    if(objDrag)
    {
      if(drag)
        posOffset = objDrag->Transform.Offset - clickPos;
      else
      {
        posOffset = clickPos - objDrag->Transform.Offset;
        double ignore;
        ExtractAngles(objDrag->Transform.Matrix,ignore,ignore,angleOffset);
        angleOffset *= pi_180;
        angleOffset -= atan2(posOffset.y,posOffset.x);
      }
    }
  }
}

void RoboCupCtrl::OnDrag(const CLICKINFO& ci)
{
  if(objDrag)
  {
    VECTOR clickPos = CalcClickPosition(ci);
    Pose2D pose = getPose2D(objDrag);
    if(drag)
      pose.translation = Vector2<double>((clickPos + posOffset).x,
                                         (clickPos + posOffset).y);
    else
    {
      posOffset = clickPos - objDrag->Transform.Offset;
      pose.rotation = atan2(posOffset.y,posOffset.x) + angleOffset;
    }
    setPose2D(objDrag,pose);
  }
}

void RoboCupCtrl::OnEndDrag(const CLICKINFO& ci)
{
  OnDrag(ci);
}

VECTOR RoboCupCtrl::CalcClickPosition(const CLICKINFO& ci)
{
  VECTOR vMin(-4000,-4000,0),
         vMax(4000,4000,0),
         vScreen = ci.GetClickPos(),
         vScene,
         v;
  for(int i = 0; i < 13; ++i)
  {
    vScene = (vMin + vMax) / 2;
    v = ci.SceneToScreenY(vScene);
    if(v.y > vScreen.y)
      vMax.y = vScene.y;
    else
      vMin.y = vScene.y;
  }
  for(i = 0; i < 20; ++i)
  {
    vScene = (vMin + vMax) / 2;
    v = ci.SceneToScreenY(vScene);
    if(v.x > vScreen.x)
      vMax.x = vScene.x;
    else
      vMin.x = vScene.x;
  }
  return TurnXYMatrix(-ci.GetZ()) * vScene / 1.3;
}

SIM3DOBJECT RoboCupCtrl::getSimRobotObject() const
{
  unsigned threadId = GetCurrentThreadId();
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    for(ProcessList::Pos j = robots[i].getFirst(); j; ++j)
      if(robots[i][j].getId() == threadId)
        return robots[i].getSimRobotObject();
  return obj;
}

int RoboCupCtrl::getRobotIndex() const
{
  unsigned threadId = GetCurrentThreadId();
  int index = 0;
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i,++index)
    for(ProcessList::Pos j = robots[i].getFirst(); j; ++j)
      if(robots[i][j].getId() == threadId)
        return index;
  return robots.getSize(); // process is currently in construction
}

int RoboCupCtrl::getProcessIndex() const
{
  unsigned threadId = GetCurrentThreadId();
  int index = 0;
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i,++index)
    for(ProcessList::Pos j = robots[i].getFirst(); j; ++j, ++index)
      if(robots[i][j].getId() == threadId)
        return index;
  return index + ProcessList::numberOfProcesses; // process is currently in construction
}

Pose2D RoboCupCtrl::getPose2D(SIM3DOBJECT obj)
{
  SYNC;
  double rotation,
         ignore;
  ExtractAngles(obj->Transform.Matrix,ignore,ignore,rotation);
  return Pose2D(double(rotation * pi_180),
                        Vector2<double>(obj->Transform.Offset.x,
                                                       obj->Transform.Offset.y));
}

void RoboCupCtrl::setPose2D(SIM3DOBJECT obj,const Pose2D& pose)
{
  for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
    if(robots[i].getSimRobotObject() == obj)
      break;
  if(i)
  {
    Pose2D p = getPose2D(obj);
    const Robot& robot = robots[i];
    const VECTOR& fl = ((SIM3DGlobal*) robot.getSimRobotObject(1))->GlobalPos.Offset,
                & fr = ((SIM3DGlobal*) robot.getSimRobotObject(2))->GlobalPos.Offset,
                & bl = ((SIM3DGlobal*) robot.getSimRobotObject(3))->GlobalPos.Offset,
                & br = ((SIM3DGlobal*) robot.getSimRobotObject(4))->GlobalPos.Offset;
    VECTOR front = fl.z < fr.z ? fl : fr,
           back = bl.z < br.z ? bl : br;
    Vector2<double> poseFront = 
      (Pose2D(Vector2<double>(front.x,front.y)) - p).translation,
                                   poseBack = 
      (Pose2D(Vector2<double>(back.x,back.y)) - p).translation;
    double angle = atan2(front.z - back.z,fabs(poseBack.x - poseFront.x));
    SYNC;
    double ax,ay,az;
    ExtractAngles(obj->Transform.Matrix,ax,ay,az);
    if(front.z < back.z)
      obj->Transform.Offset.z -= front.z - 30;
    else
      obj->Transform.Offset.z -= back.z - 30;
    if(obj->Transform.Offset.z < 50)
      obj->Transform.Offset.z = 50;
    obj->Transform.Matrix = TurnXYMatrix(pose.rotation * 180 / pi) *
                            TurnXZMatrix(ay + angle * 180 / pi);
    if(robot.isSwitchActivated(SensorData::accelerationY))
      obj->Transform.Matrix = obj->Transform.Matrix * TurnYZMatrix(90);
  }
  else
  {
    SYNC;
    obj->Transform.Matrix = TurnXYMatrix(pose.rotation * 180 / pi);
  }
  obj->Transform.Offset.x = pose.translation.x;
  obj->Transform.Offset.y = pose.translation.y;
  obj->MakeStep(GetObject("")->Transform);
  obj->MakeNewBoundingBox();
  obj->RegisterBoundingBox();
  obj->MakeTimeStep(GetObject("")->Transform);
}

void RoboCupCtrl::connect(Robot& robot)
{
  const char* gatewaySenderName = "TCPGateway.Sender.",
            * gatewayReceiverName = "TCPGateway.Receiver.";

  // open connect.cfg for actual process layout
  char conncfg[512];
  sprintf(conncfg,"%s/Src/Processes/%s/connect.cfg",File::getGTDir(),LAYOUT);
  InConfigFile stream(conncfg,"ERS-210");
  ASSERT(stream.exists());

  // attach receivers to senders
  char senderName[NAME_LENGTH_MAX],
       receiverName[NAME_LENGTH_MAX];
  while(!stream.eof())
  {
    stream >> senderName >> receiverName;
    SenderList* sender = 0;
    if(strncmp(senderName,gatewaySenderName,strlen(gatewaySenderName)) == 0)
    {
      STRING packageName = STRING(&senderName[strlen(gatewaySenderName)]);
      packageName = packageName.First(packageName.Length() - 2);
      SIM3DOBJECT obj = robot.getSimRobotObject();
      if(obj)
      {
        STRING netReceiverName = obj->ElementName + ".Receiver." +
                                 packageName + ".O";
        char robconcfg[512];
        sprintf(robconcfg,"%s/Bin/connect-for-RobotControl.cfg",File::getGTDir());
        STRING netSenderName = getSenderName(robconcfg, netReceiverName);
        if(netSenderName.Length())
          for(List<Robot>::Pos i = robots.getFirst(); i; ++i)
          {
            SIM3DOBJECT obj = robots[i].getSimRobotObject();
            if(obj)
            {
              STRING otherRobotName = obj->ElementName + ".";
              if(strncmp(netSenderName,otherRobotName,otherRobotName.Length()) == 0)
              {
                STRING otherSenderName = 
                  getSenderName(conncfg, STRING(gatewayReceiverName) + packageName + ".O");
                if(otherSenderName.Length())
                  sender = robots[i].getSender(otherSenderName);
                break;
              }
            }
          }
      }
    }
    else
      sender = robot.getSender(senderName);

    if(strncmp(receiverName,gatewayReceiverName,strlen(gatewayReceiverName)) != 0)
      Robot::connect(sender,robot.getReceiver(receiverName));
  }
}

STRING RoboCupCtrl::getSenderName(const char* fileName,const char* searchedReceiverName)
{
  // open connect.cfg for actual process layout
  InConfigFile stream(fileName,"ERS-210");
  if(stream.exists())
  {
    char senderName[NAME_LENGTH_MAX],
         receiverName[NAME_LENGTH_MAX];
    while(!stream.eof())
    {
      stream >> senderName >> receiverName;
      if(strcmp(receiverName,searchedReceiverName) == 0)
        return senderName;
    }
  }
  OutputDebugString(searchedReceiverName);
  OutputDebugString(" not found\n");
  return "";
}

unsigned RoboCupCtrl::getCurrentSystemTime() const
{
  _timeb sysTime;
  _ftime(&sysTime);
  return sysTime.time * 1000 + sysTime.millitm;
}

unsigned RoboCupCtrl::getTime() const
{
  if(simTime)
    return unsigned(time);
  else
    return unsigned(getCurrentSystemTime() + time);
}

void RoboCupCtrl::adaptColors()
{
  ColorTable64 colorTable;
  InBinaryFile stream(getLocation().getFilename("coltable.c64"));
  ASSERT(stream.exists());
  stream >> colorTable;
  int ySum[numOfColors],
      uSum[numOfColors],
      vSum[numOfColors],
      count[numOfColors];
  for(int c = 0; c < numOfColors; ++c)
    ySum[c] = uSum[c] = vSum[c] = count[c] = 0;
  for(int y = 0; y < 255; y += 4)
    for(int u = 0; u < 255; u += 4)
      for(int v = 0; v < 255; v += 4)
      {
        colorClass c = colorTable.getColorClass(y,u,v);
        ASSERT(unsigned(c) < numOfColors);
        ySum[c] += y;
        uSum[c] += u;
        vSum[c] += v;
        ++count[c];
      }
  for(c = 1; c < numOfColors; ++c)
    if(count[c])
    {
      ySum[c] /= count[c];
      uSum[c] /= count[c];
      vSum[c] /= count[c];
      if(colorTable.getColorClass(ySum[c],uSum[c],vSum[c]) != c)
      {
        int yCenter = ySum[c],
            uCenter = uSum[c],
            vCenter = vSum[c],
            minDist = 1000000;
        for(int y = 0; y < 255; y += 4)
          for(int u = 0; u < 255; u += 4)
            for(int v = 0; v < 255; v += 4)
            {
              if(colorTable.getColorClass(y,u,v) == c)
              {
                int d = (y - yCenter) * (y - yCenter) +
                        (u - uCenter) * (u - uCenter) +
                        (v - vCenter) * (v - vCenter);
                if(d < minDist)
                {
                  minDist = d;
                  ySum[c] = y;
                  uSum[c] = u;
                  vSum[c] = v;
                }
              }
            }
        
      }
      if((colorClass)c != yellowOrange)
      {
        SURFACEDESCR* s = ((SIM3DWorld*) GetObject(""))->FindSurface(ColorClasses::getColorName((colorClass) c));
        ASSERT(s);
        SENSCLASSVECTOR& scv = s->Color;
        scv[0] = SHORTREAL(ySum[c] / 255.0);
        scv[1] = SHORTREAL(uSum[c] / 255.0);
        scv[2] = SHORTREAL(vSum[c] / 255.0);
      }
    }
  SURFACEDESCR* s = ((SIM3DWorld*) GetObject(""))->FindSurface("frame");
  ASSERT(s);
  SENSCLASSVECTOR& scv = s->Color;
  scv[0] = SHORTREAL(ySum[white] / 255.0 * 0.8);
  scv[1] = SHORTREAL(uSum[white] / 255.0);
  scv[2] = SHORTREAL(vSum[white] / 255.0);
}

/*
 * Change log :
 *
 * $Log: RoboCupCtrl.cpp,v $
 * Revision 1.3  2003/11/03 20:22:34  juengel
 * Added color class yellowOrange
 *
 * Revision 1.2  2003/10/26 08:57:15  roefer
 * Symbol SIMROBXP removed
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/26 15:30:28  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.2  2003/07/02 10:27:03  loetzsch
 * change thread priorities
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.12  2003/05/11 17:03:00  risler
 * added location.cfg
 *
 * Revision 1.11  2003/04/18 15:21:25  roefer
 * Robot name handling changed
 *
 * Revision 1.10  2003/04/16 07:00:16  roefer
 * Bremen GO checkin
 *
 * Revision 1.10  2003/04/09 18:30:29  roefer
 * Nonsense in drag and drop corrected
 *
 * Revision 1.9  2003/03/24 09:42:02  roefer
 * Reduction of thread priority removed for RobotControl
 *
 * Revision 1.8  2003/03/23 17:08:18  roefer
 * New player selection in simulation
 *
 * Revision 1.7  2003/02/21 16:34:16  dueffert
 * common pi in own code, warnings removed, platform compatibility restored
 *
 * Revision 1.6  2003/02/18 19:00:51  roefer
 * Bug fixed when playing log files
 *
 * Revision 1.5  2002/11/19 15:43:03  dueffert
 * doxygen comments corrected
 *
 * Revision 1.4  2002/11/18 17:32:35  dueffert
 * RobotControl should be startable in any path now
 *
 * Revision 1.3  2002/11/02 17:16:03  roefer
 * Test for bar removed
 *
 * Revision 1.2  2002/09/22 18:40:55  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.1  2002/09/10 15:40:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.5  2002/07/28 15:43:22  roefer
 * gc kickOff corrected
 *
 * Revision 1.4  2002/07/23 13:39:39  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - new design of debugging architecture
 * - exchanged StaticQueue with MessageQueue
 *
 * Revision 1.3  2002/07/09 21:03:02  roefer
 * Extract simulation colors from color table
 *
 * Revision 1.2  2002/07/09 20:08:26  roefer
 * Extract simulation colors from color table
 *
 * Revision 1.1  2002/06/28 10:26:21  roefer
 * OUTPUT is possible in constructors
 *
 * Revision 1.9  2002/06/07 09:29:37  kosen
 * Oracle paints now multipleBalls.
 *
 * Revision 1.8  2002/06/06 17:01:18  roefer
 * Missing bar does not crash simulator anymore
 *
 * Revision 1.7  2002/06/05 23:10:26  roefer
 * Collaboration challenge bar inserted
 *
 * Revision 1.6  2002/06/04 14:47:40  loetzsch
 * several bug fixes
 *
 * Revision 1.5  2002/06/01 13:47:51  roefer
 * Use of GTMath reduced
 *
 * Revision 1.4  2002/06/01 13:36:48  roefer
 * Inter-robot communication added to RobotControl
 *
 * Revision 1.3  2002/05/24 10:16:36  roefer
 * Simulation calculates tilt and provides it as sensor reading
 *
 * Revision 1.2  2002/05/16 22:36:11  roefer
 * Team communication and GTMath bugs fixed
 *
 * Revision 1.4  2002/05/02 12:12:33  kallnik
 * GTMath
 *
 * Revision 1.3  2002/04/25 14:50:36  kallnik
 * changed double/float to double
 * added several #include GTMath
 *
 * PLEASE use double
 *
 * Revision 1.2  2002/04/23 10:38:29  risler
 * renamed headOdometry to headState
 *
 * Revision 1.1  2002/04/20 15:52:20  roefer
 * Project simpified, WATCH and WATCH_PART added
 *
 * Revision 1.12  2002/04/18 10:25:34  roefer
 * Bremen GO
 *
 * Revision 1.11  2002/04/06 09:55:53  roefer
 * Image and SensorData path through DebugQueues changed
 *
 * Revision 1.10  2002/03/24 18:15:01  loetzsch
 * continued change to blocking sensor data receivers
 *
 * Revision 1.9  2002/02/05 18:08:40  roefer
 * PlatformProcess::getRobotIndex() inserted
 *
 * Revision 1.8  2002/02/05 14:04:30  roefer
 * Oracle has optional parameter for referred robot
 *
 * Revision 1.7  2002/02/05 04:14:36  loetzsch
 * replaced the ASSERT in RoboCupCtrl::getSimRobotObject()
 * by return robots[robots.getFirst()].getSimRobotObject();
 * to give OVirtualRobotComm access to the Oracle
 *
 * Revision 1.6  2002/01/27 16:39:06  roefer
 * Renamed RoboCup.scn to GT2002.scn for aesthetic reasons
 *
 * Revision 1.5  2002/01/25 15:40:14  roefer
 * The Oracle
 *
 * Revision 1.4  2002/01/22 14:39:17  roefer
 * Drag and drop for robots and ball
 *
 * Revision 1.3  2002/01/20 23:34:28  loetzsch
 * Sending images and sensor data to processes running in RobotControl now possible
 *
 * Revision 1.2  2002/01/16 10:51:53  roefer
 * Process index table is cleared on restart of simulation
 *
 * Revision 1.1  2002/01/14 22:40:07  loetzsch
 * SimRobot / RobotControl integration started
 *
 *
 * Revision 1.3  2001/12/10 17:47:08  risler
 * change log added
 *
 */
