/**
 * @file Platform/Win32/Oracle2.cpp
 *
 * This file implements the magic Oracle that knows everything.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#include "RoboCupCtrl2.h"
#include "Tools/Player.h"
#include "Oracle2.h"

Field Oracle::field;

bool Oracle::isBlue(SimObject* me)
{
  return me->getName()[4] > '4';
}

RobotPose Oracle::getRobotPose(SimObject* me)
{
  RoboCupCtrl& ctrl = *RoboCupCtrl::getController();
  Pose2D pose = ctrl.getPose2D(me);
  pose.translation *= 1000;
  RobotPose rp;
  rp.setPose(pose);
  if(isBlue(me))
    rp.setPose(Pose2D(double(pi)) + rp.getPose());
  rp.setValidity(1);

  return rp;
}

BallModel Oracle::getBallPosition(SimObject* me)
{
  BallModel bp;
  RoboCupCtrl& ctrl = *RoboCupCtrl::getController();
  SimObject* balls = ctrl.getObjectReference("GT2004.balls");
  if(balls->getNumberOfChildNodes())
  {
    Pose2D pose = ctrl.getPose2D(balls->getChildNode(0));
    if(isBlue(me))
      pose = (Pose2D(pi) + pose);
    (Vector2<double>&) bp.seen = pose.translation * 1000;
    bp.seen.timeWhenLastSeen = SystemCall::getCurrentSystemTime();
    (Vector2<double>&) bp.communicated = pose.translation;
    bp.communicated.timeWhenLastObserved = bp.seen.timeWhenLastSeen;
    bp.validity = 1.0;
    bp.seen.timeUntilSeenConsecutively = SystemCall::getCurrentSystemTime();
    bp.seen.timeWhenFirstSeenConsecutively = 0;
 }
  return bp;
}

PlayerPoseCollection Oracle::getPlayerPoseCollection(SimObject* me)
{
  PlayerPoseCollection ppc;
  RoboCupCtrl& ctrl = *RoboCupCtrl::getController();
  addPlayers(ppc, me, ctrl.getObjectReference("GT2004.robots"));
  addPlayers(ppc, me, ctrl.getObjectReference("GT2004.extras"));
  return ppc;
}

void Oracle::addPlayers(PlayerPoseCollection& ppc, SimObject* me, SimObject* objs)
{
  for(unsigned int i = 0; i < objs->getNumberOfChildNodes(); ++i)
    if(objs->getChildNode(i) != me)
      addPlayer(ppc, me, objs->getChildNode(i));
}

void Oracle::addPlayer(PlayerPoseCollection& ppc, SimObject* me, SimObject* obj)
{
  Pose2D pose = RoboCupCtrl::getController()->getPose2D(obj);
  pose.translation *= 1000;
  if(isBlue(me))
    pose = Pose2D(double(pi)) + pose;
  PlayerPose pp;
  pp.setPose(pose);
  pp.setValidity(1);

  if(isBlue(me) == isBlue(obj))
  {
    if(ppc.numberOfOwnPlayers < 3)
      ppc.setOwnPlayerPose(ppc.numberOfOwnPlayers++, pp);
  }
  else if(ppc.numberOfOpponentPlayers < 4)
    ppc.setOpponentPlayerPose(ppc.numberOfOpponentPlayers++, pp);
}

ObstaclesModel Oracle::getObstaclesModel(const RobotPose& pose, const PlayerPoseCollection& ppc)
{
  field.placePlayers(ppc);
  ObstaclesModel obstaclesModel;
  int i;

  // set sector distances
  for(i = 0; i < ObstaclesModel::numOfSectors; i++)
  {
    double distance = field.getObstacleDistance(pose + Pose2D(ObstaclesModel::getAngleOfSector(i)), obstaclesModel.obstacleType[i]);
    if(distance > 0)
      obstaclesModel.distance[i] = (int)distance;
    else
    {
      obstaclesModel.distance[i] = ObstaclesModel::maxDistance;
      obstaclesModel.obstacleType[i] = ObstaclesPercept::unknown;
    }
  }

  // calculate corridor
  double closest = ObstaclesModel::maxDistance; 
  for(i = ObstaclesModel::getSectorFromAngle(-pi_2); i < ObstaclesModel::getSectorFromAngle(pi_2); i++)
  {
    Vector2<double> v = (pose + Pose2D(ObstaclesModel::getAngleOfSector(i)) + 
                         Pose2D(Vector2<double>(obstaclesModel.distance[i], 0))).translation;
    if(v.x < closest && v.x > 10 && v.y < 90 && v.y > -90)
      closest = v.x;
  }
  obstaclesModel.corridorInFront = (int) closest;
  obstaclesModel.frameNumber = pose.frameNumber;
  return obstaclesModel;  
}

/*
 * Change log :
 * 
 * $Log: Oracle2.cpp,v $
 * Revision 1.8  2004/03/29 18:40:21  juengel
 * Added timeUntilSeenConsecutively and timeWhenFirstSeenConsecutively to ballModel
 *
 * Revision 1.7  2004/03/01 11:46:32  juengel
 * Moved enum ObstacleType to class ObstaclesPercept.
 *
 * Revision 1.6  2004/02/28 14:14:19  juengel
 * Added ObstacleType to ObstaclesModel.
 *
 * Revision 1.5  2004/02/28 14:00:43  juengel
 * Added ObstacleType to ObstaclesModel.
 *
 * Revision 1.4  2004/02/03 13:20:48  spranger
 * renamed all references to  class BallPosition to BallModel (possibly changed include files)
 *
 * Revision 1.3  2003/12/03 23:59:26  roefer
 * Compatibility with VC2003.NET, GUI works now
 *
 * Revision 1.2  2003/10/23 15:41:40  roefer
 * Oracled obstacle model added
 *
 * Revision 1.1  2003/10/21 15:05:51  roefer
 * Added the oracle
 *
 */
