/** 
* @file Platform/Win32/ForSimRobXP/RobotConsole.cpp
*
* Implementation of RobotConsole.
*
* @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
*/

#include "RobotConsole.h"
#include "ConsoleRoboCupCtrl.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Location.h"
#include "RobotControl/Visualization/DrawingMethods.h"
#include "RobotControl/Visualization/ImageMethods.h"
#include "RobotControl/Visualization/PaintMethodsWin32.h"
#include "Representations/Motion/HeadControlMode.h"
#include "Representations/Motion/HeadMotionRequest.h"
#include "Representations/Motion/MotionRequest.h"
#define HAVE_BOOLEAN
#include "Representations/Perception/JPEGImage.h"
#include "Tools/Debugging/QueueFillRequest.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ColorCorrector.h"
#include "View.h"
#include "Xabsl2View.h"
#include "TimeView.h"
#include "SensorView.h"

class ImageView : public DirectView
{
  private:
    RobotConsole& console;
    const Image*& image;
    const ColorTable64& colorTable;
    const DebugDrawing* drawings;
    const Drawings::ImageDrawing* layers;
    const int numOfLayers;
    const int type;
  
  public:
    ImageView(RobotConsole& c,const Image*& i,const ColorTable64& ct,
      const DebugDrawing* d,const Drawings::ImageDrawing* l,int n,int t)
      : console(c),
        image(i), 
        colorTable(ct),
        drawings(d),
        layers(l),
        numOfLayers(n),
        type(t) {}
    void draw(CDC& dc);
};

void ImageView::draw(CDC& dc)
{
  SYNC_WITH(console);
  if(image)
  {
    dc.SetWindowExt(image->cameraInfo.resolutionWidth,image->cameraInfo.resolutionHeight);
    switch(type)
    {
      case Images::image:
        ImageMethods::paintImage2CDCAsYUV(dc,*image,CRect(0,0,image->cameraInfo.resolutionWidth,
                                                              image->cameraInfo.resolutionHeight));
        break;
      case Images::segmentedImage:
      {
        ColorClassImage colorClassImage;
        colorTable.generateColorClassImage(*image,colorClassImage);
        ImageMethods::paintColorClassImage2CDC(dc,colorClassImage,CRect(0,0,image->cameraInfo.resolutionWidth,
                                                                            image->cameraInfo.resolutionHeight));
        break;
      }
      case Images::correctedImage:
      {
        Image corrected(*image);
        if(corrected.cameraInfo.resolutionWidth == cameraResolutionWidth_ERS7)
          ColorCorrector::correct(corrected);
        ImageMethods::paintImage2CDCAsYUV(dc,corrected,CRect(0,0,corrected.cameraInfo.resolutionWidth,
                                                                  corrected.cameraInfo.resolutionHeight));
        break;
      }
      case Images::correctedSegmentedImage:
      {
        Image corrected(*image);
        if(corrected.cameraInfo.resolutionWidth == cameraResolutionWidth_ERS7)
          ColorCorrector::correct(corrected);
        ColorClassImage colorClassImage;
        colorTable.generateColorClassImage(corrected,colorClassImage);
        ImageMethods::paintColorClassImage2CDC(dc,colorClassImage,CRect(0,0,corrected.cameraInfo.resolutionWidth,
                                                                            corrected.cameraInfo.resolutionHeight));
        break;
      }
    }
  }
  else
  {
    dc.SetWindowExt(cameraResolutionWidth_ERS7,cameraResolutionHeight_ERS7);
    CBrush br(COLORREF(0));
    CRect rect(0, 0, cameraResolutionWidth_ERS7,cameraResolutionHeight_ERS7);
    dc.FillRect(&rect,&br);
  }
  for(int i = 0; i < numOfLayers; ++i)
    PaintMethodsWin32::paintDebugDrawingToCDC(drawings[layers[i]],dc);
}

class FieldView : public DirectView
{
  private:
    RobotConsole& console;
    const DebugDrawing* drawings;
    const Drawings::FieldDrawing* layers;
    const int numOfLayers;
    const Player::teamColor& ownColor;
  
  public:
    FieldView(RobotConsole& c,const DebugDrawing* d,
      const Drawings::FieldDrawing* l,int n,Player::teamColor& color)
      : console(c),
        drawings(d),
        layers(l),
        numOfLayers(n),
        ownColor(color) {}
    void draw(CDC& dc);
};

void FieldView::draw(CDC& dc)
{
  SYNC_WITH(console);
  if(GetDeviceCaps(dc.m_hDC,TECHNOLOGY) != DT_METAFILE)
  {
    CSize size = dc.GetViewportExt();
    dc.SetMapMode(MM_ISOTROPIC);
    dc.SetViewportExt(size);
    dc.SetViewportOrg(size.cx/2,size.cy/2);
  }
  else
    dc.SetWindowOrg(int(-xPosOpponentGoal - flagRadius),
                    int(yPosLeftFlags + flagRadius));
  dc.SetWindowExt((int(xPosOpponentGoal + flagRadius) * 2),
                  int((yPosLeftFlags + flagRadius) * -2));
  for(int i = 0; i < numOfLayers; ++i)
  {
    if(layers[i] == Drawings::fieldPolygons)
    {
      DebugDrawing field;
      DrawingMethods::paintFieldPolygons(field,ownColor);
      PaintMethodsWin32::paintDebugDrawingToCDC(field,dc);
    }
    else if(layers[i] == Drawings::fieldLines)
    {
      DebugDrawing field;
      DrawingMethods::paintFieldLines(field);
      PaintMethodsWin32::paintDebugDrawingToCDC(field,dc);
    }
    else
    {
      PaintMethodsWin32::paintDebugDrawingToCDC(drawings[layers[i]],dc);
    }
  }
}

RobotConsole::RobotConsole(MessageQueue& in,MessageQueue& out)
: Process(in,out),
  debugIn(in), debugOut(out),
  logPlayer(out),
  sr(true),
  logMessages(0)
{
  // this is a hack: call global functions to get parameters
  ctrl = (ConsoleRoboCupCtrl*) RoboCupCtrl::getController();
  robotName = RoboCupCtrl::getRobotFullName();

  teamColor = Player::blue; // will be replaced later
  InBinaryFile(getLocation().getFilename("coltable.c64")) >> colorTable;
  ColorCorrector::load();
  printMessages = true;
  calculateImage = true;
  sendOracle = false;
  xabsl2Error = false;
  destructed = false;
  for(int i = 0; i < Images::numberOfImageIDs; ++i)
    images[i] = 0;
}

RobotConsole::~RobotConsole()
{
  SYNC;
  xabsl2Infos[0].reset();
  xabsl2Infos[1].reset();
  if(logMessages)
    delete [] logMessages;
  for(int i = 0; i < Images::numberOfImageIDs; ++i)
    if(images[i])
      delete images[i];
  destructed = true;
}

void RobotConsole::addViews()
{
  for(List<ImageViewBase*>::Pos iv = ImageViewBase::list.getFirst(); iv; ++iv)
  {
    const char* name;
    Drawings::ImageDrawing* layers;
    int numOfLayers;
    Images::ImageID id;
    ImageViewBase::list[iv]->getInfo(name,layers,numOfLayers,id);
    if(id >= Images::numberOfImageIDs)
      ctrl->addView(new ImageView(*this, images[Images::rawImage], colorTable, imageDrawings, layers, numOfLayers, id),
                    robotName.substr(robotName.find_last_of(".") + 1) + "." + name);
    else
      ctrl->addView(new ImageView(*this, images[id], colorTable, imageDrawings, layers, numOfLayers, Images::image),
                    robotName.substr(robotName.find_last_of(".") + 1) + "." + name);
  }
  
  for(List<FieldViewBase*>::Pos fv = FieldViewBase::list.getFirst(); fv; ++fv)
  {
    const char* name;
    Drawings::FieldDrawing* layers;
    int numOfLayers;
    FieldViewBase::list[fv]->getInfo(name,layers,numOfLayers);
    ctrl->addView(new FieldView(*this, fieldDrawings, layers, numOfLayers, teamColor),
                  robotName.substr(robotName.find_last_of(".") + 1) + "." + name);
  }
  
  ctrl->addView(new Xabsl2View(*this, xabsl2Infos[0]), 
                robotName.substr(robotName.find_last_of(".") + 1) + ".xabsl");
  ctrl->addView(new Xabsl2View(*this, xabsl2Infos[1]), 
                robotName.substr(robotName.find_last_of(".") + 1) + ".xabslHeadControl");
  ctrl->addView(new TimeView(*this, timeInfo), 
                robotName.substr(robotName.find_last_of(".") + 1) + ".timing");
  ctrl->addView(new SensorView(*this, sensorData), 
                robotName.substr(robotName.find_last_of(".") + 1) + ".sensorData");
}

bool RobotConsole::handleMessage(InMessage& message)
{
  // Only one thread can access *this now.
  SYNC;
  if(destructed) // if object is already destructed, abort here
    return true; // avoid further processing of this message
  teamColor = message.getTeamColor();
  logPlayer.handleMessage(message);
  message.resetReadPosition();

  switch(message.getMessageID())
  {
    case idText:
    {
      char* buffer = new char[message.getMessageSize() * 2 + 1];
      message.text.readAll(buffer);
      if(printMessages)
        ctrl->printLn(buffer);
      if(logMessages)
        *logMessages << buffer << std::endl;
      delete [] buffer;
      return true;
    }
    case idImage:
      if(dkt.getMode(DebugKeyTable::sendImage) != DebugKey::disabled)
      {
        if(!images[Images::rawImage])
          images[Images::rawImage] = new Image;
        message.bin >> *images[Images::rawImage] >> cameraMatrix;
        return true;
      }
      else
        return false;
    case idJPEGImage:
    {
      if(dkt.getMode(DebugKeyTable::sendJPEGImage) != DebugKey::disabled)
      {
        if(!images[Images::rawImage])
          images[Images::rawImage] = new Image;
        JPEGImage jpi;
        message.bin >> jpi >> cameraMatrix;
        jpi.toImage(*images[Images::rawImage]);
        return true;
      }
      else
        return false;
    }
    case idDebugImage:
    {
      int id;
      message.bin >> id;
      DebugKeyTable::debugKeyID key;
      if(Images::getDebugKeyID(Images::ImageID(id), key) && dkt.getMode(key) != DebugKey::disabled)
      {
        if(!images[id])
          images[id] = new Image;
        message.bin >> *images[id];
      }
      break;
    }
    case idDebugColorClassImage:
    {
      int id;
      message.bin >> id;
      DebugKeyTable::debugKeyID key;
      if(Images::getDebugKeyID(Images::ImageID(id), key) && dkt.getMode(key) != DebugKey::disabled)
      {
        if(!images[id])
          images[id] = new Image;
        ColorClassImage c;
        message.bin >> c;
        c.convertToImage(*images[id]);
      }
      break;
    }
    case idSensorData:
    {
      SensorDataBuffer sensorDataBuffer;
      message.bin >> sensorDataBuffer;
      sensorData = sensorDataBuffer.lastFrame();
      return true;
    }
    case idWorldState:
      if(dkt.getMode(DebugKeyTable::sendWorldState) != DebugKey::disabled)
      {
        CameraInfo cameraInfo;
        message.bin >> RECEIVE_WORLDSTATE(robotPose,
          ballPosition, playerPoseCollection, obstaclesModel, robotState, cameraMatrix, cameraInfo);
      
        // paint the world state
        DebugDrawing debugDrawing;
        DrawingMethods::paintWorldState(debugDrawing,robotPose,
          ballPosition, playerPoseCollection, message.getTeamColor(), 
          message.getTimeStamp(),false);
        fieldDrawings[Drawings::worldState] = debugDrawing;

        debugDrawing = DebugDrawing();
        DrawingMethods::paintObstaclesModelForFieldView(debugDrawing, obstaclesModel, robotPose);
        fieldDrawings[Drawings::models_obstaclesField] = debugDrawing;

        debugDrawing = DebugDrawing();
        DrawingMethods::paintObstaclesModelForImageView(debugDrawing, obstaclesModel, cameraMatrix, cameraInfo);
        imageDrawings[Drawings::models_obstacles] = debugDrawing;
        return true;
      }
      else
        return false;
    case idOracledWorldState:
    {
      CameraInfo cameraInfo;
      message.bin >> RECEIVE_WORLDSTATE(oracledRobotPose, 
        oracledBallPosition,oracledPlayerPoseCollection, obstaclesModel, robotState, cameraMatrix, cameraInfo);
      DebugDrawing debugDrawing(Drawings::worldStateOracle);
      DrawingMethods::paintWorldState(debugDrawing, oracledRobotPose, 
        oracledBallPosition, oracledPlayerPoseCollection, message.getTeamColor(), 
        message.getTimeStamp(),true);
      fieldDrawings[Drawings::worldStateOracle] = debugDrawing;
      return true;        
    }
    case idPercepts:
      if(dkt.getMode(DebugKeyTable::sendPercepts) != DebugKey::disabled)
      {
        CameraInfo cameraInfo;
        message.bin >> RECEIVE_PERCEPTS(cameraMatrix,cameraInfo,ballPercept,
          landmarksPercept,linesPercept,playersPercept,obstaclesPercept, psdPercept, collisionPercept);

        DebugDrawing debugDrawing;
        DrawingMethods::paintPerceptCollectionForRadarView(debugDrawing,
          landmarksPercept, ballPercept, playersPercept, obstaclesPercept, linesPercept);
        fieldDrawings[Drawings::percepts_ballFlagsGoalsRadar] = debugDrawing;
        
        debugDrawing = DebugDrawing();
        DrawingMethods::paintPerceptCollectionForFieldView(debugDrawing,
          landmarksPercept, ballPercept, playersPercept, obstaclesPercept, linesPercept, 
          robotPose, message.getTeamColor());
        fieldDrawings[Drawings::percepts_ballFlagsGoalsField] = debugDrawing;
        
        debugDrawing = DebugDrawing();
        DrawingMethods::paintBallPerceptForImageView(debugDrawing, ballPercept, cameraMatrix, cameraInfo);
        imageDrawings[Drawings::percepts_ball] = debugDrawing;

        debugDrawing = DebugDrawing();
        DrawingMethods::paintLandmarksPerceptForImageView(debugDrawing, landmarksPercept, cameraMatrix, cameraInfo);
        imageDrawings[Drawings::percepts_flagsGoals] = debugDrawing;

        debugDrawing = DebugDrawing();
        DrawingMethods::paintFreePartOfGoalPerceptForImageView(debugDrawing, obstaclesPercept, cameraMatrix, cameraInfo);
        imageDrawings[Drawings::percepts_freePartOfGoal] = debugDrawing;

        debugDrawing = DebugDrawing();
        DrawingMethods::paintLinesPerceptForImageView(debugDrawing, linesPercept, cameraMatrix, cameraInfo);
        imageDrawings[Drawings::percepts_lines] = debugDrawing;

        debugDrawing = DebugDrawing();
        DrawingMethods::paintObstaclesPerceptForImageView(debugDrawing, obstaclesPercept, cameraMatrix, cameraInfo);
        imageDrawings[Drawings::percepts_obstacles] = debugDrawing;

        debugDrawing = DebugDrawing();
        DrawingMethods::paintPSDPerceptForImageView(debugDrawing, psdPercept, cameraMatrix, cameraInfo);
        imageDrawings[Drawings::percepts_psd] = debugDrawing;
        return true;
      }
      else
        return false;
    case idSpecialPercept:
      if(dkt.getMode(DebugKeyTable::sendSpecialPercept) != DebugKey::disabled)
      {
        Player player;
        SpecialPercept specialPercept;
        CameraMatrix cameraMatrix;
        message.bin >> player >> specialPercept >> cameraMatrix;

        DebugDrawing debugDrawing;
        DrawingMethods::paintSpecialPerceptForImageView(debugDrawing,
          specialPercept,cameraMatrix);
        imageDrawings[Drawings::percepts_special] = debugDrawing;

        debugDrawing = DebugDrawing();
        DrawingMethods::paintSpecialPerceptForFieldView(debugDrawing,
          specialPercept,robotPose, message.getTeamColor());
        fieldDrawings[Drawings::percepts_specialField] = debugDrawing;
        return true;
      }
      else
        return false;
    case idDebugDrawing:
    {
      DebugDrawing debugDrawing;
      message.bin >> debugDrawing;
      DebugKeyTable::debugKeyID temp;
      if(debugDrawing.typeOfDrawing == Drawings::drawingOnImage &&
         Drawings::getDebugKeyID(debugDrawing.imageDrawingID, temp) &&
         dkt.getMode(temp) != DebugKey::disabled)
        imageDrawings[debugDrawing.imageDrawingID] = debugDrawing;
      else if (debugDrawing.typeOfDrawing == Drawings::drawingOnField &&
         Drawings::getDebugKeyID(debugDrawing.fieldDrawingID, temp) &&
         dkt.getMode(temp) != DebugKey::disabled)
        fieldDrawings[debugDrawing.fieldDrawingID] = debugDrawing;
      return true;
    }
    case idDebugDrawing2:
    {
      char shapeType,
           id,
           typeOfDrawing;
      message.bin >> shapeType >> id >> typeOfDrawing;
 
      if((Drawings::TypeOfDrawing)typeOfDrawing == Drawings::drawingOnImage)
        id += Drawings::numberOfFieldDrawings;

      incompleteDrawings[id].addShapeFromQueue(message, (Drawings::ShapeType)shapeType);
      
      return true;
    }
    case idDebugDrawingFinished:
    {
      char id, typeOfDrawing;
      message.bin >> id;
      message.bin >> typeOfDrawing;
      DebugKeyTable::debugKeyID temp;
      if((Drawings::TypeOfDrawing)typeOfDrawing == Drawings::drawingOnField)
      {
        incompleteDrawings[id].fieldDrawingID = (Drawings::FieldDrawing)id;
        incompleteDrawings[id].typeOfDrawing = Drawings::drawingOnField;
        if(Drawings::getDebugKeyID(Drawings::FieldDrawing(id), temp) &&
           dkt.getMode(temp) != DebugKey::disabled)
          fieldDrawings[id] = incompleteDrawings[id];
        incompleteDrawings[id].reset();
      }
      else
      {
        incompleteDrawings[id + Drawings::numberOfFieldDrawings].imageDrawingID = (Drawings::ImageDrawing)id;
        incompleteDrawings[id + Drawings::numberOfFieldDrawings].typeOfDrawing = Drawings::drawingOnImage;
        if(Drawings::getDebugKeyID(Drawings::ImageDrawing(id), temp) &&
           dkt.getMode(temp) != DebugKey::disabled)
          imageDrawings[id] = incompleteDrawings[id + Drawings::numberOfFieldDrawings];
        incompleteDrawings[id + Drawings::numberOfFieldDrawings].reset();
      }
      return true;
    }
    case idXabsl2DebugMessage:
      if(!xabsl2Infos[0].handleMessage(message))
      {
        message.resetReadPosition();
        if(!xabsl2Infos[1].handleMessage(message) && !xabsl2Error)
        {
          ctrl->printLn("Wrong Xabsl2 behavior loaded");
          xabsl2Error = true;
        }
      }
      return true;
    case idStopwatch:
      return timeInfo.handleMessage(message);
  }
  return false;
}

void RobotConsole::handleConsole(InConfigMemory& stream)
{
  char command[80];
  stream >> command;
  bool result = false;
  if(!strcmp(command,"msg"))
    result = msg(stream);
  else if(!strcmp(command,"ci"))
    result = calcImage(stream);
  else if(!strcmp(command,"cp"))
    result = cameraParameters(stream);
  else if(!strcmp(command,"dk"))
    result = debugKey(stream);
  else if(!strcmp(command,"sg"))
    result = sendGeneric(stream);
  else if(!strcmp(command,"hcm"))
    result = headControlMode(stream);
  else if(!strcmp(command,"hmr"))
    result = headMotionRequest(stream);
  else if(!strcmp(command,"log"))
    result = log(stream);
  else if(!strcmp(command,"mr"))
    result = sendMotionRequest(stream);
  else if(!strcmp(command,"qfr"))
    result = queueFillRequest(stream);
  else if(!strcmp(command,"so"))
    result = sendOracledWorldStates(stream);
  else if(!strcmp(command,"sr"))
    result = solutionRequest(stream);
  else if(!strcmp(command,"gc"))
    result = gameControl(stream);
  else if(!strcmp(command,"pr"))
    result = penalizeRobot(stream);
  else if(!strcmp(command,"tr"))
    result = sendTailRequest(stream);
  else if(!strcmp(command,"xlb"))
    result = xabslLoadBehavior(stream);
  else if(!strcmp(command,"xis"))
    result = xabslInputSymbol(stream);
  else if(!strcmp(command,"xos"))
    result = xabslOutputSymbol(stream);
  else if(!strcmp(command,"xo"))
    result = xabslOption(stream);
  else if(!strcmp(command,"xbb"))
    result = xabslBasicBehavior(stream);
  if(!result)
    ctrl->printLn("Syntax Error");
}

void RobotConsole::list(const char* text, const char* required)
{
  CString s1(text),
          s2(required);
  s1.MakeUpper();
  s2.MakeUpper();
  if(s1.Find(s2) != -1)
    ctrl->print(std::string(text) + " ");
}

bool RobotConsole::msg(InConfigMemory& stream)
{
  char state[80];
  stream >> state;
  if(!strcmp(state,"off"))
  {
    printMessages = false;
    return true;
  }
  else if(!strcmp(state,"on"))
  {
    printMessages = true;
    return true;
  }
  else if(!strcmp(state,"log"))
  {
    if(logMessages)
      delete logMessages;
    stream >> state;
    std::string name(state);
    if(name.size() == 0)
      return false;
    else 
    {
      if(name.find_first_of('.') == -1)
        name = name + ".txt";
      char buf[FILENAME_MAX];
      if(name[0] != '/' && name[0] != '\\' && name[0] != '.' && (name[0] == 0 || name[1] != ':'))
        sprintf(buf,"%s/Config/Logs/", File::getGTDir());
      else
        buf[0] = 0;
      ASSERT(strlen(buf) + strlen(name.c_str()) < FILENAME_MAX);
      strcat(buf, name.c_str());
      logMessages = new std::fstream(buf, std::ios_base::out);
      return true;
    }
  }
  return false;
}

bool RobotConsole::calcImage(InConfigMemory& stream)
{
  char state[80];
  stream >> state;
  if(!strcmp(state,"off"))
  {
    calculateImage = false;
    return true;
  }
  else if(!strcmp(state,"on"))
  {
    calculateImage = true;
    return true;
  }
  return false;
}

bool RobotConsole::cameraParameters(InConfigMemory& stream)
{
  char wb[80],
       gain[80],
       shutter[80];
  stream >> wb >> gain >> shutter;
  CameraParameters cp;

  if(!strcmp(wb,"indoor"))
    cp.theWhiteBalance = CameraParameters::wb_indoor_mode;
  else if(!strcmp(wb,"outdoor"))
    cp.theWhiteBalance = CameraParameters::wb_outdoor_mode;
  else if(!strcmp(wb,"fluorescent"))
    cp.theWhiteBalance = CameraParameters::wb_fl_mode;
  else
    return false;

  if(!strcmp(gain,"low"))
    cp.theGain = CameraParameters::gain_low;
  else if(!strcmp(gain,"mid"))
    cp.theGain = CameraParameters::gain_mid;
  else if(!strcmp(gain,"high"))
    cp.theGain = CameraParameters::gain_high;
  else
    return false;

  if(!strcmp(shutter,"slow"))
    cp.theShutterSpeed = CameraParameters::shutter_slow;
  else if(!strcmp(shutter,"mid"))
    cp.theShutterSpeed = CameraParameters::shutter_mid;
  else if(!strcmp(shutter,"fast"))
    cp.theShutterSpeed = CameraParameters::shutter_fast;
  else
    return false;

  SYNC;
  debugOut.out.bin << cp;
  debugOut.out.finishMessage(idCameraParameters);
  return true;
}

bool RobotConsole::debugKey(InConfigMemory& stream)
{
  char debugKey[80],
       state[80];
  stream >> debugKey >> state;
  if(!strcmp(debugKey,"?"))
  {
    for(int i = 0; i < DebugKeyTable::numOfDebugKeys; ++i)
      list(DebugKeyTable::getDebugKeyName(DebugKeyTable::debugKeyID(i)), state);
    ctrl->printLn("");
    return true;
  }
  else
    for(int i = 0; i < DebugKeyTable::numOfDebugKeys; ++i)
      if(!strcmp(debugKey,DebugKeyTable::getDebugKeyName(DebugKeyTable::debugKeyID(i))))
        if(!strcmp(state,"off"))
        {
          SYNC;
          dkt.set(DebugKeyTable::debugKeyID(i),DebugKey::disabled);
          debugOut.out.bin << dkt;
          debugOut.out.finishMessage(idDebugKeyTable);

          // clear debug drawings not requested anymore
          switch(DebugKeyTable::debugKeyID(i))
          {
          case DebugKeyTable::sendImage:
          case DebugKeyTable::sendJPEGImage:
            if(dkt.getMode(DebugKeyTable::sendImage) == DebugKey::disabled &&
               dkt.getMode(DebugKeyTable::sendJPEGImage) == DebugKey::disabled &&
               images[Images::rawImage])
            {
              delete images[Images::rawImage];
              images[Images::rawImage] = 0;
            }
            break;
          case DebugKeyTable::sendWorldState:
            fieldDrawings[Drawings::worldState] = DebugDrawing();
            fieldDrawings[Drawings::models_obstaclesField] = DebugDrawing();
            imageDrawings[Drawings::models_obstacles] = DebugDrawing();
            break;
          case DebugKeyTable::sendPercepts:
            fieldDrawings[Drawings::percepts_ballFlagsGoalsRadar] = DebugDrawing();
            fieldDrawings[Drawings::percepts_ballFlagsGoalsField] = DebugDrawing();
            imageDrawings[Drawings::percepts_ball] = DebugDrawing();
            imageDrawings[Drawings::percepts_flagsGoals] = DebugDrawing();
            imageDrawings[Drawings::percepts_freePartOfGoal] = DebugDrawing();
            imageDrawings[Drawings::percepts_lines] = DebugDrawing();
            imageDrawings[Drawings::percepts_obstacles] = DebugDrawing();
            imageDrawings[Drawings::percepts_psd] = DebugDrawing();
            break;
          case DebugKeyTable::sendSpecialPercept:
            imageDrawings[Drawings::percepts_special] = DebugDrawing();
            fieldDrawings[Drawings::percepts_specialField] = DebugDrawing();
            break;
          default:
            {
              DebugKeyTable::debugKeyID key;
              int j;
              for(j = 0; j < Drawings::numberOfImageDrawings; ++j)
                if(Drawings::getDebugKeyID(Drawings::ImageDrawing(j), key) && key == DebugKeyTable::debugKeyID(i))
                  imageDrawings[j] = DebugDrawing();
              for(j = 0; j < Drawings::numberOfFieldDrawings; ++j)
                if(Drawings::getDebugKeyID(Drawings::FieldDrawing(j), key) && key == DebugKeyTable::debugKeyID(i))
                  fieldDrawings[j] = DebugDrawing();
              for(j = 0; j < Images::numberOfImageIDs; ++j)
                if(Images::getDebugKeyID(Images::ImageID(j), key) && key == DebugKeyTable::debugKeyID(i) && images[j])
                {
                  delete images[j];
                  images[j] = 0;
                }
            }
          }
          return true;
        }
        else if(!strcmp(state,"on"))
        {
          SYNC;
          dkt.set(DebugKeyTable::debugKeyID(i),DebugKey::always);
          debugOut.out.bin << dkt;
          debugOut.out.finishMessage(idDebugKeyTable);
          return true;
        }
        else if(!strcmp(state,"every"))
        {
          char ms[80];
          int times;
          stream >> times >> ms;
          if(times)
          {
            SYNC;
            if(!strcmp(ms,"ms"))
              dkt.set(DebugKeyTable::debugKeyID(i),DebugKey::every_n_ms,times);
            else
              dkt.set(DebugKeyTable::debugKeyID(i),DebugKey::every_n_times,times);
            debugOut.out.bin << dkt;
            debugOut.out.finishMessage(idDebugKeyTable);
            return true;
          }
        }
        else if(atoi(state))
        {
          SYNC;
          dkt.set(DebugKeyTable::debugKeyID(i),DebugKey::n_times,atoi(state));
          debugOut.out.bin << dkt;
          debugOut.out.finishMessage(idDebugKeyTable);
          return true;
        }
  return false;
}

bool RobotConsole::gameControl(InConfigMemory& stream)
{
  char state[80];
  stream >> state;
  if(!strcmp(state,"ready"))
  {
    gameControlData.data.state = ROBOCUP_STATE_READY;
    stream >> state;
    Player::teamColor kickOffColor;
    if(!strcmp(state,"blue"))
      kickOffColor = Player::blue;
    else if(!strcmp(state,"red"))
      kickOffColor = Player::red;
    else if(strcmp(state,""))
      return false;
    if(teamColor == kickOffColor)
      gameControlData.data.kickoff = ROBOCUP_KICKOFF_OWN; 
    else
      gameControlData.data.kickoff = ROBOCUP_KICKOFF_OPPONENT; 
    if(!stream.eof())
      if(teamColor == Player::blue)
        stream >> gameControlData.data.ownScore 
               >> gameControlData.data.opponentScore;
      else
        stream >> gameControlData.data.opponentScore 
               >> gameControlData.data.ownScore;
  }
  else if(!strcmp(state,"set"))
    gameControlData.data.state = ROBOCUP_STATE_SET;
  else if(!strcmp(state,"playing"))
    gameControlData.data.state = ROBOCUP_STATE_PLAYING;
  else if(!strcmp(state,"finished"))
    gameControlData.data.state = ROBOCUP_STATE_FINISHED;
  else
    return false;
  SYNC;
  debugOut.out.bin << gameControlData;
  debugOut.out.finishMessage(idGameControlData);
  return true;
}

bool RobotConsole::penalizeRobot(InConfigMemory& stream)
{
  char state[80];
  stream >> state;
  gameControlData.data.state = ROBOCUP_STATE_PENALIZED;
  if(!strcmp(state,"continue"))
  {
    gameControlData.data.state = ROBOCUP_STATE_PLAYING;
    gameControlData.data.penalty = PENALTY_NONE;
  }
  else if(!strcmp(state,"ballHolding"))
    gameControlData.data.penalty = PENALTY_BALL_HOLDING;
  else if(!strcmp(state,"keeperCharged"))
    gameControlData.data.penalty = PENALTY_KEEPER_CHARGE;
  else if(!strcmp(state,"playerCharged"))
    gameControlData.data.penalty = PENALTY_FIELD_PLAYER_CHARGE;
  else if(!strcmp(state,"illegalDefender"))
    gameControlData.data.penalty = PENALTY_ILLEGAL_DEFENDER;
  else if(!strcmp(state,"illegalDefense"))
    gameControlData.data.penalty = PENALTY_ILLEGAL_DEFENSE;
  else if(!strcmp(state,"obstruction"))
    gameControlData.data.penalty = PENALTY_OBSTRUCTION;
  else if(!strcmp(state,"pickup"))
    gameControlData.data.penalty = PENALTY_REQ_FOR_PICKUP;
  else
    return false;
  SYNC;
  debugOut.out.bin << gameControlData;
  debugOut.out.finishMessage(idGameControlData);
  return true;
}

bool RobotConsole::headMotionRequest(InConfigMemory& stream)
{
  SYNC;
  HeadMotionRequest hmr;
  double tilt = 0,
         pan = 0,
         roll = 0,
         mouth = 0;
  stream >> tilt >> pan >> roll >> mouth;
  hmr.tilt = toMicroRad(fromDegrees(tilt));
  hmr.pan = toMicroRad(fromDegrees(pan));
  hmr.roll = toMicroRad(fromDegrees(roll));
  hmr.mouth = toMicroRad(fromDegrees(mouth));
  debugOut.out.bin << hmr;
  debugOut.out.finishMessage(idHeadMotionRequest);
  return true;
}

bool RobotConsole::headControlMode(InConfigMemory& stream)
{
  char mode[80];
  stream >> mode;
  if(!strcmp("?",mode))
  {
    stream >> mode;
    for(int i = 0; i < HeadControlMode::numOfHeadControlModes; ++i) 
      list(HeadControlMode::getHeadControlModeName(HeadControlMode::HeadControlModes(i)), mode);
    ctrl->printLn("");
    return true;
  }
  else
    for(int i = 0; i < HeadControlMode::numOfHeadControlModes; ++i) 
      if(!strcmp(mode,HeadControlMode::getHeadControlModeName(HeadControlMode::HeadControlModes(i))))
      {
        SYNC;
        HeadControlMode hmc;
        hmc.headControlMode = HeadControlMode::HeadControlModes(i);
        debugOut.out.bin << hmc;
        debugOut.out.finishMessage(idHeadControlMode);
        return true;
      }
  return false;
}

bool RobotConsole::log(InConfigMemory& stream)
{
  char command[80];
  stream >> command;
  if(!strcmp("start",command) || !strcmp("stop",command))
  {
    logPlayer.record();
    return true;
  }
  else if(!strcmp("clear",command))
  {
    logPlayer._new();
    return true;
  }
  else if(!strcmp("save",command))
  {
    stream >> command;
    std::string name(command);
    if(name.size() == 0)
      return false;
    else 
    {
      if(name.find_first_of('.') == -1)
        name = name + ".log";
      if(name[0] != '\\' && (name.size() < 2 || name[1] != ':'))
        name = std::string("Logs\\") + name;
      return logPlayer.save(name.c_str());
    }
  }
  else
    return false;
}

bool RobotConsole::queueFillRequest(InConfigMemory& stream)
{
  char request[80];
  stream >> request;
  QueueFillRequest qfr;
  if(!strcmp("queue",request))
    qfr.mode = QueueFillRequest::immediateReadWrite;
  else if(!strcmp("replace",request))
    qfr.mode = QueueFillRequest::overwriteOlder;
  else if(!strcmp("reject",request))
    qfr.mode = QueueFillRequest::rejectAll;
  else if(!strcmp("collect",request))
  {
    qfr.mode = QueueFillRequest::collectNSeconds;
    stream >> qfr.seconds;
    if(!qfr.seconds)
      return false;
  }
  else if(!strcmp("save",request))
  {
    qfr.mode = QueueFillRequest::toStickNSeconds;
    stream >> qfr.seconds;
    if(!qfr.seconds)
      return false;
  }
  else
    return false;
  debugOut.out.bin << qfr;
  debugOut.out.finishMessage(idQueueFillRequest);
  return true;
}

bool RobotConsole::sendMotionRequest(InConfigMemory& stream)
{
  char request[80];
  stream >> request;
  if(!strcmp("?",request))
  {
    stream >> request;
    list("getup ", request);
    for(int i = 0; i < MotionRequest::numOfWalkType; ++i) 
      list(MotionRequest::getWalkTypeName(MotionRequest::WalkType(i)), request);
    for(i = 0; i < MotionRequest::numOfSpecialAction; ++i) 
      list(MotionRequest::getSpecialActionName(MotionRequest::SpecialActionID(i)), request);
    ctrl->printLn("");
    return true;
  }
  else if(!strcmp("getup",request))
  {
    SYNC;
    motionRequest.motionType = MotionRequest::getup;
    debugOut.out.bin << motionRequest;
    debugOut.out.finishMessage(idMotionRequest);
    return true;
  }
  else
  {
    for(int i = 0; i < MotionRequest::numOfWalkType; ++i) 
      if(!strcmp(request,MotionRequest::getWalkTypeName(MotionRequest::WalkType(i))))
      {
        SYNC;
        motionRequest.motionType = MotionRequest::walk;
        motionRequest.walkType = MotionRequest::WalkType(i);
        double rotation;
        stream >> motionRequest.walkParams.translation.x 
          >> motionRequest.walkParams.translation.y
          >> rotation;
        motionRequest.walkParams.rotation = rotation * pi_180;
        debugOut.out.bin << motionRequest;
        debugOut.out.finishMessage(idMotionRequest);
        return true;
      }
    for(i = 0; i < MotionRequest::numOfSpecialAction; ++i) 
      if(!strcmp(request,MotionRequest::getSpecialActionName(MotionRequest::SpecialActionID(i))))
      {
        SYNC;
        motionRequest.motionType = MotionRequest::specialAction;
        motionRequest.specialActionType = MotionRequest::SpecialActionID(i);
        debugOut.out.bin << motionRequest;
        debugOut.out.finishMessage(idMotionRequest);
        return true;
      }
  }
  return false;
}

bool RobotConsole::sendGeneric(InConfigMemory& stream)
{
  GenericDebugData data;
  char buffer[80];
  stream >> buffer;
  if(!strcmp(buffer,"?"))
  {
    stream >> buffer;
    for(int i = 0; i < GenericDebugData::numOfGenericDebugDataIDs; ++i)
      list(GenericDebugData::getGenericDebugDateName(GenericDebugData::GenericDebugDataID(i)), buffer);
    ctrl->printLn("");
    return true;
  }
  else
    for(int i = 0; i < GenericDebugData::numOfGenericDebugDataIDs; ++i)
      if(!strcmp(buffer,GenericDebugData::getGenericDebugDateName(GenericDebugData::GenericDebugDataID(i))))
      {
        data.id = i;
        for(i = 0; i < 10; ++i)
        {
          data.data[i] = GenericDebugData::getDefaultValue(GenericDebugData::GenericDebugDataID(data.id),i);
          stream >> data.data[i];
        }
        debugOut.out.bin << data;
        debugOut.out.finishMessage(GenericDebugData::getMessageID(GenericDebugData::GenericDebugDataID(data.id)));
        return true;
      }
  return false;
}

bool RobotConsole::sendOracledWorldStates(InConfigMemory& stream)
{
  char state[80];
  stream >> state;
  if(!strcmp(state,"off"))
  {
    sendOracle = false;
    return true;
  }
  else if(!strcmp(state,"on"))
  {
    sendOracle = true;
    return true;
  }
  return false;
}

bool RobotConsole::solutionRequest(InConfigMemory& stream)
{
  char module[80],
       solution[80];
  stream >> module >> solution;
  if(!strcmp("?",module))
  {
    for(int i = 0; i < SolutionRequest::numOfModules; ++i)
      list(SolutionRequest::getModuleName(SolutionRequest::ModuleID(i)), solution);
    ctrl->printLn("");
    return true;
  }
  else
    for(int i = 0; i < SolutionRequest::numOfModules; ++i)
      if(!strcmp(module,SolutionRequest::getModuleName(SolutionRequest::ModuleID(i))))
        if(!strcmp("?",solution))
        {
          stream >> solution;
          for(int j = -1; j < SolutionRequest::getNumOfSolutions(SolutionRequest::ModuleID(i)); ++j)
            list(SolutionRequest::getModuleSolutionName(SolutionRequest::ModuleID(i),
                                                        SolutionRequest::ModuleSolutionID(j)), solution);
          ctrl->printLn("");
          return true;
        }
        else if(!strcmp("off", solution))
        {
          SYNC;
          sr.solutions[i] = SolutionRequest::ModuleSolutionID(-1);
          debugOut.out.bin << sr;
          debugOut.out.finishMessage(idSolutionRequest);
          return true;
        }
        else
          for(int j = -1; j < SolutionRequest::getNumOfSolutions(SolutionRequest::ModuleID(i)); ++j)
            if(!strcmp(solution,SolutionRequest::getModuleSolutionName(SolutionRequest::ModuleID(i),
              SolutionRequest::ModuleSolutionID(j))))
            {
              SYNC;
              sr.solutions[i] = SolutionRequest::ModuleSolutionID(j);
              debugOut.out.bin << sr;
              debugOut.out.finishMessage(idSolutionRequest);
              return true;
            }
  return false;
}

bool RobotConsole::sendTailRequest(InConfigMemory& stream)
{
  char request[80];
  stream >> request;
  if(!strcmp("?",request))
  {
    stream >> request;
    for(int i = 0; i < MotionRequest::numOfTailRequests; ++i) 
      list(MotionRequest::getTailRequestName(MotionRequest::TailRequest(i)), request);
    ctrl->printLn("");
    return true;
  }
  else
  {
    for(int i = 0; i < MotionRequest::numOfTailRequests; ++i) 
      if(!strcmp(request,MotionRequest::getTailRequestName(MotionRequest::TailRequest(i))))
      {
        SYNC;
        motionRequest.tailRequest = MotionRequest::TailRequest(i);
        debugOut.out.bin << motionRequest;
        debugOut.out.finishMessage(idMotionRequest);
        return true;
      }
  }
  return false;
}

bool RobotConsole::xabslLoadBehavior(InConfigMemory& stream)
{
  char buffer[80];
  static const int lower[2] = {0, SolutionRequest::numOfXabslBehaviors},
                   upper[2] = {SolutionRequest::numOfXabslBehaviors, SolutionRequest::undefined};
  static const DebugKeyTable::debugKeyID activationKey[2] = 
  {
    DebugKeyTable::sendXabsl2DebugMessagesForBehaviorControl,
    DebugKeyTable::sendXabsl2DebugMessagesForHeadControl
  };
  int index = getXabslIndex(stream, buffer);
  Xabsl2Info& xabsl2Info = xabsl2Infos[index];
  if(!strcmp("?",buffer))
  {
    stream >> buffer;
    for(int i = lower[index]; i < upper[index]; ++i) 
    {
      std::string s = SolutionRequest::getXabsl2EngineFileID(SolutionRequest::xabsl2EngineID(i));
      int p = s.find_first_of("/");
      if(p >= 0)
        s = s.substr(p + 1);
      list(s.c_str(), buffer);
    }
    ctrl->printLn("");
    return true;
  }
  else
  {
    for(int i = lower[index]; i < upper[index]; ++i) 
    {
      std::string s = SolutionRequest::getXabsl2EngineFileID(SolutionRequest::xabsl2EngineID(i));
      int p = s.find_first_of("/");
      if(p >= 0)
        s = s.substr(p + 1);
      if(s == buffer)
      {
        InConfigFile file((std::string("Xabsl2/") + SolutionRequest::getXabsl2EngineFileID(SolutionRequest::xabsl2EngineID(i)) + "-ds.dat").c_str());
        if(!file.exists())
          return false;
        file >> xabsl2Info;
        xabsl2Info.selectedBehaviorControlName = SolutionRequest::getXabsl2EngineIDName(SolutionRequest::xabsl2EngineID(i));
        ((ConsoleRoboCupCtrl*) RoboCupCtrl::getController())->setXabsl2Info(index, xabsl2Info);
        xabsl2Error = false;
        {
          SYNC;
          dkt.set(activationKey[index],DebugKey::always);
          debugOut.out.bin << dkt;
          debugOut.out.finishMessage(idDebugKeyTable);
        }
        char buffer2[80];
        stream >> buffer2;
        if(!strcmp("update",buffer2))
        {
          InBinaryFile file2((std::string("Xabsl2/") + SolutionRequest::getXabsl2EngineFileID(SolutionRequest::xabsl2EngineID(i)) + "-ic.dat").c_str());
          if(!file2.exists())
            return false;
          char s[500000];
          memset(s,0,sizeof(s));
          file2.read(s,sizeof(s) - 1);
          SYNC;
          debugOut.out.bin << (char) i;
          debugOut.out.bin.write(s,strlen(s));
          debugOut.out.finishMessage(idXabsl2IntermediateCode);
        }
        ctrl->updateCommandCompletion();
        return true;
      }
    }
  }
  return false;
}

bool RobotConsole::xabslInputSymbol(InConfigMemory& stream)
{
  char buffer[80];
  Xabsl2Info& xabsl2Info = xabsl2Infos[getXabslIndex(stream, buffer)];
  if(!strcmp("?",buffer))
  {
    stream >> buffer;
    for(List<Xabsl2Info::InputSymbol>::Pos i = xabsl2Info.inputSymbols.getFirst(); i; ++i)
      list(xabsl2Info.inputSymbols[i].name.c_str(), buffer);
    ctrl->printLn("");
    return true;
  }
  else
    for(List<Xabsl2Info::InputSymbol>::Pos i = xabsl2Info.inputSymbols.getFirst(); i; ++i)
    {
      Xabsl2Info::InputSymbol& is = xabsl2Info.inputSymbols[i];
      if(is.name == buffer)
      {
        stream >> buffer;
        if(!strcmp("off",buffer) || !strcmp("on",buffer))
        {
          is.show = !strcmp("on",buffer);
          debugOut.out.bin << xabsl2Info;
          debugOut.out.finishMessage(idXabsl2DebugRequest);
          return true;
        }
      }
    }
  return false;
}

bool RobotConsole::xabslOutputSymbol(InConfigMemory& stream)
{
  char buffer[80];
  Xabsl2Info& xabsl2Info = xabsl2Infos[getXabslIndex(stream, buffer)];
  if(!strcmp("?",buffer))
  {
    stream >> buffer;
    for(List<Xabsl2Info::OutputSymbol>::Pos i = xabsl2Info.outputSymbols.getFirst(); i; ++i)
      list(xabsl2Info.outputSymbols[i].name.c_str(), buffer);
    ctrl->printLn("");
    return true;
  }
  else
    for(List<Xabsl2Info::OutputSymbol>::Pos i = xabsl2Info.outputSymbols.getFirst(); i; ++i)
    {
      Xabsl2Info::OutputSymbol& os = xabsl2Info.outputSymbols[i];
      if(os.name == buffer)
      {
        stream >> buffer;
        if(!strcmp("off",buffer) || !strcmp("on",buffer))
        {
          os.show = !strcmp("on",buffer);
          debugOut.out.bin << xabsl2Info;
          debugOut.out.finishMessage(idXabsl2DebugRequest);
          return true;
        }
        else if(!strcmp("?",buffer))
        {
          stream >> buffer;
          for(List<std::string>::Pos i = os.alternatives.getFirst(); i; ++i)
          {
            if(!strncmp((os.name + ".").c_str(), os.alternatives[i].c_str(), os.name.length() + 1))
              list(os.alternatives[i].substr(os.name.length() + 1).c_str(), buffer);
            else
              list(os.alternatives[i].c_str(), buffer);
          }
          ctrl->printLn("");
          return true;
        }
        else if(!strcmp("unchanged",buffer))
        {
          os.selectedAlternative = List<std::string>::Pos();
          debugOut.out.bin << xabsl2Info;
          debugOut.out.finishMessage(idXabsl2DebugRequest);
          return true;
        }
        else
        {
          for(List<std::string>::Pos i = os.alternatives.getFirst(); i; ++i)
            if(os.alternatives[i] == buffer || os.alternatives[i] == os.name + "." + buffer)
            {
              os.selectedAlternative = i;
              debugOut.out.bin << xabsl2Info;
              debugOut.out.finishMessage(idXabsl2DebugRequest);
              return true;
            }
        }
      }
    }
  return false;
}

bool RobotConsole::xabslOption(InConfigMemory& stream)
{
  char buffer[80];
  Xabsl2Info& xabsl2Info = xabsl2Infos[getXabslIndex(stream, buffer)];
  if(!strcmp("?",buffer))
  {
    stream >> buffer;
    for(List<Xabsl2Info::Option>::Pos i = xabsl2Info.options.getFirst(); i; ++i)
      list(xabsl2Info.options[i].name.c_str(), buffer);
    ctrl->printLn("");
    return true;
  }
  else if(!strcmp("unchanged",buffer))
  {
    xabsl2Info.selectedOption = List<Xabsl2Info::Option>::Pos();
    xabsl2Info.selectedBasicBehavior = List<Xabsl2Info::BasicBehavior>::Pos();
    debugOut.out.bin << xabsl2Info;
    debugOut.out.finishMessage(idXabsl2DebugRequest);
    return true;
  }
  else
    for(List<Xabsl2Info::Option>::Pos i = xabsl2Info.options.getFirst(); i; ++i)
    {
      Xabsl2Info::Option& o = xabsl2Info.options[i];
      if(o.name == buffer)
      {
        xabsl2Info.selectedOption = i;
        xabsl2Info.selectedBasicBehavior = List<Xabsl2Info::BasicBehavior>::Pos();
        for(int j = 0; j < 5; ++j)
          stream >> xabsl2Info.parameters[j];
        debugOut.out.bin << xabsl2Info;
        debugOut.out.finishMessage(idXabsl2DebugRequest);
        return true;
      }
    }
  return false;
}

bool RobotConsole::xabslBasicBehavior(InConfigMemory& stream)
{
  char buffer[80];
  Xabsl2Info& xabsl2Info = xabsl2Infos[getXabslIndex(stream, buffer)];
  if(!strcmp("?",buffer))
  {
    stream >> buffer;
    for(List<Xabsl2Info::BasicBehavior>::Pos i = xabsl2Info.basicBehaviors.getFirst(); i; ++i)
      list(xabsl2Info.basicBehaviors[i].name.c_str(), buffer);
    ctrl->printLn("");
    return true;
  }
  else if(!strcmp("unchanged",buffer))
  {
    xabsl2Info.selectedOption = List<Xabsl2Info::Option>::Pos();
    xabsl2Info.selectedBasicBehavior = List<Xabsl2Info::BasicBehavior>::Pos();
    debugOut.out.bin << xabsl2Info;
    debugOut.out.finishMessage(idXabsl2DebugRequest);
    return true;
  }
  else
    for(List<Xabsl2Info::BasicBehavior>::Pos i = xabsl2Info.basicBehaviors.getFirst(); i; ++i)
    {
      Xabsl2Info::BasicBehavior& bb = xabsl2Info.basicBehaviors[i];
      if(bb.name == buffer)
      {
        xabsl2Info.selectedOption = List<Xabsl2Info::Option>::Pos();
        xabsl2Info.selectedBasicBehavior = i;
        for(int j = 0; j < 5; ++j)
          stream >> xabsl2Info.parameters[j];
        debugOut.out.bin << xabsl2Info;
        debugOut.out.finishMessage(idXabsl2DebugRequest);
        return true;
      }
    }
  return false;
}

int RobotConsole::getXabslIndex(InConfigMemory& stream, char* buffer)
{
  stream >> buffer;
  if(!strcmp(buffer, "hc"))
  {
    stream >> buffer;
    return 1;
  }
  else
    return 0;
}

IMAGE_VIEW(image)
  Drawings::imageProcessor_horizon,
  Drawings::sketch,
  Drawings::imageProcessor_general,
  Drawings::percepts_ball,
  Drawings::percepts_flagsGoals,
  Drawings::percepts_freePartOfGoal,
  Drawings::percepts_lines,
  Drawings::percepts_obstacles,
  Drawings::percepts_psd,
  Drawings::models_obstacles,
  Drawings::percepts_special,
  Drawings::selfLocator
END_VIEW(image)

IMAGE_VIEW(correctedImage)
  Drawings::imageProcessor_horizon,
  Drawings::sketch,
  Drawings::imageProcessor_general,
  Drawings::percepts_ball,
  Drawings::percepts_flagsGoals,
  Drawings::percepts_freePartOfGoal,
  Drawings::percepts_lines,
  Drawings::percepts_obstacles,
  Drawings::percepts_psd,
  Drawings::models_obstacles,
  Drawings::percepts_special,
  Drawings::selfLocator
END_VIEW(correctedImage)

IMAGE_VIEW(segmentedImage)
END_VIEW(segmentedImage)
  
IMAGE_VIEW(correctedSegmentedImage)
END_VIEW(correctedSegmentedImage)
  
IMAGE_VIEW(imageProcessorGeneral)
END_VIEW(imageProcessorGeneral)

IMAGE_VIEW(segmentedImage1)
END_VIEW(segmentedImage1)

FIELD_VIEW(worldState)
  Drawings::fieldPolygons, 
  Drawings::fieldLines, 
  Drawings::selfLocatorField,
  Drawings::ballLocatorField,
  Drawings::worldState, 
  Drawings::models_obstaclesField,
  Drawings::percepts_ballFlagsGoalsField,
  Drawings::percepts_specialField
END_VIEW(worldState)
  
FIELD_VIEW(worldStateOracle)
  Drawings::fieldPolygons, 
  Drawings::fieldLines, 
  Drawings::selfLocatorField,
  Drawings::ballLocatorField,
  Drawings::worldStateOracle,
  Drawings::worldState, 
  Drawings::models_obstaclesField,
  Drawings::percepts_ballFlagsGoalsField,
  Drawings::percepts_specialField
END_VIEW(worldStateOracle)

FIELD_VIEW(radar)
  Drawings::percepts_ballFlagsGoalsRadar
END_VIEW(radar)

/*
* Change log :
*
* $Log: RobotConsole.cpp,v $
* Revision 1.34  2004/05/18 20:39:53  roefer
* Compatibiliy with new Xabsl2 intermediate code format
*
* Revision 1.33  2004/05/18 11:16:54  roefer
* Automatically send Xabsl debug key when behavior is loaded
*
* Revision 1.32  2004/05/18 11:04:54  roefer
* Separate view and optional parameter hc for xabsl head control
*
* Revision 1.31  2004/05/14 22:56:32  roefer
* Adapted to new Xabsl engine
*
* Revision 1.30  2004/04/27 19:39:37  roefer
* Support for debug images added to simulator
*
* Revision 1.29  2004/04/20 13:14:53  roefer
* All console commands now also work outside the start script
*
* Revision 1.28  2004/04/17 21:50:50  roefer
* New color corrector (BB/DDD mix) used in BB2004ImageProcessor and Simulator
* Displaying corrected images as debug image
*
* Revision 1.27  2004/04/12 18:00:45  roefer
* SensorView added
*
* Revision 1.26  2004/04/11 18:51:02  roefer
* Using team color from messages
*
* Revision 1.25  2004/04/09 18:55:16  roefer
* Better structure for views
* cp command added
* Timing view added
*
* Revision 1.24  2004/04/09 11:35:53  roefer
* Bremen Byters German Open check-in
*
* Revision 1.24  2004/03/31 09:51:50  roefer
* BBColorCorrector was loaded in wrong class
*
* Revision 1.23  2004/03/28 12:02:47  roefer
* All drawings can be switched on and off in simulator
*
* Revision 1.22  2004/03/21 19:07:48  roefer
* More realistic camera images for ERS-7
*
* Revision 1.21  2004/03/16 18:52:05  roefer
* Tab completion for console commands
*
* Revision 1.20  2004/03/15 20:39:40  roefer
* Adaptation for new game controller
*
* Revision 1.19  2004/03/15 12:50:31  tim
* Adaptions to new GameController
*
* Revision 1.18  2004/03/10 23:55:27  roefer
* ERS7 support for log files
*
* Revision 1.17  2004/02/09 08:12:46  roefer
* Separate debug drawings for percepts
*
* Revision 1.16  2004/01/28 19:36:48  roefer
* Stupid bug fixed
*
* Revision 1.15  2004/01/27 23:41:12  roefer
* off as synonym for disabled added to command sr
*
* Revision 1.14  2004/01/10 10:09:14  juengel
* Added CameraInfo to and removed Player from (SEND|RECEIVE)_(PERCEPTS|WORLDSTATE).
*
* Revision 1.13  2003/12/30 21:51:19  roefer
* Image size is now 208 x 160. Smaller images are placed in the upper left corner
*
* Revision 1.12  2003/12/30 20:33:53  roefer
* xos adds name of output symbol if required
*
* Revision 1.11  2003/12/29 16:06:33  roefer
* Ability to log text messages sent from robot added
*
* Revision 1.10  2003/12/15 11:49:07  juengel
* Introduced CameraInfo
*
* Revision 1.9  2003/12/09 18:50:01  loetzsch
* correct invocation of DrawingMethods::paintWorldState
*
* Revision 1.8  2003/12/03 23:59:26  roefer
* Compatibility with VC2003.NET, GUI works now
*
* Revision 1.7  2003/11/20 17:28:32  roefer
* xy ? command has optional pattern
*
* Revision 1.6  2003/10/23 15:41:40  roefer
* Oracled obstacle model added
*
* Revision 1.5  2003/10/21 22:37:09  roefer
* so off is default now
*
* Revision 1.4  2003/10/21 15:05:51  roefer
* Added the oracle
*
* Revision 1.3  2003/10/20 14:05:53  roefer
* Views and buttons
*
* Revision 1.2  2003/10/16 15:30:34  goehring
* Doxygen
*
* Revision 1.1  2003/10/14 07:34:16  roefer
* Support files for SimRobXP added, not finished yet
*
*/
