/**
 * @file Platform/Win32/ForSimRobXP/ConsoleRoboCupCtrl.h
 *
 * This file declares the class ConsoleRoboCupCtrl.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#ifndef __ConsoleRoboCupCtrl_H__
#define __ConsoleRoboCupCtrl_H__

#include "../RoboCupCtrl2.h"
#include "PhysicalRobot.h"
#include "Xabsl2Info.h"
#include <map>

/**
 * The class implements the SimRobot controller for RoboCup.
 */
class ConsoleRoboCupCtrl : public RoboCupCtrl
{
  private:
    DECLARE_SYNC;
    List<List<Robot>::Pos> simulated; /**< The currently selected simulated robot. */
    List<List<PhysicalRobot>::Pos> physical; /**< The currently selected physical robot. */
    List<PhysicalRobot> physicalRobots;
    List<std::string> textMessages; /**< A list of all text messages received in the current frame. */
    bool newLine; /**< States whether the last line of text was finished by a new line. */
    int nesting; /**< The number of recursion level during the execution of console files. */
    std::string buttonCommand[32], /**< Commands activated when a certain button is pressed. */
                lastCommand[3]; /**< The last joystick commands calculated. */
    unsigned lastTime; /**< The last time when joystick commands were handled. */
    int tilt, /**< The tilt of the head (for joystick control). */
        pan, /**< The roll of the head (for joystick control). */
        roll, /**< The roll of the head (for joystick control). */
        tiltId, /**< The button assignment to control the tilt of the head. */
        panId, /**< The button assignment to control the pan of the head. */
        rollId; /**< The button assignment to control the roll of the head. */
    static std::string logFile; /**< States whether the current robot constructed shall play back a log file. */
    std::map<std::string,int> completion; /**< A list for command completion. */
    const Xabsl2Info* xabsl2Infos[2]; /**< Pointers to the current Xabsl2 infos. */

    /**
     * The function adds a robot with a certain name to the set of selected robots.
     * @param name The name of the robot.
     * @return Does a robot with the specified name exist?
     */
    bool selectRobot(const char* name);

    /**
     * The function executes the specified file.
     * @param name The file to execute.
     */
    void executeFile(std::string name, bool printError = true);

    /**
     * The function read text from the stream and prints it to the console.
     * @param stream The text stream.
     */
    void echo(InConfigMemory& stream);

    /**
     * The function defines the command to be executed if a certain joystick button is pressed.
     * @param stream The text stream.
     */
    void joystickButtonCommand(InConfigMemory& stream);

    /**
     * The function sets the assignment of joystick controls to control the head.
     * @param stream The text stream.
     */
    void joystickHeadControl(InConfigMemory& stream);

    /**
     * The function prints a help text.
     */
    void help();

    /**
     * The function handles the console input for the "sc" command.
     * @param stream The stream containing the parameters of "sc".
     * @return Returns true if the parameters were correct.
     */
    bool startConnection(InConfigMemory& stream);

    /**
     * The function handles the console input for the "sl" command.
     * @param stream The stream containing the parameters of "sl".
     * @return Returns true if the parameters were correct.
     */
    bool startLogFile(InConfigMemory& stream);

    /**
     * The function creates the map for command completion.
     */
    void createCompletion();

    void handleJoystick();
    std::string calcButtonCommand(unsigned buttons) const;
    std::string calcWalk(unsigned x,unsigned y,unsigned r) const;
    std::string calcHead(unsigned acc,unsigned coolie);
    double convert(unsigned value,double threshold) const;

  public:
    /**
     * Constructor.
     */
    ConsoleRoboCupCtrl();

    /**
     * Destructor.
     */
    ~ConsoleRoboCupCtrl();

    /**
     * The function is called in each simulation step.
     */
    void execute();

    /**
     * The function is called when a movable object has been selected.
     * @param obj The object.
     */
    void onSelected(SimObject* obj);

    /**
     * The function is called when a console command has been entered.
     * @param command The command.
     */
    void onConsoleCommand(const std::string& command);

    /**
      * The function is called when the tabulator key is pressed.
      * It can replace the given command line by a new one.
      * @param command The command.
      * @param forward Complete in forward direction.
      */
    virtual void onConsoleCompletion(std::string& command, bool forward);

    /**
     * The function prints a string into the console window.
     * @param text The text to be printed.
     */
    void print(const std::string& text);

    /**
     * The function prints a string into the console window.
     * Future text will be printed on the next line.
     * @param text The text to be printed.
     */
    void printLn(const std::string& text);

    /**
     * The function returns whether the current robot constructed shall play back a log file.
     * @return Play back a log file?
     */
    static const std::string& getLogFile() {return logFile;}

    /**
     * The function sets the Xabsl2 info used by the command completion.
     * @param xabsl2Info The new Xabsl2 info.
     */
    void setXabsl2Info(int index, const Xabsl2Info& xabsl2Info) {this->xabsl2Infos[index] = &xabsl2Info;}

    /**
    * The function forces an update of the command completion table.
    */
    void updateCommandCompletion() {completion.clear();}

  protected:
    /**
     * The function is called for each line received from the router.
     * As it is called from a different thread, it must take synchronisation
     * into account.
     * @param line The line of text that has been received.
     */
    virtual void onLineReceived(const char* line) {printLn(line);}
};

#endif

/*
 * Change log :
 *
 * $Log: ConsoleRoboCupCtrl.h,v $
 * Revision 1.11  2004/05/18 11:04:53  roefer
 * Separate view and optional parameter hc for xabsl head control
 *
 * Revision 1.10  2004/04/20 13:14:53  roefer
 * All console commands now also work outside the start script
 *
 * Revision 1.9  2004/03/27 15:57:21  roefer
 * Reverse tab completion
 *
 * Revision 1.8  2004/03/16 18:52:05  roefer
 * Tab completion for console commands
 *
 * Revision 1.7  2004/01/04 16:53:53  roefer
 * Adaptation to new communication scheme
 *
 * Revision 1.6  2003/10/25 11:52:20  roefer
 * Simulator is SimRobXP now
 *
 * Revision 1.5  2003/10/21 15:05:51  roefer
 * Added the oracle
 *
 * Revision 1.4  2003/10/17 15:38:59  roefer
 * Comments for old revisions removed
 *
 * Revision 1.2  2003/10/16 15:30:34  goehring
 * Doxygen
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 */
