/** 
 * @file Platform/Win32/ForRobotControl/OVirtualRobotComm.cpp
 *
 * Implementation of OVirtualRobotComm.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#include "OVirtualRobotComm.h"
#include "../RoboCupCtrl.h"
#include "RobotControl.h"
#include "Platform/Win32/Oracle.h"

#include "Representations/Cognition/RobotState.h"
#include "Representations/Cognition/ObstaclesModel.h"
#include "Representations/Perception/CameraMatrix.h"
 
const char* OVirtualRobotComm::jointName[JointData::numOfJoint] =
{
  ".head.y",                       // headTilt
  ".head.y.z",                     // headPan
  ".head.y.z.x",                   // headRoll
  ".head.y.z.x.mouth",             // mouth
  ".head.y.z.x.left ear",          // earL
  ".head.y.z.x.right ear",         // earR
  ".right arm.y",                  // legFR1
  ".right arm.y.x",                // legFR2
  ".right arm.y.x.y",              // legFR3
  ".left arm.y",                   // legFL1
  ".left arm.y.x",                 // legFL2
  ".left arm.y.x.y",               // legFL3
  ".right leg.y",                  // legHR1
  ".right leg.y.x",                // legHR2
  ".right leg.y.x.y",              // legHR3
  ".left leg.y",                   // legHL1
  ".left leg.y.x",                 // legHL2
  ".left leg.y.x.y",               // legHL3
  ".tail.z",                       // tailPan
  ".tail.z.y"                      // tailTilt
};

const char* OVirtualRobotComm::ledName[LEDValue::numOfLED_ERS210] =
{
  ".head.y.z.x.leds.left bottom",  // redLB,
  ".head.y.z.x.leds.left middle",  // greenL,
  ".head.y.z.x.leds.left top",     // redLT,
  ".head.y.z.x.leds.right bottom", // redRB,
  ".head.y.z.x.leds.right middle", // greenR,
  ".head.y.z.x.leds.right top",    // redRT,
  ".head.y.z.x.leds.center",       // greenT,
  ".tail.z.y.blue",                // tailBlue
  ".tail.z.y.orange"               // tailRed
};

OVirtualRobotComm::OVirtualRobotComm() : 
  INIT_DEBUGGING,
  theMotorCommandsReceiver(this,"Effector.OCommandVectorData.O",true),
  INIT_RECEIVER(OdometryData,false),

  theSensorDataBufferSender(this,"Sensor.OSensorFrameVectorData.S",false),
  theImageSender(this,"FbkImageSensor.OFbkImageVectorData.S",false)
{
  // this is a hack: call global functions to get parameters
  ctrl = RoboCupCtrl::getController();
  STRING robotName = RoboCupCtrl::getRobotName();

  VERIFY(objRobot = ctrl->GetObject(robotName));
  
  spCamera = ctrl->GetSensorPort(robotName + ".head.y.z.x.camera.VALUE");
  ASSERT(spCamera.PortObject);
  imageBuffer = ctrl->NewSensorBuffer(spCamera);

  spPsd = ctrl->GetSensorPort(robotName + ".head.y.z.x.PSD.VALUE");
  ASSERT(spPsd.PortObject);

  for(int i = 0; i < JointData::numOfJoint; ++i)
  {
    objJoint[i] = (SIM3DMoveable*) ctrl->GetObject(robotName + jointName[i]);
    target[i] = 0;
  }
  for(i = 0; i < LEDValue::numOfLED_ERS210; ++i)
    objLed[i] = ctrl->GetObject(robotName + ledName[i]);
  for(i = 0; i < SensorData::numOfSensor_ERS210; ++i)
    overruleSensorData[i] = jointDataInvalidValue;

  frameNumber = 1000; // approx. 8 seconds after startup
  lastFrameNumber = frameNumber;

  SIM3DOBJECT obj = RoboCupCtrl::getController()->getSimRobotObject();
  robotNumber = obj ? obj->ElementName[4] - '1' : 0;
  getSimulatedRobots().robots[robotNumber] = this;
  lastHeight = objRobot->Transform.Offset.z / 1000;
  lastSpeed = 0;
}

int OVirtualRobotComm::main() 
{
  {
    // Only one thread can access *this now.
    SYNC;
    
    bool sendOracledWorldState;
    
    if (getSimulatedRobots().getSimRobotDocument()->IsRunning())
    {
      if (getSimulatedRobots().getState(robotNumber) == CRobotControlSimulatedRobots::activeWithoutImages)
      {
        sendOracledWorldState = true;
      }
      else 
      {
        sendOracledWorldState = getSimulatedRobots().getSendOracle();
      }
    }
    else
    {
      sendOracledWorldState = false;
    }
 
    // Receivers cannot be accessed in a synchronized way.
    // Therefore, copy them to member variables.
    motorCommands = theMotorCommandsReceiver;
    odometryData = theOdometryDataReceiver;

    if(frameNumber != lastFrameNumber)
    {
      lastFrameNumber = frameNumber;
      if(frameNumber % 4 == 0)
        theSensorDataBufferSender.send();
      if(frameNumber % 5 == 0)
      {
        if (sendOracledWorldState)
        {
          //calculate an oracled world state
          oracledRobotPose = Oracle::getRobotPose(frameNumber);
          oracledBallPosition = Oracle::getBallPosition(frameNumber);
          oracledPlayerPoseCollection = Oracle::getPlayerPoseCollection(frameNumber);
          ObstaclesModel obstaclesModel = Oracle::getObstaclesModel(oracledRobotPose, oracledPlayerPoseCollection);
          RobotState robotState;
          CameraMatrix cameraMatrix;
          CameraInfo cameraInfo;
          
          OUTPUT(idWorldState,bin,SEND_WORLD_STATE(oracledRobotPose,
            oracledBallPosition,oracledPlayerPoseCollection,obstaclesModel,robotState, cameraMatrix, cameraInfo));
          
          // send the same world state as idOracledWorldState. 
          // That message is then immediately sent back.
          OUTPUT(idOracledWorldState,bin,SEND_WORLD_STATE(oracledRobotPose,
            oracledBallPosition,oracledPlayerPoseCollection,obstaclesModel,robotState, cameraMatrix, cameraInfo));
        }
        else
        {
          // send an empty debug  drawing for oracled world states to erase old oracled world states
          DebugDrawing drawing(Drawings::worldStateOracle);
          debugIn.out.bin << drawing;
          debugIn.out.finishMessage(idDebugDrawing);
        }

        if (getSimulatedRobots().getState(robotNumber) 
          == CRobotControlSimulatedRobots::activeWithImages)
        {
          theImageSender.send();
        }
      }
    }
  }
  theDebugSender.send();

  return 8;
}

void OVirtualRobotComm::update()
{
  // Only one thread can access *this now.
  SYNC;
  // Set joint data
  for(int i = 0; i < JointData::numOfJoint; ++i)
    setJointAngle(i,motorCommands.jointDataBuffer.frame[0].data[i] / 1.0e6);
  
  // Set led values
  for(i = 0; i < LEDValue::numOfLED_ERS210; ++i)
    setLedState(i,(motorCommands.ledValue.data[0]>>i)&1);
  
  if(frameNumber % 5 == 0) 
  {
    if (getSimulatedRobots().getState(robotNumber) 
      == CRobotControlSimulatedRobots::activeWithImages)
    {
      // generate images only if requested
      ctrl->GetSensorValue(spCamera,imageBuffer);
      PSHORTREAL p = imageBuffer;
      for(int c = 0; c < 3; ++c)
      {
        for(int y = 0; y < 144; ++y)
        {
          for(int x = 0; x < 176; ++x)
          {
            theImageSender.image[y][c][x] = (unsigned char) (*p++ * 255.0);
          }
        }
      }
    }
    else
    {
      memset(theImageSender.image,0,sizeof(theImageSender.image));
    }
    theImageSender.cameraInfo.resolutionHeight = cameraResolutionHeight_ERS210;
    theImageSender.cameraInfo.resolutionWidth = cameraResolutionWidth_ERS210;
    theImageSender.frameNumber = frameNumber + 0x10000000;
  }

  // Setup sensor data
  if(theSensorDataBufferSender.numOfFrames == 4)
    for(int i = 0; i < 3; ++i)
      theSensorDataBufferSender.frame[i] = theSensorDataBufferSender.frame[i + 1];
  else
    ++theSensorDataBufferSender.numOfFrames;

  SensorData& s = theSensorDataBufferSender.frame[theSensorDataBufferSender.numOfFrames - 1];
  s.data[SensorData::headTilt] = s.refValue[SensorData::headTilt] = long(getJointAngle(JointData::headTilt) * 1e6);
  s.data[SensorData::headPan] = s.refValue[SensorData::headPan] = long(getJointAngle(JointData::headPan) * 1e6);
  s.data[SensorData::headRoll] = s.refValue[SensorData::headRoll] = long(getJointAngle(JointData::headRoll) * 1e6);
  s.data[SensorData::headBack] = s.refValue[SensorData::headBack] = 0;
  s.data[SensorData::headFront] = s.refValue[SensorData::headFront] = 0;
  SHORTREAL psd;
  ctrl->GetSensorValue(spPsd,&psd);
  // simulate psd delay
  psdBuffer.add(psd);
  psd = psdBuffer.getEntry(9);
  psd = SHORTREAL(67 + (1 - psd) * 900);
  if(psd < 100)
    psd = 100;
  else if(psd > 900)
    psd = 900;
  s.data[SensorData::psd] = s.refValue[SensorData::psd] = long(psd * 1000);
  s.data[SensorData::mouth] = s.refValue[SensorData::mouth] = long(getJointAngle(JointData::mouth) * 1e6);
  s.data[SensorData::chin] = s.refValue[SensorData::chin] = 0;
  s.data[SensorData::legFL1] = s.refValue[SensorData::legFL1] = long(getJointAngle(JointData::legFL1) * 1e6);
  s.data[SensorData::legFL2] = s.refValue[SensorData::legFL2] = long(getJointAngle(JointData::legFL2) * 1e6);
  s.data[SensorData::legFL3] = s.refValue[SensorData::legFL3] = long(getJointAngle(JointData::legFL3) * 1e6);
  // missing: pawFL
  s.data[SensorData::legHL1] = s.refValue[SensorData::legHL1] = long(getJointAngle(JointData::legHL1) * 1e6);
  s.data[SensorData::legHL2] = s.refValue[SensorData::legHL2] = long(getJointAngle(JointData::legHL2) * 1e6);
  s.data[SensorData::legHL3] = s.refValue[SensorData::legHL3] = long(getJointAngle(JointData::legHL3) * 1e6);
  // missing: pawHL
  s.data[SensorData::legFR1] = s.refValue[SensorData::legFR1] = long(getJointAngle(JointData::legFR1) * 1e6);
  s.data[SensorData::legFR2] = s.refValue[SensorData::legFR2] = long(getJointAngle(JointData::legFR2) * 1e6);
  s.data[SensorData::legFR3] = s.refValue[SensorData::legFR3] = long(getJointAngle(JointData::legFR3) * 1e6);
  // missing: pawFR
  s.data[SensorData::legHR1] = s.refValue[SensorData::legHR1] = long(getJointAngle(JointData::legHR1) * 1e6);
  s.data[SensorData::legHR2] = s.refValue[SensorData::legHR2] = long(getJointAngle(JointData::legHR2) * 1e6);
  s.data[SensorData::legHR3] = s.refValue[SensorData::legHR3] = long(getJointAngle(JointData::legHR3) * 1e6);
  // missing: pawHR
  s.data[SensorData::tailPan] = s.refValue[SensorData::tailPan] = long(getJointAngle(JointData::tailPan) * 1e6);
  s.data[SensorData::tailTilt] = s.refValue[SensorData::tailTilt] = long(getJointAngle(JointData::tailTilt) * 1e6);
  // missing: thermo
  s.data[SensorData::back] = 0;

  double ax,ay,az;
  ExtractAngles(objRobot->Transform.Matrix,ax,ay,az);
  double tilt = ay * pi_180,
         height = objRobot->Transform.Offset.z / 1000,
         speed = (height - lastHeight) * 125,
         acc = (speed - lastSpeed);
    if(acc < -5)
      acc = -5;
  lastHeight = height;
  lastSpeed = speed;
  s.data[SensorData::accelerationX] = long(9810000 * sin(tilt));
  s.data[SensorData::accelerationY] = 0;
  s.data[SensorData::accelerationZ] = long(-9810000 * cos(tilt) - acc * 1e6);

  for(i = 0; i < SensorData::numOfSensor_ERS210; ++i)
    if(isSwitchActivated(SensorData::sensors(i)))
      s.data[i] = overruleSensorData[i];

  s.frameNumber = frameNumber;

  Pose2D pose = ctrl->getPose2D(objRobot);
  if(!isSwitchActivated(SensorData::accelerationY))
    pose += odometryData - lastOdometry;
  lastOdometry = odometryData;
  ctrl->setPose2D(objRobot,pose);

  ++frameNumber;
}

/**@todo double ?*/
void OVirtualRobotComm::setJointAngle(int j,double angle)
{
  ASSERT(objJoint[j]);
  objJoint[j]->Command = (angle * 180 / pi - objJoint[j]->ActValue) / objJoint[j]->Factor;
  if(objJoint[j]->Command > 1)
    objJoint[j]->Command = 1;
  else if(objJoint[j]->Command < -1)
    objJoint[j]->Command = -1;
}

bool OVirtualRobotComm::handleMessage(InMessage& message)
{
  SYNC_WITH(getRobotControlApp());
  message >> getQueues().fromSimulatedRobots;

  return true;
}

void OVirtualRobotComm::activateSwitch(SensorData::sensors sensor, bool activate)
{
  if(activate)
    switch(sensor)
    {
      case SensorData::headBack:
      case SensorData::headFront:
        overruleSensorData[sensor] = 400000; // twice the limit, see DefaultSensorDataProcessor
        break;
      case SensorData::mouth:
        overruleSensorData[sensor] = -160000; // twice the limit, see DefaultSensorDataProcessor
        break;
      case SensorData::chin:
      case SensorData::back:
        overruleSensorData[sensor] = 1; // can only be 0 or 1
        break;
      case SensorData::accelerationY:
        overruleSensorData[sensor] = 98100000;
        break;
      default:
        ASSERT(false);
    }
  else
    overruleSensorData[sensor] = jointDataInvalidValue;
}

MAKE_PROCESS(OVirtualRobotComm);

/*
 * Change log :
 *
 * $Log: OVirtualRobotComm.cpp,v $
 * Revision 1.12  2004/05/13 06:56:23  roefer
 * Simulation of PSDs added
 * Special intrisic parameters for simulated images
 *
 * Revision 1.11  2004/01/10 10:09:14  juengel
 * Added CameraInfo to and removed Player from (SEND|RECEIVE)_(PERCEPTS|WORLDSTATE).
 *
 * Revision 1.10  2003/12/31 20:16:13  roefer
 * SensorData for ERS-7
 *
 * Revision 1.9  2003/12/31 14:29:19  roefer
 * Joints and LEDs for ERS-7
 *
 * Revision 1.8  2003/12/15 11:49:07  juengel
 * Introduced CameraInfo
 *
 * Revision 1.7  2003/12/09 19:49:24  loetzsch
 * Renamed some of the main queues of RobotControl.
 *
 * Added possibility to send messages to specific simulated or physical robots.
 *
 * Revision 1.6  2003/12/08 16:48:55  roefer
 * Acceleration limited
 *
 * Revision 1.5  2003/12/07 19:00:11  loetzsch
 * messages from all simulated robots instead of the selected one are moved
 * to the "queue from simulated robot"
 *
 * Revision 1.4  2003/12/06 21:03:16  roefer
 * Improved simulation of z-acceleration
 *
 * Revision 1.3  2003/11/18 15:46:25  dueffert
 * oracle cares about frame numbers now
 *
 * Revision 1.2  2003/10/23 15:41:40  roefer
 * Oracled obstacle model added
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:30:28  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1  2003/09/26 11:34:07  juengel
 * no message
 *
 * Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.31  2003/06/26 17:07:22  risler
 * correct psd sensor delay in simulator
 *
 * Revision 1.30  2003/05/26 08:15:53  juengel
 * Added cameraMatrix to worldState.
 *
 * Revision 1.29  2003/05/03 16:17:25  roefer
 * SensorDataBuffer integration
 *
 * Revision 1.28  2003/05/03 15:26:28  risler
 * added refValue to SensorData
 *
 * Revision 1.27  2003/05/02 18:26:18  risler
 * SensorDataBuffer added
 * replaced SensorData with SensorDataBuffer
 * full SensorData resolution now accessible
 *
 * Revision 1.26  2003/03/23 19:14:20  loetzsch
 * finished GUI for 8 simulated robots
 * - added the passive state
 * - added the state[8] variable to CRobotControlSimulatedRobots
 * - Repaint of object viewer after reset
 * - Resend debug key table after reset
 * - don't stop simulation when selected robot changes
 *
 * Revision 1.25  2003/03/23 17:08:18  roefer
 * New player selection in simulation
 *
 * Revision 1.24  2003/03/22 22:37:51  loetzsch
 * finished GUI and message routing for 8 simulated robots
 * almost all functionallity for simulated robots now is encapsulated in class
 *   CRobotControlSimulatedRobots
 *
 * Revision 1.23  2003/03/22 16:39:34  loetzsch
 * continued GUI for simulated robots
 *
 * Revision 1.22  2003/03/22 14:50:44  jhoffman
 * added comment
 *
 * Revision 1.21  2003/03/20 23:42:17  loetzsch
 * changed cast of AfxGetApp() to CRobotControlApp*
 * to getRobotControlApp()
 *
 * Revision 1.20  2003/03/10 13:58:20  juengel
 * Added ObstaclesPercept and ObstaclesModel
 *
 * Revision 1.19  2003/03/05 17:09:27  loetzsch
 * redesign of the queues and debug key tables in RobotControl
 *
 * Revision 1.18  2003/03/05 11:09:49  loetzsch
 * added class GlobalGameControlData
 * the Game Toolbar now sents GlobalGameControlData
 *
 * Revision 1.17  2003/02/21 16:34:16  dueffert
 * common pi in own code, warnings removed, platform compatibility restored
 *
 * Revision 1.16  2002/12/07 16:40:45  roefer
 * Blocking for theDebugReceiver changed
 *
 * Revision 1.15  2002/12/01 15:26:08  roefer
 * PSD fixed
 *
 * Revision 1.14  2002/11/12 10:49:02  juengel
 * New debug drawing macros - second step
 * -moved /Tools/Debugging/PaintMethods.h and . cpp
 *  to /Visualization/DrawingMethods.h and .cpp
 * -moved DebugDrawing.h and .cpp from /Tools/Debugging/
 *  to /Visualization
 *
 * Revision 1.13  2002/11/11 11:27:10  juengel
 * First Step: New debug drawing macros.
 *
 * Revision 1.12  2002/11/07 15:36:58  risler
 * removed OUTPUT in SolutionRequest streaming operator
 *
 * Revision 1.11  2002/11/06 17:56:16  risler
 * removed sending solution request
 *
 * Revision 1.10  2002/11/06 15:11:58  risler
 * removed sending solution request
 *
 * Revision 1.9  2002/10/10 16:16:10  roefer
 * PSD corrected
 *
 * Revision 1.8  2002/10/10 13:09:50  loetzsch
 * First experiments with the PSD Sensor
 * - SensorDataProcessor now calculates PSDPercept
 * - Added the PerceptBehaviorControl solution PSDTest
 * - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
 *
 * Revision 1.7  2002/09/25 12:05:44  loetzsch
 * removed BarPercept and PatternPercept
 *
 * Revision 1.6  2002/09/25 10:25:13  loetzsch
 * removed the "executeVisionModules" variable
 * from SolutionRequest and ModuleHandler.
 *
 * Revision 1.5  2002/09/22 18:40:55  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.4  2002/09/18 19:52:36  loetzsch
 * the head state is now sent from Motion to Cognition using the package.
 *
 * Revision 1.3  2002/09/18 16:35:12  loetzsch
 * made GT2003 compilable,
 * rechanged message ids
 *
 * Revision 1.2  2002/09/17 23:55:22  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.1  2002/09/10 15:40:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.13  2002/08/22 15:20:10  loetzsch
 * debug messages to the local processes are now sent in a better synchronized way.
 *
 * Revision 1.12  2002/08/13 12:58:41  loetzsch
 * - Added base class CRobotControlToolBar for tool bars
 * - Moved command handling from CRobotControlMainFrame to the tool bars
 * - Added CRobotControlConfigurationManager
 *
 * Revision 1.11  2002/08/07 10:50:33  loetzsch
 * - removed direct access to the dialogs
 * - faster starting of RobotControl application and main window
 * - general clean up
 *
 * Revision 1.10  2002/07/23 13:39:39  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - new design of debugging architecture
 * - exchanged StaticQueue with MessageQueue
 *
 * Revision 1.9  2002/06/28 10:26:21  roefer
 * OUTPUT is possible in constructors
 *
 * Revision 1.8  2002/06/13 11:43:15  risler
 * bar oracle
 *
 * Revision 1.7  2002/06/07 09:29:37  kosen
 * Oracle paints now multipleBalls.
 *
 * Revision 1.6  2002/06/04 23:27:02  loetzsch
 * 4-robots-in-RobotControl related bug fixes and improvements
 *
 * Revision 1.5  2002/06/04 14:47:40  loetzsch
 * several bug fixes
 *
 * Revision 1.4  2002/06/04 00:15:36  loetzsch
 * RobotControl now can simulate four robots.
 *
 * Revision 1.3  2002/05/24 10:16:36  roefer
 * Simulation calculates tilt and provides it as sensor reading
 *
 * Revision 1.2  2002/05/16 22:36:11  roefer
 * Team communication and GTMath bugs fixed
 *
 * Revision 1.39  2002/05/07 15:28:15  dueffert
 * new LED stuff bugfixed
 *
 * Revision 1.38  2002/04/25 20:29:58  roefer
 * New BallPercept and BallPosition, GTMath errors in SimGT2002 fixed
 *
 * Revision 1.37  2002/04/25 14:50:36  kallnik
 * changed double/float to double
 * added several #include GTMath
 *
 * PLEASE use double
 *
 * Revision 1.36  2002/04/23 10:38:29  risler
 * renamed headOdometry to headState
 *
 * Revision 1.35  2002/04/20 15:52:20  roefer
 * Project simpified, WATCH and WATCH_PART added
 *
 * Revision 1.34  2002/04/18 10:25:34  roefer
 * Bremen GO
 *
 * Revision 1.33  2002/04/16 21:36:19  dueffert
 * no message
 *
 * Revision 1.2  2002/04/10 20:53:24  jhoffman
 * oracled ball has actual system time
 *
 * Revision 1.32  2002/04/06 09:55:53  roefer
 * Image and SensorData path through DebugQueues changed
 *
 * Revision 1.31  2002/04/03 15:25:07  roefer
 * Odometry added to oracled RobotPose
 *
 * Revision 1.30  2002/04/03 14:18:52  juengel
 * camera matrix and odometry data added to streaming operator for images
 *
 * Revision 1.29  2002/03/31 10:54:06  roefer
 * Head buttons fixed
 *
 * Revision 1.28  2002/03/29 23:09:53  roefer
 * SensorData initialized to avoid detection of falling down state
 *
 * Revision 1.27  2002/03/29 09:32:42  roefer
 * Buttons for switches and crash in simulator toolbar are now working
 *
 * Revision 1.26  2002/03/24 18:15:00  loetzsch
 * continued change to blocking sensor data receivers
 *
 * Revision 1.25  2002/03/05 13:41:34  roefer
 * Button-API for OVirtualRobotComm
 *
 * Revision 1.24  2002/02/23 16:38:25  risler
 * Added values for acceleration sensors
 *
 * Revision 1.23  2002/02/21 10:38:13  risler
 * abs needs cast to long
 *
 * Revision 1.22  2002/02/07 23:24:19  loetzsch
 * now sends empty debug drawings with id "worldStateOracle" to
 * overwrite old oracled world states when no oracled world states are sent
 *
 * Revision 1.21  2002/02/07 16:29:12  loetzsch
 * no message
 *
 * Revision 1.20  2002/02/06 18:57:58  roefer
 * Simplifications for oracle and debug drawings
 *
 * Revision 1.19  2002/02/06 01:15:40  loetzsch
 * oracled world states are now first send through the local processes
 * before they are painted. (for synchronization with the painting of the
 * calculated world states)
 *
 * Revision 1.18  2002/02/05 20:02:16  risler
 * handleDebugMessage now returns bool, added debug message handling to ImageProcessor
 *
 * Revision 1.17  2002/02/05 18:10:31  roefer
 * no message
 *
 * Revision 1.16  2002/02/05 14:04:30  roefer
 * Oracle has optional parameter for referred robot
 *
 * Revision 1.15  2002/02/05 03:48:40  loetzsch
 * painting of oracled world states in ::update added
 *
 * Revision 1.14  2002/01/30 17:29:55  loetzsch
 * handleDebugMessage um Parameter timestamp erweitert
 *
 * Revision 1.13  2002/01/25 16:54:56  loetzsch
 * Messages from OVirtualRobotComm are sent now in a synchronized way
 *
 * Revision 1.12  2002/01/25 15:40:14  roefer
 * The Oracle
 *
 * Revision 1.11  2002/01/24 18:16:55  loetzsch
 * added SYNC in handeDebugMessages
 *
 * Revision 1.10  2002/01/23 13:50:42  loetzsch
 * setImageAndSensorData changed
 *
 * Revision 1.9  2002/01/23 07:48:36  loetzsch
 * setImageAndSensorData weitergemacht
 *
 * Revision 1.8  2002/01/20 23:34:27  loetzsch
 * Sending images and sensor data to processes running in RobotControl now possible
 *
 * Revision 1.7  2002/01/18 13:08:10  loetzsch
 * Also notifies the colorTable tool
 *
 * Revision 1.6  2002/01/17 15:13:17  roefer
 * Views and debug output in SimGT2002
 *
 * Revision 1.5  2002/01/17 13:35:13  loetzsch
 * queueToProcesses introduced an connected to OVirtualRobotComm
 *
 * Revision 1.4  2002/01/16 17:17:59  loetzsch
 * Process Debug queues integration to RobotControl started
 *
 * Revision 1.3  2002/01/16 12:10:32  roefer
 * Scene update during robot motion fixed
 *
 * Revision 1.2  2002/01/15 23:54:13  loetzsch
 * SolutionRequest introduced
 *
 * Revision 1.1  2002/01/14 22:52:47  loetzsch
 * SimRobot / RobotControl integration started
 *
 *
 * Revision 1.6  2001/12/10 17:47:09  risler
 * change log added
 *
 */
