/**
 * @file Platform/Aperios1.3.2/ProcessFramework.cpp
 * This file implements classes related to processes on the Aperios Platform.
 */
#include "ProcessFramework.h"
#include "IPEndpoint.h"

#ifdef __GNUC__
#include <apsys.h>
/**
 * This is a big hack. For gcc, we need entry points for all callback functions,
 * and it is not possible to generate them automatically. Therefore, we define
 * a pool of 100 entry points here. The array "entry" contains the addresses of
 * the entry points, "link" contains the functions that will actually be called.
 */
static void (*link[ENTRY_TABLE_MAX])(void*);

/**
 * Create 100 entry points.
 */
#define LINK(id) \
  extern "C" void link##id(void* msg) \
  { \
    link[id](msg); \
  } \
  GEN_ENTRY(link##id##Entry,link##id)

#define LROW(n) \
  LINK(n##0); LINK(n##1); LINK(n##2); LINK(n##3); LINK(n##4); LINK(n##5); LINK(n##6); LINK(n##7); LINK(n##8); LINK(n##9)

LROW(); LROW(1); LROW(2); LROW(3); LROW(4); LROW(5); LROW(6); LROW(7); LROW(8); LROW(9);

/**
 * Create and initialize "entry".
 */
#define ENTRY(id) link##id##Entry

#define EROW(n) \
  ENTRY(n##0), ENTRY(n##1), ENTRY(n##2), ENTRY(n##3), ENTRY(n##4), \
  ENTRY(n##5), ENTRY(n##6), ENTRY(n##7), ENTRY(n##8), ENTRY(n##9)

static void (*entry[ENTRY_TABLE_MAX])() =
{
  EROW(), EROW(1), EROW(2), EROW(3), EROW(4), EROW(5), EROW(6), EROW(7), EROW(8), EROW(9)
};

GEN_ENTRY(PrologueEntry,Prologue);

#endif // __GNUC__

int ProcessBase::blockMask = 0,
    ProcessBase::eventMask = 0;
ProcessBase* ProcessBase::theInstance = 0;

ObjectEntry ObjectEntryTable[ENTRY_TABLE_MAX + 1];

void ProcessBase::aperiosInit(OReasonMessage* reasonMsg)
{
  theInstance->Init(OSystemEvent(reasonMsg->reason,reasonMsg->param,reasonMsg->paramSize));
  Return();
}

void ProcessBase::aperiosStart(OReasonMessage* reasonMsg)
{
  theInstance->Start(OSystemEvent(reasonMsg->reason,reasonMsg->param,reasonMsg->paramSize));
  Return();
}

void ProcessBase::aperiosStop(OReasonMessage* reasonMsg)
{
  theInstance->Stop(OSystemEvent(reasonMsg->reason,reasonMsg->param,reasonMsg->paramSize));
  Return();
}

void ProcessBase::aperiosDestroy(OReasonMessage* reasonMsg)
{
  theInstance->Destroy(OSystemEvent(reasonMsg->reason,reasonMsg->param,reasonMsg->paramSize)); 
  Return();
}

void ProcessBase::awakeOnTimer(OReasonMessage* reasonMsg)
{
  setEventId(31);
  Exit();
}

    
void ProcessBase::antListenCont(antEnvMsg* msg)
{
  // When the message was send to the ANT-lib its continuation
  // field was set to the address of the Endpoint. So this calls
  // the listenCont funtion of the right IPEndpoint
  // cout << "ProcessBase::antListenCont " << msg << "\n";
  (static_cast<IPEndpoint*>(msg->continuation))->listenCont(msg);
  // cout << "ProcessBase::antListenCont done \n";
  Return();
}

void ProcessBase::antConnectCont(antEnvMsg* msg)
{
  //cout << "ProcessBase::antConnectCont " << msg << "\n";
  (static_cast<IPEndpoint*>(msg->continuation))->connectCont(msg);
  //cout << "ProcessBase::antConnectCont done \n";
  Return();
}

void ProcessBase::antReceiveCont(antEnvMsg* msg)
{
  //cout << "ProcessBase::antReceiveCont " << msg << "\n";
  (static_cast<IPEndpoint*>(msg->continuation))->receiveCont(msg);
  //cout << "ProcessBase::antReceiveCont done \n ";
  Return();
}  

void ProcessBase::antSendCont(antEnvMsg* msg)
{
  //cout << "ProcessBase::antSendCont " << msg << "\n";
  (static_cast<IPEndpoint*>(msg->continuation))->sendCont(msg);
  //cout << "ProcessBase::antSendCont done \n";
  Return();
}

void ProcessBase::antCloseCont(antEnvMsg* msg)
{
  //cout << "ProcessBase::antCloseCont " << msg << "\n";
  (static_cast<IPEndpoint*>(msg->continuation))->closeCont(msg);
  //cout << "ProcessBase::antCloseCont done \n";
  Return();
}

void ProcessBase::getAntInformation(int*& listenContSelector, int*& sendContSelector,
                 int*& receiveContSelector, int*& closeContSelector, 
                 int*& connectContSelector)
{
    listenContSelector = &antListenContSelector;
    sendContSelector = &antSendContSelector;
    receiveContSelector = &antReceiveContSelector;
    closeContSelector = &antCloseContSelector;
    connectContSelector = &antConnectContSelector;   
}

void ProcessBase::init()
{
  unsigned int id = 0;
  ObjectEntryTable[id].selector = id;
  ObjectEntryTable[id++].entry = (Entry) aperiosInit;
  ObjectEntryTable[id].selector = id;
  ObjectEntryTable[id++].entry = (Entry) aperiosStart;
  ObjectEntryTable[id].selector = id;
  ObjectEntryTable[id++].entry = (Entry) aperiosStop;
  ObjectEntryTable[id].selector = id;
  ObjectEntryTable[id++].entry = (Entry) aperiosDestroy;
  ObjectEntryTable[id].selector = id;
  ObjectEntryTable[id++].entry = (Entry) awakeOnTimer;

  for(ReceiverList* p = getProcess()->getFirstReceiver(); p; p = p->getNext())
    id = p->fillEntryTable(ObjectEntryTable,id);
  for(SenderList* q = getProcess()->getFirstSender(); q; q = q->getNext())
    id = q->fillEntryTable(ObjectEntryTable,id);
  ASSERT(id <= ENTRY_TABLE_MAX);

  // Entrys for IP-networking

  ObjectEntryTable[id].selector = id;
  antListenContSelector = id;
  ObjectEntryTable[id++].entry = (Entry) antListenCont;
  
  ObjectEntryTable[id].selector = id;
  antReceiveContSelector = id;
  ObjectEntryTable[id++].entry = (Entry) antReceiveCont;

  ObjectEntryTable[id].selector = id;
  antSendContSelector = id;
  ObjectEntryTable[id++].entry = (Entry) antSendCont;

  ObjectEntryTable[id].selector = id;
  antCloseContSelector = id;
  ObjectEntryTable[id++].entry = (Entry) antCloseCont;

  ObjectEntryTable[id].selector = id;
  antConnectContSelector = id;
  ObjectEntryTable[id++].entry = (Entry) antConnectCont;

  ASSERT(id <= ENTRY_TABLE_MAX);
 
#ifdef __GNUC__
  // add entry points to callback functions
  for(unsigned int i = 0; i < id; ++i)
  {
    link[i] = (void (*)(void*)) ObjectEntryTable[i].entry;
    ObjectEntryTable[i].entry = (Entry) entry[i];
  }
#endif // __GNUC__

  ObjectEntryTable[id].selector = UNDEF;
  ObjectEntryTable[id].entry = ENTRY_UNDEF;
}

void ProcessBase::nextFrame()
{
  if(theInstance->isRunning())
    theInstance->processNextFrame();
}

void ProcessBase::setBlockingId(int id,bool block)
{
  if(block)
    blockMask |= 1 << id;
  else
    blockMask &= ~(1 << id);
}

void ProcessBase::setEventId(int id)
{
  eventMask |= 1 << id;
  if(blockMask && (eventMask & blockMask) == blockMask)
  {
    blockMask = 0;
    nextFrame();
  }
}

/*
 * Change log :
 * 
 * $Log: ProcessFramework.cpp,v $
 * Revision 1.2  2003/12/02 20:52:41  wachter
 * Added IP-networking to aperios process-framework.
 *
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.4  2003/05/05 14:54:55  dueffert
 * comment corrected
 *
 * Revision 1.3  2003/03/06 11:52:37  dueffert
 * signed comparison warning removed
 *
 * Revision 1.2  2002/12/02 11:00:13  dueffert
 * doxygen docu corrected
 *
 * Revision 1.1  2002/09/10 15:40:04  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/07/06 20:08:09  roefer
 * Advances towards gcc
 *
 * Revision 1.2  2002/07/06 15:28:28  roefer
 * Prologue function removed
 *
 * Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.8  2002/04/21 21:03:07  roefer
 * Again, events block with &-logic
 *
 * Revision 1.7  2002/04/20 15:52:20  roefer
 * Project simpified, WATCH and WATCH_PART added
 *
 * Revision 1.6  2001/12/15 20:32:08  roefer
 * Senders and receivers are now part of the processes
 *
 * Revision 1.5  2001/12/12 13:25:02  roefer
 * Blocking sender fixed
 *
 * Revision 1.4  2001/12/10 17:47:08  risler
 * change log added
 *
 */
