/**
 * @file Platform/Aperios1.2.1/Receiver.h
 * This file contains classes related to receivers.
 * @author Thomas Rfer
 */
#ifndef __RECEIVER_H__
#define __RECEIVER_H__

#ifndef __ProcessFramework_h__
#error Never include this file directly. Include ProcessFramework.h instead.
#endif

#include "Tools/Streams/InStreams.h"
#include "Platform/GTAssert.h"

/**
 * The class is the base class for receivers.
 * A receiver is an object that reads packages from Aperios queues.
 * The class manages a global list of all receivers in an Aperios process.
 */
class ReceiverList : public OObserver
{
  private:
    PlatformProcess* process;   /**< The process this receiver is associated with. */
    ReceiverList* next;         /**< The successor of the current receiver. */
    char name[NAME_LENGTH_MAX]; /**< The name of a receiver without the module's name. */
  
  protected:
    int eventId;                /**< The id of the current receiver in the range [0..30]. */
    bool blocking;              /**< Stores whether this is a blocking receiver. */

  public:
    /**
     * The constructor.
     * @param process The process this receiver is associated with.
     * @param receiverName The Aperios connection name of the receiver without the process name.
     * @param blocking Decides whether this receiver blocks the execution of the next frame
     *                 until it has received a package.
     */
    ReceiverList(PlatformProcess* process,const char* receiverName,bool blocking);

    /**
     * Returns the begin of the list of all receivers.
     * @return The first receiver in the list, or 0 if the list ist empty.
     */
    ReceiverList*& getFirst();

    /**
     * Returns the next receiver in the list.
     * @return The next receiver in the list, or 0 if this receiver is the last one.
     */
    ReceiverList* getNext() const {return next;}

    /**
     * Returns the Aperios connection name of the receiver.
     * @return The Aperios connection name without the process name ("Receiver.type.O")
     */
    const char* getName() const {return name;}

    /**
     * Fills the Aperios entry table for global functions.
     * The function will write two entries into the table.
     * @param entryTable The Aperios entry table.
     * @param id The first empty id in the entry table.
     * @return The new first empty id in the entry table.
     */
    virtual int fillEntryTable(EntryTable* entryTable,int id) = 0;
    
    /**
     * Returns the number of entries required in the entry table for all receivers.
     * @return The required number of entries.
     */
    int getNumberOfTableEntries();
    
    /**
     * Returns whether a new package was received in the current frame.
     * This is always true if this is a blocking receiver.
     * @return Has a new package been received?
     */
    bool receivedNew() const;

    /**
     * The function must be called to finish the current frame.
     */
    void finishFrame();

    /**
     * The functions sets or resets a bit in the blocking mask.
     * After a bit is set in the blocking mask for a certain 
     * sender or receiver, a new frame will not be started before
     * this sender or receiver received an event.
     * @param id The id of the sender or receiver.
     * @param block Should it block or not?
     */
    static void setBlockingId(int id,bool block = true);

    /**
     * The function is called when an event was received. 
     * If this was the last event the process was waiting for, the next
     * frame is started, i.e. NextFrame() is called.
     * @param id The id of the sender or receiver that received an event.
     */
    static void setEventId(int id);
};

/**
 * The class implements a receiver.
 * A receiver is an object that reads packages from an Aperios queue.
 */
template<class T> class Receiver : public ReceiverList, public T
{
  protected:
    static Receiver<T>* theInstance; /**< The only instance of Receiver<T>. */

    /**
     * The function is called when a new package arrived.
     * @param msg An Open-R message that contains the package.
     */    
    virtual void handleMessage(ONotifyMessage& msg)
    {
      //2do:
      //ONotifyEvent event;
      //event.SetIndex(eventId);
      //NotifyHandler(msg,&event);
      //T& data = *static_cast<T*>(this);
      //InBinaryMemory memory(event.RCData(0)->Base(),event.RCData(0)->Size());
      //memory >> data;
      //ASSERT(memory.eof());
      setEventId(eventId);
    }

    /**
     * The Aperios connection handler.
     * @param msg A message provided by Aperios.
     */
    static void aperiosConnect(OConnectMessage* msg)
    {
      theInstance->ConnectHandler(msg);
      Return();
    }

    /**
     * The Aperios handler that is called when a package is received.
     * @param msg A message provided by Aperios that contains the package.
     */
    static void aperiosNotify(ONotifyMessage* msg)
    {
      theInstance->handleMessage(*msg);
      Return();
    }

  public:
    /**
     * The constructor.
     * @param process The process this receiver is associated with.
     * @param receiverName The Aperios connection name of the receiver without the process name.
     * @param blocking Decides whether this receiver blocks the execution of the next frame
     *                 until it has received a package.
     */
    Receiver(PlatformProcess* process,const char* receiverName,bool blocking)
    : ReceiverList(process,receiverName,blocking)
    {
      theInstance = this;
    }

    /**
     * Fills the Aperios entry table for global functions.
     * The function will write two entries into the table.
     * @param entryTable The Aperios entry table.
     * @param id The first empty id in the entry table.
     * @return The new first empty id in the entry table.
     */
    virtual int fillEntryTable(EntryTable* entryTable,int id)
    {
      eventId = (id - ENTRY_TABLE_MIN) / 2;
      ASSERT(eventId <= 31); // should test < 31
      if(blocking)
        setBlockingId(eventId); // setBlocking(eventId,blocking) should done it too
      entryTable->SetEntry(id++,(Entry) aperiosConnect);
      entryTable->SetEntry(id++,(Entry) aperiosNotify);
      return id;
    }
};

template<class T> Receiver<T>* Receiver<T>::theInstance = 0;

#endif

/*
 * Change log :
 * 
 * $Log$
 *
 */
