/**
 * @file Platform/Aperios1.2.1/ProcessFramework.h
 * This file contains classes related to processes on the Aperios Platform.
 * @author Thomas Rfer
 */
#ifndef __ProcessFramework_h__
#define __ProcessFramework_h__

#include <MCOOP.h>
#include <InitInfo.h>
#include <EntryTable.h>
#include <OPENR/OObject.h>
#include <OPENR/OSubject.h>
#include <OPENR/OObserver.h>
#include <string.h>
#include "Platform/SystemCall.h"
#include "Platform/GTAssert.h"

const unsigned int ENTRY_TABLE_MIN = 5,   /**< The default number of entries in an Aperios entry table. */
          ENTRY_TABLE_MAX = 100, /**< The maximum number of entries in an Aperios entry table. */
          NAME_LENGTH_MAX = 100, /**< The maximum length of Aperios connection names. */
          PROCESS_MAX = 1,       /**< The maximum number of processes in this address context. */
          ROBOT_MAX = 1;         /**< The maximum number of robots in this address context. */

class PlatformProcess;
class ProcessBase;

#include "Sender.h"
#include "MotorCommandsSender.h"
#include "SoundDataSender.h"
#include "Receiver.h"

//#include "../Aperios1.3.2/IPEndpoint.h"

/**
 * This class is the platform dependent base class for all processes.
 */
class PlatformProcess
{
  private:
    SenderList* firstSender;     /**< The begin of the list of all senders of this process. */
    ReceiverList* firstReceiver; /**< The begin of the list of all receivers of this process. */

  public:
    /**
     * Constructor.
     */
    PlatformProcess()
    {
      firstSender = 0;
      firstReceiver = 0;
    }

    /**
     * The function returns the begin of list of all senders.
     * Note that the function returns a reference that can be changed.
     * @return A reference to the address of the first element. 
     */
    SenderList*& getFirstSender() {return firstSender;}

    /**
     * The function returns the begin of list of all receivers.
     * Note that the function returns a reference that can be changed.
     * @return A reference to the address of the first element. 
     */
    ReceiverList*& getFirstReceiver() {return firstReceiver;}
    
    /**
     * The function returns the index of the current process.
     * @return Under Aperios, the function always returns 0.
     */    
    static int getIndex() {return 0;}

    /**
     * The function returns the index of the robot associated to the current process.
     * @return Under Aperios, the function always returns 0.
     */    
    static int getRobotIndex() {return 0;}
};

/**
 * The start function for any Aperios process.
 */
extern "C" void Prologue(InitInfo* pInfo);

/**
 * The class is a helper that allows to instantiate a class as an Aperios process.
 * ProcessBase contains the parts that need not to be implemented as a template.
 * It will only be used by the macro MAKE_PROCESS and should never be used directly.
 */
class ProcessBase : public OObject
{
  public:
    static ProcessBase* theInstance; /**< A pointer to the only instance of this class. */
    void getAntInformation(int*& listenContSelector, int*& sendContSelector,
                 int*& receiveContSelector, int*& closeContSelector, 
                 int*& connectContSelector);
    OID* getOID(){
      return (&myOID_);
    }
    
  private:
    static int blockMask,            /**< A mask with bits set for all blocking senders and receivers. */
               eventMask;            /**< A mask with bits set for all senders and receivers that received an event */
    bool running; /**< The flag states whether the process is still running. */

    /**
     * A global Aperios event handler that calls OObject::Init().
     * @param reasonMsg A message provided by Aperios.
     */
    static void aperiosInit(OReasonMessage* reasonMsg);

    /**
     * A global Aperios event handler that calls OObject::Start().
     * @param reasonMsg A message provided by Aperios.
     */
    static void aperiosStart(OReasonMessage* reasonMsg);

    /**
     * A global Aperios event handler that calls OObject::Stop().
     * @param reasonMsg A message provided by Aperios.
     */
    static void aperiosStop(OReasonMessage* reasonMsg);

    /**
     * A global Aperios event handler that calls OObject::Destroy().
     * @param reasonMsg A message provided by Aperios.
     */
    static void aperiosDestroy(OReasonMessage* reasonMsg);

    /**
     * A global event handler for timer events.
     * It is automatically called after a certain time if Process::main()
     * returned a nonzero value. It will note that the timer event occurred,
     * and will start a new frame if there are no other pending event, i.e.
     * waiting receivers.
     * @param reasonMsg A message provided by Aperios.
     */
    static void awakeOnTimer(OReasonMessage* reasonMsg);

   
    /**
     * Global event handlers for IP-networking 
     */
/*
    static void antConnectCont(antEnvMsg* msg);
    int antConnectContSelector;
    
    static void antReceiveCont(antEnvMsg* msg);
    int antReceiveContSelector;

    static void antSendCont(antEnvMsg* msg);
    int antSendContSelector;

    static void antListenCont(antEnvMsg* msg);
    int antListenContSelector;
 
    static void antCloseCont(antEnvMsg* msg);
    int antCloseContSelector;
*/
    /**
     * The function calls the member ProcessNextFrame() of the only instance of this class.
     */
    static void nextFrame();

  protected:
    /**
     * The function initializes the object and should immediately be called after
     * the construction of an object.
     */
    void init();

    /**
     * The function is called once for each frame.
     */
    virtual void processNextFrame() = 0;

    /**
     * The function returns the address of the associated process.
     * @return The address of the object.
     */
    virtual PlatformProcess* getProcess() = 0;

  public:
    /**
     * Constructor.
     */
    ProcessBase():running(true)
    {
      theInstance=this;
    }

    /**
     * The function marks the process as stopped.
     */
    void stop() {running = false;}

    /**
     * The function returns whether the process is still running.
     * @return Shall the process continue?
     */
    bool isRunning() {return running;}

    /**
     * The functions sets or resets a bit in the blocking mask.
     * After a bit is set in the blocking mask for a certain 
     * sender or receiver, a new frame will not be started before
     * this sender or receiver received an event.
     * @param id The id of the sender or receiver.
     * @param block Should it block or not?
     */
    static void setBlockingId(int id,bool block = true);

    /**
     * The function is called when an event was received. 
     * If this was the last event the process was waiting for, the next
     * frame is started, i.e. NextFrame() is called.
     * @param id The id of the sender or receiver that received an event.
     */
    static void setEventId(int id);

    /**
     * Returns the event mask.
     * @return The event mask.
     */
    static int getEventMask() {return eventMask;}

    /**
     * Resets the event mask.
     */
    static void resetEventMask() {eventMask = 0;}
  
  friend void Prologue(InitInfo* pInfo);
};

/**
 * The class is a helper that allows to instantiate a class as an Aperios process.
 * ProcessCreator contains the parts that need to be implemented as a template.
 * It will only be used by the macro MAKE_PROCESS and should never be used directly.
 */
template<class T> class ProcessCreator : public ProcessBase
{
  private:
    T* process;       /**< A pointer to the only instance of the process. */
    const char* name; /**< The name of the process. */
    int lastTime;     /**< The last time when Process::Main() was finished. */

    /**
     * The function registers the global functions for object communication.
     * It is called by OObject::Init().
     * @param event An Open-R event.
     * @return The status. Currently, the function always reports success.
     */
    virtual OStatus DoInit(const OSystemEvent& event)
    {
      setBlockingId(31);
      int id = ENTRY_TABLE_MIN;
      for(ReceiverList* p = process->getFirstReceiver(); p; p = p->getNext())
      {
        ASSERT(strlen(name) + strlen(p->getName()) + 2 <= NAME_LENGTH_MAX);
        char buf[NAME_LENGTH_MAX];
        strcpy(buf,name);
        strcat(buf,".");
        strcat(buf,p->getName());
        OServiceEntry entry;
        entry.Set(myOID_,id++);
        VERIFY(serviceManager_.RegisterServiceEntry(entry,buf) == oSUCCESS);
        entry.Set(myOID_,id++);
        VERIFY(p->SetNotifyEntry(entry) == oSUCCESS);
      }
      for(SenderList* q = process->getFirstSender(); q; q = q->getNext())
      {
        ASSERT(strlen(name) + strlen(q->getName()) + 2 <= NAME_LENGTH_MAX);
        char buf[NAME_LENGTH_MAX];
        strcpy(buf,name);
        strcat(buf,".");
        strcat(buf,q->getName());
        OServiceEntry entry;
        entry.Set(myOID_,id++);
        VERIFY(serviceManager_.RegisterServiceEntry(entry,buf) == oSUCCESS);
        entry.Set(myOID_,id++);
        VERIFY(q->SetReadyEntry(entry) == oSUCCESS);
      }
      return oSUCCESS;
    }

    /**
     * The function starts the process.
     * It is called by OObject::Start().
     * @param event An Open-R event.
     * @return The status. Currently, the function always reports success.
     */
    virtual OStatus DoStart(const OSystemEvent& event) 
    {
      for(ReceiverList* p = process->getFirstReceiver(); p; p = p->getNext())
        p->AssertReady();
      // Call processNextFrame if no blocking receivers are waiting
      setEventId(31);
      return oSUCCESS;
    };

    /**
     * The function stops the process.
     * It is called by OObject::Stop(). However, OObject::Stop() is 
     * never called. Therefore, this function is called neither.
     * @param event An Open-R event.
     * @return The status. Currently, the function always reports success.
     */
    virtual OStatus DoStop(const OSystemEvent& event)
    {
      stop();
      for(ReceiverList* p = process->getFirstReceiver(); p; p = p->getNext())
        p->DeassertReady();
      if(process)
        delete process;
      process = 0;
      return oSUCCESS;
    }

    /**
     * The function destroys the process.
     * It is called by OObject::Destroy(). However, OObject::Destroy() is 
     * never called. Therefore, this function is called neither.
     * @param event An Open-R event.
     * @return The status. Currently, the function always reports success.
     */
    virtual OStatus DoDestroy(const OSystemEvent& event) {return oSUCCESS;}

    /**
     * The function is called once for each frame.
     * It calls Process::main() and will start a timer if Process::main()
     * returned a nonzero value.
     */
    void processNextFrame()
    {
      int frameTime = process->processMain();
      if(getProcess()->getFirstSender())
        getProcess()->getFirstSender()->finishFrame();
      if(getProcess()->getFirstReceiver())
        process->getFirstReceiver()->finishFrame();
      resetEventMask();
      int currentTime = SystemCall::getCurrentSystemTime();
      if(frameTime < 0)
        frameTime = lastTime - frameTime - currentTime;
      if(frameTime)
      {
        if(frameTime < 4)
          frameTime = 4;
        TimeEventInfoWithRelativeTime timeEvent(RelativeTime(frameTime / 1000,frameTime % 1000));
        EventID eventId;
        VERIFY(SetTimeEvent(&timeEvent,myOID_,Selector(4),0,0,&eventId) == sSUCCESS);
      }
      setBlockingId(31,frameTime != 0);
      lastTime = currentTime;
    }
  
    /**
     * The function returns the address of the associated process.
     * @return The address of the object.
     */
    virtual PlatformProcess* getProcess() {return process;}

  public:
    /**
     * Constuctor. 
     * It is only called from MAKE_PROCESS.
     * @param name The name of the process.
     */
    ProcessCreator(const char* name)
    {
      this->name = name;
      VERIFY(process = new T);
      init();
      lastTime = 0;
    }

  friend void Prologue(InitInfo* pInfo);
};

/**
 * The macro MAKE_PROCESS instantiates a class as an Aperios process.
 * As a convention, it should be used in the last line of the 
 * source file. In each Aperios process, MAKE_PROCESS must exactly be used
 * once.
 * @param theClass The type of the class to be instantiated.
 */
#define MAKE_PROCESS(theClass) \
  int ProcessBase::blockMask = 0, \
      ProcessBase::eventMask = 0; \
  extern "C" void Prologue(InitInfo* pInfo) \
  { \
    extern void _HeapInitialize(size_t); \
    extern void InitGlobals(); \
    _HeapInitialize(pInfo->heapSize); \
    InitGlobals(); \
    ProcessBase::theInstance = new ProcessCreator<theClass>(#theClass); \
    ExitFromPrologue(); \
  } \
  ProcessBase* ProcessBase::theInstance = 0; \
  struct ObjectEntry \
  { \
	  Selector selector; \
	  Entry entry; \
  } ObjectEntryTable[1]

#endif //__ProcessFramework_h__

/*
 * Change log :
 * 
 * $Log$
 *
 */
