/**
* @file WalkingEngineSelector.h
* 
* Definition of class WalkingEngineSelector
*
* @author Max Risler
*/

#ifndef __WalkingEngineSelector_h_
#define __WalkingEngineSelector_h_

#include "Tools/Module/ModuleSelector.h"

#include "InvKinWalkingParameterSets.h"
#include "UNSW2003WalkingEngine.h"
#include "UDWalkingEngine.h"
#include "YetAnotherInvKinWE.h"
#include "BB2004InvKinWalkingEngine.h"
#include "MSH2004InvKinWalkingEngine.h"

/**
* @class WalkingEngineSelector
*
* A module selector for WalkingEngine modules.
*/
class WalkingEngineSelector : public ModuleSelector, public WalkingEngine
{
public:
/**
* Constructor.
* @param handler The module handler of the process.
* @param moduleID The module id of the walking engine.
* @param interfaces The paramters of the MotionControl module.
* @param pInvKinWalkingEngine Pointer to an InvKinWalkingEngine to be used to avoid having different engines for different parameter sets.
  */
  WalkingEngineSelector(ModuleHandler &handler,
    SolutionRequest::ModuleID moduleID,
    const WalkingEngineInterfaces& interfaces,
    InvKinWalkingEngine* pInvKinWalkingEngine) 
    : ModuleSelector(moduleID),
    WalkingEngine(interfaces), pInvKinWalkingEngine(pInvKinWalkingEngine)
  {
    handler.setModuleSelector(moduleID,this);
  }
  
  
  /** 
  * Is called on start and when the selected solution changes
  * to create a specific solution.
  * @param id The id of the solution to create
  * @return The created solution or 0
  */
  virtual Module* createSolution(SolutionRequest::ModuleSolutionID id)
  {
    switch(id) {
      // "Real" WalkingEngine solutions
      //
      case SolutionRequest::invKinGT2003WalkingEngine:
        return new GT2003InvKinWalkingEngine(pInvKinWalkingEngine);
      
      case SolutionRequest::invKinBB2004WalkingEngine:
        return new BB2004InvKinWalkingEngine(pInvKinWalkingEngine);

      case SolutionRequest::invKinMSH2004WalkingEngine: 
        return new MSH2004InvKinWalkingEngine(pInvKinWalkingEngine);
      
      case SolutionRequest::unsw2003WalkingEngine:
        return new UNSW2003WalkingEngine(*this);
      
      case SolutionRequest::udWalkingEngine:
        return new UDWalkingEngine(*this);
      
      case SolutionRequest::yetAnotherInvKinWE:
        return new YetAnotherInvKinWE(*this);

      // ParamInvKinWalkingEngines
      //
      case SolutionRequest::invKinUpsideDownWalkingEngine:
        return new ParamInvKinWalkingEngine(new UpsideDownWalkingParameters, pInvKinWalkingEngine);
      
      case SolutionRequest::invKinTurnWithBallWalkingEngine: 
        return new ParamInvKinWalkingEngine(new TurnWithBallWalkingParameters, pInvKinWalkingEngine);
      
      case SolutionRequest::invKinUNSW2003WalkingEngine:
        return new ParamInvKinWalkingEngine(new UNSW2003WalkingParameters, pInvKinWalkingEngine);
      
      case SolutionRequest::invKinFastForwardWalkingEngine:
        return new ParamInvKinWalkingEngine(new FastForwardWalkingParameters, pInvKinWalkingEngine);
      
      case SolutionRequest::invKinERS7WalkingEngine: 
        return new ParamInvKinWalkingEngine(new ERS7WalkingParameters, pInvKinWalkingEngine);

      case SolutionRequest::invKinERS7TurnWithBallWalkingEngine: 
        return new ParamInvKinWalkingEngine(new ERS7TurnWithBallWalkingParameters, pInvKinWalkingEngine);
    
      case SolutionRequest::invKinMSH2004TurnWithBallWalkingEngine: 
        return new ParamInvKinWalkingEngine(new MSH2004TurnWithBallWalkingParameters, pInvKinWalkingEngine);

      case SolutionRequest::invKinDDDERS7TurnWithBallWalkingEngine: 
        return new ParamInvKinWalkingEngine(new DDDERS7TurnWithBallWalkingParameters, pInvKinWalkingEngine);

      case SolutionRequest::invKinDDDERS7WalkingEngine: 
        return new ParamInvKinWalkingEngine(new DDDERS7WalkingParameters, pInvKinWalkingEngine);

      case SolutionRequest::invKinDDDERS7FastWalkingEngine: 
        return new ParamInvKinWalkingEngine(new DDDERS7FastWalkingParameters, pInvKinWalkingEngine);

      default:
        return 0;
    }
  }
  
  /** Executes the module */
  bool executeParameterized(JointData& jointData, const MotionRequest& motionRequest)
  {
    if (selectedSolution!=0) 
    {
      return ((WalkingEngine*)selectedSolution)->executeParameterized(jointData, motionRequest);
    }
    else 
    {
      //if no WE is enabled, 2do:
      jointData.data[JointData::headTilt] = jointDataInvalidValue;
      jointData.data[JointData::headPan] = jointDataInvalidValue;
      jointData.data[JointData::headRoll] = jointDataInvalidValue;
      return false;
    }
  }
    /** Handles a message */
  virtual bool handleMessage(InMessage& message)
  {
    if (selectedSolution!=0) 
      return ((WalkingEngine*)selectedSolution)->handleMessage(message);
    else return false;
  }
  
  /** The only used InvKinWalkingEngine */
  InvKinWalkingEngine* pInvKinWalkingEngine;
};

#endif // __WalkingEngineSelector_h_

/*
* Change log :
* 
* $Log: WalkingEngineSelector.h,v $
* Revision 1.21  2004/04/28 17:08:16  goehring
* Head Motion control possible when switched off
*
* Revision 1.20  2004/04/09 11:35:53  roefer
* Bremen Byters German Open check-in
*
* Revision 1.19  2004/04/08 15:33:07  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.18  2004/04/07 12:29:00  risler
* ddd checkin after go04 - first part
*
* Revision 1.5  2004/04/02 23:44:40  dthomas
* added new invKinParams for turn with ball
* integrated into rotate-around-ball
* combined with updated catch-ball head-control-mode
*
* Revision 1.4  2004/04/02 09:27:54  Marc
* Removed Opponent Team Code
*
* Revision 1.3  2004/03/31 10:08:08  dthomas
* added new parameterset for ers7 based on jolly pochie
*
* Revision 1.2  2004/03/29 10:06:24  dthomas
* added new ers7 walking engine parameters
*
* Revision 1.1.1.1  2004/03/29 08:28:46  Administrator
* initial transfer from tamara
*
* Revision 1.17  2004/03/22 21:58:13  roefer
* True odometry
*
* Revision 1.16  2004/03/17 15:38:51  thomas
* added walkType debug for debugging WalkingEngineParameterSets
* paramSets-messages send by debug-message are stored in the new debug-walkType only
*
* Revision 1.15  2004/03/17 01:10:51  cesarz
* added MSH2004TurnWithBallWalkingParameters
*
* Revision 1.14  2004/03/16 18:47:09  kindler
* - added real class for MSH2004InvKinWalkingEngine solution.
*
* Revision 1.13  2004/03/10 17:45:21  kindler
* - Added MSH2004 InvKinWalkingEngine parameter set.
*
* Revision 1.12  2004/03/08 02:29:50  roefer
* Interfaces should be const
*
* Revision 1.11  2004/03/04 18:21:21  juengel
* Added ERS7TurnWithBallWalkingParameters.
*
* Revision 1.10  2004/03/03 21:17:07  roefer
* For the sake of consistency, renamed BB* to BB2004*
*
* Revision 1.9  2004/02/04 13:41:33  roefer
* Some place holders for BB2004 modules added
*
* Revision 1.8  2004/01/21 14:31:58  loetzsch
* Module Selectors create only the selected solution.
* When the solution changes, the old solution is erased and the new
* one ist created using createSolution(..)
*
* Revision 1.7  2004/01/13 18:35:23  loetzsch
* added invkin:walkingparameterset ERS7
*
* Revision 1.6  2003/12/29 15:48:54  roefer
* Bremen Byters evo walking added
*
* Revision 1.5  2003/12/02 18:07:14  dueffert
* first working not yet calibrated version of UDWalkingEngine added
*
* Revision 1.4  2003/11/22 09:33:01  jhoffman
* added "Yet Another Inv Kin Walking Engine"
* experimental stage, doesn't do much yet
* no Fourier inside!
*
* Revision 1.3  2003/11/21 15:44:19  dueffert
* fast forward walking added
*
* Revision 1.2  2003/10/24 15:01:25  loetzsch
* added UNSW2003WalkingEngine
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.5  2003/09/25 10:12:58  juengel
* Removed some walking engines.
*
* Revision 1.4  2003/07/24 13:58:22  risler
* added parameter set UNSW2003
*
* Revision 1.3  2003/07/24 12:41:26  risler
* ParamInvKinWalkingEngine no longer is template class -> 10min boot bug fixed
*
* Revision 1.2  2003/07/02 14:56:58  roefer
* Several walking engines removed
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.21  2003/06/14 21:45:47  roefer
* GT2003 walking parameters redesign
*
* Revision 1.20  2003/06/12 21:42:27  roefer
* GT2003 walking parameters finished
*
* Revision 1.19  2003/06/09 19:59:05  roefer
* GT2003 walking engine parameters added
*
* Revision 1.18  2003/05/07 17:00:49  goehring
* Added new walking solution 'Swing'
*
* Revision 1.17  2003/04/16 07:00:16  roefer
* Bremen GO checkin
*
* Revision 1.18  2003/04/09 09:39:28  roefer
* InvKin:UNSWLargeStep added
*
* Revision 1.17  2003/04/08 18:28:28  roefer
* bodyTiltOffset added
*
* Revision 1.16  2003/04/04 17:37:38  dueffert
* a bunch of fine tuning
*
* Revision 1.15  2003/03/31 14:16:20  hebbel
* Walking parameters for turning with ball added
*
* Revision 1.14  2003/03/06 12:08:45  dueffert
* execute with parameters renamed to avoid inheritance warnings
*
* Revision 1.13  2003/03/03 17:35:08  risler
* added CMUClone parameter set
*
* Revision 1.12  2003/02/25 11:43:44  dueffert
* line feed and doxygen bugs fixed
*
* Revision 1.11  2003/02/19 16:55:17  hebbel
* Added evolved parameterset for InvKinWalkingEngine
*
* Revision 1.10  2003/02/17 16:17:47  dueffert
* sorry Jan, but Motion needs mysterious 10 minutes to start with Fourier
*
* Revision 1.9  2003/01/23 16:44:10  risler
* only one instance of InvKinWalkingEngine
* parameter sets can now be switched while walking
* added UNSWFastTurn, combining two parameter sets
*
* Revision 1.8  2003/01/18 17:45:59  risler
* added upsideDown parameters
*
* Revision 1.7  2003/01/09 14:04:27  jhoffman
* added Inv Kin Parameter Set
*
* Revision 1.6  2002/12/20 13:50:54  dueffert
* CMU2002WalkingEngine added
*
* Revision 1.4  2002/12/16 14:49:10  dueffert
* changelog added
*
* Revision 1.3  2002/09/22 18:40:54  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/11 00:06:59  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:16  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.5  2002/09/01 18:06:41  loetzsch
* removed 2 walking engines
*
* Revision 1.4  2002/08/21 11:50:42  cesarz
* GT2001PlayersPerceptor, BerlinSelfLocator, GT2001WalkingEngine,
* DoWalkingEngine, FuzzyBehaviorControl, BremenGOBehaviorControl,
* FuzzyBehaviortesterdlgbar and Behaviortesterdlgbar removed.
*
* Revision 1.3  2002/07/23 13:33:43  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.2  2002/06/17 05:06:10  risler
* added FixedSequenceWalkingEngine
*
* Revision 1.1  2002/06/01 13:38:35  risler
* reorganized walking engine selection
*
* Revision 1.3  2002/05/16 15:05:49  risler
* added walk type walkWithBall
*
* Revision 1.2  2002/05/15 08:08:53  risler
* changed default walk, added InvKin:Dortmund to selector
*
* Revision 1.1.1.1  2002/05/10 12:40:17  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.3  2002/05/03 16:10:19  risler
* added UNSWClone InvKinWE Parameters
*
* Revision 1.2  2002/04/29 13:48:20  risler
* added lastWalkType to WalkingEngine execute
*
* Revision 1.1  2002/04/26 13:35:32  risler
* DarmstadtGOWalkingEngine renamed to InvKinWalkingEngine
* added InvKinParameterSets
*
* Revision 1.18  2002/04/25 14:50:35  kallnik
* changed double/float to double
* added several #include GTMath
*
* PLEASE use double
*
* Revision 1.17  2002/04/18 10:25:33  roefer
* Bremen GO
*
* Revision 1.16  2002/04/16 21:11:54  dueffert
* UNSW walk added
*
* Revision 1.15  2002/04/16 16:41:50  dueffert
* no message
*
* Revision 1.14  2002/04/15 13:39:23  rentmeister
* DoWalkingEngine (Dortmunder WalkingEngine)
*
* Revision 1.13  2002/04/05 16:46:34  kosen
* BremenGOWalkingEngine added
*
* Revision 1.12  2002/04/04 15:07:59  rentmeister
* Walk with Ball hinzugefgt
*
* Revision 1.11  2002/03/29 19:36:53  risler
* added DarmstadtGOWalkingEngine
*
* Revision 1.10  2002/03/24 17:45:07  juengel
* Bugs FourierWalking fixed.
*
* Revision 1.9  2002/03/18 09:45:36  kallnik
* GTMathTable updated
* GTMathConfig updated
* several doubles changed in GTMathValue
*
* Revision 1.8  2002/03/12 14:31:58  jhoffman
* added fourier walking engine
*
* Revision 1.7  2002/02/28 16:28:25  risler
* added GT2001WalkingParameters
*
* Revision 1.6  2002/01/20 13:00:18  risler
* changed implementation of selecting default solution
*
* Revision 1.5  2002/01/17 14:35:48  risler
* SolutionRequest added
*
* Revision 1.4  2001/12/10 17:47:08  risler
* change log added
*
*/
