/**
* @file WalkingEngine.h
* 
* Definition of class WalkingEngine
*/

#ifndef __WalkingEngine_h_
#define __WalkingEngine_h_

#include "Tools/Module/Module.h"
#include "Tools/MessageQueue/InMessage.h"
#include "Tools/RobotConfiguration.h"
#include "Tools/Actorics/RobotVertices.h"
#include "Tools/RingBuffer.h"
#include "Representations/Motion/MotionRequest.h"
#include "Representations/Motion/JointData.h"
#include "Representations/Motion/PIDData.h"
#include "Representations/Perception/SensorDataBuffer.h"
#include "Representations/Motion/OdometryData.h"
#include "Representations/Motion/HeadState.h"
#include "InvKinWalkingParameters.h"
#include "UDParameterSet.h"

/**
* @class WalkingEngineInterfaces
*
* The interfaces of the module WalkingEngine
*/
class WalkingEngineInterfaces
{
public:
  /** Constructor */
  WalkingEngineInterfaces(
    const SensorDataBuffer& sensorDataBuffer,
    const InvKinWalkingParameters& invKinWalkingParameters,
    const UDParameters& udParameters,
    const unsigned long& walkParameterTimeStamp,
    const RingBuffer<RobotVertices, 4>& robotVerticesBuffer,
    MotionRequest::MotionID& lastMotionType,
    PIDData& pidData,
    OdometryData& odometryData,
    HeadState& headState,
    MotionRequest& executedMotionRequest)
    : sensorDataBuffer(sensorDataBuffer),
    invKinWalkingParameters(invKinWalkingParameters),
    udParameters(udParameters),
    walkParameterTimeStamp(walkParameterTimeStamp),
    robotVerticesBuffer(robotVerticesBuffer),
    lastMotionType(lastMotionType),
    pidData(pidData),
    odometryData(odometryData),
    headState(headState),
    executedMotionRequest(executedMotionRequest)
  {}

protected:
  /** The sensor data buffer containing all joint angles and others */
  const SensorDataBuffer& sensorDataBuffer;
  
  /** walking parameter sets to be used by the WalkingEngine, eg calculated by evolution behavior */
  const InvKinWalkingParameters& invKinWalkingParameters;
  const UDParameters& udParameters;
  const unsigned long& walkParameterTimeStamp;

  /** a buffer for the robot vertices corresponding to the previous sensor data frames. */
  const RingBuffer<RobotVertices, 4>& robotVerticesBuffer;
  
  /** The motion type of the previous frame */
  MotionRequest::MotionID& lastMotionType;
  
  /** The current PID servo gains */
  PIDData& pidData;
  
  /** Has to be incremented by the walking engine */
  OdometryData& odometryData;

  /** The neck height and the body tilt to be estimated by the walking engine */
  HeadState& headState;

  /** The motion request that is currently executed by the walking engine */
  MotionRequest& executedMotionRequest;
};

/**
* A generic class for walking engine modules.
*
* It generates JointData and OdometryData according to current MotionRequest
* executing walking type motions.
*/
class WalkingEngine : public Module, protected WalkingEngineInterfaces, protected RobotDimensions
{
public:
/*
* Constructor.
* @param interfaces The paramters of the WalkingEngine module.
  */
  WalkingEngine(const WalkingEngineInterfaces& interfaces)
    : WalkingEngineInterfaces(interfaces),
      RobotDimensions(getRobotConfiguration().getRobotDimensions())
  {}
  
  /**
  * Calculates the next joint data set
  * @param jointData stores calculated frame
  * @param motionRequest The current motion request 
  * @return true if next set should be calculated by WalkingEngine
  *         false if change to other module can occur
  */
  virtual bool executeParameterized(JointData& jointData,
        const MotionRequest& motionRequest) = 0;
  
  /** Destructor */
  virtual ~WalkingEngine() {}
  
private:
  /** That function is not used but has to be implemented */
  virtual void execute() {};
};

#endif //__WalkingEngine_h_

/*
* Change log :
* 
* $Log: WalkingEngine.h,v $
* Revision 1.6  2004/03/20 09:55:26  roefer
* Preparation for improved odometry
*
* Revision 1.5  2004/03/08 02:29:50  roefer
* Interfaces should be const
*
* Revision 1.4  2004/02/16 17:57:13  dueffert
* packageCognitionMotion extended with invkin parameters
*
* Revision 1.3  2004/01/01 10:58:51  roefer
* RobotDimensions are in a class now
*
* Revision 1.2  2003/12/10 16:52:53  roefer
* Added sensor data buffer to walking engine
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.9  2003/03/06 12:08:45  dueffert
* execute with parameters renamed to avoid inheritance warnings
*
* Revision 1.8  2003/01/23 16:44:10  risler
* only one instance of InvKinWalkingEngine
* parameter sets can now be switched while walking
* added UNSWFastTurn, combining two parameter sets
*
* Revision 1.7  2002/11/20 18:02:28  risler
* added PID values to GT2001MotionNetSpecialActions
* added playDead motion
*
* Revision 1.6  2002/09/22 18:40:53  risler
* added new math functions, removed GTMath library
*
* Revision 1.5  2002/09/17 23:55:22  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.4  2002/09/12 13:17:27  loetzsch
* made member variables of module interfaces protected
*
* Revision 1.3  2002/09/11 20:06:19  loetzsch
* continued experiments with modules/solutions
*
* Revision 1.2  2002/09/11 00:06:59  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:16  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.4  2002/08/30 14:35:32  dueffert
* removed unused includes
*
* Revision 1.3  2002/08/22 14:41:03  risler
* added some doxygen comments
*
* Revision 1.2  2002/07/23 13:33:43  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.1.1.1  2002/05/10 12:40:17  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.13  2002/04/29 13:48:20  risler
* added lastWalkType to WalkingEngine execute
*
* Revision 1.12  2002/04/25 14:50:35  kallnik
* changed double/float to double
* added several #include GTMath
*
* PLEASE use double
*
* Revision 1.11  2002/04/16 16:39:54  dueffert
* no message
*
* Revision 1.10  2002/04/15 13:39:23  rentmeister
* DoWalkingEngine (Dortmunder WalkingEngine)
*
* Revision 1.9  2002/04/05 16:46:34  kosen
* BremenGOWalkingEngine added
*
* Revision 1.8  2002/04/04 15:07:59  rentmeister
* Walk with Ball hinzugefgt
*
* Revision 1.7  2002/04/03 16:44:30  jhoffman
* added "stabilizeRobot" to motionControl (which is turned off as a default)
*
* Revision 1.6  2002/02/28 16:28:25  risler
* added GT2001WalkingParameters
*
* Revision 1.5  2001/12/21 14:09:38  roefer
* Added several destructors
*
* Revision 1.4  2001/12/10 17:47:07  risler
* change log added
*
*/
