/**
* @file GT2003MotionNetSpecialActions/MotionNetNode.h
*
* Contains class MotionNetNode
*
* @author Uwe Dffert, Martin Ltzsch, Max Risler
*/

#ifndef __MOTIONNETNODE_H_
#define __MOTIONNETNODE_H_

#include "Representations/Motion/MotionRequest.h"
#include "Representations/Motion/JointData.h"
#include "Representations/Motion/PIDData.h"

#define INVAL 10000
#define DEFLT 10000

/**
* Represents a node of the motion net.
* The motion net is organised in an array of nodes (MotionNetNode).
* 
* @author Uwe Dffert <dueffert@informatik.hu-berlin.de>
*/
typedef struct MotionNetNode 
{
public:
  enum NodeType {
    //! the current node is a conditional transition
    typeConditionalTransition,
    //! the current node is a transition
    typeTransition,
    //! the current node is a motor data vector
    typeData,
    //! the current node is a pid data vector
    typePID
  };
  
  //! Represent a set of values from a data line 
  short d[24];
  /*
  //possible content:
  {typeData, d[0]..d[19], interpolationMode, dataRepetitionCounter, execMotionRequest}
  {typeConditionalTransition, to_motion, via_label, 20*0, execMotionRequest}
  {typeTransition, to_label, 21*0, execMotionRequest}
  {typePIDData, joint, p, i, d, 18*0, execMotionRequest}
  */
  
  void toJointData(JointData& jointData, long& dataRepetitionCounter, bool& interpolationMode)
  {
    //head
    jointData.data[JointData::headTilt] = d[1]*1000;
    jointData.data[JointData::headPan] = d[2]*1000;
    jointData.data[JointData::headRoll] = d[3]*1000;
    //mouth
    jointData.data[JointData::mouth] = d[4]*1000;
    //tail
    jointData.data[JointData::tailPan] = d[5]*1000;
    jointData.data[JointData::tailTilt] = d[6]*1000;
    //legs
    for (int i=0;i<12;i++)
      jointData.data[JointData::legFR1 + i] = d[i + 7]*1000;
    //ears
    jointData.data[JointData::earL] = d[19]*1000;
    jointData.data[JointData::earR] = d[20]*1000;
    //timing information
    interpolationMode = (d[21] == 1);
    dataRepetitionCounter = d[22];
  }
  
  void toPIDData(PIDData& pidData)
  {
    PIDData defaultPid;
    short pv = (d[2] == DEFLT) ? defaultPid.p[d[1]] : d[2];
    short iv = (d[3] == DEFLT) ? defaultPid.i[d[1]] : d[3];
    short dv = (d[4] == DEFLT) ? defaultPid.d[d[1]] : d[4];
    pidData.setValues((JointData::JointID)d[1],pv,iv,dv);
  }

  void toExecutedMotionRequest(MotionRequest& execMR)
  {
    if (d[23]>=0)
    execMR.motionType = MotionRequest::specialAction;
    execMR.specialActionType = (MotionRequest::SpecialActionID)d[23];
  }

} MotionNetNode;

#endif

/*
* Change log :
* 
* $Log: MotionNetNode.h,v $
* Revision 1.1  2003/10/06 14:10:12  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.2  2003/03/19 09:22:02  dueffert
* executedMotionRequest support added
*
* Revision 1.1  2003/03/10 14:14:08  dueffert
* optimized MotionNet
*
* Revision 1.5  2002/11/27 13:48:50  dueffert
* doxygen docu corrected
*
* Revision 1.4  2002/11/20 18:02:29  risler
* added PID values to GT2001MotionNetSpecialActions
* added playDead motion
*
* Revision 1.3  2002/11/19 17:38:31  dueffert
* doxygen bugs corrected
*
* Revision 1.2  2002/11/19 15:43:03  dueffert
* doxygen comments corrected
*
* Revision 1.1  2002/09/10 15:36:16  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1.1.1  2002/05/10 12:40:17  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.4  2001/12/12 12:43:33  risler
* removed MotionRequest.specialActionParams and MotionNetNode.pval
*
* Revision 1.3  2001/12/10 17:47:07  risler
* change log added
*
*/
