/**
* @file DefaultSoundControl.cpp
* 
* Implementation of class SoundControl.
*
* @author <a href="mailto:Oliver.Giese@uni-dortmund.de">Oliver Giese</a>
* @author <a href="mailto:matthias.hebbel@uni-dortmund.de">Matthias Hebbel</a>
*
*/
#include "DefaultSoundControl.h"
#include "Tools/Debugging/Debugging.h"


DefaultSoundControl::DefaultSoundControl(const SoundControlInterfaces& interfaces)
: SoundControl(interfaces)
{
  playingWave = false;
  actWavePosition = 0;
  lastSound = SoundRequest::none;

  for (int i = 0; i < SoundRequest::numOfSoundIDs; i++)
  {
    waveData[i] = 0;
    waveLength[i] = 0;
    if (i != (int) SoundRequest::none)
    {
      char file[20];
      strcpy(file,"Sounds/");
      strcat(file,(const char*)soundRequest.getSoundIDName((SoundRequest::SoundID)i));
      strcat(file,".wav");
      loadWavefile(file, i);
    }
  }
}


DefaultSoundControl::~DefaultSoundControl()
{
  for (int i = 0; i < SoundRequest::numOfSoundIDs; i++)
    if (waveData[i]) delete (waveData[i]);
}


void DefaultSoundControl::execute()
{
  if(playingWave) //occupied with playing wavesound
  {
    fillWaveBuffer(&soundData, lastSound);
  }
  else //open wavefile and play it
  {
    if ((soundRequest.soundID != SoundRequest::none) && 
        (soundRequest.soundID < SoundRequest::numOfSoundIDs) &&
        (soundRequest.soundID != lastSound))
    {
      playWave(&soundData, (int)soundRequest.soundID);
    }
    lastSound = soundRequest.soundID;
  }
}


void DefaultSoundControl::playWave(SoundData* soundData, int index)
{
  if (waveData[index])
  {
    playingWave = true;
    soundData->isInUse = true;
    actWavePosition = startWavePosition[index];
    fillWaveBuffer(soundData, index);
  }
}


bool DefaultSoundControl::loadWavefile(const char* filename, int index)
{
  InBinaryFile fin(filename);
  if (!fin.exists())
  {
    OUTPUT(idText,text,"DefaultSoundControl : Error, " << filename << " not found.");
    return false;
  }
  else
  {
    if (checkWaveHeader(filename, index))
    {
      waveData[index] = new char[waveLength[index]+startWavePosition[index]];
      fin.read(waveData[index],waveLength[index]+startWavePosition[index]);
      return true;
    }
  }
return false;
}


bool DefaultSoundControl::checkWaveHeader(const char* filename, int index)
{
  InBinaryFile fin(filename);
  int headerPos = 0;
  unsigned short format;
  unsigned long sampleRate;
  unsigned long bytesPerSecond;

  // load waveheader of maximum size
  fin.read(waveHeader, 100);

  // check waveformat
  if(strncmp(waveHeader,"RIFF",4)!=0)
  {
    OUTPUT(idText,text, "DefaultSoundControl : Error: " << filename << " is no RIFF file!");
    return false;
  }
  headerPos += 8;
  if(strncmp(waveHeader+headerPos,"WAVE",4)!=0)
  {
    OUTPUT(idText,text, "DefaultSoundControl : Error: " << filename << " is no WAVE file!");
    return false;
  }
  headerPos += 4;
  if(strncmp(waveHeader+headerPos,"fmt ",4)!=0)
  {
    OUTPUT(idText,text, "DefaultSoundControl : Error: " << filename << " has wrong subchunk fmt!");
    return false;
  }
  headerPos += 10;
  memcpy(&format,waveHeader+headerPos,2);
  if(format != 1)
  {
    OUTPUT(idText,text, "DefaultSoundControl : Error: " << filename << " is not mono!");
    return false;
  }
  headerPos += 2;
  memcpy(&sampleRate,waveHeader+headerPos,4);
  if(sampleRate != 8000)
  {
    OUTPUT(idText,text, "DefaultSoundControl : Error: " << filename << ": sample rate not 8000Hz!");
    return false;
  }
  headerPos += 4;
  memcpy(&bytesPerSecond,waveHeader+headerPos,4);
  if(bytesPerSecond != 8000)
  {
    OUTPUT(idText,text, "DefaultSoundControl : Error: " << filename << ": not 8000 bytes per second!");
    return false;
  }

  //find keyword "data"
  while ((headerPos < 97) && (strncmp(waveHeader+headerPos,"data",4)!=0))
  {
    headerPos+=2;
  }
  if (strncmp(waveHeader+headerPos,"data",4)!=0)
  {
    OUTPUT(idText,text, "DefaultSoundControl : Error: " << filename << ": 'data' not found!");
    return false;
  }
  memcpy(&waveLength[index],waveHeader+headerPos+4,4);
  startWavePosition[index] = headerPos + 8;

  return true;
}


void DefaultSoundControl::fillWaveBuffer(SoundData *soundData,int index)
{
  if(actWavePosition < waveLength[index] - 256)
  {
    memcpy(soundData->pcmBuffer, waveData[index] + actWavePosition, 256);
    // convert unsigned values to signed values
    for (int i=0; i < 256; i++)
    {
      soundData->pcmBuffer[i] = soundData->pcmBuffer[i] ^ 0x80;
    }
    actWavePosition += 256;
  }
  else
  {
    memset(soundData->pcmBuffer, 0, 256);
    playingWave = false;
    soundData->isInUse = false;
  }
}



/*
* Change log :
* 
* $Log: DefaultSoundControl.cpp,v $
* Revision 1.2  2004/03/08 02:11:52  roefer
* Interfaces should be const
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/07/03 17:15:03  dueffert
* error output improved
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.10  2003/05/02 18:26:18  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.9  2003/04/16 18:06:58  risler
* bugfix: useless if removed from destructor
*
* Revision 1.8  2003/04/16 07:00:52  roefer
* Destruction problem solved
*
* Revision 1.7  2003/04/15 18:55:03  risler
* destructor corrected
*
* Revision 1.6  2003/04/15 15:52:09  risler
* DDD GO 2003 code integrated
*
* Revision 1.8  2003/04/09 11:07:13  max
* bugfix: lastsound set always
*
* Revision 1.7  2003/04/08 21:20:36  max
* preload wave files at startup
*
* Revision 1.6  2003/04/06 12:32:48  max
* added ddd sound stuff
*
* Revision 1.5  2003/03/10 18:19:11  dueffert
* const cast warning removed
*
* Revision 1.4  2002/11/28 23:07:02  hebbel
* using variable isInUse of soundData
*
* Revision 1.3  2002/11/28 14:20:53  hebbel
* Cleaned up and added support for playing *.wav files
*
* Revision 1.2  2002/09/11 00:06:58  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:16  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.14  2002/07/23 13:33:43  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.13  2002/06/04 15:33:57  dueffert
* lengths corrected
*
* Revision 1.12  2002/06/03 18:53:39  dueffert
* messagetype in AcousticMessage is now private
*
* Revision 1.11  2002/05/27 16:15:55  schley
* ballposition can be transmitted
*
* Revision 1.10  2002/05/27 15:39:04  fischer
* Added SoundState (Sender and Receiver)
*
* Revision 1.9  2002/05/22 13:46:22  fischer
* Changed SoundInProcessor interface to accept WorldState
*
* Revision 1.8  2002/05/21 15:42:30  hebbel
* acoustic signals can be detected now
*
* Revision 1.7  2002/05/21 11:44:47  hebbel
* added table for signalcoding
*
* Revision 1.6  2002/05/17 11:20:57  fischer
* ball position exchange via acoustic communication
*
* Revision 1.5  2002/05/15 10:20:00  schley
* moved the audio.cfg to SoundProtocol
*
* Revision 1.4  2002/05/15 07:28:18  hebbel
* Removed mute, uses Soundprotocol instead, plays wave for patternchallenge
*
* Revision 1.3  2002/05/14 18:52:58  hebbel
* Added variable mute
*
* Revision 1.2  2002/05/13 13:29:43  schley
* Deconstructor added and audio.cfg
*
* Revision 1.1.1.1  2002/05/10 12:40:16  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.4  2002/05/06 15:06:54  schley
* Added the PCMCreator
*
* Revision 1.3  2002/05/03 13:47:57  giese
* Added timestamp to AcousticMessage
*
* Revision 1.2  2002/04/29 18:04:11  hebbel
* Put SoundPlay to Motion Process
*
* Revision 1.1  2002/04/28 19:14:12  giese
* SoundPlay added...
*
*
* 
*/
