/**
* @file JansVisionSandbox.cpp
* This file contains a class that demonstrates walking.
* 
* @author <A href=mailto:jhoffman@informatik.hu-berlin.de>Jan Hoffmann</A>
*/

#include "JansVisionSandbox.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "Tools/Math/Matrix2x2.h"



JansVisionSandbox::JansVisionSandbox(SensorBehaviorControlInterfaces& interfaces)
: SensorBehaviorControl(interfaces), 
maxSpeedX(140),
distanceToGround(300),
distanceOffset(50),
distanceControlInterval(50),
speedX(0, 0.6, 0.0, 0, -maxSpeedX/2, maxSpeedX, 100),
speedY(0, 1, 0, 0, -50, 50, 50),
speedPhi(0, .1, 0, 0, -pi_4, pi_4, pi/10),
gridSize(4),
timeOfLastExecute(0),
localColTableInitialized(false)
{
}

void JansVisionSandbox::init() {}

void JansVisionSandbox::execute()
{
  /* The module was not active for a while */
  /*
  unsigned long currentTime = SystemCall::getCurrentSystemTime();
  if((currentTime - timeOfLastExecute) > 2000) init();
  timeOfLastExecute = currentTime;
  */
  
  /* [JH] use the "proper" way to access the color table */
  Image debugImage = image;
  if(!localColTableInitialized)
  {
    debugImage.setColorTable(&colorTable);
  }
  else 
  {
    debugImage.setColorTable(&localColTable);		
  }
  
  
  struct 
  {
    double headNear;
    double headFar;
    double body;
  } range;
  
  double sum,
    greenRight = 0, 
    greenLeft = 0, 
    targetY = 0, 
    targetPhi, 
    targetX;
  
  range.headNear  = (sensorDataBuffer.lastFrame().data[SensorData::headPsdNear] / 1000) - 100.0; 
  range.headFar   = (sensorDataBuffer.lastFrame().data[SensorData::headPsdFar] / 1000) - 100.0; 
  range.body      = (sensorDataBuffer.lastFrame().data[SensorData::bodyPsd] / 1000) - 100.0; 
  
  int x, y;
  
  distanceToGround = 350;
  
  if (range.body > distanceToGround * 1.4) // don't jump off cliff!
    targetX = 0.0; // don't use negative nuber because robot might "reverse" off table
  else
    targetX = range.body - (distanceToGround - distanceOffset - distanceControlInterval);
  //shouldn't this be = psp - distanceOffset - (distanceToGround) ?
  
  targetX *= maxSpeedX/distanceControlInterval;
  
  //OUTPUT(idText, text, "target: " << targetX << ", psd: " << psd << ", speedX: " << speedX.getVal() << ", P speed: " << speedX.getWeightP());
  
  
  struct Line
  {
    Vector2<double> offset;
    Vector2<double> normal;
    int weight;
    int belongsToLineNo;
  };
  
  Line lines[1000];
  
  long numOfFoundLines = 0;
  
  Vector2<double> grad;
  double c = cos(-5*pi/4);
  double s = sin(-5*pi/4);
  Matrix2x2<double> rotation(
    Vector2<double>(c,c),
    Vector2<double>(-s,s)
    );
  

  // find gradients that can belong to white edges
  for(y = 1; y < image.cameraInfo.resolutionHeight-1; y+=3)
  {
    for(x = 1; x < image.cameraInfo.resolutionWidth-1; x+=3)
    {
      //if ((y-image.cameraInfo.resolutionHeight/2)*(y-image.cameraInfo.resolutionHeight/2)
      //  +(x-image.cameraInfo.resolutionWidth/2)*(x-image.cameraInfo.resolutionWidth/2) < image.cameraInfo.resolutionHeight*image.cameraInfo.resolutionHeight/4)
      {
        grad = Vector2<double>(image.image[y][0][x] - image.image[y-1][0][x-1] ,
          image.image[y-1][0][x] - image.image[y][0][x-1]);
        
        if (grad.abs() > 10)
        {
          Vector2<double> gradRotated = rotation*grad;
          
          double gradAngle = gradRotated.angle();
          normalize(gradAngle);
          
          Vector2<double> normGrad(gradRotated);
          normGrad.normalize();
          normGrad *= 2;
          
          if (image.image[y-(int)normGrad.y][0][x-(int)normGrad.x] > 100)
          {
            debugImage.image[y][0][x] = 255;
            debugImage.image[y][1][x] = 128;
            debugImage.image[y][2][x] = 128;
            
            if (!((y+x)%4))
            {
              //LINE(sketch, x, y, x-gradRotated.y/10, y+gradRotated.x/10, 1, 1, 3);
              
              if (numOfFoundLines < 1000)
              {
                lines[numOfFoundLines].offset = Vector2<double>(x,y);
                lines[numOfFoundLines].normal = gradRotated.normalize();
                lines[numOfFoundLines].weight = 0;
                lines[numOfFoundLines].belongsToLineNo = -1;
                ARROW(sketch, x, y, x+gradRotated.normalize().x*10, y+gradRotated.normalize().y*10, 1, 1, 0);
                numOfFoundLines++;
              }
            }
          }
        }
        else
        {
          debugImage.image[y][0][x] = 0;
          debugImage.image[y][1][x] = 128;
          debugImage.image[y][2][x] = 128;
        }
      }
    }
  }
  

  // show the five top rated lines
  int i, j, m, maxWeight=0;
  int color = 0;
  for (m = 0; m < 5; m++)
  {
    int lineWithHighestWeight = -1;
    maxWeight = 0;

    // for all lines, calculate how many other lines are similar
    for(i = 0; i < numOfFoundLines; i++)
    {
      for (j = 0; j < numOfFoundLines; j++)
      {
        // if line is not part of another line
        if(lines[j].belongsToLineNo == -1)
        {
          // distance point to line:
          // http://mo.mathematik.uni-stuttgart.de/kurse/kurs8/seite44.html
          double distance = fabs((lines[j].offset - lines[i].offset) * lines[i].normal);  
          double projection =  lines[i].normal*lines[j].normal;         
          // if distance is small
          // AND normal points in the same direction
          if((distance < 1.5) && 
            (projection > .98))
          {
            lines[i].weight++; 
            
            if(lines[i].weight > maxWeight)
            {
              maxWeight = lines[i].weight;
              lineWithHighestWeight = i;
            }
          }
        }
      }
    }
    
    color++;
    
    // show lines that belong to the line with highest weight
    for(i = 0; i < numOfFoundLines; i++)
    {
      double distance = fabs((lines[i].offset - lines[lineWithHighestWeight].offset) * lines[lineWithHighestWeight].normal);  
      double projection =  lines[lineWithHighestWeight].normal*lines[i].normal;         

      if(lines[i].belongsToLineNo == -1)
      if((distance < 1.5) && 
        (projection > .98))
      {
        ARROW(sketch, 
          lines[i].offset.x, 
          lines[i].offset.y, 
          lines[i].offset.x+lines[i].normal.x*10, 
          lines[i].offset.y+lines[i].normal.y*10, 1, 1, color);
        lines[i].belongsToLineNo = lineWithHighestWeight;
      }
      lines[i].weight = 0;
    }

    if (maxWeight > 3)
    {
      LINE(sketch, 
        lines[lineWithHighestWeight].offset.x+.5+lines[lineWithHighestWeight].normal.y*200, 
        lines[lineWithHighestWeight].offset.y+.5-lines[lineWithHighestWeight].normal.x*200, 
        lines[lineWithHighestWeight].offset.x+.5-lines[lineWithHighestWeight].normal.y*200, 
        lines[lineWithHighestWeight].offset.y+.5+lines[lineWithHighestWeight].normal.x*200, 1, 1, color);
    }
  }
  
  
  INIT_DEBUG_IMAGE(classificationY, debugImage);
  SEND_DEBUG_IMAGE(classificationY);
  //OUTPUT(idText, text, " " << greenLeft << " : " << greenRight); 
  
  sum = greenLeft + greenRight;
  if (sum == 0) sum = 1;
  
  /** if robot is moving forward, turn where there's more
  *   green
  */
  if (speedX.getVal() > .7 * maxSpeedX)
  {
    targetPhi = 1.2 * (greenLeft - greenRight) / sum;
    //targetPhi *= .5;
  }
  /** if robot is too close to something, stick to the
  * direction it last turned to until obstacle is no
  * longer in front of robot
  */
  else 
  {
    if (speedPhi.getVal() < 0)
      targetPhi = -pi;
    else
      targetPhi = pi;
  }
  
  if (greenLeft > greenRight) 
  {
    
    ledRequest.redBottomLEDs = LEDRequest::leftOnly;
  }
  else if (greenLeft < greenRight)
  {
    ledRequest.redBottomLEDs = LEDRequest::rightOnly;
  }
  else if (greenLeft == greenRight)
  {
    if (greenLeft == 0)
      ledRequest.redBottomLEDs = LEDRequest::bothOff;
    else
      ledRequest.redBottomLEDs = LEDRequest::bothOn;
  }
  else 
    ledRequest.redBottomLEDs = LEDRequest::bothOff;
  
  ledRequest.redTopLEDs = LEDRequest::bothOff;
  
  if(sensorDataBuffer.lastFrame().data[SensorData::tailPan] < 0)
  {
    speedPhi.approximateVal(targetPhi);
    speedX.approximateVal(targetX);
    timeToCalibrate = 0;
    soundRequest.soundID = SoundRequest::okay;
    headControlMode.headControlMode = HeadControlMode::lookJustBelowHorizon;
  }
  else //stop
  {
    headControlMode.headControlMode = HeadControlMode::lookStraightAhead;
    if (SystemCall::getTimeSince(timeToCalibrate) < 2000)
      soundRequest.soundID = SoundRequest::notokay;
    else 
      soundRequest.soundID = SoundRequest::none;
    speedPhi.approximateVal(0);
    speedX.approximateVal(0);
    if (timeToCalibrate > 0)
    {
      ledRequest.redTopLEDs = LEDRequest::bothOn;
      if (SystemCall::getTimeSince(timeToCalibrate) > 2000) // wait for one sec to allow motion to subside
      {
        if(robotState.getSwitches() == RobotState::headFrontDown)
        {
          soundRequest.soundID = SoundRequest::rob005;
          calibrate();
        }
      }
    }
    else
    {
      timeToCalibrate = SystemCall::getCurrentSystemTime();
      ledRequest.redTopLEDs = LEDRequest::bothOff;
    }
  }
  
  /* set the motion request */
  motionRequest.motionType=MotionRequest::walk;
  if (fabs(speedX.getVal()) > 1)
    motionRequest.walkParams.translation.x = speedX.getVal();
  else  
    motionRequest.walkParams.translation.x = 0.0;
  if (fabs(speedY.getVal()) > 1)
    motionRequest.walkParams.translation.y = targetY;
  else 
    motionRequest.walkParams.translation.y = 0.0;
  if (fabs(speedPhi.getVal()) > .05)
    motionRequest.walkParams.rotation = speedPhi.getVal();
  else 
    motionRequest.walkParams.rotation = 0.0;
  
  
  DEBUG_DRAWING_FINISHED(sketch);
  
  
}




void JansVisionSandbox::calibrate()
{
}



bool JansVisionSandbox::handleMessage(InMessage& message)
{
  bool handled = false;
  
  switch(message.getMessageID())
  {
  case idGenericDebugData:
    GenericDebugData d;
    message.bin >> d;
    if(d.id == GenericDebugData::braitenbergPIDs)
    {
      OUTPUT(idText,text,"generic debug message (Braitenberg) handled by module JansVisionSandbox");
      
      speedX.setWeightP(d.data[0]);
      speedX.setWeightI(d.data[1]);
      speedX.setWeightD(d.data[2]);
      
      speedPhi.setWeightP(d.data[3]);
      speedPhi.setWeightI(d.data[4]);
      speedPhi.setWeightD(d.data[5]);
      
      distanceOffset = d.data[6];
      distanceControlInterval = d.data[7];
      distanceToGround = d.data[8];
    }
    handled = true;
    break;
    
  }
  return handled;
}


/*
* Change log :
* 
* $Log: JansVisionSandbox.cpp,v $
* Revision 1.1  2004/05/25 12:36:57  jhoffman
* "robertson" operator demo added
*
* Revision 1.1.1.1  2004/03/05 10:10:27  loetzsch
* created local cvs for Gnne
*
* Revision 1.1  2004/02/27 14:16:20  jhoffman
* - screen layout "save" works properly
* - delete now has "confirm" dialog
*
*/
