/**
* @file BB2004Calibrator.h
*
* Definition of class BB2004Calibrator.
*
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#ifndef __BB2004Calibrator_h_
#define __BB2004Calibrator_h_

#include "Modules/SensorBehaviorControl/SensorBehaviorControl.h"
#include "Modules/ImageProcessor/GT2004ImageProcessor/GT2004ImageProcessor.h"
#include "Modules/SelfLocator/LinesTables.h"
#include "Tools/RingBuffer.h"
#include "Tools/Evolution/BB2004Evo.h"

/**
* @class BBInvKinIndividual
*
* The class represents an individual for a set of walking parameters.
*/
class CalibrationIndividual : public BBIndividual
{
private:
  enum 
  {
    numOfGenes = 7 /**< The number of genes of the individual. */
  };
  static double geneScale[numOfGenes]; /**< An array that estimates the range of value of a certain gene. */
  double genes[numOfGenes]; /**< The genes. */
  double fitness; /**< The fitness of this individual. */

public:
  /**
  * Constructor.
  */
  CalibrationIndividual() {fitness = 1;}

  /**
  * The function initializes the current individual with another one.
  * @param initial The other individual the current one is initialized with.
  */
  virtual void init(const BBIndividual& initial);
  
  /**
  * The function interpolates between the current individual and another one.
  * The result is stored in the current individual.
  * @param other The other individual the current one is interpolated with.
  */
  virtual void interpolate(const BBIndividual& other);

  /**
  * The function extrapolates between the current individual and another one.
  * @param other The other individual the current one is extrapolated with.
  */
  virtual void extrapolate(const BBIndividual& other);

  /**
  * The function mutates the current individual.
  */
  virtual void mutate();

  /**
  * The function returns the fitness of the individual.
  * @return The fitness as a number >= 0. A higher result means a higher fitness.
  */
  virtual double getFitness() const {return fitness;}

  /**
  * The function sets the fitness of the individual.
  * @param fitness The fitness.
  */
  void setFitness(double fitness) {this->fitness = fitness;}

  /**
  * The function copies the genes into the calibration parameters.
  */
  void select() const;

  /**
  * The function prints the genes.
  */
  void dump() const;
};

/**
* @class BB2004Calibrator
* The class calibrates the correction values for the camera matrix.
*/ 
class BB2004Calibrator : public SensorBehaviorControl, public LinesTables
{
public:
  /** 
  * Constructor.
  * @param interfaces The parameters of the BB2004Calibrator module.
  */
  BB2004Calibrator(const SensorBehaviorControlInterfaces& interfaces);

  /** 
  * The function executes the module.
  */
  virtual void execute();

  /**
  * The function handles messages sent to the module.
  * @param message A message.
  * @return Was the message handled by this module?
  */
  virtual bool handleMessage(InMessage& message);

private:
  CameraMatrix cameraMatrix; /**< The camera matrix that will be set for the image processor. */
  RobotPose robotPose; /**< A dummy robot pose for the image processor. */
  BallModel ballPosition; /**< A dummy ball position for the image processor. */
  PlayerPoseCollection playerPoseCollection; /**< A dummy player pose collection for the image processor. */
  RobotState robotState; /**< A dummy robot state for the image processor. */
  CalibrationRequest calibrationRequest; /**< A dummy calibration request for the image processor. */
  LandmarksPercept landmarksPercept; /**< A dummy landmark percept for the image processor. */
  BallPercept ballPercept; /**< A dummy ball percept for the image processor. */
  LinesPercept linesPercept; /**< The lines percept that will be returned by image processor. */
  PlayersPercept playersPercept; /**< A dummy players percept for the image processor. */
  ObstaclesPercept obstaclesPercept; /**< A dummy obstacles percept for the image processor. */
  SpecialPercept specialPercept; /**< A dummy special percept for the image processor. */
  GT2004ImageProcessor imageProcessor; /**< The standard image processor used for calibration. */
  RingBuffer<SensorData, 16> buffer; /**< A buffer for the sensor data of the previous 16 frames. */
  Image imageBuffer; /**< A buffer for the previous image. */
  enum 
  {
#ifdef _WIN32
    numOfIndividuals = 100, /**< The number of individuals in the population. */
#else
    numOfIndividuals = 15, /**< The number of individuals in the population. */
#endif
    waitPerGeneration = 10000 /**< The time to wait per generation in ms. */
  };
  BBPopulation<CalibrationIndividual> population; /**< The population of individuals. */
  Vector2<double> fitness[numOfIndividuals]; /**< The accumulated (temporary) fitnesses of the individuals. */
  int evolutions, /**< A counter for the number of evolutions. */
      count[numOfIndividuals]; /**< Variables that count how many fitnesses have been accumulated in "fitnesses". */
  unsigned timeStamp; /**< The beginning of the current evaluation of the individuals. */

  /**
  * The function performs a single evolution step.
  * @param sensorData The sensor data with the same time stamp as the image in "imageBuffer".
  */
  void evolve(const SensorData& sensorData);
};
#endif// __BB2004Calibrator_h_

/*
* Change log :
* 
* $Log: BB2004Calibrator.h,v $
* Revision 1.8  2004/05/04 13:56:55  tim
* added GT2004ImageProcessor
*
* Revision 1.7  2004/04/09 11:35:52  roefer
* Bremen Byters German Open check-in
*
* Revision 1.6  2004/03/13 17:20:43  roefer
* Different parameters for simulation and real robot
*
* Revision 1.5  2004/03/13 17:14:36  roefer
* Different parameters for simulation and real robot
*
* Revision 1.4  2004/03/11 00:06:18  roefer
* Parameter tuning
*
* Revision 1.3  2004/03/10 07:59:26  roefer
* New head control mode
*
* Revision 1.2  2004/03/08 02:11:49  roefer
* Interfaces should be const
*
* Revision 1.1  2004/03/04 23:00:54  roefer
* Added (so far empty) BB2004Calibrator
*
*/
