/**
* @file Modules/SelfLocator/MonteCarloSelfLocator.h
* 
* This file contains a class for self-localization based on the Monte Carlo approach.
*
* @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
* @author <A href=mailto:asbre01@tzi.de>Andreas Sztybryc</A>
*/

#ifndef __MonteCarloSelfLocator_h_
#define __MonteCarloSelfLocator_h_

#include "Representations/Perception/LandmarksPercept.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Field.h"
#include "SelfLocator.h"

/**
* The class implements the Monte Carlo self-localization, following some ideas
* of the CMU sensor resetting localization.
*/
class MonteCarloSelfLocator : public SelfLocator
{
private:
  /**
  * The class represents a sample.
  */
  class Sample : public PoseSample
  {
  public:
    Pose2D camera; /**< Temporary representing the pose of the camera. */
    double quality; /**< The quality of the sample. */
    Sample* next; /**< The next sample in the cell cube. */
    
    /**
    * Constructor.
    */
    Sample() {quality = probability = 0;}
    
    /**
    * Constructor.
    * @param pose The pose of the sample.
    * @param probability The probability of the sample.
    */
    Sample(const Pose2D& pose,double probability) : PoseSample(pose)
      {quality = 0; this->probability = probability;}
  };
  
  /**
  * The class represents a cell in a cube that is used to determine the largest sample cluster.
  */
  class Cell
  {
  public:
    int count; /**< The number of samples in this cube. */
    Sample* first; /**< The first sample in this cube. */
    
    /**
    * Constructor.
    */
    Cell() {count = 0; first = 0;}
  };
  
  enum
  {
    SAMPLES_MAX = 100, /**< The number of samples. */
    GRID_MAX = 10, /**< The number of cells in one dimension. */
    FLAGS_MAX = 3 /**< The size of the flag buffer. */
  };
  
  enum FlagSides
  {
    LEFT_SIDE_OF_FLAG = 1, /**< A marker for left edges of flags. */
    RIGHT_SIDE_OF_FLAG = -1 /**< A marker for right edges of flags. */
  };
  
  SampleSet<Sample, SAMPLES_MAX> sampleSet;
  Flag flags[FLAGS_MAX]; /**< A buffer for previously seen flags. */
  Field field; /**< The field. */
  Boundary<double> largeField; /**< An enlarged size of the field. */
  Pose2D lastOdometry, /**< The state of the odometry at the previous call of this module. */
         templates[SAMPLES_MAX]; /**< Templates for poses replacing bad samples. */
  int numOfFlags, /**< The number of flags in the buffer. */
      numOfTemplates, /**< The number of templates generated. */
      nextTemplate; /**< The next template delivered. */
  bool sensorUpdated; /**< Did any update of the samples by a sensor reading happen? */
  int factor, /**< A factor that determines the requirements for simularity between real and assumed measurements. */
      randomFactor; /**< A factor that is increased if more templates are required. */
  
  /**
  * The function distributes the parameter in a Gaussian way.
  * @param d A value that should be distributed.
  * @return A transformation of d according to a Gaussian curve.
  */
  double sigmoid(double d) const {return exp(- d * d * factor);}
  
  /** 
  * The function updates the samples by the odometry offset.
  * In addition, it set the camera pose for all samples.
  * @param odometry The motion since the last call to this function.
  * @param camera The camera offset.
  * @param noise Dermines whether some additional noise is added to the sample poses.
  */
  void updateByOdometry(const Pose2D& odometry,
                        const Pose2D& camera,
                        bool noise);
  
  /** 
  * The function updates the samples by a single edge of a flag recognized.
  * @param flag The position of the flag.
  * @param sideOfFlag The side of the flag that was seen.
  * @param measuredBearing The bearing, under which the edge was seen.
  */
  void updateFlag(const Vector2<double>& flag,
                  FlagSides sideOfFlag,
                  double measuredBearing);
  
  /** 
  * The function updates the samples by a single goal post recognized.
  * @param goalPost The position of the goal post.
  * @param measuredBearing The bearing, under which the goal post was seen.
  */
  void updateGoalPost(const Vector2<double>& goalPost,
                      double measuredBearing);
  
  /**
  * The function re-distributes the samples according to their probabilities.
  */
  void resample();
  
  /**
  * The function determines the most probable pose from the sample distribution.
  * @param pose The pose is returned to this variable.
  * @param validity The validity of the pose is returned to this variable.
  */
  void calcPose(Pose2D& pose,double& validity);
  
  /**
  * The function adds a flag to the buffer.
  * @param flag The new flag.
  */
  void addFlag(const Flag& flag);
  
  /**
  * The function calculates the current pose of the robot from three bearings.
  * @param dir0 The bearing on the first landmark.
  * @param dir1 The bearing on the second landmark.
  * @param dir2 The bearing on the third landmark.
  * @param mark0 The position of the first landmark.
  * @param mark1 The position of the second landmark.
  * @param mark2 The position of the third landmark.
  * @param cameraOffset The offset of the camera relative to the body center.
  * @param resultingPose The calculated pose is returned to this variable.
  * @return Was the function successful?
  */
  bool poseFromBearings(double dir0,double dir1,double dir2,
                        const Vector2<double>& mark0,
                        const Vector2<double>& mark1,
                        const Vector2<double>& mark2,
                        const Vector2<double>& cameraOffset,
                        Pose2D& resultingPose) const;
  
  /**
  * The function calculates the up to two current poses of the robot from two bearings and a distance.
  * @param dir0 The bearing on the first landmark.
  * @param dir1 The bearing on the second landmark.
  * @param dist The distance of the first landmark.
  * @param mark0 The position of the first landmark.
  * @param mark1 The position of the second landmark.
  * @param cameraOffset The offset of the camera relative to the body center.
  * @param resultingPose1 One calculated pose is returned to this variable.
  * @param resultingPose2 A second calculated pose is returned to this variable.
  * @return The number of poses calculated?
  */
  int poseFromBearingsAndDistance(double dir0,double dir1,
                                  double dist,
                                  const Vector2<double>& mark0,
                                  const Vector2<double>& mark1,
                                  const Vector2<double>& cameraOffset,
                                  Pose2D& resultingPose1,
                                  Pose2D& resultingPose2) const;
  
  /**
  * The function determines a pair of landmark positions and bearings from a landmarks percept.
  * @param landmarksPercept The landmarks percept.
  * @param i The index of the entry in the percept.
  * @param mark The position of the mark is returned here.
  * @param dir The bearing on the mark is returned to this variable.
  * @param dist The distance of the mark is returned to this variable.
  * @return returns true if getting the bearing was successful.
  */
  bool getBearing(const LandmarksPercept& landmarksPercept,int i,
                  Vector2<double>& mark,
                  double& dir,double& dist) const;
  
  /**
  * The function generates pose templates from a landmark percept.
  * The pose templates can be used to initialize new samples.
  * @param landmarksPercept The landmarks percept.
  * @param odometry The odometry offset since the last call of this function.
  */
  void generatePoseTemplates(const LandmarksPercept& landmarksPercept,
                             const Pose2D& odometry);
 
  /**
  * The function returns the next pose template or a random one if no templates were determined.
  * @return A new sample.
  */
  Sample getTemplate();

  /**
  * The function draws the samples to a debug drawing.
  */
  void draw() const;

  /**
  * The function draws an arrow to a debug drawing.
  * @param pose The position and direction of the arrow.
  * @param color The color of the arrow.
  */
  void draw(const Pose2D& pose, Drawings::Color color) const;
  
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the SelfLocator module.
  */
  MonteCarloSelfLocator(const SelfLocatorInterfaces& interfaces);
  
  /**
  * The function re-distributes the samples to random poses.
  */
  void reset();
  
  /** Executes the module */
  virtual void execute();
};

#endif// __MonteCarloSelfLocator_h_

/*
* Change log :
* 
* $Log: MonteCarloSelfLocator.h,v $
* Revision 1.2  2004/03/08 02:11:47  roefer
* Interfaces should be const
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.13  2003/05/08 23:52:24  roefer
* SampleSet and SampleSetProxy added
*
* Revision 1.12  2003/03/31 21:01:44  roefer
* Moved class Field to Tools
*
* Revision 1.11  2002/12/12 22:09:41  roefer
* Progress in LinesSelfLocator
*
* Revision 1.10  2002/11/19 17:38:31  dueffert
* doxygen bugs corrected
*
* Revision 1.9  2002/11/12 10:49:02  juengel
* New debug drawing macros - second step
* -moved /Tools/Debugging/PaintMethods.h and . cpp
*  to /Visualization/DrawingMethods.h and .cpp
* -moved DebugDrawing.h and .cpp from /Tools/Debugging/
*  to /Visualization
*
* Revision 1.8  2002/11/07 13:49:06  roefer
* poseFromBearingAndDistance fixed, templates distributed and painted
*
* Revision 1.7  2002/10/14 15:17:20  dueffert
* doxygen warnings removed
*
* Revision 1.6  2002/10/14 13:14:25  dueffert
* doxygen comments corrected
*
* Revision 1.5  2002/10/10 16:16:10  roefer
* PSD corrected
*
* Revision 1.4  2002/10/02 13:30:48  loetzsch
* changed the random function
*
* Revision 1.3  2002/09/22 18:40:53  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/12 12:24:09  juengel
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:16  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.6  2002/08/30 14:34:06  dueffert
* removed unused includes
*
* Revision 1.5  2002/06/08 20:19:20  Thomas Rfer
* Again 100 samples
*
* Revision 1.4  2002/06/02 23:21:09  roefer
* Single color table and progress in LinesSelfLocator
*
* Revision 1.3  2002/05/30 06:45:42  roefer
* Sample number is again 100
*
* Revision 1.2  2002/05/28 20:29:29  roefer
* Sample number reduced to 50
*
* Revision 1.1.1.1  2002/05/10 12:40:15  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.10  2002/04/08 21:33:20  roefer
* Paderborn release of self location
*
* Revision 1.9  2002/04/02 13:10:20  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.8  2002/02/22 08:43:24  roefer
* Intelligent sensor resetting added
*
* Revision 1.7  2002/02/13 22:43:02  roefer
* First working versions of DefaultLandmarksPerceptor and MonteCarloSelfLocator
*
* Revision 1.6  2002/02/12 09:45:09  roefer
* Progress in DefaultLandmarksPerceptor and MonteCarloSelfLocator
*
* Revision 1.5  2002/02/07 16:27:51  loetzsch
* delete [] samples added at the end of execute. removed in destructor
*
* Revision 1.4  2002/02/06 10:30:11  AndySHB
* MonteCarloLocalization First Draft
*
* Revision 1.2  2001/12/10 17:47:07  risler
* change log added
*
*/
