/**
 * @file Modules/SelfLocator/LinesTables.cpp
 * 
 * This file contains a class that represents the tables used for localization based on field lines.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#include "LinesTables.h"
#include "Tools/Debugging/Debugging.h"
#include "Tools/Streams/InStreams.h"

Field LinesTables::field;
ObservationTable<280,200,25>* LinesTables::observationTable = 0;
TemplateTable* LinesTables::templateTable = 0;
int LinesTables::refCount = 0;

//#define SAVE

LinesTables::LinesTables()
{
  if(!refCount++)
  {
    int i;
    observationTable = new ObservationTable<280,200,25>[LinesPercept::numberOfLineTypes];
    templateTable = new TemplateTable[LinesPercept::numberOfLineTypes];

#ifdef SAVE
    OutBinaryFile stream("lines.tab");
    for(i = 0; i < LinesPercept::numberOfLineTypes; ++i)
    {
      observationTable[i].create(field,LinesPercept::LineType(i));
      templateTable[i].create(field,LinesPercept::LineType(i));
      stream << observationTable[i] << templateTable[i];
    }
#else
    InBinaryFile stream("lines.tab");
    if(stream.exists())
      for(i = 0; i < LinesPercept::numberOfLineTypes; ++i)
        stream >> observationTable[i] >> templateTable[i];
    else
    {
      OUTPUT(idText, text, "LinesTables : lines.tab not found");
    }
#endif
  }
}

LinesTables::~LinesTables()
{
  if(!--refCount)
  {
    delete [] observationTable;
    delete [] templateTable;
  }
}

///////////////////////////////////////////////////////////////////////
// TemplateTable

Out& operator<<(Out& stream,const TemplateTable& table)
{
  stream.write(&table,sizeof(TemplateTable));
  return stream;
}

In& operator>>(In& stream,TemplateTable& table)
{
  stream.read(&table,sizeof(TemplateTable));
  return stream;
}

void TemplateTable::create(const Field& field,LinesPercept::LineType type)
{
  Temp* temp = new Temp[TEMPLATES_MAX];
  int i;
  double dist;
  for(i = 0; i < TEMPLATES_MAX; ++i)
    for(;;)
    {
      (Pose2D&) temp[i] = field.randomPose();
      dist = field.getDistance(temp[i],type);
      if(dist < 0)
        continue; // no line in that direction
      if(type == LinesPercept::skyblueGoal || type == LinesPercept::yellowGoal)
      {
        double distToBorder = field.getDistance(temp[i],LinesPercept::border);
        if(distToBorder > 0 && distToBorder < dist)
          continue; // goal line is hidden by border
      }
      temp[i].distance = (int) dist / 10;
      break; // everything all right -> leave loop
    }
  qsort(temp,TEMPLATES_MAX,sizeof(Temp),(int (*)(const void *,const void*)) compare);
  i = 0;
  int j = 0;
  distance[j] = i;
  while(i < TEMPLATES_MAX && j < DISTANCE_MAX)
  {
    while(i < TEMPLATES_MAX && j < DISTANCE_MAX && temp[i].distance == j)
    {
      char* t = templates[i];
      t[0] = char(temp[i].translation.x / 20);
      t[1] = char(temp[i].translation.y / 20);
      t[2] = char(temp[i].getAngle() * 127 / pi);
      ++i;
    }
    distance[++j] = i;
  }
  while(j < DISTANCE_MAX)
    distance[++j] = i;
  delete [] temp;
}

int TemplateTable::compare(const Temp* t1,const Temp* t2)
{
  return t1->distance == t2->distance ? 0 : t1->distance < t2->distance ? -1 : 1;
}

Pose2D TemplateTable::sample(double realDist) const
{
  realDist /= 10;
  int dist = (int) (realDist * (0.9 + 0.2 * random()));
  if(dist < 0)
    dist = 0;
  else if(dist >= DISTANCE_MAX)
    dist = DISTANCE_MAX - 1;
  int index = distance[dist] + int((distance[dist + 1] - distance[dist]) * random());
  const char* t = templates[index < TEMPLATES_MAX ? index : TEMPLATES_MAX - 1];
  return Pose2D(t[2] * pi / 127,Vector2<double>(t[0] * 20,t[1] * 20));
}

/*
 * Change log :
 * 
 * $Log: LinesTables.cpp,v $
 * Revision 1.2  2004/03/08 01:09:34  roefer
 * Use of LinesTables restructured
 *
 * Revision 1.1  2003/10/06 14:10:14  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.1  2003/05/22 07:53:05  roefer
 * GT2003SelfLocator added
 *
 */
