/**
* @file ObstaclesLocator.h
* 
* This file contains a generic class for Obstacles Localization.
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#ifndef __ObstaclesLocator_h_
#define __ObstaclesLocator_h_

#include "Tools/Module/Module.h"

#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/LinesPercept.h"
#include "Representations/Perception/PSDPercept.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Cognition/ObstaclesModel.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Motion/OdometryData.h"

/**
* @class ObstaclesLocatorInterfaces
* 
* The interfaces of the ObstaclesLocator module.
*/
class ObstaclesLocatorInterfaces
{
public:
  /** Constructor.*/
  ObstaclesLocatorInterfaces(
    const ObstaclesPercept& obstaclesPercept,
    const LinesPercept& linesPercept,
    const PSDPercept& psdPercept,
    const CameraMatrix& cameraMatrix,
    const OdometryData& odometryData,
    const PlayerPoseCollection& playerPoseCollection,
    const RobotPose& robotPose,
    const BallModel& ballPosition,
    ObstaclesModel& obstaclesModel)
    : obstaclesPercept(obstaclesPercept),
    linesPercept(linesPercept),
    psdPercept(psdPercept),
    cameraMatrix(cameraMatrix),
    odometryData(odometryData),
    playerPoseCollection(playerPoseCollection),
    robotPose(robotPose),
    ballPosition(ballPosition),
    obstaclesModel(obstaclesModel)
  {}
  
protected:
  /** A detected player */
  const ObstaclesPercept& obstaclesPercept;

  /** A lines percept */
  const LinesPercept& linesPercept;

  /** An object detected with the PSD sensor */
  const PSDPercept& psdPercept;
  
  /** A detected player */
  const CameraMatrix& cameraMatrix;
  
  /** The odometry data that was calculated by the motion modules */
  const OdometryData& odometryData;

  /** The poses of the other players */
  const PlayerPoseCollection& playerPoseCollection;
  
  /** The pose of the robot */
  const RobotPose& robotPose;

  /** The pose of the robot */
  const BallModel& ballPosition;

  /** The obstacle model to be calculated */
  ObstaclesModel& obstaclesModel;

};

/**
* @class ObstaclesLocator 
*
* The Base class for obstacle locators
*
*/
class ObstaclesLocator : public Module, public ObstaclesLocatorInterfaces
{
public:
/*
* Constructor.
* @param interfaces The paramters of the ObstaclesLocator module.
  */
  ObstaclesLocator(const ObstaclesLocatorInterfaces& interfaces)
    : ObstaclesLocatorInterfaces(interfaces)
  {}
  
  /** Destructor */
  virtual ~ObstaclesLocator() {}
};

#endif //__ObstaclesLocator_h_

/*
 * Change log :
 * 
 * $Log: ObstaclesLocator.h,v $
 * Revision 1.3  2004/03/08 01:39:04  roefer
 * Interfaces should be const
 *
 * Revision 1.2  2004/02/03 13:19:49  spranger
 * renamed all references to  class BallPosition to BallModel
 *
 * Revision 1.1  2003/10/06 14:10:14  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:27:49  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/06/18 13:57:46  juengel
 * Added ballPosition to ObstaclesLocatorInterfaces.
 *
 * Revision 1.5  2003/05/26 08:29:13  juengel
 * Added PlayerPoseCollection and RobotPose to ObstaclesLocatorInterfaces
 *
 * Revision 1.4  2003/04/06 15:05:42  loetzsch
 * Matthias added LinesPercept to ObstaclesLocatorInterfaces
 *
 * Revision 1.3  2003/03/22 16:52:36  juengel
 * Added PSDPercept.
 *
 * Revision 1.2  2003/03/15 13:24:30  juengel
 * Added CameraMatrix to ObstaclesLocatorInterfaces
 *
 * Revision 1.1  2003/03/10 13:53:59  juengel
 * ObstaclesLocator
 *
 */
