/**
* @file DefaultMotionControl.cpp
* 
* Implementation of class DefaultMotionControl.
*
* @author Max Risler
*/

#include "DefaultMotionControl.h"

#include "Modules/WalkingEngine/WalkingEngineSelector.h"
#include "Modules/WalkingEngine/InvKinWalkingEngine.h"
#include "Modules/WalkingEngine/InvKinWalkingParameterSets.h"
#include "Modules/SpecialActions/GT2003MotionNetSpecialActions/GT2003MotionNetSpecialActions.h"

#include "Platform/SystemCall.h"
#include "Tools/Actorics/Kinematics.h"
#include "Tools/Debugging/Debugging.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "Tools/RobotConfiguration.h"

DefaultMotionControl::DefaultMotionControl(ModuleHandler& moduleHandler,
                                           const MotionControlInterfaces& interfaces) 
                                           :MotionControl(interfaces),
                                           moduleHandler(moduleHandler)
{
  // create an uninitialised motion request to set startup motion
  MotionRequest emptyRequest;
  currentMotionType = lastMotionType = emptyRequest.motionType;
  lastHeadTilt = lastHeadPan = lastHeadRoll = 0;

  WalkingEngineInterfaces walkingEngineInterfaces
    (sensorDataBuffer, invKinWalkingParameters, udParameters, walkParameterTimeStamp, robotVerticesBuffer, lastMotionType, pidData, odometryData, headState, executedMotionRequest);

  // create the InvKinWalking engine as only one instance is created
  // and used by different walk type with different parameter sets
  pInvKinWalkingEngine = new InvKinWalkingEngine(walkingEngineInterfaces);


  pWalkingEngine[MotionRequest::normal] = 
    new WalkingEngineSelector(moduleHandler, SolutionRequest::walkingEngineNormal,
    walkingEngineInterfaces, pInvKinWalkingEngine);
  
  pWalkingEngine[MotionRequest::upsideDown] = 
    new ParamInvKinWalkingEngine(new UpsideDownWalkingParameters, pInvKinWalkingEngine);

  pWalkingEngine[MotionRequest::turnWithBall] = 
    new ParamInvKinWalkingEngine(new TurnWithBallWalkingParameters, pInvKinWalkingEngine);

  pWalkingEngine[MotionRequest::turnWithBall2] = 
    new ParamInvKinWalkingEngine(new ERS7TurnWithBallWalkingParameters, pInvKinWalkingEngine);

  pWalkingEngine[MotionRequest::turnWithBall3] = 
    new ParamInvKinWalkingEngine(new MSH2004TurnWithBallWalkingParameters, pInvKinWalkingEngine);

  pWalkingEngine[MotionRequest::turnWithBall4] = 
    new ParamInvKinWalkingEngine(new ATH2004TurnWithBallWalkingParameters, pInvKinWalkingEngine);

  pWalkingEngine[MotionRequest::turnWithBall5] = 
    new ParamInvKinWalkingEngine(new DDDERS7TurnWithBallWalkingParameters, pInvKinWalkingEngine);

  pWalkingEngine[MotionRequest::turnWithBallRearOnly] = 
    new ParamRearOnlyInvKinWalkingEngine(new MSH2004TurnWithBallWalkingParameters, pInvKinWalkingEngine);

  pWalkingEngine[MotionRequest::dash] =
    new ParamInvKinWalkingEngine(new DDDERS7FastWalkingParameters, pInvKinWalkingEngine);

  pWalkingEngine[MotionRequest::debug] = 
    new ParamInvKinWalkingEngine(new QuickUNSWWalkingParameters, pInvKinWalkingEngine);

  SpecialActionsInterfaces specialActionsInterfaces(motionRequest,lastMotionType, 
    pidData, odometryData, headState, executedMotionRequest);
  pSpecialActions = new GT2003MotionNetSpecialActions(specialActionsInterfaces);

  GetupEngineInterfaces getupEngineInterfaces(lastMotionType, sensorDataBuffer, motionRequest, pidData, odometryData,
    headState, executedMotionRequest);
  pGetupEngine = new GetupEngineSelector(moduleHandler,getupEngineInterfaces);

  wakingUp = false;
  lastSpecialActionWasPlayDead = false;
}

DefaultMotionControl::~DefaultMotionControl()
{
  delete pWalkingEngine[MotionRequest::normal];
  delete pWalkingEngine[MotionRequest::upsideDown];
  delete pWalkingEngine[MotionRequest::turnWithBall];
  delete pWalkingEngine[MotionRequest::turnWithBall2];
  delete pWalkingEngine[MotionRequest::turnWithBall3];
  delete pWalkingEngine[MotionRequest::turnWithBall4];
  delete pWalkingEngine[MotionRequest::turnWithBall5];
  delete pWalkingEngine[MotionRequest::turnWithBallRearOnly];
  delete pWalkingEngine[MotionRequest::dash];
  delete pWalkingEngine[MotionRequest::debug];
  delete pSpecialActions;
  delete pGetupEngine;
  delete pInvKinWalkingEngine;

  moduleHandler.setModuleSelector(SolutionRequest::walkingEngineNormal,0);
  moduleHandler.setModuleSelector(SolutionRequest::getupEngine,0);
}

void DefaultMotionControl::wagTail(
                                   const MotionRequest& motionRequest,
                                   const double& bodyTilt,
                                   const SensorData& sensorData,
                                   JointData& jointData
                                   )
{
  if ((jointData.data[JointData::tailPan] != jointDataInvalidValue) ||
    (jointData.data[JointData::tailTilt] != jointDataInvalidValue))
    return;
  
  switch (motionRequest.tailRequest)
  {
  case MotionRequest::wagHorizontal:
    {
      int framesPerPeriod = 200;
      tailCount %= framesPerPeriod;
      tailCount++;
      jointData.data[JointData::tailPan] = toMicroRad(jointLimitTailPanP * sin(tailCount * 2 * pi / framesPerPeriod));
      jointData.data[JointData::tailTilt] = toMicroRad(jointLimitTailTiltN);
    }
    break;
  case MotionRequest::wagLeftRightHorizontal:
    {
      int framesPerPeriod = 125;
      tailCount %= framesPerPeriod;
      tailCount++;
      jointData.data[JointData::tailPan] = toMicroRad(jointLimitTailPanP * sin(tailCount * 2 * pi / framesPerPeriod));
      jointData.data[JointData::tailTilt] = toMicroRad(jointLimitTailTiltN);
    }
    break;
  case MotionRequest::wagLeftRightVertical:
    {
      int framesPerPeriod = 125;
      tailCount %= framesPerPeriod;
      tailCount++;
      jointData.data[JointData::tailPan] = toMicroRad(jointLimitTailPanP * sin(tailCount * 2 * pi / framesPerPeriod));
      jointData.data[JointData::tailTilt] = toMicroRad(jointLimitTailTiltN);
    }
    break;
  case MotionRequest::wagHorizontalFast:
    {
      int framesPerPeriod = 30;
      tailCount %= framesPerPeriod;
      tailCount++;
      jointData.data[JointData::tailPan] = toMicroRad((jointLimitTailPanP/2) * sgn(sin(tailCount * 2 * pi / framesPerPeriod)));
      jointData.data[JointData::tailTilt] = toMicroRad(jointLimitTailTiltN);
    }
    break;
  case MotionRequest::wagVertical:
    {
      int framesPerPeriod = 125;
      tailCount %= framesPerPeriod;
      tailCount++;
      jointData.data[JointData::tailTilt] = 
        toMicroRad(
        (jointLimitTailTiltP + jointLimitTailTiltN) / 2 +
        (jointLimitTailTiltP - jointLimitTailTiltN) / 2 * sin(tailCount * 2 * pi / framesPerPeriod)
        );
      jointData.data[JointData::tailPan] = 0;
    }
    break;
  case MotionRequest::wagUpDownLeft:
    {
      int framesPerPeriod = 125;
      tailCount %= framesPerPeriod;
      tailCount++;
      jointData.data[JointData::tailTilt] = 
        toMicroRad(
        (jointLimitTailTiltP + jointLimitTailTiltN) / 2 +
        (jointLimitTailTiltP - jointLimitTailTiltN) / 2 * sin(tailCount * 2 * pi / framesPerPeriod)
        );
      jointData.data[JointData::tailPan] = toMicroRad(jointLimitTailPanN);
    }
    break;
  case MotionRequest::wagUpDownRight:
    {
      int framesPerPeriod = 125;
      tailCount %= framesPerPeriod;
      tailCount++;
      jointData.data[JointData::tailTilt] = 
        toMicroRad(
        (jointLimitTailTiltP + jointLimitTailTiltN) / 2 +
        (jointLimitTailTiltP - jointLimitTailTiltN) / 2 * sin(tailCount * 2 * pi / framesPerPeriod)
        );
      jointData.data[JointData::tailPan] = toMicroRad(jointLimitTailPanP);
    }
    break;
  case MotionRequest::wagVerticalFast:
    {
      tailCount &= 7;
      int j=tailCount++;
      if (j>3) j = 8-j;
      jointData.data[JointData::tailTilt] = 160000*(j - 2);
      jointData.data[JointData::tailPan] = 0;
    }
    break;
  case MotionRequest::stayAsForced:
    jointData.data[JointData::tailTilt] = sensorData.data[SensorData::tailTilt];
    jointData.data[JointData::tailPan] = sensorData.data[SensorData::tailPan];
    break;
  case MotionRequest::stayAsForcedPan:
    jointData.data[JointData::tailTilt] = toMicroRad(jointLimitTailTiltP);
    jointData.data[JointData::tailPan] = sensorData.data[SensorData::tailPan];
    break;
  case MotionRequest::stayAsForcedTilt:
    jointData.data[JointData::tailTilt] = sensorData.data[SensorData::tailTilt];
    jointData.data[JointData::tailPan] = 0;
    break;
  case MotionRequest::tailFollowsHead:
    jointData.data[JointData::tailTilt] = sensorData.data[SensorData::headTilt];
    jointData.data[JointData::tailPan] = -sensorData.data[SensorData::headPan];
    break;
  case MotionRequest::tailParallelToGround:
    jointData.data[JointData::tailTilt] = toMicroRad(-bodyTilt);
    jointData.data[JointData::tailPan] = 0;
    break;
  case MotionRequest::noTailWag:
    jointData.data[JointData::tailTilt] = 0;
    jointData.data[JointData::tailPan] = 0;
    break;
  case MotionRequest::twoPositionSwitchVertical:
    jointData.data[JointData::tailPan] = 0;
    if (sensorData.data[SensorData::tailTilt] > 0)
      jointData.data[JointData::tailTilt] = 380000;
    else
      jointData.data[JointData::tailTilt] = -380000;
    break;
  case MotionRequest::twoPositionSwitchHorizontal:
    jointData.data[JointData::tailTilt] = 0;
    if (sensorData.data[SensorData::tailPan] > 0)
      jointData.data[JointData::tailPan] = 380000;
    else
      jointData.data[JointData::tailPan] = -380000;
    break;
  case MotionRequest::threePositionSwitchVertical:
    jointData.data[JointData::tailPan] = 0;
    if (sensorData.data[SensorData::tailTilt] > 130000)
      jointData.data[JointData::tailTilt] = 380000;
    else if(sensorData.data[SensorData::tailTilt] > -130000)
      jointData.data[JointData::tailTilt] = 0;
    else
      jointData.data[JointData::tailTilt] = -380000;
    break;
  case MotionRequest::threePositionSwitchHorizontal:
    jointData.data[JointData::tailTilt] = 0;
    if (sensorData.data[SensorData::tailPan] > 130000)
      jointData.data[JointData::tailPan] = 380000;
    else if(sensorData.data[SensorData::tailPan] > -130000)
      jointData.data[JointData::tailPan] = 0;
    else
      jointData.data[JointData::tailPan] = -380000;
    break;
  case MotionRequest::fourPositionSwitchUpDownLeftRight:
    if(sensorData.data[SensorData::tailPan] > sensorData.data[SensorData::tailTilt])
    {
      if(sensorData.data[SensorData::tailPan] > -sensorData.data[SensorData::tailTilt])
      {
        //right
        jointData.data[JointData::tailPan] = 380000;
        jointData.data[JointData::tailTilt] = 0;
      }
      else
      {
        //bottom
        jointData.data[JointData::tailPan] = 0;
        jointData.data[JointData::tailTilt] = -380000;
      }
    }
    else
    {
      if(sensorData.data[SensorData::tailPan] > -sensorData.data[SensorData::tailTilt])
      {
        //top
        jointData.data[JointData::tailPan] = 0;
        jointData.data[JointData::tailTilt] = 380000;
      }
      else
      {
        //left
        jointData.data[JointData::tailPan] = -380000;
        jointData.data[JointData::tailTilt] = 0;
      }
    }
    break;
  case MotionRequest::fourPositionSwitchCorners:
    if (sensorData.data[SensorData::tailTilt] > 0)
      jointData.data[JointData::tailTilt] = 380000;
    else
      jointData.data[JointData::tailTilt] = -380000;

    if (sensorData.data[SensorData::tailPan] > 0)
      jointData.data[JointData::tailPan] = 380000;
    else
      jointData.data[JointData::tailPan] = -380000;
    break;
  case MotionRequest::fivePositionSwitch:
    if(sensorData.data[SensorData::tailTilt] > -130000 &&
       sensorData.data[SensorData::tailTilt] < 130000 &&
       sensorData.data[SensorData::tailPan] > -130000 &&
       sensorData.data[SensorData::tailPan] < 130000)
    {
      jointData.data[JointData::tailPan] = 0;
      jointData.data[JointData::tailTilt] = 0;
    }
    else
    {
      if (sensorData.data[SensorData::tailTilt] > 0)
        jointData.data[JointData::tailTilt] = 380000;
      else
        jointData.data[JointData::tailTilt] = -380000;

      if (sensorData.data[SensorData::tailPan] > 0)
        jointData.data[JointData::tailPan] = 380000;
      else
        jointData.data[JointData::tailPan] = -380000;
    }
    break;
  case MotionRequest::eightPositionSwitch:
    // vertical
    jointData.data[JointData::tailPan] = 0;
    if (sensorData.data[SensorData::tailTilt] > 130000)
    {
      jointData.data[JointData::tailTilt] = 380000;
      if (sensorData.data[SensorData::tailPan] > 130000)
        jointData.data[JointData::tailPan] = 380000;
      else if(sensorData.data[SensorData::tailPan] > -130000)
        jointData.data[JointData::tailPan] = 0;
      else
        jointData.data[JointData::tailPan] = -380000;
    }
    else if(sensorData.data[SensorData::tailTilt] > -130000)
    {
      jointData.data[JointData::tailTilt] = 0;
      if (sensorData.data[SensorData::tailPan] > 130000)
        jointData.data[JointData::tailPan] = 380000;
      else if(sensorData.data[SensorData::tailPan] > -130000)
      {
        // if the tail is in the center it is set to the top
        jointData.data[JointData::tailPan] = 0;
        jointData.data[JointData::tailTilt] = 380000;
      }
      else
        jointData.data[JointData::tailPan] = -380000;
    }
    else
    {
      jointData.data[JointData::tailTilt] = -380000;
      if (sensorData.data[SensorData::tailPan] > 130000)
        jointData.data[JointData::tailPan] = 380000;
      else if(sensorData.data[SensorData::tailPan] > -130000)
        jointData.data[JointData::tailPan] = 0;
      else
        jointData.data[JointData::tailPan] = -380000;
    }
    break;
  case MotionRequest::tailLeftTop:
    jointData.data[JointData::tailPan] = -380000;
    jointData.data[JointData::tailTilt] = 380000;
    break;
  case MotionRequest::tailCenterTop:
    jointData.data[JointData::tailPan] =       0;
    jointData.data[JointData::tailTilt] = 380000;
    break;
  case MotionRequest::tailRightTop:
    jointData.data[JointData::tailPan] =  380000;
    jointData.data[JointData::tailTilt] = 380000;
    break;
  case MotionRequest::tailLeftCenter:
    jointData.data[JointData::tailPan] = -380000;
    jointData.data[JointData::tailTilt] =      0;
    break;
  case MotionRequest::tailCenterCenter:
    jointData.data[JointData::tailPan] =       0;
    jointData.data[JointData::tailTilt] =      0;
    break;
  case MotionRequest::tailRightCenter:
    jointData.data[JointData::tailPan] =  380000;
    jointData.data[JointData::tailTilt] =      0;
    break;
  case MotionRequest::tailLeftBottom:
    jointData.data[JointData::tailPan] = -380000;
    jointData.data[JointData::tailTilt] = -380000;
    break;
  case MotionRequest::tailCenterBottom:
    jointData.data[JointData::tailPan] =        0;
    jointData.data[JointData::tailTilt] = -380000;
    break;
  case MotionRequest::tailRightBottom:
    jointData.data[JointData::tailPan] = 380000;
    jointData.data[JointData::tailTilt] = -380000;
    break;
  }
}

void DefaultMotionControl::execute()
{
  bool change = false;
  bool stand = false;
  bool fallenDown = false;

  HeadState headState2 = headState;

  //calculate body angles
  long accX = sensorDataBuffer.lastFrame().data[SensorData::accelerationX]/1000;
  long accY = sensorDataBuffer.lastFrame().data[SensorData::accelerationY]/1000;
  long accZ = sensorDataBuffer.lastFrame().data[SensorData::accelerationZ]/1000;

  headState.bodyTilt = atan2((double)accX,(double)-accZ);
  headState.bodyRoll = atan2((double)-accY,(double)-accZ);

  //executedMotionRequest = motionRequest;
  
  for (int i = 0; i < jointDataBufferNumOfFrames; i++)
  {
    JointData &currentFrame = jointDataBuffer.frame[i];

/**/
    if(wakingUp == true)
    {
      //OUTPUT(idText, text, "wakingUp.step");
      if(wakeUpEngine.step(currentFrame, pidData)) wakingUp = false;
      return;
    }
    else if(
      lastSpecialActionWasPlayDead &&
      (
      currentMotionType != MotionRequest::specialAction 
      ||
      motionRequest.specialActionType != MotionRequest::playDead
      )
      )
    {
      //OUTPUT(idText, text, "wakingUp.start");
      lastSpecialActionWasPlayDead = false;
      wakeUpEngine.start(sensorDataBuffer.frame[0], pidData);
      wakeUpEngine.step(currentFrame, pidData);
      wakingUp = true;
    }
/**/
    // calculate next frame
    switch (currentMotionType) 
    {
    case MotionRequest::getup:
      //OUTPUT(idText, text, "getup");
      executedMotionRequest = motionRequest;
      if (!pGetupEngine->executeParameterized(currentFrame))
      {
        // getup finished, stand and change if necessary
        change = true;
        stand = true;
      }
      else
        fallenDown = true;
      break;
    case MotionRequest::specialAction:
      //OUTPUT(idText, text, "special action");
/**/
      if(motionRequest.specialActionType == MotionRequest::playDead)
      {
        lastSpecialActionWasPlayDead = true;
      }
/**/
      if(!pSpecialActions->executeParameterized(currentFrame))
      {
        // execute stand if specialAction is finished
        change = true;
        stand = true;
      }
      break;
    case MotionRequest::walk:
      executedMotionRequest = motionRequest;
      change = !pWalkingEngine[motionRequest.walkType]->executeParameterized(currentFrame,motionRequest);

      /*
        // do forward kinematic on leg joints in RobotControl to view leg position
        #ifdef _WIN32
        double x,y,z;
        Kinematics::legPositionFromJoints(Kinematics::fl,
          fromMicroRad(currentFrame.data[JointData::legFL1]),
          fromMicroRad(currentFrame.data[JointData::legFL2]),
          fromMicroRad(currentFrame.data[JointData::legFL3]),
          headState.bodyTilt-getRobotConfiguration().getBodyTiltOffset(),
          x,y,z);
          currentFrame.data[JointData::legFR1] = (long)(x * 10000.0);
          currentFrame.data[JointData::legFR2] = (long)(y * 10000.0);
          currentFrame.data[JointData::legFR3] = (long)(z * 10000.0);
        Kinematics::legPositionFromJoints(Kinematics::hr,
          fromMicroRad(currentFrame.data[JointData::legHR1]),
          fromMicroRad(currentFrame.data[JointData::legHR2]),
          fromMicroRad(currentFrame.data[JointData::legHR3]),
          headState.bodyTilt-getRobotConfiguration().getBodyTiltOffset(),
          x,y,z);
          currentFrame.data[JointData::legHL1] = (long)(x * 10000.0);
          currentFrame.data[JointData::legHL2] = (long)(y * 10000.0);
          currentFrame.data[JointData::legHL3] = (long)(z * 10000.0);
        
          {
            double b = asin((-currentFrame.data[JointData::legHL3] + currentFrame.data[JointData::legFR3])/(10000.0*lengthBetweenLegs));
            double d = atan2(-currentFrame.data[JointData::legHL3] + currentFrame.data[JointData::legFR3],
              currentFrame.data[JointData::legFR1] - currentFrame.data[JointData::legHL1]+(10000.0*lengthBetweenLegs));

          }
        #endif
      */
      
      break;
    case MotionRequest::stand:
    default:
      executedMotionRequest = motionRequest;
      // stand and change if necessary
      change = true;
      stand = true;
    }
    
    if(headState2.calculated && !fallenDown)
      headState = headState2;

    if (stand)
    {
      // execute normal walk for stand
      MotionRequest standRequest;
      standRequest.motionType = MotionRequest::walk;
      standRequest.walkType = MotionRequest::normal;
      standRequest.walkParams = Pose2D(0,0,0);
      pWalkingEngine[MotionRequest::normal]->executeParameterized(currentFrame,standRequest);
      executedMotionRequest.motionType = MotionRequest::stand;
    }
    
    lastMotionType = currentMotionType;
    // switch to new motion if needed
    if (change) 
      currentMotionType = motionRequest.motionType;
    

    headIsBlockedBySpecialActionOrWalk = 
      (
      currentFrame.data[JointData::headTilt] != jointDataInvalidValue &&
      currentFrame.data[JointData::headPan] != jointDataInvalidValue &&
      currentFrame.data[JointData::headRoll] != jointDataInvalidValue 
      );
    
    // execute HeadMotionRequest (smoothed)
    const long maxdiff=50000; //(~2pi/s)
    long diff;
    if (currentFrame.data[JointData::headTilt] == jointDataInvalidValue)
    {
      diff=headMotionRequest.tilt-lastHeadTilt;
      if (diff<-maxdiff)
      {
        currentFrame.data[JointData::headTilt] = lastHeadTilt-maxdiff;
      }
      else if (diff<maxdiff)
      {
        currentFrame.data[JointData::headTilt] = headMotionRequest.tilt;
      }
      else
      {
        currentFrame.data[JointData::headTilt] = lastHeadTilt+maxdiff;
      }
    }
    if (currentFrame.data[JointData::headPan] == jointDataInvalidValue)
    {
      diff=headMotionRequest.pan-lastHeadPan;
      if (diff<-maxdiff)
      {
        currentFrame.data[JointData::headPan] = lastHeadPan-maxdiff;
      }
      else if (diff<maxdiff)
      {
        currentFrame.data[JointData::headPan] = headMotionRequest.pan;
      }
      else
      {
        currentFrame.data[JointData::headPan] = lastHeadPan+maxdiff;
      }
    }
    if (currentFrame.data[JointData::headRoll] == jointDataInvalidValue)
    {
      diff=headMotionRequest.roll-lastHeadRoll;
      if (diff<-maxdiff)
      {
        currentFrame.data[JointData::headRoll] = lastHeadRoll-maxdiff;
      }
      else if (diff<maxdiff)
      {
        currentFrame.data[JointData::headRoll] = headMotionRequest.roll;
      }
      else
      {
        currentFrame.data[JointData::headRoll] = lastHeadRoll+maxdiff;
      }
    }
    if (currentFrame.data[JointData::mouth] == jointDataInvalidValue)
    {
      currentFrame.data[JointData::mouth] = headMotionRequest.mouth;
    }

    lastHeadTilt=currentFrame.data[JointData::headTilt];
    lastHeadPan=currentFrame.data[JointData::headPan];
    lastHeadRoll=currentFrame.data[JointData::headRoll];
    
    // the obgligatory GT tail wag
    wagTail(motionRequest,headState.bodyTilt,sensorDataBuffer.lastFrame(), currentFrame);
    
    // whether or not stabilize does something is determined in the method.
    // it is called because of averaging that needs to be done continously!!
    stabilize(lastMotionType, motionRequest, currentFrame, odometryData, sensorDataBuffer);
    if (stand) breathingMotion(motionRequest, currentFrame);

    if (motionRequest.breathe)
    {
      if(sin((double)SystemCall::getCurrentSystemTime()/200.0) > 0)
        currentFrame.data[JointData::mouth] = (long)(-100000 * (1 - cos((double)SystemCall::getCurrentSystemTime()/100.0)));
    }
  }
}

bool DefaultMotionControl::handleMessage(InMessage& message)
{ 
	switch(message.getMessageID())
  {
  case idGenericDebugData:
    {
      GenericDebugData d;
      message.bin >> d;
      if(d.id == GenericDebugData::motionStabilizer)
      {
        OUTPUT(idText,text,"generic debug message (motionStabilizer) handled by module motionStabilizer");
        
        xFore.setWeightP(d.data[0]);
        xHind.setWeightP(d.data[0]);
        yLeft.setWeightP(d.data[1]);
        yRight.setWeightP(d.data[1]);
        
        stabilizerScale = d.data[2];
      }
    }
    break;
  case idMotionNet:
    return pSpecialActions->handleMessage(message);
  case idOdometryScale:
    return pWalkingEngine[MotionRequest::normal]->handleMessage(message);
  case idInvKinWalkingParameters:
    return pWalkingEngine[MotionRequest::debug]->handleMessage(message);
  }
  return false;
}


/*
* Change log :
* 
* $Log: DefaultMotionControl.cpp,v $
* Revision 1.27  2004/05/22 14:28:56  juengel
* Activated WakeUpEngine
*
* Revision 1.26  2004/05/19 15:26:58  juengel
* WakeUpEngine
*
* Revision 1.25  2004/05/19 13:33:21  heinze
* added conditions for WakeUpEngine
*
* Revision 1.24  2004/05/03 09:51:18  heinze
* Prepared WakeUpEngine.
*
* Revision 1.23  2004/04/08 15:33:06  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.22  2004/04/07 12:28:59  risler
* ddd checkin after go04 - first part
*
* Revision 1.21  2004/03/29 17:59:29  dueffert
* sorry
*
* Revision 1.20  2004/03/29 15:24:39  dueffert
* new turn parameterset
*
* Revision 1.19  2004/03/26 21:29:44  roefer
* Headstate handling improved
*
* Revision 1.18  2004/03/26 10:01:47  dueffert
* warning removed
*
* Revision 1.17  2004/03/25 09:11:57  jhoffman
* breathing motion added (robot opens/closes mouth)
*
* Revision 1.16  2004/03/22 21:58:11  roefer
* True odometry
*
* Revision 1.15  2004/03/20 09:55:25  roefer
* Preparation for improved odometry
*
* Revision 1.14  2004/03/19 11:25:01  juengel
* Added walkType turnWithBall2.
*
* Revision 1.13  2004/03/17 21:08:39  cesarz
* Added turnWithBall3, which points to MSH2004TurnWithBallWalkingParameters.
*
* Revision 1.12  2004/03/17 15:38:51  thomas
* added walkType debug for debugging WalkingEngineParameterSets
* paramSets-messages send by debug-message are stored in the new debug-walkType only
*
* Revision 1.11  2004/03/17 09:32:11  jhoffman
* tail wag adjusted to work with ERS7
*
* Revision 1.10  2004/03/08 01:39:02  roefer
* Interfaces should be const
*
* Revision 1.9  2004/03/04 18:15:24  juengel
* removed unused comment
*
* Revision 1.8  2004/02/16 18:02:01  dueffert
* packageCognitionMotion extended with invkin parameters
*
* Revision 1.7  2004/01/21 14:31:58  loetzsch
* Module Selectors create only the selected solution.
* When the solution changes, the old solution is erased and the new
* one ist created using createSolution(..)
*
* Revision 1.6  2004/01/05 17:56:27  juengel
* Added HeadControlModes stayAsForcedPan and stayAsForcedTilt.
*
* Revision 1.5  2004/01/05 11:37:08  juengel
* Added new TailModes: wagUpDown* and wagLeftRight*.
*
* Revision 1.4  2003/12/31 23:50:38  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.3  2003/12/16 19:02:45  loetzsch
* The motion net file Config/spec_act.dat can be sent through WLAN to a robot.
*
* Revision 1.2  2003/12/10 16:52:52  roefer
* Added sensor data buffer to walking engine
*
* Revision 1.1  2003/10/06 14:10:13  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.6  2003/09/26 11:38:52  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.5  2003/09/25 10:11:57  juengel
* Removed some walk-types
*
* Revision 1.4  2003/08/04 07:54:39  risler
* no message
*
* Revision 1.3  2003/07/24 12:41:26  risler
* ParamInvKinWalkingEngine no longer is template class -> 10min boot bug fixed
*
* Revision 1.2  2003/07/02 15:42:32  loetzsch
* esoteric change (but it works now)
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.38  2003/06/13 17:29:25  jhoffman
* added debug message handling
*
* Revision 1.37  2003/05/27 16:17:58  juengel
* Added headIsBlockedBySpecialActionOrWalk.
*
* Revision 1.36  2003/05/26 16:23:37  dueffert
* no message
*
* Revision 1.35  2003/05/23 11:30:12  goehring
* MotionStabilizer signature changed
*
* Revision 1.34  2003/05/23 09:36:51  goehring
* MotionStabilizer gets all 4 Sensor Values
*
* Revision 1.33  2003/05/06 16:52:30  juengel
* Added tail-mode eightPositionSwitch.
*
* Revision 1.32  2003/05/02 18:26:18  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.31  2003/04/28 07:43:57  roefer
* Error in head motion smoothing fixed
*
* Revision 1.30  2003/04/27 07:49:35  roefer
* static vars -> member vars
*
* Revision 1.29  2003/04/19 04:36:48  pruente
* Commited changes done by Arthur at GO2003:
* - some changes
* - some changes
* - dash chanaged to evo parameters
*
* Revision 1.28  2003/04/16 07:00:16  roefer
* Bremen GO checkin
*
* Revision 1.28  2003/04/08 18:28:28  roefer
* bodyTiltOffset added
*
* Revision 1.27  2003/04/02 14:13:39  dueffert
* walkType turnWithBall added
*
* Revision 1.26  2003/04/01 17:44:21  roefer
* Mouth added to HeadMotionRequest
*
* Revision 1.25  2003/03/24 14:46:21  juengel
* Added several tail modes
*
* Revision 1.24  2003/03/10 14:18:39  dueffert
* optimized MotionNet
*
* Revision 1.23  2003/03/06 12:05:43  dueffert
* execute with parameters renamed to avoid inheritance warnings
*
* Revision 1.22  2003/03/03 17:31:53  risler
* walking engine now writes calculated bodyTilt to headState
*
* Revision 1.21  2003/02/22 00:12:07  roefer
* constant bodyTilt set to 17.5
*
* Revision 1.20  2003/02/16 08:30:38  roefer
* Constant body tilt changed to 17
*
* Revision 1.19  2003/02/11 23:12:50  roefer
* useConstantBodyPose debug key added
*
* Revision 1.18  2003/02/06 13:15:14  dueffert
* memory leak fixed
*
* Revision 1.17  2003/02/05 12:23:32  dueffert
* executedMotionRequest is set correctly by SpecialAction and therefore not overwritten by DefaultMotionControl any more
*
* Revision 1.16  2003/01/27 19:27:08  juengel
* added walkType upsideDown
*
* Revision 1.15  2003/01/23 16:44:09  risler
* only one instance of InvKinWalkingEngine
* parameter sets can now be switched while walking
* added UNSWFastTurn, combining two parameter sets
*
* Revision 1.14  2002/12/03 13:07:02  dueffert
* HeadMotionRequests smoothed to avoid destroying heads
*
* Revision 1.13  2002/11/26 17:18:13  risler
* breathing only when standing is executed (not requested)
*
* Revision 1.12  2002/11/26 13:08:05  jhoffman
* no message
*
* Revision 1.11  2002/11/25 14:49:07  jhoffman
* added "breathing" motion
*
* Revision 1.10  2002/11/20 18:02:29  risler
* added PID values to GT2001MotionNetSpecialActions
* added playDead motion
*
* Revision 1.9  2002/11/19 17:12:24  juengel
* "switch" tail modes added
*
* Revision 1.8  2002/10/11 16:59:15  roefer
* Fast tail wags added
*
* Revision 1.7  2002/09/22 18:40:53  risler
* added new math functions, removed GTMath library
*
* Revision 1.6  2002/09/19 12:15:13  loetzsch
* made compilable for the gcc compiler.
*
* Revision 1.5  2002/09/17 23:55:21  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.4  2002/09/16 17:36:04  dueffert
* anonymous contructors return &CLASS with VS, but CLASS with gcc.
*
* Revision 1.3  2002/09/11 13:40:00  loetzsch
* Changed DoxyGen comments
*
* Revision 1.2  2002/09/11 00:06:58  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:15  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.19  2002/09/03 16:00:50  juengel
* HeadControlMode follwTail, TailMode stayAsForced.
*
* Revision 1.18  2002/07/23 13:33:41  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.17  2002/07/07 09:43:27  roefer
* Memory leak closed
*
* Revision 1.16  2002/06/27 11:18:59  risler
* added additional walk type for bar challenge
*
* Revision 1.15  2002/06/22 04:35:50  dueffert
* 4 bar chall
*
* Revision 1.14  2002/06/21 04:53:38  risler
* added walktype walkWithBar
*
* Revision 1.13  2002/06/12 13:50:26  loetzsch
* changed #ifndef WIN32  to #ifndef _WIN32
*
* Revision 1.12  2002/06/12 11:26:28  dueffert
* if timestamp of last MotionRequest is too old, we will only swing on robot, not in RobotControl
*
* Revision 1.11  2002/06/10 11:05:17  risler
* added timestamp to motion request
* motioncontrol executes swing when no request was received
*
* Revision 1.10  2002/06/06 15:50:44  risler
* inverted tail tilt angle
*
* Revision 1.9  2002/06/06 12:51:44  risler
* removed bug with MotionRequest == operator / executed motion request
*
* Revision 1.8  2002/06/04 16:44:55  risler
* tailRequest added
*
* Revision 1.7  2002/06/01 13:38:35  risler
* reorganized walking engine selection
*
* Revision 1.6  2002/05/31 17:22:43  loetzsch
* added module selector for dribble walk type
*
* Revision 1.5  2002/05/16 22:36:11  roefer
* Team communication and GTMath bugs fixed
*
* Revision 1.4  2002/05/16 15:05:49  risler
* added walk type walkWithBall
*
* Revision 1.3  2002/05/15 08:08:53  risler
* changed default walk, added InvKin:Dortmund to selector
*
* Revision 1.2  2002/05/14 13:50:56  rentmeister
* DortmundWalkingParameters
*
* Revision 1.1.1.1  2002/05/10 12:40:15  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.28  2002/05/06 15:57:43  risler
* corrected bodyRoll angle
*
* Revision 1.27  2002/05/05 15:14:23  risler
* changed stand implementation
*
* Revision 1.26  2002/05/04 18:23:43  risler
* added calculation of executedMotionRequest
*
* Revision 1.25  2002/05/04 12:43:38  loetzsch
* The currently executed MotionRequest is now sent from the MotionControl
* to the BehaviorControl via the OdometryData structure
*
* Revision 1.24  2002/04/29 13:48:24  risler
* added lastWalkType to WalkingEngine execute
*
* Revision 1.23  2002/04/26 13:35:33  risler
* DarmstadtGOWalkingEngine renamed to InvKinWalkingEngine
* added InvKinParameterSets
*
* Revision 1.22  2002/04/23 15:08:44  risler
* changed MotionRequest: walk instead of normalWalk,... and walkType added
*
* Revision 1.21  2002/04/23 10:38:31  risler
* renamed headOdometry to headState
*
* Revision 1.20  2002/04/16 16:59:47  dueffert
* newWalkingParameters added
*
* Revision 1.19  2002/04/15 13:39:24  rentmeister
* DoWalkingEngine (Dortmunder WalkingEngine)
*
* Revision 1.18  2002/04/08 09:48:54  rentmeister
* galopWalk fr den Evolver
*
* Revision 1.17  2002/04/05 14:08:43  jhoffman
* stabilizer stuff
*
* Revision 1.16  2002/04/04 15:14:49  jhoffman
* added motion stabilizer
*
* Revision 1.15  2002/04/04 15:07:59  rentmeister
* Walk with Ball hinzugefgt
*
* Revision 1.14  2002/04/03 16:44:31  jhoffman
* added "stabilizeRobot" to motionControl (which is turned off as a default)
*
* Revision 1.13  2002/02/28 16:28:25  risler
* added GT2001WalkingParameters
*
* Revision 1.12  2002/02/23 16:33:07  risler
* finished GetupEngine
*
* Revision 1.11  2002/02/20 16:40:50  risler
* added GetupEngine
*
* Revision 1.10  2002/02/08 20:00:01  risler
* added DebugMotionControl
*
* Revision 1.9  2002/02/08 17:48:57  risler
* SensorData to MotionControl
*
* Revision 1.8  2002/01/19 12:42:32  risler
* fixed bug with solutionHandler
*
* Revision 1.7  2001/12/10 17:47:06  risler
* change log added
*
*/
