// RDefaultStrategy.cpp: Implementierung der Klasse RDefaultStrategy.
//
//////////////////////////////////////////////////////////////////////

#include "RDefaultStrategy.h"
#include "Tools/Location.h"

typedef Geometry::Line Line;

//////////////////////////////////////////////////////////////////////
// Konstruktion/Destruktion
//////////////////////////////////////////////////////////////////////

RDefaultStrategy::RDefaultStrategy(RasterImageProcessor &processor):
	RasterStrategy(processor),
	ballFactor(1),
	fieldFactor(2),
	playerFactor(1),
	goalFactor(1),
	lmFactor(1),
	edgeScanner(processor),
	fieldDSA()
{	
	//load parameters
	InBinaryFile stream(getLocation().getFilename("active.rip"));

	if(stream.exists()){ 
		for (int i = 0;i<RasterStrategy::numberOfActivationKeys;i++)
			stream.read(&active[i],1);
	}
    else{ 
		active[RasterStrategy::preScan] = true;
		active[RasterStrategy::postScan] = true;
		active[RasterStrategy::ball] = true;
		active[RasterStrategy::flag] = true;
		active[RasterStrategy::goal] = true;
		active[RasterStrategy::lines] = true;
		active[RasterStrategy::foes] = true;
	};	
	
	InBinaryFile stream2(getLocation().getFilename("ps.thr"));
	
	if(stream2.exists()) stream2 >> postThreshold;
	else postThreshold = 30;

	OUTPUT(idText,text,"RIP Activated");
	OUTPUT(idText,text,"preScan: " << active[RasterStrategy::preScan]);
	OUTPUT(idText,text,"postScan: " << active[RasterStrategy::postScan]);
	OUTPUT(idText,text,"ball: " << active[RasterStrategy::ball]);
	OUTPUT(idText,text,"flags: " << active[RasterStrategy::flag]);
	OUTPUT(idText,text,"goals: " << active[RasterStrategy::goal]);
	OUTPUT(idText,text,"lines: " << active[RasterStrategy::lines]);
	OUTPUT(idText,text,"foes: " << active[RasterStrategy::foes]);
	OUTPUT(idText,text,"ps-thr: " << postThreshold);
	
	edgeScanner.threshold = postThreshold;

	//configure strategy
	rip->marginX = 2;
	rip->marginY = 2;
	
	rip->rasterWidth  = rip->image.cameraInfo.resolutionWidth  / rip->marginX;
	rip->rasterHeight = rip->image.cameraInfo.resolutionHeight / rip->marginY;
	
	height = rip->rasterHeight;
	width = rip->rasterWidth;


	ballSpecialist = new RBallSpecialist2(processor,*this);
	rip->setSpecialist(ballSpecialist);
	linePattern = green + white;	

	//set red enemy Color as default
	enemyColor = red;
	teamColor = blue;

	enemySpecialist = new REnemySpecialist(processor,*this);
	rip->setSpecialist(enemySpecialist);

	fieldSpecialist = new RFieldSpecialist(processor,*this);
	rip->setSpecialist(fieldSpecialist);

	boxSpecialist = new BoxSpecialist(processor,*this);
	rip->setSpecialist(boxSpecialist);

	bridgeSpecialist = new RBridgeSpecialist(processor,*this);
	rip->setSpecialist(bridgeSpecialist);

	for (int x=0;x<width;x++)
		for (int y=0;y<height;y++)
		{
			//rip->raster[x][y] = Vector2<int>(x*rip->marginX,y*rip->marginY);
			//isEdge[x][y] = false;
		}
		
  //OUTPUT(idText, text, "still alive!!");
}

RDefaultStrategy::~RDefaultStrategy()
{	
  delete ballSpecialist;
  delete fieldSpecialist;
  delete enemySpecialist;
  delete boxSpecialist;
  delete bridgeSpecialist;
}

void RDefaultStrategy::init()
{	
	ballSpecialist->init();
	fieldSpecialist->init(); 
	boxSpecialist->init();
	enemySpecialist->init();
	bridgeSpecialist->init();

	ballSpecialist->preScanNeeded = active[RasterStrategy::ball];
	enemySpecialist->preScanNeeded = active[RasterStrategy::foes];
	boxSpecialist->preScanNeeded = active[RasterStrategy::goal];
	fieldSpecialist->preScanNeeded = true;/*active[RasterStrategy::lines];*/
	bridgeSpecialist->preScanNeeded = true;

	if (!rip->cameraMatrix.isValid)
	{
		enemySpecialist->preScanNeeded = false;
	}

	
	for (int i=0;i<5;i++) diffBuffer[i] = 0;
	width = rip->image.cameraInfo.resolutionWidth/rip->marginX;
	height = rip->image.cameraInfo.resolutionWidth/rip->marginY;
}



void RDefaultStrategy::postProcessing(){	
	if (active[RasterStrategy::ball])  ballSpecialist->executePostProcessing();
	//if (active[RasterStrategy::lines]) fieldSpecialist->executePostProcessing();
	if (active[RasterStrategy::foes])  enemySpecialist->executePostProcessing();
	//TODO(schmidtb): split activation of goal and flag detection.
	if (active[RasterStrategy::goal])  boxSpecialist->executePostProcessing();
	bridgeSpecialist->executePostProcessing();
	fieldSpecialist->executePostProcessing();
}

void RDefaultStrategy::execute()
{
		
	if (active[RasterStrategy::preScan])preScan();
	//if (active[RasterStrategy::postScan])postScan();
	postProcessing();

}


void RDefaultStrategy::preScan()
{
  horizon = rip->getHorizon(); 
  Vector2<double> direction = horizon.direction;
  Vector2<double> base = horizon.base;
  //Gibt an ob bei der Suche unter dem Horizont (oder auch Geraden die parallel zum Horizont verlaufen)
  //vor oder nach dem Schnitt-Punkt mit der jeweiligen Zeile gesucht werden muss
  bool beforeHorizon = (direction.y > 0);
  Vector2<double> pDirection = direction;
  pDirection.rotateLeft();
  pDirection *= 20;
  
  //horizon-direction parameters should not be near 0
  if (0 <= direction.y && direction.y < 0.001 ) direction.y = 0.001;
  if (0 > direction.y && direction.y > -0.001 ) direction.y = -0.001;
  if (0 <= direction.x && direction.x < 0.001 ) direction.x = 0.001;
  if (0 > direction.x && direction.x > -0.001 ) direction.x = -0.001;

  if (fabs(direction.x)<0.25){
	lineXActive = 0;
	lineYActive =1;
  }
  else{
	lineXActive = 0;
	lineYActive = 1;
  }

  Vector2<double> upperBase = base + pDirection;
  Vector2<double> lowerBase = base - pDirection;

  //OUTPUT(idText,text,"h-base: " << horizon.base.x << " " << horizon.base.y);
  //OUTPUT(idText,text,"h-base: " << horizon.base.x << " " << horizon.base.y);
  //OUTPUT(idText,text,"h-base: " << horizon.base.x << " " << horizon.base.y);


//  OUTPUT(idText,text,"before-hor: " << beforeHorizon);	
  
//pre scan, scan on raster in lexical-ordering of (y,x)
  for (int y=1; y<height-1; y++)
	{
  		lastColor    = noColor;
		currentColor = noColor;
		insideBall = false;
		insideBox = false;
		insideField = false;
		insideLM = false;
		insidePlayer = false;	
		insideShirt = false;
		insideBridge = false;
		
		BoxColor = noColor;
		bridgeColor = noColor;

		int cross = 0;
		int upperCross = 0;
		int lowerCross = 0;

		double t = 0;

		t = ((double)(y * rip->marginY ) - base.y)/direction.y;	
		cross = (int)((base.x + t*direction.x)/rip->marginX);

		t = ((double)(y * rip->marginY ) - upperBase.y)/direction.y;
		upperCross = (int)((upperBase.x + t*direction.x)/rip->marginX);

		t = ((double)(y * rip->marginY ) - lowerBase.y)/direction.y;
		lowerCross = (int)((lowerBase.x + t*direction.x)/rip->marginX);


		if (cross < 0) cross = 0;
		else if (cross > width - 1) cross = width - 1;

		if (upperCross < 0) upperCross = 0;
		else if (upperCross > width - 1) upperCross = width - 1;

		if (lowerCross < 0) lowerCross = 0;
		else if (lowerCross > width - 1) lowerCross = width - 1;
		
		if (lowerCross > upperCross)
		{
			int temp = lowerCross;
			lowerCross = upperCross;
			upperCross = temp;
		}

		bool underHorizonFinished = false;
		bool horizonSpacePassed = false;


    for(int x=1; x<width-1; x++)
	{	
		currentColor = getColorFromRaster(x,y);
		checkBridge(x,y);
		checkRedLine(x,y);
		//defining the shape of the scan lines
		if (y%2 == 0 || (x >= lowerCross) && (x < upperCross))
		{	


			//only under horizon
			if((x<cross && beforeHorizon) || (x>cross && !beforeHorizon))
			{						  
				if (ballSpecialist->preScanNeeded &&(y%ballFactor + x%ballFactor == 0))
					checkBall(x,y,0);	
		
				if (enemySpecialist->preScanNeeded && x%playerFactor + y%playerFactor == 0)
					checkPlayer(x,y,0);
			}
			else if (x==cross && beforeHorizon)
			{
				underHorizonFinished = true;

				if (enemySpecialist->preScanNeeded 
					&& (currentColor == gray || currentColor == red || 
					currentColor == blue || currentColor == black))
					enemySpecialist->invokeOnPreScan(x,y);
				if (currentColor == orange && insideBall){
					ballSpecialist->invokeOnPreScan(x,y);
				}
			}
			
			//the whole image (depending on scanlines)
			if (boxSpecialist->preScanNeeded && x%goalFactor + y%goalFactor == 0)
				checkBox(x,y,0);

			lastColor = currentColor;		
		}
		else if(x==upperCross)
		{
			horizonSpacePassed = true;
			checkLastLinePoint(ballFactor,orange,y,(RasterSpecialist&) *(ballSpecialist));
			lastColor = currentColor;
		}
	}
	
	if(y%2 == 0 && !underHorizonFinished)
	{			
		checkLastLinePoint(ballFactor,orange,y,(RasterSpecialist&) *(ballSpecialist));

	}
	checkLastLinePoint(goalFactor,BoxColor,y,(RasterSpecialist&) *(boxSpecialist));
	checkLastLinePoint(1,bridgeColor,y,(RasterSpecialist&) *(bridgeSpecialist));
	checkLastLinePoint(1,red,y,(RasterSpecialist&) *(fieldSpecialist));
  }
}

void RDefaultStrategy::postScan(){ 
	//making perpendicular scans to the horizon like GridImageProcessor does
	//TODO(schmidtb): CODE CLEANUP !!!!!
	horizon = rip->getHorizon();
	Geometry::Line cHorizon;
	cHorizon.direction.x = horizon.direction.y;
	cHorizon.direction.y = -horizon.direction.x;

	cHorizon.base.x = (rip->image.cameraInfo.resolutionWidth - 1)/2;
	cHorizon.base.y = (rip->image.cameraInfo.resolutionHeight -1)/2;
	Vector2<double>horCenter;
	Geometry::getIntersectionOfLines(horizon,cHorizon,horCenter);

	int margin = 3;
	Vector2<int> leftBorder(
		(int)(horCenter.x - horizon.direction.x * 500),
		(int)(horCenter.y - horizon.direction.y * 500));
	Vector2<int> rightBorder(
		(int) (horCenter.x + horizon.direction.x * 500),
		(int) (horCenter.y + horizon.direction.y * 500));
	Vector2<int> topRight(
		rip->image.cameraInfo.resolutionWidth-2,
		rip->image.cameraInfo.resolutionHeight-2);
	Vector2<int> bottomLeft(1,1);

	Geometry::clipLineWithRectangleCohenSutherland(
		bottomLeft,topRight,leftBorder,rightBorder);
	int horizonWidth = (leftBorder - rightBorder).abs();
	int steps = (int)((horizonWidth/(double)margin)*1.1);
	//int count = 0;

	Vector2<int> leftToRight = rightBorder - leftBorder;	
	Vector2<double> south = horizon.direction;
	south.rotateLeft();
	Vector2<double> step = horizon.direction*margin;
	Vector2<double> temp;
	Vector2<int> scan;
	Vector2<int> hScan;
	bool found = true;

	hScan.x = (int)((horCenter.x - step.x*(steps/2)) + 0.5);
	hScan.y = (int)((horCenter.y - step.y*(steps/2)) + 0.5);
	DOT(imageProcessor_general, hScan.x,hScan.y,
		Drawings::red, Drawings::red);
	/*OUTPUT(idText,text,"hScan: " << hScan.x << " " << hScan.y);
	OUTPUT(idText,text,"step: " << step.x << " " << step.y);
	OUTPUT(idText,text,"steps: " << steps);
	OUTPUT(idText,text,"base: " << horizon.base.x << " " << horizon.base.y);
	OUTPUT(idText,text,"leftBorder: " << leftBorder.x << " " << leftBorder.y);
	OUTPUT(idText,text,"horCenter: " << hScan.x << " " << hScan.y);
	*/
	edgeScanner.threshold = 50;
	Vector2<double> dir = south;
	//OUTPUT(idText,text,"dir: " << dir.x << " " << dir.y);
	
	Vector2<int> edge;
//	colorClass color = noColor;
	for (int i=0;i<steps;i++)
	{
		scan.x = (int)((double)hScan.x + (double)i*step.x + 0.5);
		scan.y = (int)((double)hScan.y + (double)i*step.y + 0.5);
		if (scan.x < 1 || scan.x > rip->image.cameraInfo.resolutionWidth-2 
			||scan.y < 1 || scan.y > rip->image.cameraInfo.resolutionHeight-2)
		{ 
			// continue, if scanline don't intersects the image
			if(!edgeScanner.findStart(scan,dir))
				continue;
		}

		edgeScanner.setDirection(dir);
		fieldDSA.reset(scan.x,scan.y);
		//fill edge-buffer and classify points with state-machine
		do{
			found = edgeScanner.bufferedScan(scan.x,scan.y);
			fieldDSA.update(scan.x,scan.y,edgeScanner);			
		}while (found);
		
		if (fieldDSA.foundObstacle())
		{
			rip->addObstaclePoints(
				fieldDSA.nearPoint(),
				fieldDSA.farPoint(),
				fieldDSA.getObstacleType());
		}

		for (int j = 0;j<fieldDSA.size();j++)
		{
			Vector2<int> point = fieldDSA.getEdge(j);
//			int test = fieldDSA.getType(j);

			switch(fieldDSA.getType(j)){
			case RFieldStateMachine::line: 
				rip->addFieldPoint(point.x,point.y,LinesPercept::field);
				//fieldSpecialist->addLinePoint(LinesPercept::field,point,i);
				DOT(imageProcessor_general, point.x,point.y,
					Drawings::white, Drawings::skyblue);
				break;
			case RFieldStateMachine::border: 
				rip->addFieldPoint(point.x,point.y,LinesPercept::border);
				//fieldSpecialist->addLinePoint(LinesPercept::border,point,i);
				DOT(imageProcessor_general, point.x,point.y,
					Drawings::green, Drawings::skyblue);
				break;
			case RFieldStateMachine::yellowGoal: 
				rip->addFieldPoint(point.x,point.y,LinesPercept::yellowGoal);
				//fieldSpecialist->addLinePoint(LinesPercept::yellowGoal,point,i);
				if (active[RasterStrategy::goal]) 
					boxSpecialist->invokeOnPostScan(point.x,point.y);
				DOT(imageProcessor_general, point.x,point.y,
					Drawings::yellow, Drawings::skyblue);
				break;
			case RFieldStateMachine::skyblueGoal: 
				rip->addFieldPoint(point.x,point.y,LinesPercept::skyblueGoal);
				//fieldSpecialist->addLinePoint(LinesPercept::skyblueGoal,point,i);
				if (active[RasterStrategy::goal]) 
					boxSpecialist->invokeOnPostScan(point.x,point.y);
				DOT(imageProcessor_general, point.x,point.y,
					Drawings::skyblue, Drawings::skyblue);
				break;
			case RFieldStateMachine::redRobot: 
				DOT(imageProcessor_general, point.x,point.y,
					Drawings::skyblue, Drawings::red);
				break;
			case RFieldStateMachine::blueRobot:
				DOT(imageProcessor_general, point.x,point.y,
					Drawings::skyblue, Drawings::blue);
				break;
			case RFieldStateMachine::ball:
				DOT(imageProcessor_general, point.x,point.y,
					Drawings::orange, Drawings::orange);
				break;
			default: 
				DOT(imageProcessor_general, point.x,point.y,
					Drawings::black, Drawings::skyblue);
				break;
			}
		}
	}
}

void RDefaultStrategy::checkPlayer(int x,int y,int prePost){
	
	bool isNewPoint = (currentColor == red || blue || gray || black);	
	bool isClosingPoint = (currentColor == green || currentColor == white);

	if (!insidePlayer){
		if (isNewPoint){
		if (prePost == 0) enemySpecialist->invokeOnPreScan(x,y);
		else enemySpecialist->invokeOnPostScan(x,y);
		insidePlayer = true;
		}
	}
	else if (isClosingPoint){
		if (prePost == 0) enemySpecialist->invokeOnPreScan(x-playerFactor,y);
		else enemySpecialist->invokeOnPostScan(x,y+playerFactor);
		insidePlayer = false;
	}
}

void RDefaultStrategy::checkBall(int x,int y,int prePost){
	bool isNewPoint = currentColor == orange;	
	if (!insideBall){
		if (isNewPoint){
			if (prePost == 0) ballSpecialist->invokeOnPreScan(x,y);
			else ballSpecialist->invokeOnPostScan(x,y);
			insideBall = true;
		}
	}
	else if (!isNewPoint){
		if (prePost == 0) ballSpecialist->invokeOnPreScan(x-ballFactor,y);
		else ballSpecialist->invokeOnPostScan(x,y+ballFactor);
		insideBall = false;
	}

/*	if (!insideBall){
		if (isNewPoint){
			ballSpecialist->in(x,y);
			insideBall = true;
		}
	}
	else if (!isNewPoint){
		ballSpecialist->out(x-ballFactor,y);
		insideBall = false;
	}
*/
}

void RDefaultStrategy::checkField(int x,int y,int prePost){

	bool isNewPoint;

	isNewPoint = (currentColor == white);

	if (!insideField){
		if (isNewPoint){
			if (prePost == 0) fieldSpecialist->invokeOnPreScan(x,y);
			else fieldSpecialist->invokeOnPostScan(x,y);
			insideField = true;
		}
	}
	else if(!isNewPoint){
		if (prePost == 0) fieldSpecialist->invokeOnPreScan(x-fieldFactor,y);
		else fieldSpecialist->invokeOnPostScan(x,y+fieldFactor);
		insideField = false;
	
	}
}


void RDefaultStrategy::checkBox(int x, int y,int prePost){

	bool newBoxPoint;

	
	newBoxPoint = (currentColor == yellow || currentColor == skyblue || currentColor == pink);


	if(!insideBox) {
		if (newBoxPoint) {

			boxSpecialist->invokeOnPreScan(x,y);			
			BoxColor = currentColor;
			insideBox = true;
		}
	}
	else if(currentColor!=BoxColor){

		boxSpecialist->invokeOnPreScan(x-goalFactor,y);
		BoxColor = noColor;
		insideBox = false;
	} 
	
}

void RDefaultStrategy::checkRedLine(int x,int y){
	bool newRedPoint = currentColor == red;

	if (!insideField) {
		if (newRedPoint){
			fieldSpecialist->invokeOnPreScan(x,y);
			insideField = true;
		}
	}
	else if (!newRedPoint){
		fieldSpecialist->invokeOnPreScan(x,y);
		insideField = false;
	}
}

void RDefaultStrategy::checkBridge(int x, int y){

	bool newBoxPoint = 
		(currentColor == yellow || currentColor == skyblue || currentColor == orange);
	
	if(!insideBridge) {
		if (newBoxPoint) {

			bridgeSpecialist->invokeOnPreScan(x,y);			
			bridgeColor = currentColor;
			insideBridge = true;
		}
	}
	else if(currentColor!=bridgeColor){

		bridgeSpecialist->invokeOnPreScan(x-goalFactor,y);
		bridgeColor = noColor;
		insideBridge = false;
	} 
	
}

void RDefaultStrategy::checkLastLinePoint(int factor,colorClass color,
										  int line,RasterSpecialist& spec){
		if (color == getColorFromRaster(width - width%factor -1 ,line))
			spec.invokeOnPreScan(width - width%factor - 1 ,line);
}


/*
* Change log :
* 
* $Log: RDefaultStrategy.cpp,v $
* Revision 1.78  2004/06/15 12:36:45  kerdels
* enabled line detection in rip and adjusted align-to-ramp
*
* Revision 1.77  2004/06/09 13:05:57  schmidtb
* removed memory leaks
*
* Revision 1.76  2004/05/27 11:18:48  schmidtb
* new version with further work for open challenge
*
* Revision 1.74  2004/05/25 13:27:34  schmidtb
* modified version of rip for open-challenge
*
* Revision 1.73  2004/05/24 15:44:53  deom
* merged vesions of RastImageProcessor from hierophant to tamara
*
* Revision 1.72  2004/05/24 14:45:04  deom
* commented out bridgeSpecialist
*
* Revision 1.71  2004/05/22 19:03:40  deom
* -now recognizes the red line
* -all the data required for the positionning are computed
*
* Revision 1.71  2004/05/22 16:01:50  pg_besc
* -modified version of rip for bridge-recognition
*
* Revision 1.70  2004/04/22 16:57:26  pg_besc
* new version of RBallSpecialist2, now omidirectional scans for detection
*
* Revision 1.69  2004/04/20 07:50:27  pg_besc
* new version of pre scan
*
* Revision 1.68  2004/04/15 19:09:17  pg_besc
* merged code
*
* Revision 1.67  2004/03/25 15:27:54  pg_besc
* made some changes
*
* Revision 1.66  2004/03/17 22:31:34  schmidtb
* integrated RFieldStateMachine and horizonal grid scan
*
* Revision 1.65  2004/03/17 18:27:45  koh
* warnings and errors removed
*
* Revision 1.64  2004/03/17 17:05:18  koh
* added enemyOnlySpecialist to strategy;
* added enemySpecialist2 to strategy;
*
* Revision 1.63  2004/03/14 12:51:57  deom
* added an accurate scan method
*
* Revision 1.62  2004/03/12 14:43:27  deom
* *** empty log message ***
*
* Revision 1.61  2004/03/11 20:49:39  schmidtb
* new version of rip , merged code
*
* Revision 1.60  2004/03/11 12:19:00  deom
* debbuged RFieldspecialist
*
* Revision 1.59  2004/03/09 09:37:57  neubach
* - added enemyColors as startingPoints for linepairs
*
* Revision 1.58  2004/03/04 09:54:46  schmidtb
* color correction integrated
*
* Revision 1.57  2004/03/03 14:05:31  hamerla
* - black added
*
* Revision 1.56  2004/03/03 12:53:20  schmidtb
* color correction integrated
*
* Revision 1.55  2004/03/01 14:17:24  koh
* added new strategy "RFlexibleStrategy" + new specialist "EnemyOnlySpecialist";
* changed references to "RDefaultStrategy" to references to "RasterStrategy" in RFieldSpecialist
* added Geometry::Line horizon to "RasterStrategy"
*
* Revision 1.54  2004/02/28 17:16:47  schmidtb
* debugged and made some changes
*
* Revision 1.53  2004/02/26 14:31:32  schmidtb
* comment some outputs
*
* Revision 1.52  2004/02/25 15:46:14  schmidtb
* enabled scans
*
* Revision 1.51  2004/02/22 19:12:54  deom
* now recognize simple lines
*
* Revision 1.50  2004/02/19 14:22:15  neubach
* - debugged Code of RIP
*
* Revision 1.49  2004/02/18 16:35:46  schmidtb
* removed errors
*
* Revision 1.48  2004/02/18 15:58:26  jess4279
* fixed comment
*
* Revision 1.47  2004/02/18 14:56:19  neubach
* new Segmentation established, code not cleared at all
*
* Revision 1.43  2004/02/05 11:51:18  schmidtb
* now BoxSpecialist recognizes landmarks and goals
*
* Revision 1.42  2004/02/04 23:19:38  schmidtb
* debugged and added calculation for angles to BoxSpecialist
*
* Revision 1.41  2004/02/04 19:09:32  schmidtb
* *** empty log message ***
*
* Revision 1.40  2004/02/04 15:05:20  schmidtb
* warnings removed
*
* Revision 1.39  2004/02/04 13:00:49  schmidtb
* new version of BoxSpecialist
*
* Revision 1.38  2004/02/03 23:09:18  hyung
* Array based on rasterWidth now
*
* ! somehow it seems that my code produces warnings for which I'm
* very very sorry, but my compiler told me always 0 errors, 0 warnings....
* does somebody know the reason?
*
* Revision 1.37  2004/02/03 21:18:27  hyung
* some minor changes,
*
* size of Segmentborder-Array (TrikotErkennung) basically adaptive to rip-resolution
*
* Revision 1.36  2004/02/03 11:47:55  neubach
* - ballSpecialist and enemySpecialist enabled
* - ball && enemy && (box || field) consumes too much cpu-power!?
*
* Revision 1.35  2004/02/02 13:42:12  schmidtb
* merged sources of RIP. added som functions.
*
* Revision 1.34  2004/01/31 11:45:02  hyung
* modified enemyValidity-calculation;
* established basical enviroment for TrikotErkennung, based on Arrays and Lists, changes will take affect only #ifdef TrikotErkennung!
*
* Revision 1.33  2004/01/26 22:21:09  schmidtb
* bugfixed BoxSpecialist
*
* Revision 1.32  2004/01/26 20:37:56  schmidtb
* removed errors and warnings, merged versions of RGoalSpecialist
*
* Revision 1.31  2004/01/23 15:44:01  deom
* New specialist :: BoxSpecialist
* Recognizes both landmarks and goals
*
* Revision 1.30  2004/01/23 10:25:02  neubach
* - warning removed
*
* Revision 1.29  2004/01/21 11:04:30  loetzsch
* removed some OUTPUTs
*
* Revision 1.28  2004/01/21 10:38:08  schumann
* warnings removed
*
* Revision 1.27  2004/01/21 03:25:36  hyung
* TrikotErkennung update
*
* Revision 1.26  2004/01/20 17:10:01  deom
* no message
*
* Revision 1.25  2004/01/20 10:23:57  schmidtb
* fixed bugs in RGoalSpecialist
*
* Revision 1.24  2004/01/19 16:18:45  schmidtb
* GoalSpecialist recognizes Landmarks
*
* Revision 1.23  2004/01/19 12:47:42  hyung
* #ifDef TrikotErkennung mit alternativem Einstiegspunkt fr Gegnererkennung
*
* Revision 1.22  2004/01/17 13:59:31  schmidtb
* scan under horizon fixed
*
* Revision 1.21  2004/01/16 17:25:06  deom
* no message
*
* Revision 1.20  2004/01/12 15:10:37  schmidtb
* new GoalSpecialist
*
* Revision 1.17  2003/12/18 17:44:46  hyung
* established new version of method checkGoal(int x,int y)
*
* Revision 1.16  2003/12/17 16:38:04  schmidtb
* Merged versions of RDefaultStrategy.
*
* Revision 1.15  2003/12/16 13:45:28  roefer
* Warnings removed
*
* Revision 1.14  2003/12/16 10:29:18  neubach
* - added calculation of farest point (from horizon) for a playerSegment
* - added conversion point on cameraPicture -> point on Field
*
* Revision 1.13  2003/12/15 14:28:34  juengel
* no message
*
* Revision 1.12  2003/12/15 13:55:32  schmidtb
* Merged and patched new version of RasterImageProcessor.
*
* Revision 1.11  2003/12/15 11:49:08  juengel
* Introduced CameraInfo
*
* Revision 1.10  2003/12/09 14:14:34  schumann
* removed code that caused assertions
*
* Revision 1.9  2003/12/08 15:07:53  schmidtb
* new version of RIP
*
* Revision 1.8  2003/12/08 13:15:00  dueffert
* memory leak fixed
*
* Revision 1.7  2003/12/04 09:51:23  schmidtb
* better BallSpecialist
*
* Revision 1.6  2003/12/02 21:59:02  schmidtb
* New version of RasterImageProcessor
*
* Revision 1.5  2003/11/29 07:37:39  roefer
* Removed memory leaks
*
* Revision 1.4  2003/11/28 14:50:01  dueffert
* bugs and warnings fixed
*
* Revision 1.3  2003/11/20 10:26:56  schmidtb
* Ball Detection added
*
* Revision 1.2  2003/11/13 10:41:29  schmidtb
* renewed RBallSpeciaslist and Strategy
*
* Revision 1.1  2003/11/12 13:13:20  schmidtb
* new RasterImageProcessor added
*
*
*/
