/**
* @file MSH2004PixelFilter.h
*
* Definition (and implementation) of class MSH2004PixelFilter
*
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/

#ifndef __MSH2004PixelFilter_h_
#define __MSH2004PixelFilter_h_

//#define CT32K_LAYOUT

#include "../ImageProcessorTools/FastSUSANNoiseReduction.h"
#include "../ImageProcessorTools/MSH2004ColorCorrector.h"
#include "Tools/Math/Geometry.h"
#ifdef CT32K_LAYOUT
#include "Representations/Perception/ColorTable32K.h"
#else
#include "Representations/Perception/ColorTable64.h"
#endif

/**
* @class MSH2004PixelFilter
*
* Utility class which applies pixel-level corrections to images, making sure that even
* if a pixel is scanned multiple times, the correction is applied only once.
* This is needed especially for color correction, because repetead applications result
* in color distortion (in the opposite direction of the original one).
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/ 
class MSH2004PixelFilter
{
public:

  /** Smoothing threshold, controls the amount of blur and noise suppression in the filtering */
  enum {SMOOTHING = 16};

  /** Constructor */
#ifdef CT32K_LAYOUT  
  MSH2004PixelFilter(ColorTable& colorTable): noiseFilter(SMOOTHING), colorTable((ColorTable32K&)colorTable)
#else
  MSH2004PixelFilter(ColorTable& colorTable): noiseFilter(SMOOTHING), colorTable((ColorTable64&)colorTable)
#endif  
  {
    geometryCorrection.setupRadialCorrectionERS7();
    reset();
  };

  /** Default destructor 
  * ~MSH2004PixelFilter();
  */
  
  /** Reset function, clears the touchedPixel matrix for the processing of a new frame*/
  inline void reset()
  {
    memset(touchedPixel, 0, sizeof(touchedPixel));
  }
  
  /** Filters a pixel in an image.
  * Performs noise reduction and color correction, but no geometrical correction
  * If repeatedly invoked on the same pixel, it does nothing.
  * @param img The source image
  * @param x X coordinate of the given pixel
  * @param y Y coordinate of the given pixel
  */
  inline void filterPixel(Image* img, int x, int y)
  {
    if (!touchedPixel[y][x]){
      noiseFilter.filterPixel((*img), x, y);
      colorCorrector.correctPixel((*img), x, y);
      touchedPixel[y][x] = true;
    }
  }

  /** Calculates the color class of a pixel in an image.
  * Performs noise reduction, color correction, and geometrical (radial) correction
  * If repeatedly invoked on the same pixel, fetches the previously computed result
  * from its internal cache.
  * @param img The source image
  * @param x X coordinate of the given pixel
  * @param y Y coordinate of the given pixel
  * @return The color class of the given pixel
  */
  inline colorClass getColorClass(const Image& img, int x, int y)
  {
    int sX, sY;
    unsigned char Y, U, V, color;
    if (!touchedPixel[y][x])
    {
      geometryCorrection.radialDistortionCorrectionFast(x, y, sX, sY);
      noiseFilter.getFilteredPixel(img, sX, sY, Y, U, V);
      colorCorrector.getCorrectedPixel(sX, sY, Y, U, V);
#ifdef CT32K_LAYOUT
      color = colorTable.getColorClassFast(Y,U,V);
#else
			color = colorTable.colorClasses[Y>>2][U>>2][V>>2];
#endif
      colorCache[y][x] = color;
      touchedPixel[y][x] = true;
    }
    else 
    {
      color = colorCache[y][x];
    }
    return (colorClass) color;
  }

  /** Returns the chosen filtered spectrum of a pixel in an image.
  * Performs color correction, and geometrical (radial) correction, no noise reduction.
  * @param img The source image
  * @param x X coordinate of the given pixel
  * @param y Y coordinate of the given pixel
  * @return The color class of the given pixel
  */
  inline unsigned char getSpectrum(const Image& img, int x, int y, int spectrum)
  {
    int sX, sY;
    geometryCorrection.radialDistortionCorrectionFast(x, y, sX, sY);
    unsigned char color;
    color = img.image[sY][spectrum][sX];
    colorCorrector.getCorrectedColor(sX, sY, color, spectrum);
    return color;
  }

  /**  Copies a given image in its internal cache, to be used as a pixel 
  * source by the filterImagePixel function.
  * @param img The source image
  */
  inline void setSource(const Image& img)
  {
    pixelCache = img;
  }
  
  /** Filters a pixel from the previously set source (see above method) and stores the result
  * in a target destination image.
  * Performs noise reduction, color correction, and geometrical (radial) correction
  * @param dst Destination image pointer
  * @param x X coordinate of the given pixel
  * @param y Y coordinate of the given pixel
  */
  inline void filterImagePixel(Image* dst, int x, int y)
  {
    int sX, sY;
    geometryCorrection.radialDistortionCorrectionFast(x, y, sX, sY);
    if (!touchedPixel[sY][sX]){
      noiseFilter.filterPixel(pixelCache, sX, sY);
      colorCorrector.correctPixel(pixelCache, sX, sY);
      touchedPixel[sY][sX] = true;
    }
    for (int i=0; i<3; i++)
      dst->image[y][i][x] = pixelCache.image[sY][i][sX];
  }
  
private:

  /** ColorTable reference*/
#ifdef CT32K_LAYOUT
  const ColorTable32K& colorTable;
#else
  const ColorTable64& colorTable;
#endif
  /** Source image cached */
  Image pixelCache;

  /** Color classified pixel cache */
  unsigned char colorCache[cameraResolutionHeight_ERS7][cameraResolutionWidth_ERS7];

  /** Geometrical distortion correction filter */
  Geometry geometryCorrection;

  /** Noise reduction filter */
  const FastSUSANNoiseReduction noiseFilter;

  /** Color correction filter */
  const MSH2004ColorCorrector colorCorrector;

  /** Keeps track of the pixels which have already been processed in the current frame */
  bool touchedPixel[cameraResolutionHeight_ERS7][cameraResolutionWidth_ERS7]; 
  
};

#endif

/*
* Change log :
* 
* $Log: MSH2004PixelFilter.h,v $
* Revision 1.5  2004/05/19 09:46:21  roefer
* Removed definition of CT32K_LAYOUT because it influences other image processors
*
* Revision 1.4  2004/05/17 11:57:04  nistico
* Imported ball detection from RasterImageProcessor (integrated in MSH by Bernd in New Orleans)
*
* Revision 1.3  2004/05/14 16:31:01  nistico
* Compilation switch ColorTable32K <-> ColorTable64
*
* Revision 1.2  2004/04/18 14:48:32  nistico
* Cleaned up and better documented
*
* Revision 1.1  2004/04/18 11:57:46  nistico
* Removed MSH2004ImageProcessor2 (integrated all changes into MSH2004ImageProcessor)
*
* Revision 1.1  2004/04/08 16:21:04  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.2  2004/03/29 11:16:15  nistico
* Compiles, and runs very well.
* It goes to replace the first version, after a bit more of testing
*
* Revision 1.1  2004/03/28 18:35:21  nistico
* Temporarily created MSH2004ImageProcessor2, it has some important advancements compared to
* the MSH2004ImageProcessor1 but it is untested on real robots yet, hence the creation of the
* new module.
* It will be removed soon, don't use it!
*
*
*/
