/**
* @file MSH2004GoalRecognizer.h
*
* Definition of class MSH2004GoalRecognizer
*
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#ifndef __MSH2004GoalRecognizer_h_
#define __MSH2004GoalRecognizer_h_

#include "Representations/Perception/Image.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/ColorTable.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "MSH2004PixelFilter.h"

#include "Tools/Math/Geometry.h"
#include "Tools/Debugging/DebugImages.h"

/**
* @class MSH2004GoalRecognizer
*
* This is a (slightly) modified version of GT GoalRecognizer which supports MSH stuff
*
* The goal recognizer recognizes the goals in the image.
*
* The recognizer scans horizontal lines in the image left to right.
* Indications for a goal are grouped and checked vertical.
*
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/ 

class MSH2004GoalRecognizer //: public ImageProcessorInterfaces
{
public:
  MSH2004GoalRecognizer(
    const Image& image, 
    const CameraMatrix& cameraMatrix, 
    const ColorTable& colorTable, 
    MSH2004PixelFilter& advPixFilter,
    ObstaclesPercept& obstaclesPercept,
    LandmarksPercept& landmarksPercept,
    int border
    );

  MSH2004GoalRecognizer(
    const Image& image, 
    const CameraMatrix& cameraMatrix, 
    const ColorTable& colorTable, 
    MSH2004PixelFilter& advPixFilter,
    int goalIndicationAboveHorizon,
    int goalIndicationBelowHorizon,
    ObstaclesPercept& obstaclesPercept,
    LandmarksPercept& landmarksPercept,
    int border
    );
  
  ~MSH2004GoalRecognizer();

  void getGoalPercept(LandmarksPercept& landmarksPercept);

  struct ColoredPartsCheck
  {
    Vector2<int> firstPoint;
    Vector2<int> lastPoint;
    int rangeOfColor;
    int numberOfColoredPixels;
    
    enum{maxNumberOfParts = 20};
    
    int numberOfLargeParts;
    int sizeOfLargePart[maxNumberOfParts];
    Vector2<int> largePartBegin[maxNumberOfParts];
    Vector2<int> largePartEnd[maxNumberOfParts];
    Vector2<double> largePartBeginAngles[maxNumberOfParts];
    Vector2<double> largePartEndAngles[maxNumberOfParts];
    bool largePartBeginIsOnBorder[maxNumberOfParts];
    bool largePartEndIsOnBorder[maxNumberOfParts];
    
    ColoredPartsCheck() 
    {
      numberOfLargeParts = 0;
    }
    
    bool determineLargePart(int size, bool beginIsOnBorder, bool endIsOnBorder, CameraMatrix cameraMatrix, CameraInfo cameraInfo)
    {
      bool foundLargePart = false;
//      if(numberOfColoredPixels > 3)
      {
        if(rangeOfColor > size)
        {
          sizeOfLargePart[numberOfLargeParts] = rangeOfColor;
          largePartBegin[numberOfLargeParts].x = firstPoint.x;
          largePartBegin[numberOfLargeParts].y = firstPoint.y;
          largePartEnd[numberOfLargeParts].x = lastPoint.x;
          largePartEnd[numberOfLargeParts].y = lastPoint.y;
          largePartBeginIsOnBorder[numberOfLargeParts] = beginIsOnBorder;
          largePartEndIsOnBorder[numberOfLargeParts] = endIsOnBorder;
          
          Vector2<double> minAngles, maxAngles;
          Geometry::calculateAnglesForPoint(largePartBegin[numberOfLargeParts], cameraMatrix, cameraInfo, largePartBeginAngles[numberOfLargeParts]);
          Geometry::calculateAnglesForPoint(largePartEnd[numberOfLargeParts], cameraMatrix, cameraInfo, largePartEndAngles[numberOfLargeParts]);
          
          numberOfLargeParts++;
          foundLargePart = true;
          LINE(imageProcessor_flagsAndGoals, firstPoint.x, firstPoint.y, lastPoint.x, lastPoint.y,
            2, Drawings::ps_solid, Drawings::pink);
        }
        
        if(numberOfLargeParts >= maxNumberOfParts)
        {
          numberOfLargeParts = maxNumberOfParts-1;
        }
      }
      return foundLargePart;
    }
  };

private:

  MSH2004PixelFilter& advPixFilter;

  /** width of the image beavel not to be touched by the image processor in case a convolutional operator is applied 
  * (to prevent access out of allocated memory space) */
  int border; 

  /** Calculates the scan lines near the horizon for the goals */
  void calculateScanLinesParallelToHorizon();
  
  /** Calculates the scan lines near the horizon for the goals */
  void calculateScanLinesParallelToHorizon(
    int distanceAboveHorizon,
    int distanceBelowHorizon,
    int numberOfScanLines
    );
  
  /** Scans horizontal for goals */
  void scanHorizontalForGoals();

  /** Calculates the vertical scan lines for the goals */
  void calculateVerticalGoalScanLines();

  /** Scans vertical for goals */
  void scanLinesForGoals();

  DECLARE_DEBUG_IMAGE(imageProcessorGoals);

  /** A reference to the image that is scanned for a goal */
  const Image& image;

  /** A reference to the camera matrix that describes position and rotation of the camera when the image was aquired */
  const CameraMatrix& cameraMatrix;

  /** A reference to the color table */
  const ColorTable& colorTable;

  int goalIndicationAboveHorizon;
  int goalIndicationBelowHorizon;
  bool useFixedScanLines;

  /** A reference to the obstacles percept */
  ObstaclesPercept& obstaclesPercept;

  /** A reference to the obstacles percept */
  LandmarksPercept& landmarksPercept;

  /** The color class of the opponent goal */
  colorClass colorOfOpponentGoal;

  /** The color class of the own goal */
  colorClass colorOfOwnGoal;

  /** A representation of the horizon */
  Geometry::Line horizonLine, verticalLine;

  /** The number of horizontal scan lines */
  int numberOfHorizontalScanLines;

  enum{maxNumberOfHorizontalScanLines = 32};
  enum{maxNumberOfGoalScanLines = 255};

  /** representation of the left points of the horizontal scan lines */
  Vector2<int> leftPoint[maxNumberOfHorizontalScanLines];
  
  /** representation of the right points of the horizontal scan lines */
  Vector2<int> rightPoint[maxNumberOfHorizontalScanLines];

  /** the number of indications for goals */
  int numberOfGoalIndications;

  /** left point of an indications for a goal */
  Vector2<int> goalIndicationLeft[maxNumberOfGoalScanLines];
  
  /** left point of an indications for a goal */
  Vector2<int> goalIndicationCenter[maxNumberOfGoalScanLines];
  
  /** right point of an indication for a goal */
  Vector2<int> goalIndicationRight[maxNumberOfGoalScanLines];

  /** true if the left end of the goal indication is on the border of the image */
  bool leftOfGoalIndicationIsOnBorder[maxNumberOfGoalScanLines];
  
  /** true if the right end of the goal indication is on the border of the image */
  bool rightOfGoalIndicationIsOnBorder[maxNumberOfGoalScanLines];

  /** the color class of the goal indication */
  colorClass colorOfGoalIndication[maxNumberOfGoalScanLines];

  /** The number of vertical scan lines where a goal is searched*/
  int numberOfGoalScanLines;

  /** representation of the top points of the goal scan lines */
  Vector2<int> topGoalPoint[maxNumberOfGoalScanLines];
  
  /** representation of the bottom points of the goal scan lines */
  Vector2<int> bottomGoalPoint[maxNumberOfGoalScanLines];
  
  bool scanLineToCheckBestAngle[maxNumberOfGoalScanLines];
  
  /** */
  int indexOfGoalIndication[maxNumberOfGoalScanLines];
  
  /** */
  colorClass colorOfGoalScanLine[maxNumberOfGoalScanLines];

};

#endif

/*
* Change log :
* 
* $Log: MSH2004GoalRecognizer.h,v $
* Revision 1.3  2004/04/18 11:57:46  nistico
* Removed MSH2004ImageProcessor2 (integrated all changes into MSH2004ImageProcessor)
*
* Revision 1.1  2004/04/08 16:21:03  wachter
* GT04 checkin of Microsoft-Hellounds
*
*/
