/**
* @file MSH2004FlagSpecialist.cpp
*
* This file contains a class for Image Processing.
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
*/

#include "MSH2004FlagSpecialist.h"

#include "Tools/FieldDimensions.h"
#include "Tools/Player.h"
#include "Tools/Math/Common.h"
#include "Representations/Perception/ColorTable32K.h"

//color table 32K
#define COLOR_CLASS(y,u,v) (colorClass)((ColorTable32K&) colorTable).colorClassesUnpacked[((y>>4)<<12)|((v>>2)<<6)|(u>>2)]
//~ #define COLOR_CLASS(y,u,v) ((ColorTable32K&) colorTable).getColorClassFast(y, u, v)


MSH2004FlagSpecialist::MSH2004FlagSpecialist()
{
}

void MSH2004FlagSpecialist::init(const Image& image)
{
  for(int i = 0; i < 6; i++)
  {
    numberOfBoundingBoxes[i] = 0;
  }
  INIT_DEBUG_IMAGE(imageProcessorFlags, image);
}

void MSH2004FlagSpecialist::searchFlags
(
 const Image& image, 
 const ColorTable& colorTable,
 const CameraMatrix& cameraMatrix,
 MSH2004PixelFilter& advPixFilter,
 colorClass color,
 bool pinkIsTop,
 const Geometry::Line horizonLine,
 int x, int y
 )
{
  bool gridPointIsCovered = false;
  
  int i, j;
  
  for(i = 0; i < 6; i++)
  {
    for(j = 0; j < numberOfBoundingBoxes[i]; j++)
    {
      if(x >= boundingBoxLeft[j][i] && x <= boundingBoxRight[j][i] &&
        y >= boundingBoxBottom[j][i] && y <= boundingBoxTop[j][i] )
      {
        gridPointIsCovered = true;
      }
    }// j
  }//i

  if(gridPointIsCovered) return;
  
  start.x = x; start.y = y;
  
  //find up
  findEndOfFlag(image, colorTable, advPixFilter, start, horizonLine.direction, 
    up, north, color, pinkIsTop, pinkIsTop);
  //find right
  findEndOfFlag(image, colorTable, advPixFilter, start, horizonLine.direction, 
    right, east, color, pinkIsTop, pinkIsTop);
  //find down
  findEndOfFlag(image, colorTable, advPixFilter, start, horizonLine.direction, 
    down, south, color, pinkIsTop, pinkIsTop);
  //find left
  findEndOfFlag(image, colorTable, advPixFilter, start, horizonLine.direction, 
    left, west, color, pinkIsTop, pinkIsTop);
  
  //go to center of pink
  start.x = (west.x + east.x) / 2;
  start.y = (west.y + east.y) / 2;
  //find north
  findEndOfFlag(image, colorTable, advPixFilter, start, horizonLine.direction, 
    up, north, color, pinkIsTop, pinkIsTop);
  //find south
  findEndOfFlag(image, colorTable, advPixFilter, start, horizonLine.direction, 
    down, south, color, pinkIsTop, pinkIsTop);
  
  
  double distance = sqrt((double)sqr(south.x-north.x) + sqr(south.y-north.y));
  int numberOfHorizontalScans = 5;
  //scan horizontal lines
  Vector2<double>step;
  step.x = (south.x - north.x) / (numberOfHorizontalScans + 1.0);
  step.y = (south.y - north.y) / (numberOfHorizontalScans + 1.0);
  
  Geometry::Line leftVerticalLine;
  leftVerticalLine.base = horizonLine.base;
  leftVerticalLine.direction.x = -horizonLine.direction.y;
  leftVerticalLine.direction.y = horizonLine.direction.x;
  
  double leftMin = 1000, rightMax = -1000;
  bool leftValid = true, rightValid = true;

  bool pointIsGood;
  bool startIsInTop = true;
  for(i = 1; i <= numberOfHorizontalScans; i++)
  {
    if(i==(numberOfHorizontalScans + 1)/2) 
    {
      i++; //no scan in the middle of the flag
      startIsInTop = false;
    }
    start.x = (int)(north.x + step.x * i);
    start.y = (int)(north.y + step.y * i); 
    //find right
    pointIsGood = findEndOfFlag(image, colorTable, advPixFilter, start, horizonLine.direction, 
      right, east, color, pinkIsTop, startIsInTop);
    double rightDistance = 0.5 + Geometry::getDistanceToLine(leftVerticalLine, Vector2<double>(east.x,east.y));
    if(rightDistance > rightMax)
    {
      rightMax = rightDistance;
      if(!pointIsGood)
        rightValid = false;
      else
        rightValid = true;
    }
    
    //find left
    pointIsGood = findEndOfFlag(image, colorTable, advPixFilter, start, horizonLine.direction, 
      left, west, color, pinkIsTop, startIsInTop);
    double leftDistance = -0.5 + Geometry::getDistanceToLine(leftVerticalLine, Vector2<double>(west.x,west.y));
    if(leftDistance < leftMin)
    {
      leftMin = leftDistance;
      if(!pointIsGood)
        leftValid = false;
      else
        leftValid = true;
    }
  }
  
  bool topValid = true, bottomValid = true;
  distance = sqrt((double)sqr(east.x-west.x) + sqr(east.y-west.y));
  int numberOfVerticalScans = 3;
  //scan vertical lines
  step.x = (east.x - west.x) / (numberOfVerticalScans + 1.0);
  step.y = (east.y - west.y) / (numberOfVerticalScans + 1.0);
  
  double topMax = -1000, bottomMin = 1000;
  
  for(i = 1; i <= numberOfVerticalScans; i++)
  {
    //    if(i==(numberOfHorizontalScans + 1)/2) i++;
    start.x = (int)(west.x + step.x * i);
    start.y = (int)(west.y + step.y * i); 
    //find top
    pointIsGood = findEndOfFlag(image, colorTable, advPixFilter, start, horizonLine.direction, 
      up, north, color, pinkIsTop, pinkIsTop);
    double topDistance = 0.5 + Geometry::getDistanceToLine(horizonLine, Vector2<double>(north.x,north.y));
    if(topDistance > topMax)
    {
      topMax = topDistance;
      if(!pointIsGood)
        topValid = false;
      else
        topValid = true;
    }
    //find bottom
    pointIsGood = findEndOfFlag(image, colorTable, advPixFilter, start, horizonLine.direction, 
      down, south, color, pinkIsTop, pinkIsTop);
    double bottomDistance = -0.5 + Geometry::getDistanceToLine(horizonLine, Vector2<double>(south.x,south.y));
    if(bottomDistance < bottomMin)
    {
      bottomMin = bottomDistance;
      if(!pointIsGood)
        bottomValid = false;
      else
        bottomValid = true;
    }
  }
  Flag::FlagType flagType = Flag::pinkAboveYellow;

  if(pinkIsTop)
  {
    switch(color)
    {
    case yellow:
      flagType = Flag::pinkAboveYellow;
      break;
    case skyblue:
      flagType = Flag::pinkAboveSkyblue;
      break;
    }
  }
  else
  {
    switch(color)
    {
    case yellow:
      flagType = Flag::yellowAbovePink;
      break;
    case skyblue:
      flagType = Flag::skyblueAbovePink;
      break;
    }
  }

  boundingBoxLeft[numberOfBoundingBoxes[flagType]][flagType] = leftMin;
  boundingBoxRight[numberOfBoundingBoxes[flagType]][flagType] = rightMax;
  boundingBoxTop[numberOfBoundingBoxes[flagType]][flagType] = topMax;
  boundingBoxBottom[numberOfBoundingBoxes[flagType]][flagType] = bottomMin;

  boundingBoxLeftValid[numberOfBoundingBoxes[flagType]][flagType] = leftValid;
  boundingBoxRightValid[numberOfBoundingBoxes[flagType]][flagType] = rightValid;
  boundingBoxTopValid[numberOfBoundingBoxes[flagType]][flagType] = topValid;
  boundingBoxBottomValid[numberOfBoundingBoxes[flagType]][flagType] = bottomValid;
  
  numberOfBoundingBoxes[flagType]++;
  if(numberOfBoundingBoxes[flagType] >= maxNumberOfBoundingBoxes)
  {
    numberOfBoundingBoxes[flagType] = maxNumberOfBoundingBoxes - 1;
  }
 }

bool MSH2004FlagSpecialist::findEndOfFlag
(
 const Image& image,
 const ColorTable& colorTable,
 MSH2004PixelFilter& advPixFilter,
 const Vector2<int> start,
 Vector2<double> horizonDirection,
 Direction directionToGo,
 Vector2<int>& destination,
 colorClass color,
 bool pinkIsTop,
 bool startIsInTop
 )
{
  int blackCounter = 0;
  colorClass currentColorClass;
  colorClass topColor = pink;
  colorClass bottomColor = pink;
  colorClass startColor;
  
  if(pinkIsTop) bottomColor = color; else topColor = color;
  if(startIsInTop) startColor = topColor; else startColor = bottomColor;
  
  destination = start;
  if(startColor == green) return false;

  Vector2<int> lastInsideFlag, lastDestination;
  lastInsideFlag = start;
  
  
  Vector2<double> direction;
  switch(directionToGo)
  {
  case up:
    direction.x = horizonDirection.y;
    direction.y = -horizonDirection.x;
    break;
  case right:
    direction = horizonDirection;
    break;
  case down:
    direction.x = -horizonDirection.y;
    direction.y = horizonDirection.x;
    break;
  case left:
    direction.x = -horizonDirection.x;
    direction.y = -horizonDirection.y;
    break;
  }
  enum {incX, decX, incY, decY} mode;
  if(direction.y < -fabs(direction.x)) mode = decY;
  else if(direction.y > fabs(direction.x)) mode = incY;
  else if(direction.x < -fabs(direction.y)) mode = decX;
  else mode = incX;
  
  
  Vector2<int> diff;
  
  bool goOn = true;
  while(goOn)
  {
    switch(mode)
    {
    case incX:
      diff.x++;
      diff.y = (int)(diff.x * direction.y / direction.x);
      break;
    case decX:
      diff.x--;
      diff.y = (int)(diff.x * direction.y / direction.x);
      break;
    case incY:
      diff.y++;
      diff.x = (int)(diff.y * direction.x / direction.y);
      break;
    case decY:
      diff.y--;
      diff.x = (int)(diff.y * direction.x / direction.y);
      break;
    }
    lastDestination = destination;
    destination = start + diff;
    
    DEBUG_IMAGE_SET_PIXEL_Y(imageProcessorFlags, destination.x, destination.y, 180)
    
      if(destination.x < 1 || destination.x >= image.cameraInfo.resolutionWidth - 2 ||
        destination.y < 1 || destination.y >= image.cameraInfo.resolutionHeight - 2)
    {
      goOn = false;
      destination = lastInsideFlag;
      return false;
    }
    else
    {
      currentColorClass = advPixFilter.getColorClass(image, destination.x, destination.y);
    
      if(directionToGo == left || directionToGo == right)
      {
        if(currentColorClass == startColor)
          lastInsideFlag = destination;
        else blackCounter++;
      }
      
      else if(directionToGo == up)
      {
        if(currentColorClass == topColor)
        {
          lastInsideFlag = destination;
        }
        else if(currentColorClass != bottomColor) blackCounter++;
      }
      else // down
      {
        if(currentColorClass == bottomColor)
        {
          lastInsideFlag = destination;
        }
        else if(currentColorClass != topColor) blackCounter++;
      }

      if(blackCounter > 10)
      {
        goOn = false;
        destination = lastInsideFlag;
      }
    }// if inside image
  }//while goOn
  
  DEBUG_IMAGE_SET_PIXEL_DARK_GREEN(imageProcessorFlags, destination.x, destination.y)
  return true;
}

void MSH2004FlagSpecialist::getFlagPercept
(
 const CameraMatrix& cameraMatrix, 
 const CameraInfo& cameraInfo, 
 const Geometry::Line horizonLine,
 LandmarksPercept& landmarksPercept
 )
{
  int flip = getPlayer().getTeamColor() == Player::blue ? -1 : 1;

  Vector2<double> verticalDirection;
  verticalDirection.x = -horizonLine.direction.y;
  verticalDirection.y = horizonLine.direction.x;

  double factor = cameraInfo.focalLength;
  for(int flagType = 0; flagType < 6; flagType++)
  {
    // find best bounding box
    for(int i = 0; i < numberOfBoundingBoxes[flagType]; i++)
    {
    }
    bestBoundingBox[flagType] = 0;

    if(numberOfBoundingBoxes[flagType] > 0)
    {
      
    Vector2<double> right, left, top, bottom;

    right = horizonLine.base + horizonLine.direction * boundingBoxRight[bestBoundingBox[flagType]][flagType];
    left = horizonLine.base + horizonLine.direction * boundingBoxLeft[bestBoundingBox[flagType]][flagType];

    top = horizonLine.base - verticalDirection * boundingBoxTop[bestBoundingBox[flagType]][flagType];
    bottom = horizonLine.base - verticalDirection * boundingBoxBottom[bestBoundingBox[flagType]][flagType];

    //Vector2<double> rightTop, leftTop, rightBottom, leftBottom;
    right = horizonLine.base + 
      horizonLine.direction * boundingBoxRight[bestBoundingBox[flagType]][flagType]
      - verticalDirection * 
      (boundingBoxTop[bestBoundingBox[flagType]][flagType] + 
       boundingBoxBottom[bestBoundingBox[flagType]][flagType] 
      ) / 2;

    left = horizonLine.base + 
      horizonLine.direction * boundingBoxLeft[bestBoundingBox[flagType]][flagType]
      - verticalDirection * 
      (boundingBoxTop[bestBoundingBox[flagType]][flagType] + 
       boundingBoxBottom[bestBoundingBox[flagType]][flagType] 
      ) / 2;

    top = horizonLine.base 
      - verticalDirection * boundingBoxTop[bestBoundingBox[flagType]][flagType]
      + horizonLine.direction * 
      (boundingBoxLeft[bestBoundingBox[flagType]][flagType] + 
       boundingBoxRight[bestBoundingBox[flagType]][flagType] 
      ) / 2;

    bottom = horizonLine.base 
      - verticalDirection * boundingBoxBottom[bestBoundingBox[flagType]][flagType]
      + horizonLine.direction * 
      (boundingBoxLeft[bestBoundingBox[flagType]][flagType] + 
       boundingBoxRight[bestBoundingBox[flagType]][flagType] 
      ) / 2;

	double centerX = cameraInfo.opticalCenter.x;
	double centerY = cameraInfo.opticalCenter.y;

	Vector3<double> vectorToLeft(factor,
                                                  centerX - left.x,
                                                  centerY - left.y);
    Vector3<double> vectorToRight(factor,
                                                  centerX - right.x,
                                                  centerY - right.y);
    Vector3<double> vectorToTop(factor,
                                                  centerX - top.x,
                                                  centerY - top.y);
    Vector3<double> vectorToBottom(factor,
                                                  centerX - bottom.x,
                                                  centerY - bottom.y);

    Vector3<double>
      vectorToLeftWorld = cameraMatrix.rotation * vectorToLeft,
    vectorToRightWorld = cameraMatrix.rotation * vectorToRight,
    vectorToTopWorld = cameraMatrix.rotation * vectorToTop,
    vectorToBottomWorld = cameraMatrix.rotation * vectorToBottom;
       
    double 
      leftAngle = atan2(vectorToLeftWorld.y,vectorToLeftWorld.x),
      rightAngle = atan2(vectorToRightWorld.y,vectorToRightWorld.x),
      topAngle = atan2(vectorToTopWorld.z,sqrt((double)sqr(vectorToTopWorld.x) + sqr(vectorToTopWorld.y)) ),
      bottomAngle = atan2(vectorToBottomWorld.z,sqrt((double)sqr(vectorToBottomWorld.x) + sqr(vectorToBottomWorld.y)) );


    Vector2<double>flagPosition;
    
    switch (flagType)
    {
    case Flag::pinkAboveYellow:
      flagPosition.x = xPosBackFlags * flip;
      flagPosition.y = yPosRightFlags * flip;
      break;
    case Flag::pinkAboveSkyblue:
      flagPosition.x = xPosFrontFlags * flip;
      flagPosition.y = yPosRightFlags * flip;
      break;
    case Flag::yellowAbovePink:
      flagPosition.x = xPosBackFlags * flip;
      flagPosition.y = yPosLeftFlags * flip;
      break;
    case Flag::skyblueAbovePink:
      flagPosition.x = xPosFrontFlags * flip;
      flagPosition.y = yPosLeftFlags * flip;
      break;
    }

    
    ConditionalBoundary boundary;

    boundary.addX(leftAngle,!boundingBoxLeftValid[bestBoundingBox[flagType]][flagType]);
    boundary.addX(rightAngle,!boundingBoxRightValid[bestBoundingBox[flagType]][flagType]);
    boundary.addY(topAngle,!boundingBoxTopValid[bestBoundingBox[flagType]][flagType]);
    boundary.addY(bottomAngle,!boundingBoxBottomValid[bestBoundingBox[flagType]][flagType]);
    landmarksPercept.addFlag((Flag::FlagType)flagType, flagPosition, boundary);
    } //if(numberOfBoundingBoxes[flagType]) > 0)
  }
  Vector2<double> cameraOffset(cameraMatrix.translation.x,
                                              cameraMatrix.translation.y);

  estimateOffsetForFlags(landmarksPercept, cameraOffset); 
  SEND_DEBUG_IMAGE(imageProcessorFlags);
}

void MSH2004FlagSpecialist::estimateOffsetForFlags
(
 LandmarksPercept& landmarksPercept,
 const Vector2<double>& cameraOffset
 )
{
  for(int i = 0;i < landmarksPercept.numberOfFlags; ++i)
  {
    Flag& flag = landmarksPercept.flags[i];

    /** @todo improve, isOnBorder(flag.x.?) not checked */
    double distance;
    double direction = flag.x.getCenter();

    if(!flag.isOnBorder(flag.y.min) && !flag.isOnBorder(flag.y.max))
    {
      if(flag.y.min != flag.y.max)
      {
        distance = flagHeight / (tan(flag.y.max) - tan(flag.y.min)) + flagRadius;
        flag.distanceValidity = 0.8;
      }
      else
      {
        distance = 4500;
        flag.distanceValidity = 0.05;
      }
    }
    else
    {
      distance = flagRadius / sin(flag.x.getSize() / 2);
      if(!flag.isOnBorder(flag.x.min) && !flag.isOnBorder(flag.x.max)) // Flag touches no vertical border
        flag.distanceValidity = 0.7;
      else
        flag.distanceValidity = 0.2;
    }

    if(!flag.isOnBorder(flag.x.min) && !flag.isOnBorder(flag.x.max)) // Flag touches no vertical border
      flag.angleValidity = 0.8;
    else
      flag.angleValidity = 0.7;
    
    Pose2D p = Pose2D(cameraOffset) + Pose2D(direction) 
                       + Pose2D(Vector2<double>(distance,0));
//    flag.distance = p.translation.abs();
//    flag.angle = atan2(p.translation.y,p.translation.x);

      flag.distance = p.translation.abs();
      flag.angle = direction;
    
       
    if (flag.distance > 6000)
    {
      flag.distance = 6000;
      flag.distanceValidity=0; // flags far away are meassured very bad
    }
    else if (flag.distance > 3000) 
      flag.distanceValidity*=0.5; // flags medium far away are meassured medium bad
  }
}


/*
* Change log :
* 
* $Log: MSH2004FlagSpecialist.cpp,v $
* Revision 1.4  2004/04/18 18:14:53  nistico
* USE_INTRINSIC layout removed.
* All functions properly replicated in intrinsic version.
* However, image processor (MSH2004) making use of them get distorted visualization
* of percepts, because drawing functions use the old parameters.
* It has to be decided wheter to fully move to intrinsic, or discard it.
*
* Revision 1.3  2004/04/18 11:57:46  nistico
* Removed MSH2004ImageProcessor2 (integrated all changes into MSH2004ImageProcessor)
*
* Revision 1.1  2004/04/08 16:21:03  wachter
* GT04 checkin of Microsoft-Hellounds
*
*/
