/**
* @file OrangeCalibration.cpp
*
* Implementation of class OrangeCalibration
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#include "OrangeCalibration.h"
#include "Tools/FieldDimensions.h"

OrangeCalibration::OrangeCalibration
(
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo,
 ColorTable64& colorTable64,
 ColorTableCuboids& colorTableCuboids,
 ColorTableReferenceColor& colorTableReferenceColor
 )
 :
cameraMatrix(cameraMatrix),
cameraInfo(cameraInfo),
colorTable64(colorTable64), 
colorTableCuboids(colorTableCuboids), 
colorTableReferenceColor(colorTableReferenceColor)
{
}

OrangeCalibration::~OrangeCalibration()
{
}

void OrangeCalibration::init()
{
  for(int i = 0; i < maxNumberOfScanLines + 2; i++)
  {
    numberOfSegments[i] = 0;
    foundSegmentWithNoColorBelowOrange[i] = false;
  }
  lineIndexOfLastSegment = -1;
}

void OrangeCalibration::addSegment(LineSegment& newSegment, int lineIndex)
{
  colorClass colorOfSegment = colorTable64.getColorClass(newSegment.averageIntensity[0], newSegment.averageIntensity[1], newSegment.averageIntensity[2]);

  // initialize if a new line starts
  if(lineIndexOfLastSegment != lineIndex) 
  {
  }  
  lineIndexOfLastSegment = lineIndex;

  if(colorOfSegment == noColor)
  {
    numberOfSegments[lineIndex + 1] = 0;
    foundSegmentWithNoColorBelowOrange[lineIndex + 1] = true;
  }

  colorOfSegment = colorTableReferenceColor.getColorClass(newSegment.averageIntensity[0], newSegment.averageIntensity[1], newSegment.averageIntensity[2]);
  if( (colorOfSegment == orange || colorOfSegment == yellowOrange) &&
    newSegment.beginIsBelowHorizon ) 
  {
    addOrangeReferenceColorSegment(newSegment, lineIndex);
  }
  
  //if the segment is the field border, forget all previously seen ball segments on that line.
  if(newSegment.isFieldBorder_forDetection(colorTable64, cameraMatrix) )
  {
    numberOfSegments[lineIndex + 1] = 0;
    LINE(imageProcessor_ball3, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y, 
      0, Drawings::ps_solid, Drawings::black);
  }
}

void OrangeCalibration::addOrangeReferenceColorSegment(LineSegment& newSegment, int lineIndex)
{
  //If there is a previous segment on that line ...
  //... and its end is close to the begin of the new one ...
  if(
    numberOfSegments[lineIndex + 1] != 0 &&
    Geometry::distance(segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].end, newSegment.begin ) < 12 )
  {
    //... combine them.
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].end = newSegment.end;
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].endIsOnBorder = newSegment.endIsOnBorder;
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].endIsBelowHorizon = newSegment.endIsBelowHorizon;
    
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].averageIntensity[0] += newSegment.averageIntensity[0];
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].averageIntensity[0] /= 2;
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].averageIntensity[1] += newSegment.averageIntensity[1];
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].averageIntensity[1] /= 2;
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].averageIntensity[2] += newSegment.averageIntensity[2];
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].averageIntensity[2] /= 2;
    
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].length = (int)Geometry::distance(
      segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].begin,
      segments[lineIndex + 1][numberOfSegments[lineIndex + 1]-1].end );
  }
  else
  {
    segments[lineIndex + 1][numberOfSegments[lineIndex + 1]] = newSegment;
    numberOfSegments[lineIndex + 1]++;
  }
}

void OrangeCalibration::calibrateOrange()
{
  int accumulatedLengthOfBallSegments = 0;
  int numberOfBallSegments = 0;

  int lineIndex;
  // remove all segments with wrong size
  for(lineIndex = 1; lineIndex < maxNumberOfScanLines + 1; lineIndex++)
  {
    for(int segmentIndex = 0; segmentIndex < numberOfSegments[lineIndex]; segmentIndex++)
    {
      //calculate the distance to the bottom end of the segment
      Vector2<int> pointOnField;
      if(Geometry::calculatePointOnField(segments[lineIndex][segmentIndex].end.x, segments[lineIndex][segmentIndex].end.y, cameraMatrix, cameraInfo, pointOnField) )
      {
        // calculate the size that an ball in that distance has in the image
        int expectedSizeOfBall = Geometry::getSizeByDistance(cameraInfo, 2 * ballRadius, pointOnField.abs());
        
        // filter the segment if its size differs to much
        if(
          //segment is to large
          segments[lineIndex][segmentIndex].length > expectedSizeOfBall * 1.5 + 4 ||
          //segent is to small and not touching the image border
           (segments[lineIndex][segmentIndex].length < expectedSizeOfBall / 5 &&
           !segments[lineIndex][segmentIndex].beginIsOnBorder &&
           !segments[lineIndex][segmentIndex].endIsOnBorder
           )
           )
        {
          LINE(imageProcessor_ball3, segments[lineIndex][segmentIndex].begin.x, segments[lineIndex][segmentIndex].begin.y, segments[lineIndex][segmentIndex].end.x, segments[lineIndex][segmentIndex].end.y,
            1, Drawings::ps_solid, Drawings::red );
          //remove the current segment
          if(segmentIndex < numberOfSegments[lineIndex] - 1)
          {
            segments[lineIndex][segmentIndex] = segments[lineIndex][numberOfSegments[lineIndex] - 1];
            segmentIndex--;
          }
          numberOfSegments[lineIndex] -= 1;
        }
        else
        {
          accumulatedLengthOfBallSegments += segments[lineIndex][segmentIndex].length;
          numberOfBallSegments++;
        }
      }// if(Geometry::calculatePointOnField(...
    }// for(int segmentIndex = 0; segmentIndex < numberOfSegments[lineIndex]; segmentIndex++)
  }// for(int lineIndex = 1; lineIndex < maxNumberOfScanLines + 1; lineIndex++)

  int averageLengthOfBallSegments = 0;
  if(numberOfBallSegments > 0) averageLengthOfBallSegments = accumulatedLengthOfBallSegments / numberOfBallSegments;

  for(lineIndex = 1; lineIndex < maxNumberOfScanLines + 1; lineIndex++)
  {
    for(int segmentIndex = 0; segmentIndex < numberOfSegments[lineIndex]; segmentIndex++)
    {
      // no orange segment of ball size was found next left or right
/*      if(numberOfSegments[lineIndex - 1] == 0 && 
        numberOfSegments[lineIndex + 1] == 0)
      {
        LINE(imageProcessor_ball2, 
          segments[lineIndex][segmentIndex].begin.x, segments[lineIndex][segmentIndex].begin.y,
          segments[lineIndex][segmentIndex].end.x, segments[lineIndex][segmentIndex].end.y,
          1, Drawings::ps_solid, Drawings::gray);
      }
      else*/
      {
        // the segment is not to small compared to the other segments and is not yellow
        if(segments[lineIndex][segmentIndex].length > averageLengthOfBallSegments / 2 &&
           colorTable64.getColorClass(segments[lineIndex][segmentIndex].averageIntensity[0], segments[lineIndex][segmentIndex].averageIntensity[1], segments[lineIndex][segmentIndex].averageIntensity[2]) 
           != yellow)
        {
/*          LINE(imageProcessor_ball2, 
            segments[lineIndex][segmentIndex].begin.x, segments[lineIndex][segmentIndex].begin.y,
            segments[lineIndex][segmentIndex].end.x, segments[lineIndex][segmentIndex].end.y,
            1, Drawings::ps_solid, Drawings::blue);
*/
          colorTable64.addCuboidToColorClass(orange, 
            segments[lineIndex][segmentIndex].averageIntensity[0] - 2,
            segments[lineIndex][segmentIndex].averageIntensity[1] - 2,
            segments[lineIndex][segmentIndex].averageIntensity[2] - 2,
            (2 * segments[lineIndex][segmentIndex].averageIntensity[0] + 
             segments[lineIndex][segmentIndex].maxIntensity[0] )
             / 3,
            segments[lineIndex][segmentIndex].averageIntensity[1] + 2,
            segments[lineIndex][segmentIndex].averageIntensity[2] + 2
            );
        }//if(segments[lineIndex][segmentIndex].length > averageLengthOfBallSegments / 2)
      }
    }// for(int segmentIndex = 0; segmentIndex < numberOfSegments[lineIndex]; segmentIndex++)
  }// for(int lineIndex = 1; lineIndex < maxNumberOfScanLines + 1; lineIndex++)
//  DEBUG_DRAWING_FINISHED(imageProcessor_ball2);
}


/*
* Change log :
* 
* $Log: OrangeCalibration.cpp,v $
* Revision 1.4  2004/05/07 15:16:25  nistico
* All geometry calculations are making use of intrinsic functions.
* I updated most of the images processors to make correct use of this.
* Please test if there are any problems, because i'm going to remove the
* old code soon.
*
* Revision 1.3  2004/02/12 14:12:38  juengel
* changed drawings
*
* Revision 1.2  2003/12/15 11:49:06  juengel
* Introduced CameraInfo
*
* Revision 1.1  2003/12/04 09:44:03  juengel
* Added OrangeCalibration
*
*/
