/**
* @file GroundDetector.h
*
* Definition of class GroundDetector
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#ifndef __GroundDetector_h_
#define __GroundDetector_h_

#include "Representations/Perception/ColorTable64.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/LinesPercept.h"
#include "Tools/Math/Histogram.h"

#include "Tools/Debugging/DebugImages.h"
#include "Tools/Math/Geometry.h"

#include "SegmentBasedDetector.h"
#include "../ColorTableCuboids.h"
#include "../ColorTableReferenceColor.h"
#include "../ColorSpaceUsageCounter.h"

#include "../LightingChangeTester.h"

/**
* @class GroundDetector
*
* The ground detector detects the ground and obstacles on the ground based on the segments provided by the image processor.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/ 

class GroundDetector
{
public:
  GroundDetector(
    const CameraMatrix& cameraMatrix, 
    const CameraInfo& cameraInfo, 
    ColorSpaceUsageCounter& colorSpaceUsageCounter,
    Vector3<int>& averageIntensityOverLastImages,
    ColorTable64& colorTable64, 
    ColorTableCuboids& colorTableCuboids, 
    ColorTableReferenceColor& colorTableReferenceColor, 
    Geometry::Line& horizonLine,
    int& goalIndicationAboveHorizon,
    int& goalIndicationBelowHorizon,
    bool& foundGoal,
    ObstaclesPercept& obstaclesPercept,
    LinesPercept& linesPercept
    );
  
  ~GroundDetector();
  
  void init();
  void addSegment(LineSegment& newSegment, int lineIndex);

  void detectObstacles(LineSegment& newSegment, int lineIndex);

  void createObstaclesPercept();
  void createGoalIndication();
  double getDistanceFromFieldBorderToHorizon();
  bool lightingHasChanged();

private:
  enum {maxNumberOfScanLines = 240}; /**< The maximum number scan lines. */
  enum {maxNumberOfSegmentsPerScanLine = 20}; /**< The maximum number of segments per line. */
  LineSegment segments[maxNumberOfScanLines + 2][maxNumberOfSegmentsPerScanLine];
  int numberOfSegments[maxNumberOfScanLines + 2];

  colorClass colorOfLastSegment;
  colorClass colorOfLastColoredSegment;

  bool detectedGround[maxNumberOfScanLines + 2];
  ObstaclesPercept::ObstacleType obstacleType[maxNumberOfScanLines + 2];
  Vector2<int> nearGround[maxNumberOfScanLines + 2];
  Vector2<int> farGround[maxNumberOfScanLines + 2];
  bool farGroundIsOnBorder[maxNumberOfScanLines + 2];
  
  bool foundEdgeBetweenBorderAndField;
  bool horizonIsInTheImage;

  bool groundFound;

  bool borderPointFound;
  Vector2<int> borderPointInImage;
  Vector2<int> borderPointOnField;

  bool borderPointAdded;
  double distanceFromBorderPointToHorizon;

  bool skyblueGoalPointFound;
  bool yellowGoalPointFound;
  
  Vector2<int> topGoalPointInImage;
  Vector2<int> bottomGoalPointInImage;
  Vector2<int> goalPointOnField;

  double accumulatedDistanceFromTopGoalPointsToHorizon;
  double accumulatedDistanceFromBottomGoalPointsToHorizon;
  int numberOfGoalSegments;

  int& goalIndicationAboveHorizon;
  int& goalIndicationBelowHorizon;
  bool& foundGoal;

  bool linePointsFound;
  Vector2<int> linePoint1OnField;
  Vector2<int> linePoint2OnField;

  bool whiteSegmentFound;
  Geometry::SetOfPoints<int, maxNumberOfScanLines + 2> firstGreenPoints;


  enum {notFound, found, validated, rejected} skyblueCalibration, yellowCalibration;
  Vector3<int> skyblueForCalibration;
  Vector3<int> yellowForCalibration;

  int lineIndexOfLastSegment;

  LightingChangeTester lightingChangeTester;

  /** A reference to the camera matrix that describes position and rotation of the camera when the image was aquired */
  const CameraMatrix& cameraMatrix;

  /** A reference to the camera info that describes the opening angle and the resolution of the image */
  const CameraInfo& cameraInfo;

  ColorSpaceUsageCounter& colorSpaceUsageCounter;

  Vector3<int>& averageIntensityOverLastImages;

  /** A reference to the color table */
  ColorTable64& colorTable64;

  /** A reference to the color table */
  ColorTableReferenceColor& colorTableReferenceColor;

  /** A reference to the color table */
  ColorTableCuboids& colorTableCuboids;

  Geometry::Line& horizonLine;

  ObstaclesPercept& obstaclesPercept;

  LinesPercept& linesPercept;
};

#endif // __GroundDetector_h_

/*
* Change log :
* 
* $Log: GroundDetector.h,v $
* Revision 1.9  2004/03/01 11:56:44  juengel
* Recognition of obstacle types added.
*
* Revision 1.8  2003/12/15 11:47:03  juengel
* Introduced CameraInfo
*
* Revision 1.7  2003/12/01 16:28:31  juengel
* Added lighting change tester.
*
* Revision 1.6  2003/11/26 10:59:12  juengel
* Calculation of goal indications added.
*
* Revision 1.5  2003/11/10 11:33:22  juengel
* added calibration and detection versions for isFieldBorder, isGoal and isFieldLine
*
* Revision 1.4  2003/10/31 11:46:51  juengel
* Improved creation of obstaclesPercept
*
* Revision 1.3  2003/10/29 13:08:25  juengel
* removed method "updateGreenInColorTable"
*
* Revision 1.2  2003/10/23 07:24:21  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.1  2003/10/06 14:10:13  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.5  2003/09/26 15:27:48  juengel
* Renamed DataTypes to representations.
*
* Revision 1.4  2003/09/26 11:38:52  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.3  2003/08/29 13:09:22  juengel
* LinesPercept is detected.
*
* Revision 1.2  2003/08/25 17:26:52  juengel
* clean up
*
* Revision 1.1  2003/08/18 11:47:53  juengel
* Added segment based detectors.
*
*/
