/**
* @file GroundDetector.cpp
*
* Implementation of class GroundDetector
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#include "GroundDetector.h"

GroundDetector::GroundDetector
(
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo,
 ColorSpaceUsageCounter& colorSpaceUsageCounter,
 Vector3<int>& averageIntensityOverLastImages,
 ColorTable64& colorTable64,
 ColorTableCuboids& colorTableCuboids,
 ColorTableReferenceColor& colorTableReferenceColor,
 Geometry::Line& horizonLine,
 int& goalIndicationAboveHorizon,
 int& goalIndicationBelowHorizon,
 bool& foundGoal,
 ObstaclesPercept& obstaclesPercept,
 LinesPercept& linesPercept
 )
 : 
cameraMatrix(cameraMatrix),
cameraInfo(cameraInfo),
colorSpaceUsageCounter(colorSpaceUsageCounter),
averageIntensityOverLastImages(averageIntensityOverLastImages),
colorTable64(colorTable64), 
colorTableCuboids(colorTableCuboids), 
colorTableReferenceColor(colorTableReferenceColor), 
horizonLine(horizonLine),
goalIndicationAboveHorizon(goalIndicationAboveHorizon),
goalIndicationBelowHorizon(goalIndicationBelowHorizon),
foundGoal(foundGoal),
obstaclesPercept(obstaclesPercept),
linesPercept(linesPercept)
{
  distanceFromBorderPointToHorizon = -15;
}

GroundDetector::~GroundDetector()
{
}

void GroundDetector::init()
{
  for(int i = 0; i < maxNumberOfScanLines + 2; i++)
  {
    detectedGround[i] = false;
  }
  lineIndexOfLastSegment = -1;
  firstGreenPoints.init();
  numberOfGoalSegments = 0;
  accumulatedDistanceFromTopGoalPointsToHorizon = 0;
  accumulatedDistanceFromBottomGoalPointsToHorizon = 0;
  lightingChangeTester.startImage();
}

void GroundDetector::addSegment(LineSegment& newSegment, int lineIndex)
{
  // initialize if a new line starts
  if(lineIndexOfLastSegment != lineIndex) 
  {
    if(yellowCalibration == validated)
      colorTable64.addColorClass(yellow, yellowForCalibration.x, yellowForCalibration.y, yellowForCalibration.z, 4);
    if(skyblueCalibration == validated)
      colorTable64.addColorClass(skyblue, skyblueForCalibration.x, skyblueForCalibration.y, skyblueForCalibration.z, 4);

    foundEdgeBetweenBorderAndField = false;
    borderPointFound = false;
    borderPointAdded = false;
    linePointsFound = false;
    yellowGoalPointFound = false;
    skyblueGoalPointFound = false;
    yellowCalibration = notFound;
    skyblueCalibration = notFound;
    groundFound = false;
    whiteSegmentFound = false;

    colorOfLastSegment = noColor;
    colorOfLastColoredSegment = noColor;
  }
  lineIndexOfLastSegment = lineIndex;

  colorClass colorOfNewSegment = colorTable64.getColorClass(newSegment.averageIntensity[0], newSegment.averageIntensity[1], newSegment.averageIntensity[2]);

  //check for green below the goals, field lines and borders.
  if(colorOfNewSegment == green)
  {
    groundFound = true;
    if(borderPointFound && newSegment.length > 6) 
    {
      Geometry::calculatePointOnField(borderPointInImage.x, borderPointInImage.y, cameraMatrix, cameraInfo, borderPointOnField);
      linesPercept.add(LinesPercept::border, borderPointOnField);
      borderPointAdded = true;
      distanceFromBorderPointToHorizon = Geometry::getDistanceToLine(horizonLine, Vector2<double>(borderPointInImage.x, borderPointInImage.y));
      borderPointFound = false;
    }
    if(linePointsFound && newSegment.length > 6) 
    {
      linesPercept.add(LinesPercept::field, linePoint1OnField);
      linesPercept.add(LinesPercept::field, linePoint2OnField);
      linePointsFound = false;
    }
    if(yellowGoalPointFound)
    {
      Geometry::calculatePointOnField(bottomGoalPointInImage.x, bottomGoalPointInImage.y, cameraMatrix, cameraInfo, goalPointOnField);
      linesPercept.add(LinesPercept::yellowGoal, goalPointOnField);
      yellowGoalPointFound = false;
      accumulatedDistanceFromTopGoalPointsToHorizon += Geometry::getDistanceToLine(horizonLine, Vector2<double>(topGoalPointInImage.x, topGoalPointInImage.y));
      accumulatedDistanceFromBottomGoalPointsToHorizon += Geometry::getDistanceToLine(horizonLine, Vector2<double>(bottomGoalPointInImage.x, bottomGoalPointInImage.y));
      numberOfGoalSegments++;
    }
    if(skyblueGoalPointFound) 
    {
      Geometry::calculatePointOnField(bottomGoalPointInImage.x, bottomGoalPointInImage.y, cameraMatrix, cameraInfo, goalPointOnField);
      linesPercept.add(LinesPercept::skyblueGoal, goalPointOnField);
      skyblueGoalPointFound = false;
      accumulatedDistanceFromTopGoalPointsToHorizon += Geometry::getDistanceToLine(horizonLine, Vector2<double>(topGoalPointInImage.x, topGoalPointInImage.y));
      accumulatedDistanceFromBottomGoalPointsToHorizon += Geometry::getDistanceToLine(horizonLine, Vector2<double>(bottomGoalPointInImage.x, bottomGoalPointInImage.y));
      numberOfGoalSegments++;
    }
    if(skyblueCalibration == found && newSegment.length > 3)
    {
      DOT(imageProcessor_calibration1, newSegment.begin.x, newSegment.begin.y, Drawings::skyblue, Drawings::skyblue);
      skyblueCalibration = validated;
    }
    if(yellowCalibration == found && newSegment.length > 3)
    {
      DOT(imageProcessor_calibration1, newSegment.begin.x, newSegment.begin.y, Drawings::yellow, Drawings::yellow);
      yellowCalibration = validated;
    }
  }
  else // the new segment is not green
  {
    yellowCalibration = rejected;
    skyblueCalibration = rejected;
  }
  if(newSegment.length > 6)
  {
    borderPointFound = false;
    linePointsFound = false;
    yellowGoalPointFound = false;
    skyblueGoalPointFound = false;
  }

  bool newSegmentIsBorderOrGoalOrFieldLine = false;

  if(newSegment.isFieldBorder_forDetection(colorTable64, cameraMatrix))
  {
    foundEdgeBetweenBorderAndField = true;
    newSegmentIsBorderOrGoalOrFieldLine = true;
    borderPointFound = true;
    yellowCalibration = rejected;
    skyblueCalibration = rejected;
    borderPointInImage = newSegment.end;
  }

  if(newSegment.isFieldBorder_forCalibration(colorTableCuboids, colorTableReferenceColor, cameraMatrix))
  {
    LINE(imageProcessor_calibration1, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y, 
      1, Drawings::ps_solid, Drawings::black);
    colorTable64.addColorClass(white, newSegment.averageIntensity[0], newSegment.averageIntensity[1], newSegment.averageIntensity[2], 4);
  }

  colorClass goalColor = newSegment.isGoal_forDetection(colorTable64);
  if( goalColor != noColor)
  {
    newSegmentIsBorderOrGoalOrFieldLine = true;
    topGoalPointInImage = newSegment.begin;
    bottomGoalPointInImage = newSegment.end;
    if(goalColor == skyblue) skyblueGoalPointFound = true;
    if(goalColor == yellow) yellowGoalPointFound = true;
  }

  goalColor = newSegment.isGoal_forCalibration(colorTableReferenceColor);
  if( goalColor != noColor)
  {
    if(goalColor == skyblue)
    {
      LINE(imageProcessor_calibration1, newSegment.begin.x, newSegment.begin.y, 
        newSegment.end.x, newSegment.end.y, 1, Drawings::ps_solid, Drawings::skyblue);
      
      skyblueCalibration = found;
      skyblueForCalibration.x = newSegment.averageIntensity[0];
      skyblueForCalibration.y = newSegment.averageIntensity[1];
      skyblueForCalibration.z = newSegment.averageIntensity[2];
    }
    if(goalColor == yellow)
    {
      LINE(imageProcessor_calibration1, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y, 
        1, Drawings::ps_solid, Drawings::yellow);
      yellowCalibration = found;
      yellowForCalibration.x = newSegment.averageIntensity[0];
      yellowForCalibration.y = newSegment.averageIntensity[1];
      yellowForCalibration.z = newSegment.averageIntensity[2];
    }
  }
  if(newSegment.isFieldLine(colorTable64, cameraMatrix) && groundFound)
  {
    LINE(imageProcessor_calibration1, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y, 
      1, Drawings::ps_solid, Drawings::gray);
    newSegmentIsBorderOrGoalOrFieldLine = true;
    linePointsFound = true;
    Geometry::calculatePointOnField(newSegment.begin.x, newSegment.begin.y, cameraMatrix, cameraInfo, linePoint1OnField);
    Geometry::calculatePointOnField(newSegment.end.x, newSegment.end.y, cameraMatrix, cameraInfo, linePoint2OnField);
  }// if( newSegment.isFieldBorder(colorTableReferenceColor, cameraMatrix) )
  if(!newSegmentIsBorderOrGoalOrFieldLine)
  {
    detectObstacles(newSegment, lineIndex);
  }//if(!newSegmentIsBorderOrGoalOrFieldLine)


  bool colorIsFrequent = colorSpaceUsageCounter.isColorFrequent(newSegment.averageIntensity[0], newSegment.averageIntensity[1], newSegment.averageIntensity[2], 60);
  bool lineIsLong = newSegment.length > 30;
  bool noChannel1Diversity = newSegment.maxIntensity[1] - newSegment.minIntensity[1] < 20;
  bool foundEdgeOrGoalAbove = (foundEdgeBetweenBorderAndField /*|| skyblueGoalPointFound*/);
  bool beginIsBelowHorizon = newSegment.beginIsBelowHorizon;
/*  bool greenIsLevel1Green = 
    colorTableCuboids.getColorClass(newSegment.averageIntensity[0], newSegment.averageIntensity[1], newSegment.averageIntensity[2])
    == green;
*/
  bool greenIsLevel1Green = (
    newSegment.averageIntensity[0] < averageIntensityOverLastImages.x * 1.3 && 
    newSegment.averageIntensity[0] > averageIntensityOverLastImages.x * 0.7);

  //visualisation of all conditions for green
  if(greenIsLevel1Green && lineIsLong)
  { LINE(imageProcessor_calibration2, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y, 5, Drawings::ps_solid, Drawings::black); }
  if(foundEdgeOrGoalAbove && beginIsBelowHorizon && lineIsLong)
  { LINE(imageProcessor_calibration2, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y, 3, Drawings::ps_solid, Drawings::white); }
  if(colorIsFrequent && lineIsLong)
  { LINE(imageProcessor_calibration2, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y, 1, Drawings::ps_solid, Drawings::green); }

  // update green
  if(
    colorIsFrequent &&
    lineIsLong &&
    noChannel1Diversity &&
    foundEdgeOrGoalAbove &&
    beginIsBelowHorizon && 
    greenIsLevel1Green &&
    !linePointsFound &&
    !borderPointFound
    )
  {
    colorTable64.addColorClass(green, newSegment.averageIntensity[0], newSegment.averageIntensity[1], newSegment.averageIntensity[2], 4);
    if(colorOfNewSegment == green)
    {
      colorTable64.addCuboidToColorClass(green, 
      (newSegment.averageIntensity[0] + newSegment.minIntensity[0]) / 2, 
      (newSegment.averageIntensity[1] + newSegment.minIntensity[1]) / 2, 
      (newSegment.averageIntensity[2] + newSegment.minIntensity[2]) / 2, 
      newSegment.maxIntensity[0], 
      (newSegment.averageIntensity[1] + newSegment.maxIntensity[1]) / 2, 
      (newSegment.averageIntensity[2] + newSegment.maxIntensity[2]) / 2
      );
    }
    colorTableReferenceColor.expandGreen(newSegment.minIntensity, newSegment.maxIntensity);

    lightingChangeTester.addColor(newSegment.averageIntensity[0], newSegment.averageIntensity[1], newSegment.averageIntensity[2]);

    LINE(imageProcessor_ground, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y,
    2, Drawings::ps_solid, Drawings::green);
  }
  
  colorOfLastSegment = colorOfNewSegment;
  if(colorOfNewSegment != noColor) colorOfLastColoredSegment = colorOfNewSegment;
  
/*  if(colorOfNewSegment == white) whiteSegmentFound = true;
  if(colorOfNewSegment == green) 
  {
    if(!newSegment.beginIsOnBorder && !groundFound && whiteSegmentFound) 
    {
      DOT(imageProcessor_horizon, newSegment.begin.x, newSegment.begin.y, Drawings::white, Drawings::green);
      firstGreenPoints.add(newSegment.begin);
    }
  }
*/
}// void GroundDetector::addSegment(LineSegment& newSegment, int lineIndex)


void GroundDetector::detectObstacles(LineSegment& newSegment, int lineIndex)
{
  colorClass colorOfNewSegment = colorTable64.getColorClass(newSegment.averageIntensity[0], newSegment.averageIntensity[1], newSegment.averageIntensity[2]);
  
  if( (colorOfNewSegment == green || colorOfNewSegment == orange)
    && newSegment.endIsBelowHorizon) 
  {
    LINE(imageProcessor_obstacles, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y,
      3, Drawings::ps_solid, Drawings::red);
    
    if(!detectedGround[lineIndex+1] /*&& newSegment.length > 10*/)
    {
      farGround[lineIndex+1] = newSegment.begin;
      nearGround[lineIndex+1] = newSegment.end;
      int lineHeight = 2 * Geometry::calculateLineSize(newSegment.begin, cameraMatrix, cameraInfo);
      if(!newSegment.beginIsBelowHorizon) lineHeight = 10;
      farGroundIsOnBorder[lineIndex+1] = 
        newSegment.begin.x < lineHeight ||
        newSegment.begin.y < lineHeight ||
        newSegment.begin.x > cameraInfo.resolutionWidth - lineHeight ||
        newSegment.begin.y > cameraInfo.resolutionHeight - lineHeight;
      detectedGround[lineIndex+1] = true;
  
      switch(colorOfLastColoredSegment)
      {
      case red: 
        if(getPlayer().getTeamColor() == Player::red)
          obstacleType[lineIndex+1] = ObstaclesPercept::teammate;
        else
          obstacleType[lineIndex+1] = ObstaclesPercept::opponent;
        break;
      case blue: obstacleType[lineIndex+1] = ObstaclesPercept::opponent; 
        if(getPlayer().getTeamColor() == Player::blue)
          obstacleType[lineIndex+1] = ObstaclesPercept::teammate;
        else
          obstacleType[lineIndex+1] = ObstaclesPercept::opponent;
        break;
      default: obstacleType[lineIndex+1] = ObstaclesPercept::unknown;
      }
      switch(colorOfLastSegment)
      {
      case white: obstacleType[lineIndex+1] = ObstaclesPercept::border; break;
      case yellow: obstacleType[lineIndex+1] = ObstaclesPercept::goal; break;
      case skyblue: obstacleType[lineIndex+1] = ObstaclesPercept::goal; break;
      }
      
      LINE(imageProcessor_obstacles, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y,
        0, Drawings::ps_solid, Drawings::blue);
    }
    else
    {
      if(
        Geometry::distance(nearGround[lineIndex+1], newSegment.begin) < 
        Geometry::calculateLineSize(newSegment.begin, cameraMatrix, cameraInfo) * 4 
        || Geometry::distance(nearGround[lineIndex+1], newSegment.begin) < 10
        )
      {
        DOT(imageProcessor_obstacles, newSegment.begin.x, newSegment.begin.y, Drawings::red, Drawings::white);
        nearGround[lineIndex+1] = newSegment.end;
      }
      else
      {
        DOT(imageProcessor_obstacles, newSegment.begin.x, newSegment.begin.y, Drawings::green, Drawings::white);
        farGround[lineIndex+1] = newSegment.begin;
        nearGround[lineIndex+1] = newSegment.end;
        int lineHeight = 4 * Geometry::calculateLineSize(newSegment.begin, cameraMatrix, cameraInfo);
        // line Height is not calculated correctly for points above the horizon
        if(!newSegment.beginIsBelowHorizon || lineHeight < 15) lineHeight = 15;
        farGroundIsOnBorder[lineIndex+1] = 
          newSegment.begin.x < lineHeight ||
          newSegment.begin.y < lineHeight ||
          newSegment.begin.x > cameraInfo.resolutionWidth - lineHeight ||
          newSegment.begin.y > cameraInfo.resolutionHeight - lineHeight;
      }
    }
  }//segment is green and below horizon

  //filter green segments behind obstacles
  if(newSegment.endIsOnBorder && !newSegment.beginIsOnBorder && colorOfNewSegment != green && colorOfNewSegment != orange)
  {
    DOT(imageProcessor_obstacles, newSegment.end.x, newSegment.end.y, Drawings::pink, Drawings::pink);
    if(detectedGround[lineIndex+1])
    {
      if(
        Geometry::distance(nearGround[lineIndex+1], newSegment.end) < 
        Geometry::calculateLineSize(newSegment.begin, cameraMatrix, cameraInfo) * 4 )
      {
        nearGround[lineIndex+1] = newSegment.end;
      }
      else detectedGround[lineIndex+1] = false;
    }
    else
    {
      detectedGround[lineIndex+1] = true;
      farGround[lineIndex+1] = newSegment.end;
      nearGround[lineIndex+1] = newSegment.end;

      switch(colorOfLastColoredSegment)
      {
      case red: 
        if(getPlayer().getTeamColor() == Player::red)
          obstacleType[lineIndex+1] = ObstaclesPercept::teammate;
        else
          obstacleType[lineIndex+1] = ObstaclesPercept::opponent;
        break;
      case blue: obstacleType[lineIndex+1] = ObstaclesPercept::opponent; 
        if(getPlayer().getTeamColor() == Player::blue)
          obstacleType[lineIndex+1] = ObstaclesPercept::teammate;
        else
          obstacleType[lineIndex+1] = ObstaclesPercept::opponent;
        break;
      default: obstacleType[lineIndex+1] = ObstaclesPercept::unknown;
      }
      switch(colorOfLastSegment)
      {
      case white: obstacleType[lineIndex+1] = ObstaclesPercept::border; break;
      case yellow: obstacleType[lineIndex+1] = ObstaclesPercept::goal; break;
      case skyblue: obstacleType[lineIndex+1] = ObstaclesPercept::goal; break;
      }
    }
  }//if(newSegment.endIsOnBorder)
}

bool GroundDetector::lightingHasChanged()
{
  return lightingChangeTester.lightingHasChanged();
}

void GroundDetector::createGoalIndication()
{
  if(numberOfGoalSegments >= 2)
  {
    foundGoal = true;
    if(
      accumulatedDistanceFromTopGoalPointsToHorizon < +2000 &&
      accumulatedDistanceFromTopGoalPointsToHorizon > -2000 &&
      accumulatedDistanceFromBottomGoalPointsToHorizon < +2000 &&
      accumulatedDistanceFromBottomGoalPointsToHorizon > -2000 
      )
    {
      goalIndicationAboveHorizon = (int)(accumulatedDistanceFromTopGoalPointsToHorizon / numberOfGoalSegments);
      goalIndicationBelowHorizon = (int)(accumulatedDistanceFromBottomGoalPointsToHorizon / numberOfGoalSegments);
    }
    else
    {
      OUTPUT(idText, text, accumulatedDistanceFromTopGoalPointsToHorizon);
      OUTPUT(idText, text, accumulatedDistanceFromBottomGoalPointsToHorizon);
    }
  }
  else
  {
    foundGoal = false;
  }
}

double GroundDetector::getDistanceFromFieldBorderToHorizon()
{
//  if(borderPointAdded) 
  return distanceFromBorderPointToHorizon;
//  else return -15;
}

void GroundDetector::createObstaclesPercept()
{
  Vector2<int> farPointOnField;
  Vector2<int> nearPointOnField;
  bool farPointIsOnField;
  bool nearPointIsOnField;
  for(int lineIndex = 1; lineIndex < maxNumberOfScanLines + 1; lineIndex++)
  {
    if(detectedGround[lineIndex])
    {
      farPointIsOnField = Geometry::calculatePointOnField(farGround[lineIndex].x, farGround[lineIndex].y, cameraMatrix, cameraInfo, farPointOnField);
      nearPointIsOnField = Geometry::calculatePointOnField(nearGround[lineIndex].x, nearGround[lineIndex].y, cameraMatrix, cameraInfo, nearPointOnField);
      if(!farPointIsOnField)
      {
        Vector2<double> angles;
        Geometry::calculateAnglesForPoint(farGround[lineIndex], cameraMatrix, cameraInfo, angles);
        farPointOnField.x = (int)(cos(angles.x) * 4000);
        farPointOnField.y = (int)(sin(angles.x) * 4000);
      }
      if(/*farPointIsOnField &&*/ nearPointIsOnField && 
        Geometry::distance(farGround[lineIndex], farGround[lineIndex-1]) < 60
        )
      {
        obstaclesPercept.add(
          nearPointOnField, 
          farPointOnField, 
          farGroundIsOnBorder[lineIndex],
          obstacleType[lineIndex]
          );
      }
    }
  }
  
  Geometry::Line line;
  Vector2<int> point1, point2;
  firstGreenPoints.linearRegression(line);

  if(Geometry::getIntersectionPointsOfLineAndRectangle(
    Vector2<int>(0,0),
    Vector2<int>(cameraInfo.resolutionWidth - 1, cameraInfo.resolutionHeight - 1),
    line, point1, point2) )
  {
    LINE(imageProcessor_horizon, point1.x, point1.y, point2.x, point2.y, 1, Drawings::ps_solid, Drawings::red);
  }
}

/*
* Change log :
* 
* $Log: GroundDetector.cpp,v $
* Revision 1.14  2004/03/01 11:56:43  juengel
* Recognition of obstacle types added.
*
* Revision 1.13  2004/02/08 11:01:46  juengel
* Added visualization.
*
* Revision 1.12  2003/12/15 11:47:03  juengel
* Introduced CameraInfo
*
* Revision 1.11  2003/12/01 16:28:31  juengel
* Added lighting change tester.
*
* Revision 1.10  2003/11/26 11:06:15  juengel
* Calculation of goal indications added.
*
* Revision 1.9  2003/11/11 17:01:01  juengel
* improved green calibration
*
* Revision 1.8  2003/11/10 11:33:22  juengel
* added calibration and detection versions for isFieldBorder, isGoal and isFieldLine
*
* Revision 1.7  2003/11/05 16:48:16  juengel
* Green is not updated, if the segment is the border.
*
* Revision 1.6  2003/11/03 20:27:30  juengel
* Added color class yellowOrange
*
* Revision 1.5  2003/10/31 11:46:51  juengel
* Improved creation of obstaclesPercept
*
* Revision 1.4  2003/10/30 18:28:15  juengel
* Added parameter colorTableCuboids to LineSegment::isFieldBorder
*
* Revision 1.3  2003/10/29 13:08:25  juengel
* removed method "updateGreenInColorTable"
*
* Revision 1.2  2003/10/23 07:24:21  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.1  2003/10/06 14:10:13  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.5  2003/09/01 15:23:33  juengel
* new drawings
*
* Revision 1.4  2003/08/30 10:19:52  juengel
* DebugDrawings clean-up 1
*
* Revision 1.3  2003/08/29 13:09:21  juengel
* LinesPercept is detected.
*
* Revision 1.2  2003/08/25 17:26:52  juengel
* clean up
*
* Revision 1.1  2003/08/18 11:47:53  juengel
* Added segment based detectors.
*
*/
