
/** 
* @file FastSUSANNoiseReduction.h
* Declaration of file FastSUSANNoiseReduction.
*
* @author <A href=mailto:walter.nistico@uni-dortmund.de>Walter Nistico</A>
*/

#ifndef _FastSUSANNoiseReduction_h_
#define _FastSUSANNoiseReduction_h_

#include "Representations/Perception/Image.h"

/**
* @class FastSUSANNoiseReduction
*
* This class represents a non-linear image noise-reduction filter.
* The main feature of S.U.S.A.N. filters is their capability to smooth out the noise while preserving image structures
* like lines, edges, borders, and so on, hence if the smoothing threshold is set correctly, the resulting image tipically exhibits no blur.
* This implementation has been significantly modified from the original SUSAN algorithm, in order to reach a vastly reduced
* computational cost making it suitable for real-time processing, so strictly speaking, this is NOT a SUSAN filter, yet it shares the same basic idea.
*
* @author <A href=mailto:walter.nistico@uni-dortmund.de>Walter Nistico</A>
*/

class FastSUSANNoiseReduction 
{

public:

  //~ enum ColorSpectra {componentA=0, componentB, componentC};
  
  /** Constructor */
  FastSUSANNoiseReduction(int smoothingThreshold);

  /** Destructor */
  ~FastSUSANNoiseReduction();

  /** 
  * Filters a chosen pixel of an image.
  * IMPORTANT NOTE: since it's a filter which makes use of a convolution kernel of size 3x3, 
  * always make sure that you don't access the 1 pixel wide beavel of the image,
  * (so it should always be x>0, x<width-1, y>0, y<width-1)
  * otherwise you'll be accessing memory out of the allocated space, with obvious consequences.
  * @param source the source image to be filtered
  * @param posX the x coordinate of the chosen pixel
  * @param posY the y coordinate of the chosen pixel
  * @param valA the first spectrum (ex. Y (luminance)) of the filtered pixel, returned
  * @param valB the second spectrum (ex. U (crominance)) of the filtered pixel, returned
  * @param valC the third spectrum (ex. V (crominance)) of the filtered pixel, returned
  */
  inline void getFilteredPixel(const Image& source, int posX, int posY, unsigned char& valA, unsigned char& valB, unsigned char& valC) const
  {
    valA = getFilteredPixelSpectrum(source, posX, posY, 0);
    valB = getFilteredPixelSpectrum(source, posX, posY, 1);
    valC = getFilteredPixelSpectrum(source, posX, posY, 2);
  }

  /** 
  * Filters a chosen pixel of an image, directly modifying it.
  * IMPORTANT NOTE: since it's a filter which makes use of a convolution kernel of size 3x3, 
  * always make sure that you don't access the 1 pixel wide beavel of the image,
  * (so it should always be x>0, x<width-1, y>0, y<width-1)
  * otherwise you'll be accessing memory out of the allocated space, with obvious consequences.
  * @param source the source image to be filtered
  * @param posX the x coordinate of the chosen pixel
  * @param posY the y coordinate of the chosen pixel
  */
  inline void filterPixel(Image& source, int posX, int posY) const
  {
    source.image[posY][0][posX] = getFilteredPixelSpectrum(source, posX, posY, 0);
    source.image[posY][1][posX] = getFilteredPixelSpectrum(source, posX, posY, 1);
    source.image[posY][2][posX] = getFilteredPixelSpectrum(source, posX, posY, 2);
  }

  /** 
  * Filters a whole image.
  * IMPORTANT NOTE: since it's a filter which makes use of a convolution kernel of size 3x3, 
  * always make sure that you don't access the 1 pixel wide beavel of the image,
  * (so it should always be x>0, x<width-1, y>0, y<width-1)
  * otherwise you'll be accessing memory out of the allocated space, with obvious consequences.
  * @param source the source image to be filtered
  * @param destination the resulting image
  */
  void getFilteredImage(const Image& source, Image& destination) const;
  
private:

  /**
  * A LookUpTable containing a correlation function, in this implementation it's a "rect" for efficiency reasons
  */
  char Susan_LUT[127];

  /**
  * Initializes the LookUpTable
  */
  void setupSusanLUT(int threshold);

  /**
  * The correlation function, precomputed
  */
	inline char correlation(int delta) const
  {
		return Susan_LUT[(delta>>2)+63];
	}

  /** 
  * Filters a single spectrum (ex. Y or U or V) of a chosen pixel of an image.
  * IMPORTANT NOTE: since it's a filter which makes use of a convolution kernel of size 3x3, 
  * always make sure that you don't access the 1 pixel wide beavel of the image,
  * (so it should always be x>0, x<width-1, y>0, y<width-1)
  * otherwise you'll be accessing memory out of the allocated space, with obvious consequences.
  * @param image the source image to be filtered
  * @param posx the x coordinate of the chosen pixel
  * @param posy the y coordinate of the chosen pixel
  *	@param spectrum the chosen image spectrum
  * @return the filtered value
  */
  inline unsigned char getFilteredPixelSpectrum(const Image& image, int posx, int posy, int spectrum) const
  {
    register unsigned int USAN;
    register int counter;
    register unsigned char center;
    register int tempCorrelation;
    register unsigned char neighbours0;
    register unsigned char neighbours1;
    register unsigned char neighbours2;
    register unsigned char neighbours3;
    
    unsigned char resp; 
    int sp = static_cast<int> (spectrum);
    
    center = image.image[posy][sp][posx];          
    neighbours0 = image.image[posy-1][sp][posx];          
    neighbours1 = image.image[posy+1][sp][posx];          
    neighbours2 = image.image[posy][sp][posx-1];          
    neighbours3 = image.image[posy][sp][posx+1];          
    
    tempCorrelation = correlation(neighbours0-center);
    USAN = neighbours0 & tempCorrelation;
    counter = tempCorrelation;
    tempCorrelation = correlation(neighbours1-center);
    USAN += neighbours1 & tempCorrelation;
    counter += tempCorrelation;
    tempCorrelation = correlation(neighbours2-center);
    USAN += neighbours2 & tempCorrelation;
    counter += tempCorrelation;
    tempCorrelation = correlation(neighbours3-center);
    USAN += neighbours3 & tempCorrelation;
    counter += tempCorrelation;
    
    if (counter == -4) { // most likely situation, typically above 85% for smoothin threshold >= 8 (which usually is the case) 
      resp = (unsigned char) (USAN>>2);
    }
    else 
    if (counter == -3) {// second most likely situation 
      USAN += center;
      resp = (unsigned char) (USAN>>2);
    }
    else 
    if (counter == -2) {
      resp = (unsigned char) (USAN>>1);
    }
    else 
    if (counter == -1) {
      USAN += center;
      resp = (unsigned char) (USAN>>1);
    }
    else 
    { // counter == 0
      unsigned char swap;
      if (neighbours0>neighbours1){
        swap = neighbours1;
        neighbours1 = neighbours0;
        neighbours0 = swap;
      }
      if (neighbours2>neighbours3){
        swap = neighbours3;
        neighbours3 = neighbours2;
        neighbours2 = swap;
      }
      if (neighbours2>neighbours0){
        neighbours0 = neighbours2;
      }
      if (neighbours3<neighbours1){
        neighbours1 = neighbours3;
      }
      USAN = neighbours0 + neighbours1;
      resp = (unsigned char) (USAN>>1);
    }
    return resp;
  }

};

#endif   //  _FastSUSANNoiseReduction_h_

/*
* Change log :
* 
* $Log: FastSUSANNoiseReduction.h,v $
* Revision 1.3  2004/04/08 15:33:06  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.2  2004/03/19 11:04:58  nistico
* Some corrections and restructuring
*
* Revision 1.1  2004/01/23 17:07:33  nistico
* Added FastSUSANNoiseReduction.*, a non-linear image noise reduction filter, to ImageProcessorTools
* Added debug visualization of segmented image in ColorTable32KImageProcessor, and experimental support of FastSUSANNoiseReduction
*
*
*/
