/** 
* @file ColorTableCuboids.cpp
* Implementation of class ColorTableCuboids.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

#include "ColorTableCuboids.h"
#include "Platform/SystemCall.h"


ColorTableCuboids::ColorTableCuboids()
{
  setThresholds(green, 100, 150, 100, 150, 100, 150);
}

ColorTableCuboids::~ColorTableCuboids()
{
}

int ColorTableCuboids::getThresholdMin(colorClass color, int channel) const
{
  return thresholdMin[color][channel];
}

int ColorTableCuboids::getThresholdMax(colorClass color, int channel) const
{
  return thresholdMax[color][channel];
}

void ColorTableCuboids::setThresholds(colorClass color, int minY, int minU, int minV, int maxY, int maxU, int maxV)
{
  lastTimeWhenThresholdWasSet[color] = SystemCall::getCurrentSystemTime();
  thresholdMin[color][0] = minY; thresholdMin[color][1] = minU; thresholdMin[color][2] = minV;
  thresholdMax[color][0] = maxY; thresholdMax[color][1] = maxU; thresholdMax[color][2] = maxV;
}

void ColorTableCuboids::clear()
{
  for(int color = 0; color < numOfColors; color++)
  {
    for(int channel = 0; channel < 3; channel++)
    {
      thresholdMin[color][channel] = 255;
      thresholdMax[color][channel] = 0;
    }
  }

}

colorClass ColorTableCuboids::getColorClass
(
 const unsigned char y, 
 const unsigned char u, 
 const unsigned char v
 ) const
{
  if(
    y > thresholdMin[orange][0] && 
    y < thresholdMax[orange][0] &&   
    u > thresholdMin[orange][1] && 
    u < thresholdMax[orange][1] && 
    v > thresholdMin[orange][2] &&
    v < thresholdMax[orange][2] &&
    SystemCall::getTimeSince(lastTimeWhenThresholdWasSet[orange]) < 2500 )
  {
    return orange;
  }
  else if(
    y > thresholdMin[green][0] && 
    y < thresholdMax[green][0] &&   
    u > thresholdMin[green][1] && 
    u < thresholdMax[green][1] && 
    v > thresholdMin[green][2] &&
    v < thresholdMax[green][2] &&
    SystemCall::getTimeSince(lastTimeWhenThresholdWasSet[green]) < 2500000 )
  {
    return green;
  }
  else if(
    y > thresholdMin[white][0] && 
    y < thresholdMax[white][0] &&   
    u > thresholdMin[white][1] && 
    u < thresholdMax[white][1] && 
    v > thresholdMin[white][2] &&
    v < thresholdMax[white][2] &&
    SystemCall::getTimeSince(lastTimeWhenThresholdWasSet[white]) < 5000 )
  {
    return white;
  }  
  else if(
    y > thresholdMin[skyblue][0] && 
    y < thresholdMax[skyblue][0] &&   
    u > thresholdMin[skyblue][1] && 
    u < thresholdMax[skyblue][1] && 
    v > thresholdMin[skyblue][2] &&
    v < thresholdMax[skyblue][2] &&
    SystemCall::getTimeSince(lastTimeWhenThresholdWasSet[skyblue]) < 5000 )
  {
    return skyblue;
  }
  else if(
    y > thresholdMin[yellow][0] && 
    y < thresholdMax[yellow][0] &&   
    u > thresholdMin[yellow][1] && 
    u < thresholdMax[yellow][1] && 
    v > thresholdMin[yellow][2] &&
    v < thresholdMax[yellow][2] &&
    SystemCall::getTimeSince(lastTimeWhenThresholdWasSet[yellow]) < 5000 )
  {
    return yellow;
  }
  else return noColor;
}

void ColorTableCuboids::generateColorClassImage
(
 const Image& image, 
 ColorClassImage& colorClassImage) const
{
  colorClass color;
  colorClassImage.width = image.cameraInfo.resolutionWidth;
  colorClassImage.height = image.cameraInfo.resolutionHeight;

  for (register int y=0; y < image.cameraInfo.resolutionHeight; y++) 
  {
    for (register int x=0; x < image.cameraInfo.resolutionWidth; x++) 
    {
      color = getColorClass(image.image[y][0][x],image.image[y][1][x],image.image[y][2][x]);
      colorClassImage.image[y][x] = color;
    }
  }
}


In& operator>>(In& stream, ColorTableCuboids& colorTableCuboids)
{
  int thresholdMin[numOfColors][3];
  int thresholdMax[numOfColors][3];
  for(int color = 0; color < numOfColors; color++)
  {
    for(int channel = 0; channel < 3; channel++)
    {
      stream >> thresholdMin[color][channel];
      stream >> thresholdMax[color][channel];
    }
    colorTableCuboids.setThresholds((colorClass)color, 
      thresholdMin[color][0],
      thresholdMin[color][1],
      thresholdMin[color][2],
      thresholdMax[color][0],
      thresholdMax[color][1],
      thresholdMax[color][2]);
  }
  return stream;
}

Out& operator<<(Out& stream, const ColorTableCuboids& colorTableCuboids)
{
  for(int color = 0; color < numOfColors; color++)
  {
    for(int channel = 0; channel < 3; channel++)
    {
      stream << colorTableCuboids.getThresholdMin((colorClass)color, channel);
      stream << colorTableCuboids.getThresholdMax((colorClass)color, channel);
    }
  }

  return stream;
}

/*
* Change log :
* 
* $Log: ColorTableCuboids.cpp,v $
* Revision 1.4  2003/12/30 23:55:01  roefer
* Image size is now 208 x 160. Smaller images are placed in the upper left corner
*
* Revision 1.3  2003/12/15 11:46:14  juengel
* Introduced CameraInfo
*
* Revision 1.2  2003/10/29 13:13:43  juengel
* added method "clear"
*
* Revision 1.1  2003/10/23 07:15:17  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor,
* added ColorTableCuboids.
*
*/
