/**
* @file Modules/ImageProcessor/ImageProcessorTools/ColorCorrector.h
* 
* This file contains a class that represents a table used for color correction.
* It is a mixture among the BB2004, DDD2004 and MSH2004 correction.
* Everything is static so it is loaded only once in the simulator.
*
* @author <A href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</A>
* @author Max Risler
* @author <A href="mailto:roefer@tzi.de">Thomas Rfer</A>
*/

#ifndef __ColorCorrector_h_
#define __ColorCorrector_h_

#include "Representations/Perception/Image.h"

#define MSH

/**
* @class ColorCorrector
* The class represents a table for color correction.
*/
class ColorCorrector
{
private:
  enum {maxRadius = 140, centerRadius = 10};
  static unsigned char radiusTable[cameraResolutionHeight_ERS7][cameraResolutionWidth_ERS7]; /**< The radius table. */
  static unsigned char correctionTable[maxRadius][256][3]; /**< The correction table. */

#ifdef MSH
  enum {maxRadialOrder = 10, maxColorOrder = 10};
  static int radialOrder;
  static int colorOrder;
  static double radialP[3 * maxRadialOrder];
  static double colorP[3 * maxColorOrder];

  static unsigned char colorDistortionCorrection(const unsigned char radius_i, const unsigned char color, 
                                                 const unsigned char channel);
#endif

#ifdef _WIN32  
  static unsigned char distortionTable[maxRadius][256][3]; /**< The distortion table. */
#endif
  static bool loaded; /**< Determines whether the corrector has already been loaded. */

  /**
  * The function calculates the corresponding radius for an image coordinate.
  * @param x The x coordinate of a pixel.
  * @param y The y coordinate of a pixel.
  * @return The distance from the center of the image.
  */
  static unsigned char calcRadius(int x, int y)
  {
    int r = (int) (Vector2<double>(x - cameraResolutionWidth_ERS7 / 2,
                                   y - cameraResolutionHeight_ERS7 / 2).abs() - centerRadius);
    if(r < 0)
      r = 0;
    return (unsigned char) r;
  }

public:
  /**
  * The functions returns a corrected intensity of a pixel.
  * @param x The x coordinate of the pixel.
  * @param y The y coordinate of the pixel.
  * @param c The color channel corrected.
  * @param intensity The intensity of the pixel in color channel c.
  * @return The corrected intensity.
  */
  static unsigned char correct(const int x, const int y, 
                               const int c, const unsigned char intensity)
  {
    return correctionTable[radiusTable[y][x]][intensity][c];
  }

  /**
  * The functions returns the corrected color of a pixel.
  * @param x The x coordinate of the pixel.
  * @param y The y coordinate of the pixel.
  * @param intensityY The intensity of the pixel in color channel Y.
  * @param intensityU The intensity of the pixel in color channel U.
  * @param intensityV The intensity of the pixel in color channel V.
  */
  static void correct(const int x, const int y, 
                      unsigned char& intensityY, 
                      unsigned char& intensityU, 
                      unsigned char& intensityV)
  {
    int radius = radiusTable[y][x];
    intensityY = correctionTable[radius][intensityY][0];
    intensityU = correctionTable[radius][intensityU][1];
    intensityV = correctionTable[radius][intensityV][2];
  }

  /**
  * The functions corrects all pixels of an image.
  * @param image The image that is corrected.
  */
  static void correct(Image& image);

#ifdef _WIN32
  /**
  * The functions returns a disturbed color of a pixel.
  * @param x The x coordinate of the pixel.
  * @param y The y coordinate of the pixel.
  * @param intensityY The intensity of the pixel in color channel Y.
  * @param intensityU The intensity of the pixel in color channel U.
  * @param intensityV The intensity of the pixel in color channel V.
  */
  static void distort(const int x, const int y, 
                      unsigned char& intensityY, 
                      unsigned char& intensityU, 
                      unsigned char& intensityV)
  {
    int radius = radiusTable[y][x];
    intensityY = distortionTable[radius][intensityY][0];
    intensityU = distortionTable[radius][intensityU][1];
    intensityV = distortionTable[radius][intensityV][2];
  }

  /**
  * The functions disturbs all pixels of an image.
  * @param image The image that is disturbed.
  */
  static void distort(Image& image);

#endif

  /**
  * Loads the calibration image and computes the lookup table
  */
  static void load();

  /**
  * The function disables the color correction.
  * The table is cleared.
  */
  static void disable();
};

#endif// __ColorCorrector_h_

/*
* Change log :
* 
* $Log: ColorCorrector.h,v $
* Revision 1.4  2004/05/15 14:31:21  nistico
* ColorCorrector now by default loads coefficients from Config/ folder.
* If other coefficients are present in the folder pointed by location.cfg,
* these override the default ones.
*
* Revision 1.3  2004/04/24 08:14:03  roefer
* Author name was missing
*
* Revision 1.2  2004/04/22 14:28:46  roefer
* ColorCorrector now supports MSH and DDD color correction
*
* Revision 1.1  2004/04/17 21:50:51  roefer
* New color corrector (BB/DDD mix) used in BB2004ImageProcessor and Simulator
* Displaying corrected images as debug image
*
*/
