/**
* @file GridImageProcessor2.h
* 
* This file contains a class for Image Processing.
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#ifndef __GridImageProcessor2_h_
#define __GridImageProcessor2_h_

#include "Modules/ImageProcessor/ImageProcessor.h"
#include "Tools/Math/Geometry.h"
#include "Tools/RingBufferWithSum.h"
#include "Tools/Debugging/DebugImages.h"
#include "../ImageProcessorTools/ColorSpaceUsageCounter.h"
#include "../ImageProcessorTools/ColorTableReferenceColor.h"
#include "../ImageProcessorTools/ColorTableCuboids.h"
#include "../ImageProcessorTools/SegmentBasedAnalysis/SegmentBasedDetector.h"
#include "../ImageProcessorTools/SegmentBasedAnalysis/GroundDetector.h"
#include "../ImageProcessorTools/SegmentBasedAnalysis/BallDetector.h"
#include "../ImageProcessorTools/SegmentBasedAnalysis/OrangeCalibration.h"
#include "../ImageProcessorTools/SegmentBasedAnalysis/LineSegmentation.h"
//#include "../ImageProcessorTools/ManualCalibration.h"
#include "GridImageProcessor2Parameters.h"

/**
* @class GridImageProcessor2
*
* A solution of the ImageProcessor module.
*/
class GridImageProcessor2 : public ImageProcessor
{
public:
  GridImageProcessor2(const ImageProcessorInterfaces& interfaces);
  ~GridImageProcessor2();
  
  /** Executes the module */
  virtual void execute();
  virtual bool handleMessage(InMessage& message);
  
private:
//  ManualCalibration manualCalibration;

  ColorSpaceUsageCounter colorSpaceUsageCounter1;
  ColorSpaceUsageCounter colorSpaceUsageCounter2;
  ColorSpaceUsageCounter& lastColorSpaceUsageCounter;
  ColorSpaceUsageCounter& currentColorSpaceUsageCounter;
  RingBufferWithSum<20> averageIntensity[3];
  int accumulatedIntensity[3];
  int numberOfScannedPixelsOnLongLines;
  int numberOfScannedPixels;
  Vector3<int> averageIntensityOverLastImages;

  RingBufferWithSum<20> averageEdgeness[3];
  int accumulatedEdgeness[3];
  Vector3<int> averageEdgenessOverLastImages;

  GroundDetector groundDetector;
  BallDetector ballDetector;
  OrangeCalibration orangeCalibration;

  enum{maxNumberOfScanLines = 240};
  
  /** The number of scan lines that fit into the image */
  int numberOfScanLines;
  
  /** Calculates the scan lines for the grid below the horizon*/
  int calculateVerticalScanLines();
  
  /** Scans the vertical grid lines*/
  void scanVertical(
    int longLineIndex, 
    bool lineIsLong,
    const Vector2<int>& start, 
    const Vector2<int>& end);
  
  /** Scans the vertical grid lines*/
  void scanVertical2(
    int longLineIndex, 
    const Vector2<int>& start, 
    const Vector2<int>& end);

  /** Scans the vertical grid lines*/
  void scanVertical3(
    const Vector2<int>& start, 
    const Vector2<int>& end);

  /** A representation of the horizon */
  Geometry::Line horizonLine, verticalLine;
  
  enum {maxNumberOfLineSegments = 40};

  int numberOfLineSegments[maxNumberOfScanLines];

  /** */
  ColorTable64 colorTable64;
  ColorTableCuboids colorTableCuboids;
  ColorTableReferenceColor colorTableReferenceColor;

  int goalIndicationAboveHorizon;
  int goalIndicationBelowHorizon;
  bool foundGoal;

  GridImageProcessor2Parameters parameters;
  
  /** Debug */
  void visualize();
  
  void drawSegment(LineSegment lineSegment);

  /** */
  int returnCondition;

  DECLARE_DEBUG_IMAGE(imageProcessorGeneral);
  DECLARE_DEBUG_IMAGE(imageProcessorScanLines);
  DECLARE_DEBUG_IMAGE(imageProcessorGradients);
  DECLARE_DEBUG_IMAGE(imageProcessorBall);
  DECLARE_DEBUG_COLOR_CLASS_IMAGE(segmentedImage1);
  DECLARE_DEBUG_COLOR_CLASS_IMAGE(segmentedImage2);
  DECLARE_DEBUG_COLOR_CLASS_IMAGE(segmentedImage3);
  DECLARE_DEBUG_IMAGE(classificationY);
  DECLARE_DEBUG_IMAGE(classificationU);
  DECLARE_DEBUG_IMAGE(classificationV);
  DECLARE_DEBUG_IMAGE(colorFrequency);
};
#endif// __GridImageProcessor2_h_

/*
* Change log :
* 
* $Log: GridImageProcessor2.h,v $
* Revision 1.15  2004/03/08 01:39:01  roefer
* Interfaces should be const
*
* Revision 1.14  2004/02/12 14:16:20  juengel
* changed visualization of segmented image 1, 2, 3
*
* Revision 1.13  2003/12/15 11:47:03  juengel
* Introduced CameraInfo
*
* Revision 1.12  2003/12/04 17:39:00  juengel
* Moved LineSegmentation to own class.
*
* Revision 1.11  2003/12/01 16:26:12  juengel
* Added calculation of "pixel usage" goal pixels are not counted yet.
*
* Revision 1.10  2003/11/26 11:43:31  juengel
* Goal recognition added.
*
* Revision 1.9  2003/11/12 17:40:14  juengel
* color class based ball detection is done on all scan lines now
*
* Revision 1.8  2003/11/10 11:37:12  juengel
* Added extra ball scan lines.
*
* Revision 1.7  2003/11/07 11:21:19  juengel
* GridImageProcessor2 has more parameters now.
*
* Revision 1.6  2003/11/03 20:16:09  juengel
* color class images can be sent from robot now
*
* Revision 1.5  2003/10/29 13:06:19  juengel
* New constructor for LineSegment.
*
* Revision 1.4  2003/10/23 07:24:20  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.3  2003/10/12 20:18:07  juengel
* ColorTable is not const anymore.
*
* Revision 1.2  2003/10/12 11:51:15  juengel
* Added ManualCalibration.
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.6  2003/09/01 10:16:17  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.5  2003/08/25 17:21:49  juengel
* Added some debug images.
*
* Revision 1.4  2003/08/18 12:09:23  juengel
* Added segment based detection.
*
* Revision 1.3  2003/07/29 12:47:09  juengel
* Moved calculateHorizon to Geometry
*
* Revision 1.2  2003/07/21 13:44:07  juengel
* clean up
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.32  2003/06/13 14:53:31  juengel
* Added scanLinesForObstacles.
*
* Revision 1.31  2003/06/12 16:55:22  juengel
* Moved RangeArray to its own file.
*
* Revision 1.30  2003/05/26 08:21:20  juengel
* Improved recognition of goals.
*
* Revision 1.29  2003/05/05 12:17:15  juengel
* Moved some parameters to the parameters struct.
*
* Revision 1.28  2003/04/26 11:35:26  juengel
* Three scan lines are used to check vertical for goal indications.
*
* Revision 1.27  2003/04/23 16:14:00  juengel
* Changed calculation and scan of horizontal lines, added calculation of best angle two own and opponent goal.
*
* Revision 1.26  2003/04/03 18:50:08  juengel
* Improved AutoCalibration.
*
* Revision 1.25  2003/04/01 13:59:59  juengel
* Added ColorTableReferenceColor.
*
* Revision 1.24  2003/03/31 20:55:34  juengel
* calculateSingleBallPercept
*
* Revision 1.23  2003/03/28 14:34:55  juengel
* Added new circle calculation methods.
*
* Revision 1.22  2003/03/25 11:03:09  juengel
* New calculation of ball circle.
*
* Revision 1.21  2003/03/12 12:36:49  juengel
* clean up
*
* Revision 1.20  2003/03/11 11:30:22  juengel
* Added useColorTable
*
* Revision 1.19  2003/03/10 13:56:57  juengel
* Worked at goal recognition
*
* Revision 1.18  2003/03/08 11:58:32  juengel
* improved recognition of ball
*
* Revision 1.17  2003/03/06 14:29:07  juengel
* Added several returns for time measurement.
*
* Revision 1.16  2003/03/05 17:18:33  juengel
* Improved flag perception.
*
* Revision 1.15  2003/03/04 14:54:23  juengel
* Added player perception.
*
* Revision 1.14  2003/02/27 12:55:40  juengel
* Clean up.
*
* Revision 1.13  2003/02/25 14:00:58  juengel
* dies und das
*
* Revision 1.12  2003/02/13 09:40:01  juengel
* added auto calibration of green tolerance
* improved recognition of flags
*
* Revision 1.11  2003/02/11 17:08:59  juengel
* recognition of flags added
*
* Revision 1.10  2003/02/09 15:12:59  juengel
* Added scanLinesForLinesGoalsBallAndPlayers.
*
* Revision 1.9  2003/02/09 13:02:10  juengel
* Added scanLinesForLines.
*
* Revision 1.8  2003/02/09 10:09:15  juengel
* clean up
*
* Revision 1.7  2003/02/08 18:41:46  juengel
* big change
*
* Revision 1.6  2003/01/30 22:32:35  juengel
* Added LinesPerceptor functionality.
*
* Revision 1.5  2003/01/13 10:33:16  juengel
* pattern matching for flags
*
* Revision 1.4  2002/12/03 19:01:43  juengel
* Comment added.
*
* Revision 1.3  2002/12/02 12:02:58  juengel
* Line clipping for flag predictions added.
*
* Revision 1.2  2002/12/01 17:39:37  juengel
* Added calculation of the horizon and prediction of flags.
*
* Revision 1.1  2002/11/26 14:19:25  juengel
* GridImageProcessor2 added.
*
*/
