/**
* @file GT2004ImageProcessor.h
*
* Definition of class GT2004ImageProcessor
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#ifndef __GT2004ImageProcessor_h_
#define __GT2004ImageProcessor_h_

#include "Modules/ImageProcessor/ImageProcessor.h"
#ifndef CT32K_LAYOUT
#include "../ImageProcessorTools/ManualCalibration.h"
#endif
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Debugging/DebugImages.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ColorCorrector.h"
#include "GT2004ImageProcessorTools.h"
#include "GT2004GoalRecognizer.h"
#include "GT2004BallSpecialist.h"
#include "GT2004BeaconDetector.h"

/**
* @class GT2004ImageProcessor
*
* The lines image processor recognizes characteristic lines in the image.
* Four types of lines are distinguished:
* edges between the skyblue goal and the field, edges between the yellow goal 
* and the field, edges between the border and the field, and edges between the
* field lines and the field.
*
* The module scans vertical and horizontal lines in the image from top to bottom
* and from left to right. As the green of the field is very dark, all edges are
* characterized by a big difference of the y-channel of adjacent pixels. An
* increase in the y-channel followed by a decrease is an indication for an edge.
*
* The projection of the pixels on the field plane is used to determine their
* relative position to the robot.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/ 
class GT2004ImageProcessor : public ImageProcessor
{
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the GT2004ImageProcessor module.
  */
  GT2004ImageProcessor(const ImageProcessorInterfaces& interfaces);

  /** Executes the module */
  virtual void execute();

  /** Handles an incoming message
  * @param message The message
  */
  virtual bool handleMessage(InMessage& message);

private:
#ifndef CT32K_LAYOUT
  ManualCalibration manualCalibration;
#endif
  double xFactor, /**< Factor to convert the pixel coordinate space to the anglular coordinate space. */
         yFactor; /**< Factor to convert the pixel coordinate space to the anglular coordinate space. */
  int yThreshold; /**< Brightness increase threshold. */
  int vThreshold; /**< Brightness decrease threshold. */
  int orangeCount,  /**< Number of columns with ball points. */
      noOrangeCount, /**< Number of columns without a ball point. */
      noRedCount, /**< Number of columns without a red robot point. */
      noBlueCount, /**< Number of columns without a blue robot point. */
      noGoalCount, /**< Number of columns without a opponent goal seen. */
      closestBottom; /**< Closest bottom point on the grid. */
  Vector2<int> firstRed, /**< First red robot point in a cluster. */
               closestRed, /**< Closest red robot point in a cluster. */
               lastRed, /**< Last red robot point in a cluster. */
               firstBlue, /**< First blue robot point in a cluster. */
               closestBlue, /**< Closest blue robot point in a cluster. */
               lastBlue, /**< Last blue robot point in a cluster. */
               firstFlag, /**< First flag point in a cluster. */
               lastFlag; /**< Last flag point in a cluster. */
  bool goalAtBorder; /**< Is the first goal point at the image border? */
  int longestBallRun;
  Vector2<int> ballCandidate;

  CameraMatrix cmTricot; /**< Camera matrix without tricot height. */

  ColorCorrector colorCorrector; /**< The color correction tool. */

  GT2004BeaconDetector beaconDetector; /**< The beacon detector */

  GT2004GoalRecognizer goalRecognizer; /**< The goal recognizer. */

  GT2004BallSpecialist ballSpecialist; /**< The ball specialist. */

  ImageInfo imageInfo; /**< Additional information about the current image */

  /**
  * The function scans columns for line points.
  */
  void scanColumns();

  /** 
  * The function scans rows for line points. 
  */
  void scanRows();

  /** 
  * The function scans a line for line points. 
  * @param start The start point of the line.
  * @param end The end point of the line.
  * @param vertical Vertical lines are scanned for more information.
  * @param noLines Should the line not be scanned for points on field lines or borders?
  */
  void scan(const Vector2<int>& start, const Vector2<int>& end,
            bool vertical, bool noLines);
  
  /** 
  * The function clusters points of red and blue robots.
  * @param bottomPoint The bottom point of the current scan column.
  * @param redFound Has a red robot point been found? In that case, the last
  *                 entry in the lines percept is that point.
  * @param blueFound Has a blue robot point been found? In that case, the last
  *                  entry in the lines percept is that point.
  */
  void clusterRobots(const unsigned char* bottomPoint, bool redFound, bool blueFound);

  /** 
  * The function clusters flag points.
  * @param flagType The type of the flag.
  * @param point The center of the pink area.
  */
  //void clusterFlags(Flag::FlagType flagType, const Vector2<int>& point);

  /**
  * The function filters the percepts, i.e. it removes potential misreadings.
  */
  void filterPercepts();

  /** 
  * The function converts an address to pixel coordinates.
  * @param p An address in image.image.
  * @return The x- and y-coordinates of the corresponding pixel.
  */
  Vector2<int> getCoords(const unsigned char* p) const
  {
    const int diff(p - &image.image[0][0][0]);
    return Vector2<int>(diff % cameraResolutionWidth_ERS7, diff / (cameraResolutionWidth_ERS7 * 6));
  }

  //!@name Helpers for grid drawing
  //!@{
  const unsigned char* last;
  Drawings::Color lineColor;
  void plot(const unsigned char* p,Drawings::Color color);
  //!@}

  double angleBetweenDirectionOfViewAndGround;

  int numberOfScannedPixels;

  DECLARE_DEBUG_IMAGE(imageProcessorPlayers);
  DECLARE_DEBUG_IMAGE(imageProcessorGeneral);
  DECLARE_DEBUG_COLOR_CLASS_IMAGE(segmentedImage1);
  DECLARE_DEBUG_IMAGE(imageProcessorBall);
};

#endif// __GT2004ImageProcessor_h_

/*
* $Log: GT2004ImageProcessor.h,v $
* Revision 1.3  2004/05/06 16:03:56  nistico
* Supports ColorTable32K through CT32K_LAYOUT switch located into
* GT2004ImageProcessorTools.h
*
* Revision 1.2  2004/05/05 13:27:38  tim
* removed unnecessary lines of code
*
* Revision 1.1  2004/05/04 13:40:19  tim
* added GT2004ImageProcessor
*
*/
