/**
* @file GT2004FlagSpecialist.h
* 
* This file contains a class for Image Processing.
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
*/

#ifndef __GT2004FlagSpecialist_h_
#define __GT2004FlagSpecialist_h_

#include "Tools/Debugging/DebugImages.h"

#include "Representations/Perception/ColorTable.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Tools/Math/Geometry.h"

/**
* As part of the GridImageProcessor the GT2004FlagSpecialist searches for
* flag landmarks starting at a point in an Image determined before
*/
class GT2004FlagSpecialist  
{
public:
  GT2004FlagSpecialist(const ColorCorrector&);
  
  /** Initializes the flag specialist.*/
  void init(const Image& image);
  
  /** Searches for the flag in the image, starting from the specified point. */
  void searchFlags(
    const Image& image,
    const ColorTable& colorTable,
    const CameraMatrix& cameraMatrix,
    colorClass color,
    bool pinkIsTop,
    const Geometry::Line horizonLine,
    int x, int y);
  
  void getFlagPercept(
    const CameraMatrix& cameraMatrix, 
    const CameraInfo& cameraInfo, 
    const Geometry::Line horizonLine,
    LandmarksPercept& landmarksPercept
    );

private:
  const ColorCorrector& colorCorrector;

  int imageWidth;
  int imageHeight;

// search for flags variables
  Vector2<int> north;
  Vector2<int> east;
  Vector2<int> south;
  Vector2<int> west;
  Vector2<int>start;
  Vector2<int>destination;

  void estimateOffsetForFlags(
   LandmarksPercept& landmarksPercept,
   const Vector2<double>& cameraOffset
   );

  enum {maxNumberOfBoundingBoxes = 6};
  enum {numberOfHorizontalScans = 5, numberOfVerticalScans = 3};

  double boundingBoxTop[maxNumberOfBoundingBoxes][6];
  double boundingBoxBottom[maxNumberOfBoundingBoxes][6];
  double boundingBoxLeft[maxNumberOfBoundingBoxes][6];
  double boundingBoxRight[maxNumberOfBoundingBoxes][6];

  bool boundingBoxTopValid[maxNumberOfBoundingBoxes][6];
  bool boundingBoxBottomValid[maxNumberOfBoundingBoxes][6];
  bool boundingBoxLeftValid[maxNumberOfBoundingBoxes][6];
  bool boundingBoxRightValid[maxNumberOfBoundingBoxes][6];

  int numberOfBoundingBoxes[6];

  int bestBoundingBox[6];

  enum Direction{up, down, left, right};
  enum DebugType{initial, vertical, horizontal};

  bool findEndOfFlag(
    const Image& image,
	const CameraInfo& bwCameraInfo, 
    const ColorTable& colorTable,
    const Vector2<int> start,
    Vector2<double> horizonDirection,
    Direction direction,
    Vector2<int>& destination,
    colorClass color,
    bool pinkIsTop,
    bool startIsInTop,
    DebugType type,
    int& countPixel,
    int& countTop,
    int& countBottom
    );

public:  
  /** Used for visualization of the flag scans*/
  DECLARE_DEBUG_IMAGE(imageProcessorFlags);
};


#endif// __GT2004FlagSpecialist_h_

/*
* $Log: GT2004FlagSpecialist.h,v $
* Revision 1.1  2004/05/04 13:40:19  tim
* added GT2004ImageProcessor
*
*/
