/**
* @file GT2004BeaconDetector.h
* 
* Declaration of class GT2004BeaconDetector.
*
* @author <a href="mailto:timlaue@tzi.de">Tim Laue</a>
*/ 

#ifndef GT2004BeaconDetector_h_
#define GT2004BeaconDetector_h_

#include "Tools/Math/Vector2.h"
#include "Tools/Math/Matrix2x2.h"
#include "GT2004FlagSpecialist.h"

#define MAX_NUMBER_OF_PINK_RUNS 20


/**
* @class BresenhamParameters
*
* A class which stores and computes the 
* parameters needed by the Bresenham algorithm
*/
class BresenhamParameters
{
public:
  /** Constructor: Computes parameters for a line
  * @param start The start point of the line
  * @param end The end point of the line
  */
  BresenhamParameters(const Vector2<int>& start, const Vector2<int> end)
  {
    int dx(end.x - start.x);
    int dy(end.y - start.y);
    int incX = ((dx>0) ? 1:-1);
    int incY = ((dy>0) ? 1:-1);
    int absDx(abs(dx));
    int absDy(abs(dy));
    alongX = (absDy < absDx);
    if(alongX)
    {
      baseError = -absDx;
      delta = 2*absDy;
      standardOffset.x = incX;
      correctionOffset.y = incY;
      numberOfPixels = absDx;
    }
    else
    {
      baseError = -absDy;
      delta = 2*absDx;
      standardOffset.y = incY;
      correctionOffset.x = incX;
      numberOfPixels = absDy;
    }
    resetError = 2*baseError;
  }

  /** Increase x-values, if true*/
  bool alongX;
  /** The error per step*/
  int delta;
  /** The initial error value*/
  int baseError;
  /** Resets the error to a value less than zero*/
  int resetError;
  /** The standard offset per step*/
  Vector2<int> standardOffset;
  /** The additional offset, if the error is above zero*/
  Vector2<int> correctionOffset;
  /** The number of pixels*/
  int numberOfPixels;
};


/**
* @class Run
*
* Describes a sequence of pixels of the same colour
*/
class Run
{
public:
  /** Constructor*/
  Run():length(0) {}

  /** The first point*/
  Vector2<int> start;
  /** The last point*/
  Vector2<int> end;
  /** The first point of the corresponding scan line*/
  Vector2<int> scanLineStart;
  /** The length of the run*/
  int length;
  /** The color*/
  colorClass color;
};


/**
* @class TransformedRun
*
* Special class for comparing and clustering runs
*/
class TransformedRun
{
public:
  /** Computes values
  * @param run An original run
  * @param rotMat The rotation matrix
  * @param numOfRun The number of the transformed run
  */
  void transform(const Run& run, const Matrix2x2<double>& rotMat, int numOfRun)
  {
    start.x = (double)(run.start.x - run.scanLineStart.x);
    start.y = (double)(run.start.y - run.scanLineStart.y);
    end.x = (double)(run.end.x - run.scanLineStart.x);
    end.y = (double)(run.end.y - run.scanLineStart.y);
    start = (rotMat*start);
    end = (rotMat*end);
    start.x += (double)run.scanLineStart.x;
    start.y += (double)run.scanLineStart.y;
    end.x += (double)run.scanLineStart.x;
    end.y += (double)run.scanLineStart.y;
    this->numOfRun = numOfRun;
  }

  /** The first point*/
  Vector2<double> start;
  /** The last point*/
  Vector2<double> end;
  /** Number of corresponding real run*/
  int numOfRun;
};


/**
* @class GT2004BeaconDetector
*
* A class for detecting beacons in images
*/
class GT2004BeaconDetector
{
public:
  /** Constructor */ 
  GT2004BeaconDetector(const Image& image, const CameraMatrix& cameraMatrix,
                       const ImageInfo& imageInf,
                       const ColorTable& colorTable, 
                       const ColorCorrector& colorCorrector,
                       LandmarksPercept& landmarksPercept);

  /** Executes the beacon detection*/
  void execute();

  /** Gains information about pink from the current color table*/
  void analyzeColorTable();

private:
  /** The image*/
  const Image& image;
  /** The camera matrix*/
  const CameraMatrix& cameraMatrix;
  /** Additional Image information*/
  const ImageInfo& imageInf;
  /** The color table*/
  const ColorTable& colorTable;
  /** The landmarks percept*/
  LandmarksPercept& landmarksPercept;
  /** The number of possible beacons*/
  int numOfBeaconCandidates;
  /** The list of candidates*/
  Run beaconCandidates[MAX_NUMBER_OF_PINK_RUNS];
  /** A buffer for transformed candidates*/
  TransformedRun transformedCandidates[MAX_NUMBER_OF_PINK_RUNS];
  /** The base offset for horizontal scan lines*/
  const double horizontalBaseOffset;
  /** The number of horizontal scan lines*/
  const int numOfHorizontalScanLines;
  /** The growth factor of the distance between two scan lines*/
  const double horizontalOffsetModifier;
  /** The maximum distance between the merged runs*/
  const int clusteringDistanceTolerance;
  /** The minimum length of a pink run*/
  const int minPinkRunLength;
  /** The minimum U channel value of a pink pixel*/
  unsigned char minPinkUValue;
  /** A flag specialist*/
  GT2004FlagSpecialist flagSpecialist;
  /** The color corrector*/
  const ColorCorrector& colorCorrector;

  /** Adds a pink run to the list of candidates
  * @param pinkRun A pink run
  * @return true, if the run has been appended
  */
  bool addCandidate(const Run& pinkRun);

  /** Scans along a line*/ 
  void scanForPink(const Vector2<int>& start, const Vector2<int>& end);

  /** Scans for one other beacon part*/
  bool scanForBeaconPart(const Vector2<int>& start, const Vector2<int>& end,
                         Vector2<int>& position, Vector2<int>& edge, colorClass& color); 

  /** Clusters the pink elements to find the pink beacon parts*/
  void clusterPinkBeaconParts();

  /** Tries to detect a beacon near a pink part*/
  bool analyzeBeacon(const Vector2<int>& position, const double pinkRunWidth);
};

#endif //GT2004BeaconDetector_h_

/*
* $Log: GT2004BeaconDetector.h,v $
* Revision 1.3  2004/05/18 18:39:25  nistico
* BeaconDetector improved, the landmarks are recognized much more often,
* and the width is estimated correctly most of the time.
* Some things are still less then ideal, though (i'll post something on the forum),
*
* Revision 1.2  2004/05/14 12:19:24  tim
* fixed bug
*
* Revision 1.1  2004/05/04 13:40:19  tim
* added GT2004ImageProcessor
*
*/
