/**
* @file GT2004BeaconDetector.cpp
* 
* Implementation of class GT2004BeaconDetector.
*
* @author <a href="mailto:timlaue@tzi.de">Tim Laue</a>
*/ 


#include "Representations/Perception/Image.h"
#ifdef CT32K_LAYOUT
#include "Representations/Perception/ColorTable32K.h"
#else
#include "Representations/Perception/ColorTable64.h"
#endif
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Tools/FieldDimensions.h"
#include "Tools/RingBuffer.h"
#include "Tools/Math/Vector2.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ColorCorrector.h"
#include "GT2004ImageProcessorTools.h"
#include "GT2004BeaconDetector.h"

const int Y(0),
          U(cameraResolutionWidth_ERS7),     /**< Relative offset of U component. */
          V(2 * cameraResolutionWidth_ERS7); /**< Relative offset of V component. */


GT2004BeaconDetector::GT2004BeaconDetector(const Image& image, 
                       const CameraMatrix& cameraMatrix,
                       const ImageInfo& imageInf,
                       const ColorTable& colorTable, 
                       const ColorCorrector& colorCorrector,
                       LandmarksPercept& landmarksPercept):
                       image(image), cameraMatrix(cameraMatrix), 
                       imageInf(imageInf), colorTable(colorTable), 
                       colorCorrector(colorCorrector),
                       landmarksPercept(landmarksPercept),
                       flagSpecialist(colorCorrector),
                       horizontalBaseOffset(1.5), //was 1
                       numOfHorizontalScanLines(12), //was 7
                       horizontalOffsetModifier(0.8), // was 1.5
                       clusteringDistanceTolerance(2), // was 2
                       minPinkRunLength(2) // was 2
{}


void GT2004BeaconDetector::execute()
{
  flagSpecialist.init(image);
  numOfBeaconCandidates = 0;
  double dist(horizontalBaseOffset);
  Geometry::Line scanLine;
  scanLine.direction = imageInf.horizon.direction;
  const Vector2<double> beaconScanOffsetBase(0.0, -10.0); // start a bit below the horizon
  scanLine.base = imageInf.horizon.base-beaconScanOffsetBase;
  for(int i=0; i < numOfHorizontalScanLines; i++)
  {
    scanLine.base += (-imageInf.vertLine.direction*dist);
    dist += horizontalOffsetModifier;
    Vector2<int> startPoint, endPoint;
    if(Geometry::getIntersectionPointsOfLineAndRectangle(
       Vector2<int>(0,0), imageInf.maxImageCoordinates, scanLine, startPoint, endPoint))
    {
      LINE(imageProcessor_general, startPoint.x, startPoint.y, endPoint.x, endPoint.y, 1, Drawings::ps_dash, Drawings::white );
      scanForPink(startPoint, endPoint);
    }
    else
    {
      break;
    }
  }
  if(numOfBeaconCandidates)
  {
    clusterPinkBeaconParts();
  }
  flagSpecialist.getFlagPercept(cameraMatrix, image.cameraInfo, imageInf.horizon, landmarksPercept);
}


void GT2004BeaconDetector::scanForPink(const Vector2<int>& start, 
                                       const Vector2<int>& end)
{
  BresenhamParameters bresParams(start, end);
  int error(bresParams.baseError);
  Vector2<int> pixel(start), lastPixel(start);
  int Y(0),U(0),V(0),lastY(0),lastU(0),lastV(0);
  Run currentRun;
  bool isPinkRun(false);
  bool currentPixelPink(false);
  /*while((bresParams.alongX && (pixel.x != end.x)) || 
        (!bresParams.alongX && (pixel.y != end.y)))*/
  for(int i=0; i<bresParams.numberOfPixels; i++)
  {
    U = image.image[pixel.y][1][pixel.x];
    U = ColorCorrector::correct(pixel.x, pixel.y, 1, U);
    if(U > minPinkUValue)
    {
      Y = image.image[pixel.y][0][pixel.x];
      Y = ColorCorrector::correct(pixel.x, pixel.y, 0, Y);
      V = image.image[pixel.y][2][pixel.x];
      V = ColorCorrector::correct(pixel.x, pixel.y, 2, V);
      currentPixelPink = (COLOR_CLASS(Y,U,V) == pink);
    }
    else
    {
      currentPixelPink = false;
    }
    if(!isPinkRun)
    {
      if(currentPixelPink)
      {
        currentRun.scanLineStart = start;
        currentRun.start = pixel;
        currentRun.length = 1;
        isPinkRun = true;
      }
    }
    else //if(isPinkRun)
    {
      if(currentPixelPink)// || (COLOR_COMPONENT_DIST(U,lastU) < params.maxPinkUContrast))
      {
        currentRun.length++;
        lastY = Y;
        lastU = U;
        lastV = V;
      }
      else
      {
        isPinkRun = false;
        if(currentRun.length >= minPinkRunLength)
        {
          currentRun.end = lastPixel;
          if(!addCandidate(currentRun)) break; //List is full, stop searching
          LINE(imageProcessor_general, currentRun.start.x, currentRun.start.y, 
               currentRun.end.x, currentRun.end.y, 1, Drawings::ps_solid, Drawings::pink);
        }
      }
    }    
    //Compute next position according to Bresenham algorithm
    lastPixel = pixel;
    pixel += bresParams.standardOffset;
    error += bresParams.delta;
    if(error > 0)
    {
      pixel += bresParams.correctionOffset;
      error += bresParams.resetError;
    }
  }
  //Finish the last Run
  if(isPinkRun)
  {
    currentRun.end = pixel;
    addCandidate(currentRun);
    LINE(imageProcessor_general, currentRun.start.x, currentRun.start.y, 
               currentRun.end.x, currentRun.end.y, 1, Drawings::ps_solid, Drawings::pink);
  }
}


bool GT2004BeaconDetector::addCandidate(const Run& pinkRun)
{
  bool returnValue(numOfBeaconCandidates<MAX_NUMBER_OF_PINK_RUNS);
  if(returnValue)
  {
    if((numOfBeaconCandidates>0) &&
       (pinkRun.scanLineStart == beaconCandidates[numOfBeaconCandidates-1].scanLineStart) &&
       ((beaconCandidates[numOfBeaconCandidates-1].end - pinkRun.start).abs()<4))
    {
      beaconCandidates[numOfBeaconCandidates-1].end = pinkRun.end;
    }
    else
    {
      beaconCandidates[numOfBeaconCandidates++] = pinkRun;
    }
  }
  return returnValue;
}


void GT2004BeaconDetector::clusterPinkBeaconParts()
{
  //Transform pink runs
  double rotation(imageInf.horizon.direction.angle());
  double ca(cos(-rotation));
  double sa(sin(-rotation));
  Matrix2x2<double> rotMat(Vector2<double>(ca,sa),Vector2<double>(-sa,ca));
  for(int i=0; i<numOfBeaconCandidates; i++)
  {
    transformedCandidates[i].transform(beaconCandidates[i], rotMat, i);
  }
  //Sort by starting position from left to right
  for(int j=0; j<numOfBeaconCandidates; j++)
  {
    int leftest(j);
    for(int k=j; k<numOfBeaconCandidates; k++)
    {
      if(transformedCandidates[k].start.x<transformedCandidates[leftest].start.x)
      {
        leftest = k;
      }
    }
    if(leftest != j)
    {
      TransformedRun help(transformedCandidates[j]);
      transformedCandidates[j] = transformedCandidates[leftest];
      transformedCandidates[leftest] = help;
    }
  }
  //Find overlapping runs
  int beginOfBeacon(0), endOfBeacon(0);
  double xRight(transformedCandidates[0].end.x);
  for(int l=1; l<numOfBeaconCandidates; l++)
  {
    if(transformedCandidates[l].start.x < xRight  + (double)clusteringDistanceTolerance)
    {
      if(transformedCandidates[l].end.x > xRight)
      {
        xRight = transformedCandidates[l].end.x;
        endOfBeacon = l;
      }
    }
    else
    {
      Vector2<int> leftPosition(beaconCandidates[transformedCandidates[beginOfBeacon].numOfRun].start);
      Vector2<int> rightPosition(beaconCandidates[transformedCandidates[endOfBeacon].numOfRun].end);
      Vector2<int> centerPosition(leftPosition + (rightPosition-leftPosition)/2);
      LINE(imageProcessor_general, leftPosition.x, leftPosition.y, 
           rightPosition.x, rightPosition.y, 1, Drawings::ps_solid, Drawings::red);
      if(!analyzeBeacon(centerPosition, (rightPosition-leftPosition).abs()))
        if(!analyzeBeacon(leftPosition, (rightPosition-leftPosition).abs()))
          analyzeBeacon(rightPosition, (rightPosition-leftPosition).abs());
      beginOfBeacon = l;
      endOfBeacon = l;
      xRight = transformedCandidates[l].end.x;
    }
  }
  Vector2<int> leftPosition(beaconCandidates[transformedCandidates[beginOfBeacon].numOfRun].start);
  Vector2<int> rightPosition(beaconCandidates[transformedCandidates[endOfBeacon].numOfRun].end);
  Vector2<int> centerPosition(leftPosition + (rightPosition-leftPosition)/2);
  LINE(imageProcessor_general, leftPosition.x, leftPosition.y, 
       rightPosition.x, rightPosition.y, 1, Drawings::ps_solid, Drawings::red);
  if(!analyzeBeacon(centerPosition, (rightPosition-leftPosition).abs()))
    if(!analyzeBeacon(leftPosition, (rightPosition-leftPosition).abs()))
      analyzeBeacon(rightPosition, (rightPosition-leftPosition).abs());
}


bool GT2004BeaconDetector::analyzeBeacon(const Vector2<int>& position, const double pinkRunWidth)
{
  Vector2<int> beaconCenter(position);
  Vector2<int> start, end;
  Geometry::Line vertLine;
  vertLine.direction = imageInf.vertLine.direction;
  vertLine.base.x = (double)beaconCenter.x;
  vertLine.base.y = (double)beaconCenter.y;
  Geometry::getIntersectionPointsOfLineAndRectangle(Vector2<int>(0,0), 
              imageInf.maxImageCoordinates, vertLine, start, end);
  Vector2<int> pos, pos2, pinkBottomEdge, pinkTopEdge, colorBottomEdge;
  colorClass bottomColor(white),topColor(white), baseColor(white);
  Flag::FlagType beaconType(Flag::pinkAboveYellow);
  // Scan to bottom of pink beacon part
  if(scanForBeaconPart(beaconCenter, end, pos, pinkBottomEdge, bottomColor))
  {
    if(bottomColor == white || bottomColor == gray)
    {
      // Scan to top of pink beacon part
      if(scanForBeaconPart(beaconCenter, start, pos, pinkTopEdge, topColor))
      {
        if(topColor == skyblue)
        {
          beaconType = Flag::skyblueAbovePink;
        }
        else if(topColor == yellow)
        {
          beaconType = Flag::yellowAbovePink;
        }
        else
        {
          return false; // unknown color above pink
        }
      }
      else
      {
        return false; // pink part is at the upper border of the image
      }
    }
    // Scan to bottom of color part
    else if((bottomColor == skyblue) || (bottomColor == yellow))
    {
      if(scanForBeaconPart(pos, end, pos2, colorBottomEdge, baseColor))
      {
        if(baseColor == white || baseColor == gray)
        {
          if(bottomColor == skyblue)
          {
            beaconType = Flag::pinkAboveSkyblue;
          }
          else
          {
            beaconType = Flag::pinkAboveYellow;
          }
        }
        else
        {
          return false; // unknown color below the colored part
        }
      }
      else
      {
        return false; // colored part is at the bottom of the image
      }
    }
    else
    {
      return false; // unknown color below pink
    }
  }
  else
  {
    return false; // pink is at the lower border of the image
  }
  //~ DOT(imageProcessor_ball4, pos.x, pos.y, Drawings::white, Drawings::black);
  //~ DOT(imageProcessor_ball4, pos2.x, pos2.y, Drawings::black, Drawings::gray);
  // Is the upper pink edge still missing?
  if(beaconType == Flag::pinkAboveYellow || beaconType == Flag::pinkAboveSkyblue)
  {
    if(scanForBeaconPart(beaconCenter, start, pos, pinkTopEdge, topColor))
    {
     // addToLandmarksPercept(beaconType, pinkTopEdge, colorBottomEdge, leftBorder, rightBorder);
      // flagSpecialist.searchFlags(image, colorTable, cameraMatrix, bottomColor, true, 
                                 // imageInf.horizon, pinkBottomEdge.x, pinkBottomEdge.y);
      DOT(imageProcessor_ball4, pinkBottomEdge.x, pinkBottomEdge.y, Drawings::black, Drawings::green);
      DOT(imageProcessor_ball4, pinkTopEdge.x, pinkTopEdge.y, Drawings::white, Drawings::black);
      DOT(imageProcessor_ball4, colorBottomEdge.x, colorBottomEdge.y, Drawings::black, Drawings::white);  
      Vector2<int> topPosition(pinkBottomEdge);
      Vector2<int> toColor(pinkBottomEdge - colorBottomEdge);
      Vector2<int> toPink(pinkTopEdge - pinkBottomEdge);
      double distanceToColor = toColor.abs();
      double distanceToPink = toPink.abs();
      double similarityColor, similarityPink;
      if (distanceToColor > pinkRunWidth)
        similarityColor = (pinkRunWidth/distanceToColor);
      else
        similarityColor = (distanceToColor/pinkRunWidth);
      if (distanceToPink > pinkRunWidth)
        similarityPink = (pinkRunWidth/distanceToPink);
      else
        similarityPink = (distanceToPink/pinkRunWidth);
      if (similarityColor > similarityPink)
        topPosition += toColor;
      else
        topPosition += toPink;
      DOT(imageProcessor_ball4, topPosition.x, topPosition.y, Drawings::black, Drawings::gray);
      DOT(imageProcessor_ball4, pinkBottomEdge.x, topPosition.y+(pinkBottomEdge.y-topPosition.y)/2, Drawings::black, Drawings::blue);
      flagSpecialist.searchFlags(image, colorTable, cameraMatrix, bottomColor, true, 
                                 imageInf.horizon, pinkBottomEdge.x, topPosition.y+(pinkBottomEdge.y-topPosition.y)/2);
    }
    else
    {
      // compute a theoretical position of the pink top edge:
      Vector2<int> topPosition(pinkBottomEdge);
      Vector2<int> relVec(pinkBottomEdge - colorBottomEdge);
      topPosition += relVec;
     // addToLandmarksPercept(beaconType, topPosition, colorBottomEdge, leftBorder, rightBorder);
      // flagSpecialist.searchFlags(image, colorTable, cameraMatrix, bottomColor, true, 
                                 // imageInf.horizon, pinkBottomEdge.x, pinkBottomEdge.y);
      DOT(imageProcessor_ball4, pinkBottomEdge.x, topPosition.y+(pinkBottomEdge.y-topPosition.y)/2, Drawings::black, Drawings::blue);
      DOT(imageProcessor_ball4, pinkBottomEdge.x, pinkBottomEdge.y, Drawings::black, Drawings::red);
      DOT(imageProcessor_ball4, topPosition.x, topPosition.y, Drawings::white, Drawings::black);
      flagSpecialist.searchFlags(image, colorTable, cameraMatrix, bottomColor, true, 
                                 imageInf.horizon, pinkBottomEdge.x, topPosition.y+(pinkBottomEdge.y-topPosition.y)/2);
    }
  }
  else
  { // pink is bottom
    // compute a theoretical position of the color top edge:
    Vector2<int> topPosition(pinkTopEdge); // ??????
    Vector2<int> relVec(pinkTopEdge - pinkBottomEdge);// ??????
    topPosition += relVec;// ??????
    //addToLandmarksPercept(beaconType, topPosition, pinkBottomEdge, leftBorder, rightBorder);
    //~ flagSpecialist.searchFlags(image, colorTable, cameraMatrix, topColor, false, 
                               //~ imageInf.horizon, pinkTopEdge.x, pinkTopEdge.y);
    DOT(imageProcessor_ball4, pinkTopEdge.x, pinkTopEdge.y+(pinkBottomEdge.y-pinkTopEdge.y)/2, Drawings::black, Drawings::blue);
    DOT(imageProcessor_ball4, pinkTopEdge.x, pinkTopEdge.y, Drawings::black, Drawings::yellow);
    DOT(imageProcessor_ball4, pinkBottomEdge.x, pinkBottomEdge.y, Drawings::white, Drawings::black);
    
    flagSpecialist.searchFlags(image, colorTable, cameraMatrix, topColor, false, 
                               imageInf.horizon, pinkTopEdge.x, pinkTopEdge.y+(pinkBottomEdge.y-pinkTopEdge.y)/2);
  }
  DEBUG_DRAWING_FINISHED(imageProcessor_ball4);
  return true;
}


bool GT2004BeaconDetector::scanForBeaconPart(const Vector2<int>& start, const Vector2<int>& end,
                                             Vector2<int>& position, Vector2<int>& edge, colorClass& color)
{
  const int yThreshold(10);
  const int vThreshold(5);
  Vector2<int> actual(start),
               diff(end - start),
               step(sgn(diff.x),sgn(diff.y)),
               size(abs(diff.x),abs(diff.y));
  bool isVertical(abs(diff.y) > abs(diff.x));
  int count,
      sum;

  // init Bresenham
  if(isVertical)
  {
    count = size.y;
    sum = size.y / 2;
  }
  else
  {
    count = size.x;
    sum = size.x / 2;
  }

  if(count > 7)
  {
    int i;
    RingBuffer<const unsigned char*,7> pixel;
    //the image is scanned vertically
    for(i = 0; i < 6; ++i) // fill buffer
    {
      pixel.add(&image.image[actual.y][0][actual.x]);
      
      // Bresenham
      if(isVertical)
      {
        actual.y += step.y;
        sum += size.x;
        if(sum >= size.y)
        {
          sum -= size.y;
          actual.x += step.x;
        }
      }
      else        
      {
        actual.x += step.x;
        sum += size.y;
        if(sum >= size.x)
        {
          sum -= size.x;
          actual.y += step.y;
        }
      }
    }
    // Scan along line
    for(; i <= count; ++i)
    {
      pixel.add(&image.image[actual.y][0][actual.x]);
      // Bresenham
      if(isVertical)
      {
        actual.y += step.y;
        sum += size.x;
        if(sum >= size.y)
        {
          sum -= size.y;
          actual.x += step.x;
        }
      }
      else        
      {
        actual.x += step.x;
        sum += size.y;
        if(sum >= size.x)
        {
          sum -= size.x;
          actual.y += step.y;
        }
      }

      // line point state machine
      const unsigned char* p3 = pixel[3],
                         * p4 = pixel[4];
      int p3Y(ColorCorrector::correct(actual.x,actual.y,0,p3[Y]));
      int p4Y(ColorCorrector::correct(actual.x,actual.y,0,p4[Y]));
      int p3V(ColorCorrector::correct(actual.x,actual.y,2,p3[V]));
      int p4V(ColorCorrector::correct(actual.x,actual.y,2,p4[V]));
      if(p3Y < p4Y - yThreshold || p3V < p4V - vThreshold || p3Y > p4Y + yThreshold)
      {
        const unsigned char* p0 = pixel[0];
        unsigned char p0Y(ColorCorrector::correct(actual.x,actual.y,0,p0[Y]));
        unsigned char p0U(ColorCorrector::correct(actual.x,actual.y,1,p0[U]));
        unsigned char p0V(ColorCorrector::correct(actual.x,actual.y,2,p0[V]));
        const unsigned char* p6 = pixel[6];
        unsigned char p6Y(ColorCorrector::correct(actual.x,actual.y,0,p6[Y]));
        unsigned char p6U(ColorCorrector::correct(actual.x,actual.y,1,p6[U]));
        unsigned char p6V(ColorCorrector::correct(actual.x,actual.y,2,p6[V]));
        colorClass c0(COLOR_CLASS(p0Y,p0U,p0V)), c6(COLOR_CLASS(p6Y,p6U,p6V));
        if(c0 != c6)
        {
          color = c0;
          int diff(p0 - &image.image[0][0][0]);
          position = Vector2<int>(diff % cameraResolutionWidth_ERS7, diff / (cameraResolutionWidth_ERS7 * 6));
          int edgeDiff(p4 - &image.image[0][0][0]);
          edge = Vector2<int>(edgeDiff % cameraResolutionWidth_ERS7, edgeDiff / (cameraResolutionWidth_ERS7 * 6));
          return true;
        }
      }
    }
  }
  return false;
}


void GT2004BeaconDetector::analyzeColorTable()
{
  Vector3<int> nearPoint,farPoint;
#ifdef CT32K_LAYOUT
  ((ColorTable32K&) colorTable).getBoxAroundColorClass(pink,nearPoint,farPoint);
#else
  ((ColorTable64&) colorTable).getBoxAroundColorClass(pink,nearPoint,farPoint);
#endif
  minPinkUValue = nearPoint.y;
}


/*
 * $Log: GT2004BeaconDetector.cpp,v $
 * Revision 1.5  2004/05/18 18:39:25  nistico
 * BeaconDetector improved, the landmarks are recognized much more often,
 * and the width is estimated correctly most of the time.
 * Some things are still less then ideal, though (i'll post something on the forum),
 *
 * Revision 1.4  2004/05/15 15:00:22  nistico
 * Beacon scanlines begin slightly below the horizon.
 * Scanline spacing changed.
 *
 * Revision 1.3  2004/05/14 12:19:24  tim
 * fixed bug
 *
 * Revision 1.2  2004/05/06 16:03:56  nistico
 * Supports ColorTable32K through CT32K_LAYOUT switch located into
 * GT2004ImageProcessorTools.h
 *
 * Revision 1.1  2004/05/04 13:40:19  tim
 * added GT2004ImageProcessor
 *
 */
