/**
* @file GT2004BallSpecialist.h
* 
* This file contains a class for Image Processing.
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
* @author Max Risler
* @author Ronnie Brunn
* @author Michael Kunz
*/

#ifndef __GT2004BallSpecialist_h_
#define __GT2004BallSpecialist_h_

#include "Tools/Math/Geometry.h"
#include "Tools/Debugging/DebugImages.h"
#include "Platform/GTAssert.h"

#include "Representations/Perception/ColorTable.h"
#include "Representations/Perception/Image.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/BallPercept.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ColorCorrector.h"
#include "Tools/Math/Common.h"

/**
The BallSpecialist finds a single ball in an image.
*/
class GT2004BallSpecialist  
{
public:
  GT2004BallSpecialist(const ColorCorrector& colorCorrector);
  
  /** 
  * Searches for the ball in the image, starting from the secified point. 
  */
  void searchBall
  (
    const Image& image,
    const ColorTable& colorTable,
    const CameraMatrix& cameraMatrix,
    int x, int y,
    BallPercept& ballPercept
  );
  
  /**
  * Returns the Similarity of the given color to orange.
  */
  unsigned char getSimilarityToOrange (int y, int u, int v)
  {
      double r = y + 1.140 * (u - 128);
      double g = y - 0.394 * (v - 128) - 0.581 * (u - 128);
      double b = y + 2.028 * (v - 128);
      if(r < 0) r = 0; if(r > 255) r = 255;
      if(g < 0) g = 0; if(g > 255) g = 255;
      if(b < 0) b = 0; if(b > 255) b = 255;
	  return (unsigned char) min(max((r-b-max(0,r-2*g)-max(2*g-r,0)),0),255);
  }

private:
  /**
  * The class represents a ball point.
  */
  class BallPoint : public Vector2<int>
  {
  public:
    BallPoint() : atBorder(false), greenIsClose(true), yellowIsClose(false), hardEdge(true){};
    bool greenIsClose;
    bool yellowIsClose;
    bool hardEdge;
    bool atBorder;

    BallPoint& operator=(const Vector2<int>& other)
    {
      x = other.x;
      y = other.y;

      return *this;
    }
  };

  class BallPointList
  {
  public:
    BallPointList() : number(0) {};

    enum {maxNumberOfPoints = 400}; /**< The maximum number of ball points. */
    BallPoint ballPoints[maxNumberOfPoints]; /**< The ball points. */
    int number; /**< The actual number of ball points. */

    void add(const BallPoint& ballPoint)
    {
      ASSERT(number < maxNumberOfPoints);
      ballPoints[number++] = ballPoint;
      DOT(imageProcessor_ball2, ballPoint.x / 2, ballPoint.y / 2,
        (ballPoint.hardEdge) ? Drawings::blue : Drawings::orange, 
        (ballPoint.atBorder) ? Drawings::black :
        (ballPoint.greenIsClose) ? Drawings::green :
        (ballPoint.yellowIsClose) ? Drawings::yellow :
        Drawings::white
      );
    }

    BallPoint& operator[] (int i) { return ballPoints[i]; }
    const BallPoint& operator[] (int i) const { return ballPoints[i]; }
  };

  /** Scan for the ball starting at a given trigger point */
  void scanForBallPoints
  (
   const Image& image,
   const CameraInfo& bwCameraInfo,
   const ColorTable& colorTable,
   int x, int y,
   BallPointList& ballPoints,
   int& countAmbiguous,
   int& countOrange,
   int& maxOrangePerLine,
   int& countPixel
  );

  /** Finds the end of the ball */
  bool findEndOfBall(
    const Image& image,
    const CameraInfo& bwCameraInfo,
    const ColorTable& colorTable,
    const BallPoint& start,
    const Vector2<int>& step,
    BallPoint& destination,
    int& countAmbiguous,
    int& countOrange,
    int& maxOrangePerLine,
    int& countPixel
    );

  /**
  * The function tries to calculate the ball percept by using the Levenberg-Marquardt
  * algorithm. The function fails if less than 3 points are available.
  * @return true if ball percept was created
  */
  bool createBallPerceptLevenbergMarquardt
  (
    const BallPointList& ballPoints,
    Vector2<int>& center,
    double& radius
  );

  /*
  * Check if a list of points is inside a given circle
  */
  bool checkIfPointsAreInsideBall(const BallPointList& ballPoints, Vector2<int>& center, double radius);

  /**
  * The function checks whether a ball percept is plausible and will add it if so.
  * @param centerX The x-coordinate of the center of the ball.
  * @param centerY The y-coordinate of the center of the ball.
  * @param radius The radius of the ball.
  */
  void addBallPercept
  (
    const Image& image,
    const CameraInfo& bwCameraInfo,
    const CameraMatrix& cameraMatrix,
    const Vector2<int>& center,
    double radius,
    BallPercept& ballPercept
  );  

  const ColorCorrector& colorCorrector;
};

#endif// __BallSpecialist_h_

/*
* $Log: GT2004BallSpecialist.h,v $
* Revision 1.2  2004/05/14 16:37:42  nistico
* Ghost balls on landmark improvements:
* -introduced ambiguous (pink, red, yellow) color check in ball recognition
* -some parameter tuning
*
* Revision 1.1  2004/05/04 13:40:19  tim
* added GT2004ImageProcessor
*
*/
