/**
* @file GT2004BallSpecialist.cpp
* This file contains a class for Image Processing.
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
* @author Max Risler
* @author Ronnie Brunn
* @author Michael Kunz
*/

#include "GT2004BallSpecialist.h"

#include "Tools/FieldDimensions.h"
#include "Tools/Math/Common.h"
#include "Tools/Math/MVTools.h"
#include "Tools/Math/Matrix_nxn.h"
#include "Tools/Math/Vector_n.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "GT2004ImageProcessorTools.h"

GT2004BallSpecialist::GT2004BallSpecialist
(
 const ColorCorrector& colorCorrector
)
:
 colorCorrector(colorCorrector)
{
}

void GT2004BallSpecialist::searchBall
(
 const Image& image, 
 const ColorTable& colorTable,
 const CameraMatrix& cameraMatrix,
 int x, int y,
 BallPercept& ballPercept
)
{
  BallPointList ballPoints;
  Vector2<int> center;
  double radius;
  int countOrange = 0;
  int countAmbiguous = 0; 
  int maxOrangePerLine = 0;
  int countPixel = 0;
  CameraInfo bwCameraInfo = image.cameraInfo;
  bwCameraInfo.resolutionHeight*=2;
  bwCameraInfo.resolutionWidth*=2;
  bwCameraInfo.opticalCenter.x*=2;
  bwCameraInfo.opticalCenter.y*=2;
  bwCameraInfo.focalLength*=2;
  bwCameraInfo.focalLengthInv/=2;

  scanForBallPoints(image, bwCameraInfo, colorTable, x, y, ballPoints, countAmbiguous, countOrange, maxOrangePerLine, countPixel);

  //ignore ball if not enough points are orange
  //OUTPUT (idText, text, "GT2004BallSpecialist: " << countOrange * 100 / countPixel << "% orange");
  int i;
  int numberOfHardEdgePoints = 0;
  int numberOfPointsInYellow = 0;

  for(i = 0; i < ballPoints.number; i++)
  {
    if (ballPoints[i].yellowIsClose && !ballPoints[i].atBorder) numberOfPointsInYellow++;
    if (ballPoints[i].hardEdge) numberOfHardEdgePoints++;
  }
  //OUTPUT (idText, text, "NOPoints: " << ballPoints.number << " GhostYellow: " << numberOfPointsInYellow << " HardEdge: " << numberOfHardEdgePoints);
  
  if ((countOrange > countAmbiguous) && 
    (countOrange > countPixel / 6) && 
    (numberOfHardEdgePoints * 5 >= ballPoints.number * 3) && 
    (numberOfPointsInYellow * 4 <= ballPoints.number * 3) 
  )
  {
    //try only points near green first
    BallPointList testPoints;
    for(i = 0; i < ballPoints.number; i++)
      if (ballPoints[i].greenIsClose) 
        testPoints.add(ballPoints[i]);
    if (
      testPoints.number >= ballPoints.number / 2 &&
      createBallPerceptLevenbergMarquardt(testPoints, center, radius) &&
      checkIfPointsAreInsideBall(ballPoints, center, radius))
    {
      addBallPercept(image, bwCameraInfo, cameraMatrix, center, radius, ballPercept);
    }
    else
    {
      //now all points if not at border
      testPoints.number = 0;
      for(i = 0; i < ballPoints.number; i++)
        if (!ballPoints[i].atBorder) 
          testPoints.add(ballPoints[i]);
      if (
        createBallPerceptLevenbergMarquardt(testPoints, center, radius) &&
        checkIfPointsAreInsideBall(ballPoints, center, radius))
      {
        addBallPercept(image, bwCameraInfo, cameraMatrix, center, radius, ballPercept);
      }
      else
      {
        //take all points if nothing else works
        if (createBallPerceptLevenbergMarquardt(ballPoints, center, radius))
        {
          addBallPercept(image, bwCameraInfo, cameraMatrix, center, radius, ballPercept);
        }
      }
    }
  }

  DEBUG_DRAWING_FINISHED(imageProcessor_ball1);
  DEBUG_DRAWING_FINISHED(imageProcessor_ball2);
}

void GT2004BallSpecialist::scanForBallPoints
(
 const Image& image,
 const CameraInfo& bwCameraInfo,
 const ColorTable& colorTable,
 int x, int y,
 BallPointList& ballPoints,
 int& countAmbiguous, 
 int& countOrange,
 int& maxOrangePerLine,
 int& countPixel
)
{
  // search for ball variables
  BallPoint north;
  BallPoint east;
  BallPoint south;
  BallPoint west;

  BallPoint start;
  Vector2<int>step;
  BallPoint destination;
  
//  int xStep;
//  int yStep;

  start.x = x * 2; start.y = y * 2;
  BallPoint start2;
  
  DOT(imageProcessor_ball2, x, y, Drawings::black, Drawings::white);

  //find north ///////////////////////////////////////////
  step.x = 0; step.y = -1;
  findEndOfBall(image, bwCameraInfo, colorTable, start, step, north, countAmbiguous, countOrange, maxOrangePerLine, countPixel);
  if(north.atBorder)
  {
    start2 = north - step;
    //find east
    step.x = 1; step.y = 0;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countAmbiguous, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
    //find west
    step.x = -1; step.y = 0;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countAmbiguous, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
  }
  else
  {
    ballPoints.add(north);
  }

  //find east ///////////////////////////////////////////
  step.x = 1; step.y = 0;
  findEndOfBall(image, bwCameraInfo, colorTable, start, step, east, countAmbiguous, countOrange, maxOrangePerLine, countPixel);
  if(east.atBorder)
  {
    start2 = east - step;
    //find north
    step.x = 0; step.y = -1;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countAmbiguous, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
    //find south
    step.x = 0; step.y = 1;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countAmbiguous, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
  }
  else
  {
    ballPoints.add(east);
  }

  //find south ///////////////////////////////////////////
  step.x = 0; step.y = 1;
  findEndOfBall(image, bwCameraInfo, colorTable, start, step, south, countAmbiguous, countOrange, maxOrangePerLine, countPixel);
  if(south.atBorder)
  {
    start2 = south - step;
    //find east
    step.x = 1; step.y = 0;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countAmbiguous, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
    //find west
    step.x = -1; step.y = 0;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countAmbiguous, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
  }
  else
  {
    ballPoints.add(south);
  }

  //find west ///////////////////////////////////////////
  step.x = -1; step.y = 0;
  findEndOfBall(image, bwCameraInfo, colorTable, start, step, west, countAmbiguous, countOrange, maxOrangePerLine, countPixel);
  if(west.atBorder)
  {
    start2 = west - step;
    //find north
    step.x = 0; step.y = -1;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countAmbiguous, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
    //find south
    step.x = 0; step.y = 1;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countAmbiguous, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
  }
  else
  {
    ballPoints.add(west);
  }

  //
  if( (south.y - north.y) > (east.x - west.x) )
  {
    if ((north.y + south.y) / 2 != start.y)
    {
      start.y = (north.y + south.y) / 2;
      //find east
      step.x = 1; step.y = 0;
      findEndOfBall(image, bwCameraInfo, colorTable, start, step, east, countAmbiguous, countOrange, maxOrangePerLine, countPixel);
      if(!east.atBorder)
      {
        ballPoints.add(east);
      }
      //find west
      step.x = -1; step.y = 0;
      findEndOfBall(image, bwCameraInfo, colorTable, start, step, west, countAmbiguous, countOrange, maxOrangePerLine, countPixel);
      if (!west.atBorder)
      {
        ballPoints.add(west);
      }
      //////////
      start.x = (west.x + east.x) / 2;
    }
  }
  else
  {
    if ((west.x + east.x) / 2 != start.x)
    {
      start.x = (west.x + east.x) / 2;
      //find north
      step.x = 0; step.y = -1;
      findEndOfBall(image, bwCameraInfo, colorTable, start, step, north, countAmbiguous, countOrange, maxOrangePerLine, countPixel);
      if(!north.atBorder)
      {
        ballPoints.add(north);
      }
      //find south
      step.x = 0; step.y = 1;
      findEndOfBall(image, bwCameraInfo, colorTable, start, step, south, countAmbiguous, countOrange, maxOrangePerLine, countPixel);
      if(!south.atBorder)
      {
        ballPoints.add(south);
      }
      //////////
      start.y = (north.y + south.y) / 2;
    }
  }
  //

  // find in diagonal
  
//  for(xStep = -1; xStep <= 1; xStep += 2)
  for (step.x = -1; step.x <= 1; step.x += 2)
  {
    //for(yStep = -1; yStep <= 1; yStep += 2)
	for (step.y = -1; step.y <= 1; step.y += 2)
    {
      //step.x = xStep; step.y = yStep;
      findEndOfBall(image, bwCameraInfo, colorTable, start, step, destination, countAmbiguous, countOrange, maxOrangePerLine, countPixel);
      if (!destination.atBorder)
      {
        ballPoints.add(destination);
      }
    } //for(int yStep ...
  } //for(int xStep ...
}

bool GT2004BallSpecialist::findEndOfBall
(
 const Image& image,
 const CameraInfo& bwCameraInfo,
 const ColorTable& colorTable,
 const BallPoint& start,
 const Vector2<int>& step,
 BallPoint& destination,
 int& countAmbiguous,
 int& countOrange,
 int& maxOrangePerLine,
 int& countPixel
)
{
  /*
  stopColors are colors indicating the end of the ball:
    green, yellow, skyblue
  more than 3 pixels of a stopColor indicate the end of the ball
  */
  int stopColorCounter = 0;
  int currentOrange = 0;
  int currentAmbiguous = 0;
  //~ int ambiguousThisFar = 0;
  int len = 0;
  int stopLen = 0;
  bool isAmbiguous = false;
  
  colorClass currentColorClass;

  Vector2<int> firstStopColor = start;
  unsigned char lastGoodColorOrangeSim = 0;
  unsigned char currentOrangeSim = 0;
  unsigned char prevOrangeSim = 0;
  Vector2<int> lastPoint = start;
  destination = start;
  destination.greenIsClose = false;
  destination.yellowIsClose = false;
  destination.atBorder = false;
  destination.hardEdge = true;

  bool isOrange = false;
 
  bool goOn = true;
  while(goOn)
  {
    lastPoint = destination;
    destination += step;
    /////////////
    if(destination.x < 0 || destination.x >= bwCameraInfo.resolutionWidth||
      destination.y < 0 || destination.y >= bwCameraInfo.resolutionHeight)
    {
      destination.atBorder = true;
      countPixel += len;
      goOn = false;
    }
    else
    {
      currentColorClass = CORRECTED_COLOR_CLASS(
        destination.x / 2,destination.y / 2,
        image.getHighResY(destination.x,destination.y),
        image.image[destination.y / 2][1][destination.x / 2],
        image.image[destination.y / 2][2][destination.x / 2]);
      
      // counting all orange pixels on the scanned horz./vert./diag. lines
      len++;
      if ( currentColorClass == orange )
      {
        currentOrange++;
      }
      isAmbiguous = (currentColorClass == pink ||
        currentColorClass == yellow ||
        currentColorClass == red
      );
      
      if (isAmbiguous)
      {
        currentAmbiguous++;
      }

      LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::green);
      
      prevOrangeSim = currentOrangeSim;
      currentOrangeSim = getSimilarityToOrange(
        colorCorrector.correct(destination.x / 2,destination.y / 2, 0, image.getHighResY(destination.x,destination.y)),
        colorCorrector.correct(destination.x / 2,destination.y / 2, 1, image.image[destination.y / 2][1][destination.x / 2]),
        colorCorrector.correct(destination.x / 2,destination.y / 2, 2, image.image[destination.y / 2][2][destination.x / 2])
        );
      isOrange = (currentOrangeSim > 30) || (currentColorClass == orange);
      
      if(currentColorClass == green || 
        currentColorClass == yellow ||
        currentColorClass == skyblue ||
        currentColorClass == red ||
        currentColorClass == blue ||
        currentColorClass == pink || //added
        !isOrange
        )
      {
        if (stopColorCounter == 0)
        {
          firstStopColor = destination;
          lastGoodColorOrangeSim = prevOrangeSim;
          stopLen = len;
          //~ ambiguousThisFar = currentAmbiguous;
        }
        if (currentColorClass == green) destination.greenIsClose = true;
        if (currentColorClass == yellow) destination.yellowIsClose = true;
        stopColorCounter++;

        if (isOrange)
        {
          LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::orange);
        }
        else
        {
          LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::red);
        }
        
        if(stopColorCounter > 8) 
        {
          destination = firstStopColor;
          countPixel += stopLen;
          goOn = false;
        }
      }
      else
      {
        destination.greenIsClose = false;
        destination.yellowIsClose = false;
        stopColorCounter = 0;
      }
      
    } // else destination in range
  } //  while(goOn)

  destination -= step;

  // compute sum of all orange pixels and max-pixels-per-line
  countOrange += currentOrange;
  countAmbiguous += currentAmbiguous;//ambiguousThisFar;
  maxOrangePerLine = max ( maxOrangePerLine, currentOrange);
  if (destination.greenIsClose) destination.yellowIsClose = false;

/*  if (!destination.atBorder)
  {
    OUTPUT(idText, text, "Orange in: " << lastGoodColorOrangeSim << " OrangeOut: " << currentOrangeSim);
  }*/
  if ( lastGoodColorOrangeSim > 2 * currentOrangeSim ) destination.hardEdge = true;
    else destination.hardEdge = false;

  if (!destination.atBorder)
  {
    if (destination.greenIsClose)
    {
      LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::blue);
    }
    else
      if (destination.yellowIsClose)
      {
        LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::skyblue);
      }
      else
      {
        LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::pink);
      }
  }
  return true;
}



bool GT2004BallSpecialist::createBallPerceptLevenbergMarquardt
(
 const BallPointList& ballPoints,
 Vector2<int>& center,
 double& radius
)
{
	if (ballPoints.number < 3)
		return false;

	double Mx = 0.0;
  double My = 0.0;
  double Mxx = 0.0;
  double Myy = 0.0;
  double Mxy = 0.0;
  double Mz = 0.0;
  double Mxz = 0.0;
  double Myz = 0.0;
  
	for (int i = 0; i < ballPoints.number; ++i)
	{
    double x = ballPoints[i].x;
    double y = ballPoints[i].y;
    double xx = x*x;
    double yy = y*y;
    double z = xx + yy;

    Mx += x;
    My += y;
    Mxx += xx;
    Myy += yy;
    Mxy += x*y;
    Mz += z;
    Mxz += x*z;
    Myz += y*z;
	}

	try
  {
    Matrix_nxn<double, 3> M;
	  double Matrix[9] = 
    {
      Mxx, Mxy, Mx,
      Mxy, Myy, My,
      Mx, My, ballPoints.number
    };
	  M = Matrix;

	  Vector_n<double, 3> v;
	
	  v[0] = -Mxz;
	  v[1] = -Myz;
	  v[2] = -Mz;

	  Vector_n<double, 3> BCD;
    BCD = M.solve(v);

	  center.x = (int)(-BCD[0] / 2.0);
	  center.y = (int)(-BCD[1] / 2.0);

	  double tmpWurzel = BCD[0]*BCD[0]/4.0 + BCD[1]*BCD[1]/4.0 - BCD[2];
	
	  if (tmpWurzel < 0.0)
      return false;

  	radius = sqrt(tmpWurzel);
  }
  catch (MVException)
  {
    return false;
  }
  catch (...)
  {
    OUTPUT(idText, text, "Unknown exception in GT2004BallSpecialist::createBallPerceptsFromXPoints");
    return false;
  }

	return true;
}

bool GT2004BallSpecialist::checkIfPointsAreInsideBall(const BallPointList& ballPoints, Vector2<int>& center, double radius)
{
  for(int i = 0; i < ballPoints.number; i++)
    if (Geometry::distance(center, ballPoints[i]) > radius * 1.1)
      return false;

  return true;
}

void GT2004BallSpecialist::addBallPercept
  (
    const Image& image,
    const CameraInfo& bwCameraInfo,
    const CameraMatrix& cameraMatrix,
    const Vector2<int>& center,
    double radius,
    BallPercept& ballPercept
  )
{
  // test if ball is below horizon
  //~ double xFactor = tan(bwCameraInfo.openingAngleWidth / 2) / (bwCameraInfo.resolutionWidth / 2),
    //~ yFactor = tan(bwCameraInfo.openingAngleHeight / 2) / (bwCameraInfo.resolutionHeight / 2);
  double factor = bwCameraInfo.focalLengthInv;
  Vector3<double> 
    vectorToCenter(1, (bwCameraInfo.opticalCenter.x - center.x) * factor, (bwCameraInfo.opticalCenter.y - center.y) * factor);
  Vector3<double> 
    vectorToCenterWorld = cameraMatrix.rotation * vectorToCenter;
  
  //Is the point above the horizon ? - return
//  if(vectorToCenterWorld.z <= -5 * yFactor) //for perfect horizon
    if(vectorToCenterWorld.z < 1 * factor)
  {
    Vector2<double>angles;
    Geometry::calculateAnglesForPoint(center, cameraMatrix, bwCameraInfo, angles);
    ballPercept.add(
      bwCameraInfo,
      center,
      radius,
      angles, 
      Geometry::pixelSizeToAngleSize(radius, bwCameraInfo), 
      cameraMatrix.translation, 
      cameraMatrix.isValid);
  }
}

/*
* $Log: GT2004BallSpecialist.cpp,v $
* Revision 1.3  2004/05/14 16:37:42  nistico
* Ghost balls on landmark improvements:
* -introduced ambiguous (pink, red, yellow) color check in ball recognition
* -some parameter tuning
*
* Revision 1.2  2004/05/07 15:16:24  nistico
* All geometry calculations are making use of intrinsic functions.
* I updated most of the images processors to make correct use of this.
* Please test if there are any problems, because i'm going to remove the
* old code soon.
*
* Revision 1.1  2004/05/04 13:40:19  tim
* added GT2004ImageProcessor
*
*/
