/**
* @file GT2003ImageProcessor.cpp
*
* Implementation of class GT2003ImageProcessor
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#include "GT2003ImageProcessor.h"
#include "Tools/Math/Common.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "Tools/FieldDimensions.h"
#include "Tools/RingBuffer.h"
#include "Representations/Perception/ColorTable64.h"

#define PLOT(p,c) COMPLEX_DRAWING(imageProcessor_general,plot(p,c))

const int Y = 0,
          U = cameraResolutionWidth_ERS7, /**< Relative offset of U component. */
          V = 2 * cameraResolutionWidth_ERS7; /**< Relative offset of V component. */

//#define COLOR_CLASS(y,u,v) colorTable.getColorClass(y,u,v)
// This is a hack to speed up color table access. It obviously only works with ColorTable64
#define COLOR_CLASS(y,u,v) (colorClass)((ColorTable64&) colorTable).colorClasses[(y) >> 2][(u) >> 2][(v) >> 2]

GT2003ImageProcessor::GT2003ImageProcessor(const ImageProcessorInterfaces& interfaces)
: ImageProcessor(interfaces),
manualCalibration(image, cameraMatrix, (ColorTable64&)colorTable, calibrationRequest)
{
  yThreshold = 15;
  vThreshold = 8;
}

void GT2003ImageProcessor::execute()
{
  INIT_DEBUG_IMAGE(imageProcessorPlayers, image);

  //~ xFactor = image.cameraInfo.resolutionWidth / tan(image.cameraInfo.openingAngleWidth / 2.0) / 2.0;
  //~ yFactor = image.cameraInfo.resolutionHeight / tan(image.cameraInfo.openingAngleHeight / 2.0) / 2.0;
  xFactor = yFactor = image.cameraInfo.focalLength;

  numberOfScannedPixels = 0;
  // reset everything
  landmarksPercept.reset(image.frameNumber);
  linesPercept.reset(image.frameNumber);
  ballPercept.reset(image.frameNumber);
  playersPercept.reset(image.frameNumber);
  obstaclesPercept.reset(image.frameNumber);
  flagSpecialist.init(image);

  // variations of the camera matrix for different object heights
  cmBall = cameraMatrix;
  cmBall.translation.z -= ballRadius;
  cmTricot = cameraMatrix;
  cmTricot.translation.z -= 100; // upper tricot border



  Vector3<double> vectorInDirectionOfViewCamera(1,0,0);
  Vector3<double> vectorInDirectionOfViewWorld;
  vectorInDirectionOfViewWorld = cameraMatrix.rotation * vectorInDirectionOfViewCamera;

  angleBetweenDirectionOfViewAndGround = 
    toDegrees(
    atan2(
    vectorInDirectionOfViewWorld.z,
    sqrt(sqr(vectorInDirectionOfViewWorld.x) + sqr(vectorInDirectionOfViewWorld.y))
    )
    );

//  OUTPUT(idText,text,"W:" << angleBetweenDirectionOfViewAndGround);

  horizon = Geometry::calculateHorizon(cameraMatrix, image.cameraInfo);

  LINE(imageProcessor_horizon,
    (int)(horizon.base.x - 120 * horizon.direction.x), (int)(horizon.base.y - 120 * horizon.direction.y), 
    (int)(horizon.base.x + 120 * horizon.direction.x), (int)(horizon.base.y + 120 * horizon.direction.y), 
    1, Drawings::ps_solid, Drawings::white );
  DEBUG_DRAWING_FINISHED(imageProcessor_horizon);
  
//  obstaclesNearPointOnField2.x = 0; obstaclesNearPointOnField2.y = 0;
//  obstaclesFarPointOnField2IsOnImageBorder = false;

//  obstaclesFarPointOnField1.x = 0; obstaclesFarPointOnField1.y = 0;
//  obstaclesFarPointOnField2.x = 0; obstaclesFarPointOnField2.y = 0;
  scanColumns();
  
/*  if(Geometry::distance(obstaclesFarPointOnField1, obstaclesFarPointOnField2) < 400)
  {
    obstaclesPercept.add(obstaclesNearPointOnField2, obstaclesFarPointOnField2, obstaclesFarPointOnField2IsOnImageBorder);
  }
*/
/*
  if(obstaclesPercept.numberOfPoints < 3)
  {
    obstaclesPercept.numberOfPoints = 0;
  }
*/
  scanRows();
  filterPercepts();

  landmarksPercept.cameraOffset = cameraMatrix.translation;
  flagSpecialist.getFlagPercept(cameraMatrix, image.cameraInfo, horizon, landmarksPercept);

  GoalRecognizer goalRecognizer(image, cameraMatrix, colorTable, obstaclesPercept, landmarksPercept);
  goalRecognizer.getGoalPercept(landmarksPercept);

  INFO(printPixelUsage, idText, text, "used pixels: " << 100 * numberOfScannedPixels / (image.cameraInfo.resolutionWidth * image.cameraInfo.resolutionHeight) << " %");

  SEND_DEBUG_IMAGE(imageProcessorPlayers);
  DEBUG_DRAWING_FINISHED(imageProcessor_general);
  DEBUG_DRAWING_FINISHED(imageProcessor_ball1);
}

void GT2003ImageProcessor::scanColumns()
{
  // Reset horizontal counters
  orangeCount = 0;
  noOrangeCount = noRedCount = noBlueCount = 100;
  numberOfBallPoints = 0;
  closestBottom = closestBottom = 40000;
  goalAtBorder = goalAtBorder = false;
  
  const int scanLineSpacing = 3,
            desiredNumberOfScanLines = 60;

  Geometry::Line scanLine,
                 lineAboveHorizon,
                 lineBelowHorizon[2];
  Vector2<double> intersection;
  Vector2<int> topPoint,
               bottomPoint;
  lastFlag = Vector2<int>(-100,0);
  int i;
  for(i = 0; i < 6; ++i)
    flagCount[i] = 0;

  // scan lines are perpendicular to horizon
  scanLine.direction.x = -horizon.direction.y;
  scanLine.direction.y = horizon.direction.x;

  // upper boundary for scan lines
  lineAboveHorizon.direction = horizon.direction;
  lineAboveHorizon.base = horizon.base - scanLine.direction * 50;
    
  // lower boundaries for scan lines
  for(i = 0; i < 2; ++i)
  {
    lineBelowHorizon[i].direction = horizon.direction;
    lineBelowHorizon[i].base = horizon.base + scanLine.direction * 30 * (i + 1);
  }      

  // calculate and scan lines
  for(i = 0; i < desiredNumberOfScanLines; i++)
  {
    scanLine.base.x = image.cameraInfo.resolutionWidth / 2 + (i - desiredNumberOfScanLines / 2) * horizon.direction.x * scanLineSpacing;
    scanLine.base.y = image.cameraInfo.resolutionHeight / 2 + (i - desiredNumberOfScanLines / 2) * horizon.direction.y * scanLineSpacing;
    
    //Does the scan line intersect with the image?
    if(Geometry::getIntersectionPointsOfLineAndRectangle(
                                        Vector2<int>(0,0),
                                        Vector2<int>(image.cameraInfo.resolutionWidth - 1,image.cameraInfo.resolutionHeight - 1),
                                        scanLine, 
                                        topPoint, bottomPoint))
    {
      Geometry::getIntersectionOfLines(lineAboveHorizon, scanLine, intersection);
      
      //Is the bottomPoint below the horizon?
      if(scanLine.direction.y / (intersection.y - bottomPoint.y) < 0)
      {

        //Is the intersection with the horizon point inside the image?
        if(intersection.x < image.cameraInfo.resolutionWidth && intersection.x >= 0 &&
          intersection.y < image.cameraInfo.resolutionHeight && intersection.y >= 0) 
        {
          topPoint.x = (int)intersection.x;
          topPoint.y = (int)intersection.y;
        }
        int lineType = i & 3; // different lengths of scan lines
        if(lineType)
        {
          Geometry::getIntersectionOfLines(lineBelowHorizon[lineType == 3 ? 0 : lineType - 1], scanLine, intersection);
          //Is the intersection with the horizon point inside the image?
          if(intersection.x < image.cameraInfo.resolutionWidth && intersection.x >= 0 &&
            intersection.y < image.cameraInfo.resolutionHeight && intersection.y >= 0) 
          {
            bottomPoint.x = (int)intersection.x;
            bottomPoint.y = (int)intersection.y;
            scan(topPoint, bottomPoint, true, true);
          }
        }
        else
          scan(topPoint, bottomPoint, true, false);
      }
    }
  }

  // finish clustering
  for(i = 0; i < 3; ++i)
  {
    clusterBalls(0,0,0,0,false,false);
    clusterRobots(0,false,false);
  }
  clusterFlags(Flag::numberOfFlagTypes,Vector2<int>(1000,0));
}

void GT2003ImageProcessor::scanRows()
{
  const int scanLineSpacing = 15,
            desiredNumberOfScanLines = 30;
  Geometry::Line scanLine = horizon;

  // start below horizon
  scanLine.base.x += -horizon.direction.y * 10;
  scanLine.base.y += horizon.direction.x * 10;

  for(int i = 1; i < desiredNumberOfScanLines; i++)
  {
    scanLine.base.x += -horizon.direction.y * scanLineSpacing;
    scanLine.base.y += horizon.direction.x * scanLineSpacing;
    
    Vector2<int> topPoint,
                 bottomPoint;
    //Does the scan line intersect with the image?
    if(Geometry::getIntersectionPointsOfLineAndRectangle(
                                        Vector2<int>(0,0),
                                        Vector2<int>(image.cameraInfo.resolutionWidth - 1,image.cameraInfo.resolutionHeight - 1),
                                        scanLine,
                                        topPoint, bottomPoint))
      if(rand() > RAND_MAX / 2)
        scan(topPoint, bottomPoint, false, false);
      else
        scan(bottomPoint, topPoint, false, false);
  }
}

void GT2003ImageProcessor::scan(const Vector2<int>& start, const Vector2<int>& end,
                                bool vertical, bool noLines)
{
  LinesPercept::LineType typeOfLinesPercept = LinesPercept::field;
  int pixelsSinceGoal = 0;
  int pixelsBetweenGoalAndObstacle = 0;
  
  Vector2<int> actual = start,
               diff = end - start,
               step(sgn(diff.x),sgn(diff.y)),
               size(abs(diff.x),abs(diff.y));
  bool isVertical = abs(diff.y) > abs(diff.x);
  int count,
      sum;

  // init Bresenham
  if(isVertical)
  {
    count = size.y;
    sum = size.y / 2;
  }
  else
  {
    count = size.x;
    sum = size.x / 2;
  }

  if(count > 7)
  {
    int numberOfPoints[LinesPercept::numberOfLineTypes],
        i;
    for(i = 0; i < LinesPercept::numberOfLineTypes; ++i)
      numberOfPoints[i] = linesPercept.numberOfPoints[i];
    RingBuffer<const unsigned char*,7> pixel;
    //the image is scanned vertically
    for(i = 0; i < 6; ++i) // fill buffer
    {
      pixel.add(&image.image[actual.y][0][actual.x]);
      
      // Bresenham
      if(isVertical)
      {
        actual.y += step.y;
        sum += size.x;
        if(sum >= size.y)
        {
          sum -= size.y;
          actual.x += step.x;
        }
      }
      else        
      {
        actual.x += step.x;
        sum += size.y;
        if(sum >= size.x)
        {
          sum -= size.x;
          actual.y += step.y;
        }
      }
    }
    lineColor = Drawings::numberOfColors;

    int redCount = 0,
        blueCount = 0,
        greenCount = 0,
        noGreenCount = 100,
        orangeCountVert = 0,
        noOrangeCountVert = 100,
        pinkCount = 0,
        noPinkCount = 100,
        yellowCount = 0,
        noYellowCount = 100,
        skyblueCount = 0,
        noSkyblueCount = 100;
    const unsigned char* firstRed = 0,
                       * lastRed = 0,
                       * firstBlue = 0,
                       * lastBlue = 0,
                       * firstGreen = 0,
                       * firstOrange = 0,
                       * lastOrange = 0,
                       * firstPink = 0,
                       * lastPink = 0,
                       * pFirst = pixel[2],
                       * up = pFirst;
    enum{justUP, greenAndUP, down} state = justUP;
    bool borderFound = false, // there can be only one border point per scan line
         greenAboveOrange = false,
         greenBelowOrange = false;
    Flag::FlagType flagType = Flag::numberOfFlagTypes; // none

    // obstacles
    enum{noGroundFound, foundBeginOfGround, foundEndOfGround} groundState = noGroundFound;
    Vector2<int> firstGround(0,0);
    int numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;
    int numberOfPixelsSinceLastGround = 0;
    int numberOfGroundPixels = 0;
    int numberOfWhiteObstaclePixels = 0;
    int numberOfNotWhiteObstaclePixels = 0;
    bool beginOfGroundIsAtTheTopOfTheImage = false;

    for(; i <= count; ++i)
    {
      pixelsSinceGoal++;
      numberOfScannedPixels++;
      pixel.add(&image.image[actual.y][0][actual.x]);

      // Bresenham
      if(isVertical)
      {
        actual.y += step.y;
        sum += size.x;
        if(sum >= size.y)
        {
          sum -= size.y;
          actual.x += step.x;
        }
      }
      else        
      {
        actual.x += step.x;
        sum += size.y;
        if(sum >= size.x)
        {
          sum -= size.x;
          actual.y += step.y;
        }
      }

      // line point state machine
      const unsigned char* p3 = pixel[3],
                         * p4 = pixel[4];
      //UP: y-component increases ////////////////////////////////////////
      if(p3[Y] > p4[Y] + yThreshold)
      {
        up = p3;
        const unsigned char* p6 = pixel[6];
        if(COLOR_CLASS(p6[Y],p6[U],p6[V]) == green)
          state = greenAndUP;
        else 
          state = justUP;
      }
    
      //DOWN: y-component decreases //////////////////////////////////////
      else if(p3[Y] < p4[Y] - yThreshold || p3[V] < p4[V] - vThreshold)
      {
        const unsigned char* p0 = pixel[0];
        // is the pixel in the field ??
        if(state != down && // was an "up" pixel above?
           COLOR_CLASS(p0[Y],p0[U],p0[V]) == green) // is there green 3 pixels below ?
        {
          const unsigned char* p6 = pixel[6];
          colorClass c = COLOR_CLASS(p6[Y],p6[U],p6[V]);
          if(!noLines || c == skyblue || c == yellow)
          {
            Vector2<int> coords = getCoords(p3),
                         pointOnField; //position on field, relative to robot

            if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
            {
              Vector2<int> upCoords = getCoords(up);
              int height = (coords - upCoords).abs();
              int distance = (int) sqrt(sqr(cameraMatrix.translation.z) + sqr(pointOnField.abs())),
                lineHeight = Geometry::getSizeByDistance(image.cameraInfo, 25, distance);

              const unsigned char* p5 = pixel[5];
              //The bright region is assumed to be on the ground. If it is small enough, it is a field line.
              if(height < 3 * lineHeight && state == greenAndUP &&
                 (COLOR_CLASS(p5[Y],p5[U],p5[V]) == white ||
                  COLOR_CLASS(p4[Y],p4[U],p4[V]) == white))
              {
                linesPercept.add(LinesPercept::field,pointOnField);
                if(Geometry::calculatePointOnField(upCoords.x, upCoords.y, cameraMatrix, image.cameraInfo, pointOnField))
                  linesPercept.add(LinesPercept::field,pointOnField);
              }
              else if(height >= 3 && !borderFound)
              {
                switch(c)
                {
                  case skyblue:
                    if(vertical)
                    {
                      borderFound = true;
                      if(getPlayer().getTeamColor() == Player::red && !linesPercept.numberOfPoints[LinesPercept::skyblueGoal])
                        goalAtBorder = pointOnField.abs() < closestBottom;
                      closestBottom = 40000;
                      linesPercept.add(LinesPercept::skyblueGoal,pointOnField);
                      typeOfLinesPercept = LinesPercept::skyblueGoal;
                      pixelsSinceGoal = 0;

                      lastRed = lastBlue = 0;
                      redCount = blueCount = 0;
                    }
                    break;
                  case yellow:
                    if(vertical)
                    {
                      borderFound = true;
                      if(getPlayer().getTeamColor() == Player::blue && !linesPercept.numberOfPoints[LinesPercept::yellowGoal])
                        goalAtBorder = pointOnField.abs() < closestBottom;
                      closestBottom = 40000;
                      linesPercept.add(LinesPercept::yellowGoal,pointOnField);
                      typeOfLinesPercept = LinesPercept::yellowGoal;
                      pixelsSinceGoal = 0;

                      lastRed = lastBlue = 0;
                      redCount = blueCount = 0; 
                    }
                    break;
                  case white:
                    if(state != greenAndUP && height > lineHeight * 3 && (vertical || height > 30))
                    {
                      borderFound = true;
                      linesPercept.add(LinesPercept::border,pointOnField);
                      typeOfLinesPercept = LinesPercept::border;
                      lastRed = lastBlue = 0;
                      redCount = blueCount = 0; 
                    }
                    break;
                }
              }
            }
          }
          state = down;
        }
      }
      if(vertical)
      {
        // in scanColumns, recognize player and ball points
        colorClass c3 = COLOR_CLASS(p3[Y],p3[U],p3[V]);
        ++noOrangeCountVert;
        ++noGreenCount;
        ++noPinkCount;
        ++noYellowCount;
        ++noSkyblueCount;
        switch(c3)
        {
          case blue: // count blue, remember last
            if(blueCount == 3)
              firstBlue = pixel[6];
            ++blueCount;
            lastBlue = p3;
            firstGreen = 0;
            break;
          case gray: // drop green above gray (i.e. robot legs)
            if(firstGreen && noGreenCount < 8)
              firstGreen = 0;
            break;
          case green: // find first green below a robot or a ball
            if(!firstGreen)
            { 
              firstGreen = p3;
              // check if this green is below a ball
              if(orangeCountVert > 2 && noOrangeCountVert < 6)
                greenBelowOrange = true;
            }
            if(noGreenCount > 6)
              greenCount = 1;
            else
              ++greenCount;
            if(greenCount == 2 && noPinkCount < 5 && pinkCount >= 2)
              flagType = Flag::pinkAboveGreen;
            noGreenCount = 0;
            break;
          case red:
            if(orangeCountVert < 6 || noOrangeCountVert > 4 || redCount > orangeCountVert)
            { // count red, remember last
              if(redCount == 3)
                firstRed = pixel[6];
              ++redCount;
              lastRed = p3;
              firstGreen = 0;
              break;
            }
            // no break, red below orange is interpreted as orange
          case orange: // find beginning and end of a ball
            if(noOrangeCountVert > 6)
            {
              orangeCountVert = 1;
              firstOrange = p3;
              lastOrange = p3;
              greenAboveOrange = noGreenCount < 6;
            }
            else
            {
              ++orangeCountVert;
              if(orangeCountVert > 2) // ignore robot if ball is below, distance would be wrong
              {
                lastRed = lastBlue = 0;
                redCount = blueCount = 0; 
              }
              lastOrange = p3;
            }
            firstGreen = 0;
            noOrangeCountVert = 0;
            greenBelowOrange = false;
            break;
          case pink:
            if(noPinkCount > 6)
            {
              pinkCount = 1;
              firstPink = p3;
            }
            else
              ++pinkCount;
            if(pinkCount == 2)
            {
              if(noYellowCount < 5 && yellowCount >= 2)
                flagType = Flag::yellowAbovePink;
              else if(noSkyblueCount < 5 && skyblueCount >= 2)
                flagType = Flag::skyblueAbovePink;
              else if(noGreenCount < 5 && greenCount >= 2)
                flagType = Flag::greenAbovePink;
            }
            lastPink = p3;
            noPinkCount = 0;
            break;
          case yellow:
            if(noYellowCount > 6)
              yellowCount = 1;
            else
              ++yellowCount;
            if(yellowCount == 2 && noPinkCount < 5 && pinkCount >= 2)
              flagType = Flag::pinkAboveYellow;
            noYellowCount = 0;
            break;
          case skyblue:
            if(noSkyblueCount > 6)
              skyblueCount = 1;
            else
              ++skyblueCount;
            if(skyblueCount == 2 && noPinkCount < 5 && pinkCount >= 2)
              flagType = Flag::pinkAboveSkyblue;
            noSkyblueCount = 0;
            break;
          case white:
            switch(flagType)
            {
              case Flag::pinkAboveGreen:
                if(noGreenCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              case Flag::pinkAboveYellow:
                if(noYellowCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              case Flag::pinkAboveSkyblue:
                if(noSkyblueCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              case Flag::greenAbovePink:
                if(noPinkCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              case Flag::yellowAbovePink:
                if(noPinkCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              case Flag::skyblueAbovePink:
                if(noPinkCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
            }
            break;
          
        }//switch(c3)

        // obstacles state machine
        if(count > 90 && cameraMatrix.isValid)
        {
          DEBUG_IMAGE_SET_PIXEL_BLACK(imageProcessorPlayers, getCoords(pixel[0]).x, getCoords(pixel[0]).y);
          colorClass color = COLOR_CLASS(pixel[0][Y], pixel[0][U], pixel[0][V]);
          if(groundState == noGroundFound)
          {
            DEBUG_IMAGE_SET_PIXEL_YELLOW(imageProcessorPlayers, getCoords(pixel[0]).x, getCoords(pixel[0]).y);
            if(color == green || color == orange)
            {
              if(numberOfPixelsSinceFirstOccurrenceOfGroundColor == 0)
              {
                firstGround = getCoords(pixel[0]);
              }
              numberOfPixelsSinceFirstOccurrenceOfGroundColor++;
              if(numberOfPixelsSinceFirstOccurrenceOfGroundColor > 3 || color == orange)
              {
                Vector2<int> coords = getCoords(pixel[0]);
                int lineHeight = Geometry::calculateLineSize(coords, cameraMatrix, image.cameraInfo);
                if(i < lineHeight * 4) beginOfGroundIsAtTheTopOfTheImage = true;
                groundState = foundBeginOfGround;
                pixelsBetweenGoalAndObstacle = pixelsSinceGoal;
              }
            }
            else
            {
              numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;
              if(color == white) numberOfWhiteObstaclePixels++;
              else numberOfNotWhiteObstaclePixels++;
            }
          }
          else if(groundState == foundBeginOfGround)
          {
            DEBUG_IMAGE_SET_PIXEL_GREEN(imageProcessorPlayers, getCoords(pixel[0]).x, getCoords(pixel[0]).y);
            if(color != green && color != orange)
            {
              groundState = foundEndOfGround;
              numberOfPixelsSinceLastGround = 0;
              numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;
            }
            else
            {
              numberOfGroundPixels++;
            }
          }
          else if(groundState == foundEndOfGround)
          {
            numberOfPixelsSinceLastGround++;
            DEBUG_IMAGE_SET_PIXEL_RED(imageProcessorPlayers, getCoords(pixel[0]).x, getCoords(pixel[0]).y);
            if(color == green || color == orange)
            {
              numberOfPixelsSinceFirstOccurrenceOfGroundColor++;
              if(numberOfPixelsSinceFirstOccurrenceOfGroundColor > 3 || color == orange)
              {
                groundState = foundBeginOfGround;
//                pixelsBetweenGoalAndObstacle = pixelsSinceGoal;
                Vector2<int> coords = getCoords(pixel[0]),
                             pointOnField; //position on field, relative to robot
                int lineHeight = Geometry::calculateLineSize(coords, cameraMatrix, image.cameraInfo);
                if(numberOfPixelsSinceLastGround > lineHeight * 4)
                {
                  firstGround = getCoords(pixel[0]);
                  numberOfGroundPixels = 0;
                  beginOfGroundIsAtTheTopOfTheImage = false;
                }
              }
            }
            else
            {
              numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;  
              if(color == white) numberOfWhiteObstaclePixels++;
              else numberOfNotWhiteObstaclePixels++;
            }
          }
        } // if(count > 100)

        PLOT(p3,ColorClasses::colorClassToDrawingsColor(c3));
      }//if(vertical)
    }

    const unsigned char* pLast = pixel[3];
    PLOT(pLast,Drawings::numberOfColors);

    if(vertical)
    { // line scanning finished, analyze results (only in scanColumns)
      int goal = getPlayer().getTeamColor() == Player::red ? LinesPercept::skyblueGoal
                                                           : LinesPercept::yellowGoal;
      if(linesPercept.numberOfPoints[goal] == numberOfPoints[goal]) // no goal point found in this column
      {
        Vector2<int> coords = getCoords(pLast),
                     pointOnField;
        if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
        {
          int dist = pointOnField.abs();
          if(dist < closestBottom)
            closestBottom = dist;
        }
      }
              
      // ball points found?
      if(orangeCountVert > 2 && (greenAboveOrange || greenBelowOrange || !noLines && noOrangeCountVert <= 5) || 
         orangeCountVert > 20)
      { 
        if(!greenAboveOrange) // drop border and goal points
          for(i = 1; i < LinesPercept::numberOfLineTypes; ++i)
            linesPercept.numberOfPoints[i] = numberOfPoints[i];
        if(noOrangeCountVert <= 5)
          pLast = lastOrange;
        clusterBalls(pFirst,pLast,firstOrange,lastOrange,greenAboveOrange,greenBelowOrange);
      }
      else
        clusterBalls(pFirst,pLast,0,0,greenAboveOrange,greenBelowOrange);

      bool redFound = false,
           blueFound = false;

      // red robot point found?
      if(redCount > blueCount && (firstGreen && redCount > 2 || redCount > 20))
      { // drop border and goal points
        for(i = 1; i < LinesPercept::numberOfLineTypes; ++i)
          linesPercept.numberOfPoints[i] = numberOfPoints[i];
        if(redCount > 20)
        { // if robot is close, use upper boundary of tricot to calculate distance
          Vector2<int> coords = getCoords(firstRed),
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cmTricot, image.cameraInfo, pointOnField) &&
             pointOnField.abs() < 500)
          {
            DOT(imageProcessor_general, coords.x, coords.y, Drawings::red, Drawings::white);
            linesPercept.add(LinesPercept::redRobot,pointOnField);
            typeOfLinesPercept = LinesPercept::redRobot;
            redFound = true;
          }
        }
        if(!redFound)
        { // otherwise, use foot point
          Vector2<int> coords = getCoords(firstGreen),
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
          {
            Vector2<int> redCoords = getCoords(lastRed);
            if((redCoords - coords).abs() < Geometry::getSizeByDistance(image.cameraInfo, 100, pointOnField.abs()))
            {
              linesPercept.add(LinesPercept::redRobot,pointOnField);
              typeOfLinesPercept = LinesPercept::redRobot;
              redFound = true;
            }
          }
        }
      }
      // blue robot point found?
      else if(blueCount > redCount && (firstGreen && blueCount > 2 || blueCount > 10))
      { // drop border and goal points
        for(i = 1; i < LinesPercept::numberOfLineTypes; ++i)
          linesPercept.numberOfPoints[i] = numberOfPoints[i];
        if(blueCount > 10)
        { // if robot is close, use upper boundary of tricot to calculate distance
          Vector2<int> coords = getCoords(firstBlue),
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cmTricot, image.cameraInfo, pointOnField) &&
             pointOnField.abs() < 500)
          {
            DOT(imageProcessor_general, coords.x, coords.y, Drawings::pink, Drawings::white);
            linesPercept.add(LinesPercept::blueRobot,pointOnField);
            typeOfLinesPercept = LinesPercept::blueRobot;
            blueFound = true;
          }
        }
        if(!blueFound)
        { // otherwise, use foot point
          Vector2<int> coords = getCoords(firstGreen),
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
          {
            Vector2<int> blueCoords = getCoords(lastBlue);
            if((blueCoords - coords).abs() < Geometry::getSizeByDistance(image.cameraInfo, 100, pointOnField.abs()))
            {
              linesPercept.add(LinesPercept::blueRobot,pointOnField);
              typeOfLinesPercept = LinesPercept::blueRobot;
              blueFound = true;
            }
          }
        }
      }
      // cluster the robot points found
      clusterRobots(pLast, redFound, blueFound);

      // obstacles found?
      bool addObstaclesPoint = false;
      Vector2<int> newObstaclesNearPoint;
      Vector2<int> newObstaclesFarPoint;
      bool newObstaclesFarPointIsOnImageBorder = false;
      if(count > 90 && cameraMatrix.isValid &&
        (
         numberOfGroundPixels > numberOfWhiteObstaclePixels ||
         numberOfGroundPixels > 20 ||
         numberOfNotWhiteObstaclePixels > numberOfWhiteObstaclePixels * 2
         )
        )
      {
        Vector2<int> firstGroundOnField;
        bool firstGroundOnFieldIsOnField = 
          Geometry::calculatePointOnField(firstGround.x, firstGround.y, cameraMatrix, image.cameraInfo, firstGroundOnField);

        Vector2<int> imageBottom = getCoords(pixel[0]);
        Vector2<int> imageBottomOnField;
        bool imageBottomIsOnField = 
          Geometry::calculatePointOnField(imageBottom.x, imageBottom.y, cameraMatrix, image.cameraInfo, imageBottomOnField);

        if(groundState == foundBeginOfGround)
        {
          if(firstGroundOnFieldIsOnField)
          {
            addObstaclesPoint = true;
            newObstaclesNearPoint = imageBottomOnField;
            newObstaclesFarPoint = firstGroundOnField;
            newObstaclesFarPointIsOnImageBorder = beginOfGroundIsAtTheTopOfTheImage;
          }
        }
        else if(groundState == foundEndOfGround)
        {
          int lineHeight = Geometry::calculateLineSize(imageBottom, cameraMatrix, image.cameraInfo);
          if(numberOfPixelsSinceLastGround < lineHeight * 4)
          {
            if(firstGroundOnFieldIsOnField)
            {
              addObstaclesPoint = true;
              newObstaclesNearPoint = imageBottomOnField;
              newObstaclesFarPoint = firstGroundOnField;
              newObstaclesFarPointIsOnImageBorder = beginOfGroundIsAtTheTopOfTheImage;
            }
          }
          else if(imageBottomIsOnField)
          {
            addObstaclesPoint = true;
            newObstaclesNearPoint = imageBottomOnField;
            newObstaclesFarPoint = imageBottomOnField;
          }
        }
        else if(groundState == noGroundFound)
        {
          if(imageBottomIsOnField &&
            // the carpet often is not green under the robot
            imageBottomOnField.abs() > 150)
          {
            addObstaclesPoint = true;
            newObstaclesNearPoint = imageBottomOnField;
            newObstaclesFarPoint = imageBottomOnField;
          }
        }
      }// if(count > 90 && cameraMatrix.isValid)
      if(addObstaclesPoint)
      {
        if(
/*          (
          Geometry::distance(obstaclesFarPointOnField1, obstaclesFarPointOnField2) < 400 ||
          Geometry::distance(obstaclesFarPointOnField2, newObstaclesFarPoint) < 400
          ) && 
*/
          angleBetweenDirectionOfViewAndGround > -80.0 &&
          !(newObstaclesFarPoint.x == 0 && newObstaclesFarPoint.y == 0)
          )

        {
          ObstaclesPercept::ObstacleType obstacleType = ObstaclesPercept::unknown;
          switch(typeOfLinesPercept)
          {
          case LinesPercept::skyblueGoal: 
            if(pixelsBetweenGoalAndObstacle < 15) 
              obstacleType = ObstaclesPercept::goal; 
            break;
          case LinesPercept::yellowGoal: 
            if(pixelsBetweenGoalAndObstacle < 15) 
              obstacleType = ObstaclesPercept::goal; 
            break;
          case LinesPercept::blueRobot:
            if(getPlayer().getTeamColor() == Player::blue)
              obstacleType = ObstaclesPercept::teammate;
            else 
              obstacleType = ObstaclesPercept::opponent;
            break;
          case LinesPercept::redRobot: 
            if(getPlayer().getTeamColor() == Player::red) 
              obstacleType = ObstaclesPercept::teammate;
            else 
              obstacleType = ObstaclesPercept::opponent;
            break;
          }
          obstaclesPercept.add(
            newObstaclesNearPoint, 
            newObstaclesFarPoint,
            newObstaclesFarPointIsOnImageBorder,
            obstacleType
            );
        }

/*        obstaclesFarPointOnField1 = obstaclesFarPointOnField2;
        obstaclesFarPointOnField2 = newObstaclesFarPoint;
    
        obstaclesFarPointOnField2IsOnImageBorder = newObstaclesFarPointIsOnImageBorder;
        obstaclesNearPointOnField2 = newObstaclesNearPoint;
        */
      }
    }//if(vertical)
  }
}

void GT2003ImageProcessor::clusterBalls(const unsigned char* topPoint,
                                        const unsigned char* bottomPoint,
                                        const unsigned char* first,
                                        const unsigned char* last,
                                        bool greenAboveOrange,
                                        bool greenBelowOrange)
{
  if(first) // ball found
  {
    if(first != topPoint) // not at image border
    {
      (Vector2<int>&) ballPoints[numberOfBallPoints] = getCoords(first);
      DOT(imageProcessor_general,ballPoints[numberOfBallPoints].x,
                              ballPoints[numberOfBallPoints].y,
                              Drawings::orange,Drawings::white);
      ballPoints[numberOfBallPoints].isBottom = false;
      ballPoints[numberOfBallPoints++].greenIsClose = greenAboveOrange;
    }
    if(last != bottomPoint) // not at image border
    {
      (Vector2<int>&) ballPoints[numberOfBallPoints] = getCoords(last);
      DOT(imageProcessor_general,ballPoints[numberOfBallPoints].x,
                              ballPoints[numberOfBallPoints].y,
                              Drawings::orange,Drawings::white);
      ballPoints[numberOfBallPoints].isBottom = true;
      ballPoints[numberOfBallPoints++].greenIsClose = greenBelowOrange;
    }
    if(!orangeCount)
      firstBall = (getCoords(first) + getCoords(last)) / 2;
    lastBall = (getCoords(first) + getCoords(last)) / 2;
    ++orangeCount;
    noOrangeCount = 0;
  }
  else // no ball found
  {
    Vector2<int> coords = getCoords(bottomPoint),
                 pointOnField,
                 ballOnField;
    // count number of columns without ball points, but ignore columns that are
    // not long enough to contain the ball
    if(noOrangeCount > 1 || !bottomPoint || !numberOfBallPoints ||
       (Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField) &&
        Geometry::calculatePointOnField(ballPoints[numberOfBallPoints - 1].x, 
                                        ballPoints[numberOfBallPoints - 1].y,
                                        cameraMatrix, image.cameraInfo, ballOnField) &&
        ballOnField.abs() > pointOnField.abs()))
    {
      if(++noOrangeCount == 2 && // ignore small balls close to but not below red robots
         (!numberOfBallPoints || numberOfBallPoints > 2 || noRedCount > 3 || 
          !Geometry::calculatePointOnField(ballPoints[numberOfBallPoints - 1].x, 
                                           ballPoints[numberOfBallPoints - 1].y,
                                           cameraMatrix, image.cameraInfo, ballOnField) ||
           closestRed.abs() > ballOnField.abs()))
      {
        createBallPercept();
        orangeCount = 0;
        numberOfBallPoints = 0;
      }
    }
  }
}

void GT2003ImageProcessor::createBallPercept()
{
  filterBallPoints();
  if(!createBallPerceptFrom3Points(numberOfBallPoints < 2 * orangeCount)) // if succeeded, a percept was generated
  {
    if(!numberOfBallPoints || orangeCount > 5) // if no ball points or a lot of them
    { // calculate center of orange region
      Vector2<int> centerInImage = (firstBall + lastBall) / 2;                   
      addBallPercept(centerInImage.x, centerInImage.y, (lastBall - firstBall).abs() / 2.0);
    }
    if(!createBallPerceptFrom3Points(false)) // if succeeded, a percept was generated
    { // otherwise, less than 3 points or all points are on a straight line
      Vector2<int> closestPoint(40000,0);
      for(int i = 0; i < numberOfBallPoints; ++i)
        if(ballPoints[i].isBottom)
        {
          Vector2<int> pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(ballPoints[i].x, ballPoints[i].y,
                                             cameraMatrix, image.cameraInfo, pointOnField) &&
             pointOnField.abs() < closestPoint.abs())
            closestPoint = pointOnField;
        }
      double distance;
      if(closestPoint.abs() < 40000) // point found
      {
        distance = (Vector2<double>(closestPoint.x,closestPoint.y) - 
                    Vector2<double>(cameraMatrix.translation.x,
                                    cameraMatrix.translation.y)).abs();
        double tilt = atan2(distance,cameraMatrix.translation.z);
        // point is below ball, add offset to calculate center
        distance += ballRadius * (cos(tilt) + (sin(tilt) - 1) * tan(tilt));
      }
      else
      {
        Vector2<int> farthestPoint(0,0);
        for(int i = 0; i < numberOfBallPoints; ++i)
          if(!ballPoints[i].isBottom)
          {
            Vector2<int> pointOnField; //position on field, relative to robot
            if(Geometry::calculatePointOnField(ballPoints[i].x, ballPoints[i].y,
                                               cameraMatrix, image.cameraInfo, pointOnField) &&
               pointOnField.abs() > farthestPoint.abs())
              farthestPoint = pointOnField;
          }
        if(farthestPoint.abs() > 0) // point found
        {
          distance = (Vector2<double>(farthestPoint.x,farthestPoint.y) - 
                      Vector2<double>(cameraMatrix.translation.x,
                                      cameraMatrix.translation.y)).abs();
          double tilt = atan2(distance,cameraMatrix.translation.z),
                 subs = ballRadius * (cos(tilt) + (sin(tilt) + 1) * tan(tilt));
          // point is above ball, subtract offset to calculate center
          distance -= subs;
        }
        else
          return; // no point found
      }
      Vector2<int> center = (ballPoints[0] + ballPoints[numberOfBallPoints - 1]) / 2,
                   pointOnField; //position on field, relative to robot
      if(Geometry::calculatePointOnField(center.x, center.y,
                                         cameraMatrix, image.cameraInfo, pointOnField))
      {
        Vector2<double> offset = Vector2<double>(pointOnField.x,pointOnField.y) - 
                                 Vector2<double>(cameraMatrix.translation.x,
                                                 cameraMatrix.translation.y);
        double direction = atan2(offset.y,offset.x);
        Vector2<double> ballPosition(cameraMatrix.translation.x + cos(direction) * distance,
                                       cameraMatrix.translation.y + sin(direction) * distance); 
        Geometry::Circle circle;
        circle.center.x = center.x;
        circle.center.y = center.y;
        circle.radius = (ballPoints[0] - ballPoints[numberOfBallPoints - 1]).abs() / 2;
        if(checkOrangeInsideBall(circle)) 
          addBallPercept((int)circle.center.x, (int)circle.center.y, circle.radius);
      }
    }
  }
}

void GT2003ImageProcessor::filterBallPoints()
{
  // Filter top points without a corresponding bottom point that are above
  // top points with a corresponding bottom point. This removes outliers that
  // lack a bottom point because the scan line ended too early.
  int i;
  for(i = 0; i < numberOfBallPoints; ++i)
    if(!ballPoints[i].isBottom && // top point without corresponding bottom point
       (i == numberOfBallPoints - 1 || !ballPoints[i + 1].isBottom))
    {
      Vector2<int> pointOnField;
      if(Geometry::calculatePointOnField(ballPoints[i].x, ballPoints[i].y,
                                         cameraMatrix, image.cameraInfo, pointOnField))
      {
        int count = 0,
            dist = pointOnField.abs();
        for(int j = 0; j < numberOfBallPoints; ++j)
          if(!ballPoints[j].isBottom && // top point _with_ corresponding bottom point
             j < numberOfBallPoints - 1 && ballPoints[j + 1].isBottom &&
             Geometry::calculatePointOnField(ballPoints[j].x, ballPoints[j].y,
                                             cameraMatrix, image.cameraInfo, pointOnField) &&
             dist > pointOnField.abs())
            ++count;
        if(count > 1)
        {
          --orangeCount;
          --numberOfBallPoints;
          for(int j = i; j < numberOfBallPoints; ++j)
            ballPoints[j] = ballPoints[j + 1];
          --i;
        }
      }
    }          

  // Filter bottom points that are above top points. They must be outliers.
  for(i = 0; i < numberOfBallPoints; ++i)
    if(ballPoints[i].isBottom)
    {
      Vector2<int> pointOnField;
      if(Geometry::calculatePointOnField(ballPoints[i].x, ballPoints[i].y,
                                         cameraMatrix, image.cameraInfo, pointOnField))
      {
        int count = 0,
            dist = pointOnField.abs();
        for(int j = 0; j < numberOfBallPoints; ++j)
          if(!ballPoints[j].isBottom && 
             Geometry::calculatePointOnField(ballPoints[j].x, ballPoints[j].y,
                                             cameraMatrix, image.cameraInfo, pointOnField) &&
             dist > pointOnField.abs())
            ++count;
        if(count > 1)
        {
          --orangeCount;
          --numberOfBallPoints;
          if(i > 0 && !ballPoints[i - 1].isBottom)
          {
            --numberOfBallPoints;
            --i;
          }
          for(int j = i; j < numberOfBallPoints; ++j)
            ballPoints[j] = ballPoints[j + 1];
          --i;
        }
      }
    }          

  // Filter top points that are below bottom points. They must be outliers.
  for(i = 0; i < numberOfBallPoints; ++i)
    if(!ballPoints[i].isBottom)
    {
      Vector2<int> pointOnField;
      if(Geometry::calculatePointOnField(ballPoints[i].x, ballPoints[i].y,
                                         cameraMatrix, image.cameraInfo, pointOnField))
      {
        int count = 0,
            dist = pointOnField.abs();
        for(int j = 0; j < numberOfBallPoints; ++j)
          if(ballPoints[j].isBottom && 
             Geometry::calculatePointOnField(ballPoints[j].x, ballPoints[j].y,
                                             cameraMatrix, image.cameraInfo, pointOnField) &&
             dist < pointOnField.abs())
            ++count;
        if(count > 1)
        {
          --numberOfBallPoints;
          --orangeCount;
          if(i < numberOfBallPoints - 1 && ballPoints[i + 1].isBottom)
          {
            --numberOfBallPoints;
            for(int j = i; j < numberOfBallPoints; ++j)
              ballPoints[j] = ballPoints[j + 2];
          }
          else
            for(int j = i; j < numberOfBallPoints; ++j)
              ballPoints[j] = ballPoints[j + 1];
          --i;
        }
      }
    }          

  // Remove all points if no point is below the horizon
  int numberOfPointsBelowHorizon = 0;
  for(i = 0; i < numberOfBallPoints; ++i)
  {
    Vector2<int> pointOnField;
    if(Geometry::calculatePointOnField(ballPoints[i].x, ballPoints[i].y, cameraMatrix, image.cameraInfo, pointOnField))
      numberOfPointsBelowHorizon++;
  }
  if(numberOfPointsBelowHorizon == 0)
    numberOfBallPoints = 0;
}

bool GT2003ImageProcessor::createBallPerceptFrom3Points(bool onlyIfGreenIsClose)
{
  int point1, point2, point3;
  if(select3Points(point1, point2, point3, onlyIfGreenIsClose))
  {
    Vector2<int> center = 
      cutMiddlePerpendiculars(ballPoints[point1], ballPoints[point2], ballPoints[point3]);
    double ballRadiusInImage = (center - ballPoints[point1]).abs();

    addBallPercept(center.x, center.y, ballRadiusInImage);
    return true;
  }
  return false;
}

bool GT2003ImageProcessor::select3Points(int& point1, int& point2, int& point3,
                                         bool onlyIfGreenIsClose) const
{
  double maxDist = 0;
  int i;

  // first select the two points with largest distance from each other
  for(i = 0; i < numberOfBallPoints - 1; ++i)
    if(!onlyIfGreenIsClose || ballPoints[i].greenIsClose)
      for(int j = i + 1; j < numberOfBallPoints ; ++j)
        if(!onlyIfGreenIsClose || ballPoints[j].greenIsClose)
        {
          double dist = Vector2<double>(ballPoints[i].x - ballPoints[j].x,
                                        ballPoints[i].y - ballPoints[j].y).abs();
          if(dist > maxDist)
          {
            maxDist = dist;
            point1 = i;
            point2 = j;
          }
        }
  if(maxDist > 0)
  {
    // then select a third point that is farest away from the other two.
    // point 3 must form a curve with points 1 and 2, so the sum of both distances
    // must be larger then the distance between the first two points.
    maxDist += 0.1;
    point3 = -1;
    for(i = 0; i < numberOfBallPoints; ++i)
      if((!onlyIfGreenIsClose || ballPoints[i].greenIsClose) && i != point1 && i != point2)
      {
        double dist = Vector2<double>(ballPoints[i].x - ballPoints[point1].x,
                                      ballPoints[i].y - ballPoints[point1].y).abs() +
                      Vector2<double>(ballPoints[i].x - ballPoints[point2].x,
                                      ballPoints[i].y - ballPoints[point2].y).abs();
        if(dist > maxDist)
        {
          maxDist = dist;
          point3 = i;
        }
      }
    return point3 != -1;
  }
  else
    return false;
}

Vector2<int> GT2003ImageProcessor::cutMiddlePerpendiculars(Vector2<int>& v1,
                                                           Vector2<int>& v2,
                                                           Vector2<int>& v3) const
{
  Pose2D p1(atan2((double)(v1.y - v3.y), (double)(v1.x - v3.x)) + pi_2,
            Vector2<double>((v1.x + v3.x) / 2, (v1.y + v3.y) / 2)),
         p2(atan2((double)(v2.y - v3.y), (double)(v2.x - v3.x)) + pi_2,
            Vector2<double>((v2.x + v3.x) / 2, (v2.y + v3.y) / 2)),
         p = p2 - p1;
  double sinAngle = sin(p.getAngle()); 
  if(sinAngle)
  {
    p = p1 + Pose2D(Vector2<double>(p.translation.x - p.translation.y * cos(p.getAngle()) / sinAngle, 0));
    return Vector2<int>(int(p.translation.x), int(p.translation.y));
  }
  else
    return v1;
}

void GT2003ImageProcessor::addBallPercept
(
 int centerX,
 int centerY,
 double radius
 )
{
  Vector2<double>angles;
  Geometry::calculateAnglesForPoint(Vector2<int>(centerX, centerY), cameraMatrix, image.cameraInfo, angles);
  ballPercept.add(
    angles, 
    Geometry::pixelSizeToAngleSize(radius, image.cameraInfo), 
    cameraMatrix.translation, 
    cameraMatrix.isValid);
}

bool GT2003ImageProcessor::checkOrangeInsideBall(const Geometry::Circle& circle)
{
  CIRCLE(imageProcessor_ball1,
    (int)circle.center.x, 
    (int)circle.center.y, 
    (int)circle.radius,
    1, Drawings::ps_solid, Drawings::gray);

  int left = int(circle.center.x - circle.radius),
      right = int(circle.center.x + circle.radius),
      top = int(circle.center.y - circle.radius),
      bottom = int(circle.center.y + circle.radius);
  if(left < 0)
    left = 0;
  if(right >= image.cameraInfo.resolutionWidth)
    right = image.cameraInfo.resolutionWidth - 1;
  if(top < 0)
    top = 0;
  if(bottom >= image.cameraInfo.resolutionHeight)
    bottom = image.cameraInfo.resolutionHeight - 1;
  if(left >= right || top >= bottom)
    return false; // no part of circle in image -> cannot be seen
  else
  { // sample ball region
    int xSize = right - left,
        ySize = bottom - top,
        maxCount = min(xSize * ySize, 50),
        count = 0;
    for(int i = 0; i < maxCount; ++i)
    {
      int x = left + int(xSize * random()),
          y = top + int(ySize * random());
      DOT(imageProcessor_ball1, x, y, Drawings::black, Drawings::white);
      const unsigned char* p = &image.image[y][0][x];
      if(COLOR_CLASS(p[Y],p[U],p[V]) == orange)
        ++count;
    }
    if(count * 3 < maxCount)
      return false; // not enough orange pixels -> no ball
  }
  return true;
}

bool GT2003ImageProcessor::checkOrangeInsideBall(const Vector2<double>& offset)
{
  Geometry::Circle circle;
  if(Geometry::calculateBallInImage(offset,cameraMatrix, image.cameraInfo, circle))
  { // calculate square around ball and clip it with image border
    return checkOrangeInsideBall(circle);
  }
  return false;
}

void GT2003ImageProcessor::clusterRobots(const unsigned char* bottomPoint,
                                         bool redFound, bool blueFound)
{
  Vector2<int> coords = getCoords(bottomPoint),
               pointOnField;
  int noRedCount2 = noRedCount;

  if(redFound)
  {
    lastRed = 
      linesPercept.points[LinesPercept::redRobot][linesPercept.numberOfPoints[LinesPercept::redRobot] - 1];
    if(noRedCount > 2)
      firstRed = closestRed = lastRed;
    else if(closestRed.abs() > lastRed.abs())
      closestRed = lastRed;
    noRedCount = 0;
  }
  // count number of columns without robot points, but ignore columns that are
  // not long enough to contain the robot. Drop single robot points of the other 
  // robot color.
  else if(noRedCount > 2 || !bottomPoint ||
          (Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField) &&
          closestRed.abs() > pointOnField.abs()))
  {
    if(++noRedCount == 3 && (firstRed != lastRed || noBlueCount > 4))
    {
      SinglePlayerPercept percept;
      percept.direction = (atan2((double)firstRed.y,(double)firstRed.x) + atan2((double)lastRed.y,(double)lastRed.x)) / 2;
      double distance = closestRed.abs();
      percept.offset.x = cos(percept.direction) * distance;
      percept.offset.y = sin(percept.direction) * distance;
      percept.validity = 1;
      playersPercept.addRedPlayer(percept);
    }
  }

  if(blueFound)
  {
    lastBlue = 
      linesPercept.points[LinesPercept::blueRobot][linesPercept.numberOfPoints[LinesPercept::blueRobot] - 1];
    if(noBlueCount > 2)
      firstBlue = closestBlue = lastBlue;
    else if(closestBlue.abs() > lastBlue.abs())
      closestBlue = lastBlue;
    noBlueCount = 0;
  }
  // count number of columns without robot points, but ignore columns that are
  // not long enough to contain the robot. Drop single robot points of the other 
  // robot color.
  else if(noBlueCount > 2 || !bottomPoint ||
          (Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField) &&
          closestBlue.abs() > pointOnField.abs()))
  {
    if(++noBlueCount == 3 && (firstBlue != lastBlue || noRedCount2 > 4))
    {
      SinglePlayerPercept percept;
      percept.direction = (atan2((double)firstBlue.y,(double)firstBlue.x) + atan2((double)lastBlue.y,(double)lastBlue.x)) / 2;
      double distance = closestBlue.abs();
      percept.offset.x = int(cos(percept.direction) * distance);
      percept.offset.y = int(sin(percept.direction) * distance);
      percept.validity = 1;
      playersPercept.addBluePlayer(percept);
    }
  }
}

void GT2003ImageProcessor::clusterGoal()
{
  // add goal percepts
  if(linesPercept.numberOfPoints[LinesPercept::yellowGoal])
  {
    Vector2<int> pointOnField = linesPercept.points[LinesPercept::yellowGoal][linesPercept.numberOfPoints[LinesPercept::yellowGoal] / 2] * 2,
                 point;
    Geometry::calculatePointInImage(pointOnField, cameraMatrix, image.cameraInfo, point);
  }
  if(linesPercept.numberOfPoints[LinesPercept::skyblueGoal])
  {
    Vector2<int> pointOnField = linesPercept.points[LinesPercept::skyblueGoal][linesPercept.numberOfPoints[LinesPercept::skyblueGoal] / 2] * 2,
                 point;
    Geometry::calculatePointInImage(pointOnField, cameraMatrix, image.cameraInfo, point);
  }

  // calculate free part of goal
  int goal = getPlayer().getTeamColor() == Player::blue 
             ? LinesPercept::yellowGoal : LinesPercept::skyblueGoal;
  double maxWidth = 2 * ballRadius;
  Vector2<double> angles;
  Geometry::calculateAnglesForPoint(Vector2<int>((int)image.cameraInfo.opticalCenter.x, (int)image.cameraInfo.opticalCenter.y),cameraMatrix, image.cameraInfo, angles);
  Pose2D camera(angles.x, Vector2<double>(cameraMatrix.translation.x,
                                          cameraMatrix.translation.y));
  if(linesPercept.numberOfPoints[goal])
  {
    bool goalAtBorder2 = linesPercept.points[goal][linesPercept.numberOfPoints[goal] - 1].abs() < closestBottom,
         isFirst = true;
    Vector2<double> first = (Pose2D(linesPercept.points[goal][0]) - camera).translation,
                    last = first;
    for(int i = 1; i < linesPercept.numberOfPoints[goal]; ++i)
    {
      Vector2<double> point = (Pose2D(linesPercept.points[goal][i]) - camera).translation;
      if(fabs(point.y - last.y) > 150)
      {
        if(isFirst && goalAtBorder)
        {
          Vector2<double> point2 = (Pose2D(linesPercept.points[goal][linesPercept.numberOfPoints[goal] - 1]) - camera).translation;
          first = point2 + (first - point2) * 700 / (first - point2).abs();
        }
        double width = fabs(last.y - first.y);
        if(width > maxWidth)
        {
          maxWidth = width;
          addGoal(camera, first, last);
        }
        first = point;
        isFirst = false;
      }
      last = point;
    }
    if(!isFirst && goalAtBorder2)
    {
      Vector2<double> point2 = (Pose2D(linesPercept.points[goal][0]) - camera).translation;
      last = point2 + (last - point2) * 700 / (last - point2).abs();
    }
    double width = fabs(last.y - first.y);
    if(width > maxWidth)
      addGoal(camera, first, last);
  }
}

void GT2003ImageProcessor::addGoal(const Pose2D& camera, 
                                   const Vector2<double>& first, const Vector2<double>& last)
{
  bool firstIsClose = false,
       lastIsClose = false;
  for(int i = 0; i < playersPercept.numberOfRedPlayers + playersPercept.numberOfBluePlayers; ++i)
  {
    SinglePlayerPercept& p = i < playersPercept.numberOfRedPlayers 
                      ? playersPercept.redPlayers[i]
                      : playersPercept.bluePlayers[i - playersPercept.numberOfRedPlayers];
    Vector2<double> robot = (Pose2D(p.offset) - camera).translation;
    if(robot.x > 0 && fabs(robot.y - first.y) < 200)
      firstIsClose = true;
    if(robot.x > 0 && fabs(robot.y - last.y) < 200)
      lastIsClose = true;
  }

  Vector2<double> first2 = (camera + Pose2D(first)).translation,
                  last2 = (camera + Pose2D(last)).translation;
  double firstDir = atan2(first2.y, first2.x),
         lastDir = atan2(last2.y, last2.x),
         factor;
  if(firstIsClose == lastIsClose)
    factor = 0.5;
  else if(firstIsClose)
    factor = 0.7;
  else
    factor = 0.3;

  obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::opponentGoal] = true;
  obstaclesPercept.angleToFreePartOfGoal[ObstaclesPercept::opponentGoal] = firstDir + factor * (lastDir - firstDir);
  double distance = ((first2 + last2) / 2).abs();
  obstaclesPercept.distanceToFreePartOfGoal[ObstaclesPercept::opponentGoal] = distance;
  obstaclesPercept.widthOfFreePartOfGoal[ObstaclesPercept::opponentGoal] = distance ? 2 * atan((first - last).abs() / 2 / distance) : pi;
}

void GT2003ImageProcessor::clusterFlags(Flag::FlagType flagType, const Vector2<int>& point)
{
  if((point - lastFlag).abs() > 9)
  {
    if(lastFlag.x >= 0)
    {
      Flag::FlagType bestType = Flag::numberOfFlagTypes;
      int maxCount = 0;
      bool ambiguous = true;
      for(int i = 0; i < 6; ++i)
        if(flagCount[i] > maxCount)
        {
          bestType = Flag::FlagType(i);
          maxCount = flagCount[i];
          ambiguous = false;
        }
        else if(flagCount[i] == maxCount)
          ambiguous = true;
      if(!ambiguous)
      {
        Vector2<int> center = (firstFlag + lastFlag) / 2;
        static const colorClass otherColor[6] = {yellow, green, skyblue, yellow, green, skyblue};
        flagSpecialist.searchFlags(
          image, colorTable, cameraMatrix, otherColor[bestType], 
          bestType <= Flag::pinkAboveSkyblue,
          horizon, center.x, center.y);
      }
    }
    for(int i = 0; i < 6; ++i)
      flagCount[i] = 0;
    firstFlag = point;
  }
  lastFlag = point;
  if(flagType != Flag::numberOfFlagTypes)
    ++flagCount[flagType];
}

void GT2003ImageProcessor::filterPercepts()
{
  // Close robots produce far ghost robots. Remove them.
  int i;
  for(i = 0; i < playersPercept.numberOfRedPlayers + playersPercept.numberOfBluePlayers; ++i)
  {
    SinglePlayerPercept& pi = i < playersPercept.numberOfRedPlayers 
                      ? playersPercept.redPlayers[i]
                      : playersPercept.bluePlayers[i - playersPercept.numberOfRedPlayers];
    double iDist = pi.offset.abs(),
           iAngle = atan2(pi.offset.y, pi.offset.x),
           ignoreAngle = atan2(150, iDist);
    for(int j = 0; j < playersPercept.numberOfRedPlayers + playersPercept.numberOfBluePlayers; ++j)
      if(i != j)
      {
        SinglePlayerPercept& pj = j < playersPercept.numberOfRedPlayers 
                            ? playersPercept.redPlayers[j]
                            : playersPercept.bluePlayers[j - playersPercept.numberOfRedPlayers];
        if(iDist < pj.offset.abs() &&
           fabs(atan2(pj.offset.y, pj.offset.x) - iAngle) < ignoreAngle)
        {
          if(j < playersPercept.numberOfRedPlayers)
            pj = playersPercept.redPlayers[--playersPercept.numberOfRedPlayers];
          else
            pj = playersPercept.bluePlayers[--playersPercept.numberOfBluePlayers];
          if(i > j)
          {
            i = j; // check pi again
            break;
          }
          else
            --j; // check pj again
        }
      }
  }

  // If there are too few line points of a certain type, remove them all, they may be misreadings
  for(i = 0; i < LinesPercept::numberOfLineTypes; ++i)
    if(linesPercept.numberOfPoints[i] < 3)
      linesPercept.numberOfPoints[i] = 0;

  // if many more field line points than border points have been found,
  // it is very likely that the border points also result from field lines
  // So remove them.
  if(linesPercept.numberOfPoints[LinesPercept::field] > 
     3 * linesPercept.numberOfPoints[LinesPercept::border])
    linesPercept.numberOfPoints[LinesPercept::border] = 0;
}

Vector2<int> GT2003ImageProcessor::getCoords(const unsigned char* p) const
{
  int diff = p - &image.image[0][0][0];
  return Vector2<int>(diff % cameraResolutionWidth_ERS7,diff / (cameraResolutionWidth_ERS7 * 6));
}

void GT2003ImageProcessor::plot(const unsigned char* p,Drawings::Color color)
{
  if(lineColor == Drawings::numberOfColors)
  {
    last = p;
    lineColor = color;
  }
  else if(color != lineColor)
  {
    Vector2<int> p1 = getCoords(last),
                 p2 = getCoords(p);
    LINE(imageProcessor_general,p1.x,p1.y,p2.x,p2.y,0,Drawings::ps_solid,lineColor);
    last = p;
    lineColor = color;
  }
}

bool GT2003ImageProcessor::handleMessage(InMessage& message)
{
  switch(message.getMessageID())
  {
    case idLinesImageProcessorParameters:
      GenericDebugData d;
      message.bin >> d;
      yThreshold = (int)d.data[1];
      vThreshold = (int)d.data[2];
      execute();
      return true;
  }
  return false;
}

/*
* Change log :
* 
* $Log: GT2003ImageProcessor.cpp,v $
* Revision 1.22  2004/05/07 15:16:24  nistico
* All geometry calculations are making use of intrinsic functions.
* I updated most of the images processors to make correct use of this.
* Please test if there are any problems, because i'm going to remove the
* old code soon.
*
* Revision 1.21  2004/04/18 11:58:24  nistico
* Removed CT32K_LAYOUT
*
* Revision 1.20  2004/04/07 12:28:59  risler
* ddd checkin after go04 - first part
*
* Revision 1.2  2004/03/29 15:19:52  Marc
* Intruduced the Black and White Image
* Normal Images (not Jpeg) images were now send as Color Image with BW
*
* The image has now 6 planes !
*
* Revision 1.1.1.1  2004/03/29 08:28:47  Administrator
* initial transfer from tamara
*
* Revision 1.19  2004/03/12 17:22:26  nistico
* - Fixed a problem in CT32K_LAYOUT with localization
*
* Revision 1.18  2004/03/08 01:38:59  roefer
* Interfaces should be const
*
* Revision 1.17  2004/03/04 17:51:17  nistico
* -Bug with MSH2004ColorCorrection fixed
*
* Revision 1.16  2004/03/03 17:18:41  nistico
* -Introduced fast ColorTable32K access through conditional compilation
* For Dortmund: uncomment (define) CT32K_LAYOUT in GT2003ImageProcessor.h
* For other universities: leave undefined :-)
*
* Revision 1.15  2004/03/02 00:26:21  roefer
* Warnings removed
*
* Revision 1.14  2004/03/01 11:56:44  juengel
* Recognition of obstacle types added.
*
* Revision 1.13  2003/12/31 23:50:36  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.12  2003/12/30 20:12:02  roefer
* Image size is now 208 x 160. Smaller images are placed in the upper left corner
*
* Revision 1.11  2003/12/26 13:15:29  roefer
* calculation of V corrected
*
* Revision 1.10  2003/12/16 13:44:19  roefer
* cameraInfo used for offset calculation
*
* Revision 1.9  2003/12/15 11:47:04  juengel
* Introduced CameraInfo
*
* Revision 1.8  2003/12/01 16:26:12  juengel
* Added calculation of "pixel usage" goal pixels are not counted yet.
*
* Revision 1.7  2003/11/15 17:09:02  juengel
* changed BallPercept
*
* Revision 1.6  2003/11/12 16:19:35  goehring
* frameNumber added to percepts
*
* Revision 1.5  2003/11/05 16:43:59  juengel
* Added drawing imageProcessor_ball2
*
* Revision 1.4  2003/11/05 16:38:48  goehring
* FrameNumber added
*
* Revision 1.3  2003/10/31 11:46:06  juengel
* Replaced fixed image height and width by constant from RobotDimensions.h
*
* Revision 1.2  2003/10/12 20:20:15  juengel
* ColorTable is not const anymore.
* Added manual calibration.
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.10  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.9  2003/09/26 11:38:51  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.8  2003/09/01 10:05:08  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.7  2003/08/30 10:16:42  juengel
* DebugDrawings clean-up 1
*
* Revision 1.6  2003/08/18 12:07:49  juengel
* Moved method calculateLineSize to class Geometry.
*
* Revision 1.5  2003/07/29 12:48:35  juengel
* Moved calculateHorizon, getDistanceBySize and getSizeByDistance to Geometry.
*
* Revision 1.4  2003/07/21 13:42:21  juengel
* Moved ColorTableAuto and GoalRecognizer to ImageProcessorTools.
*
* Revision 1.3  2003/07/03 14:01:41  roefer
* Ball recognition at lower image border improved
*
* Revision 1.2  2003/07/03 10:11:16  roefer
* Ball position clipped close to body
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.21  2003/06/28 08:08:06  juengel
* Moved horizon one pixel to the left.
*
* Revision 1.20  2003/06/27 11:18:12  juengel
* Distance to a ball above the horizon is calculated by size.
*
* Revision 1.19  2003/06/26 13:22:39  mkunz
* extended the shorter scanlines
*
* Revision 1.18  2003/06/25 18:44:13  juengel
* Added nearPoints, farPoints[maxNumberOfPoints] and farPointIsOnImageBorder to ObstaclesPercept.
*
* Revision 1.17  2003/06/24 08:16:32  dueffert
* calculation optimized
*
* Revision 1.16  2003/06/22 17:09:08  juengel
* Added filter for ObstaclesPercept.
*
* Revision 1.15  2003/06/21 12:51:52  juengel
* ObstaclesPercept is calculated when cameraMatrix.isValid only.
*
* Revision 1.14  2003/06/20 21:12:45  roefer
* If camera matrix is invalid, ball distance is calculated from size
*
* Revision 1.13  2003/06/18 13:59:18  juengel
* Improved perception of obstacles.
*
* Revision 1.12  2003/06/17 19:43:31  juengel
* Orange and green are "ground color" for ObstaclesPercept.
*
* Revision 1.11  2003/06/13 14:56:09  juengel
* Points in ObstaclesPercept are filled.
*
* Revision 1.10  2003/06/12 16:57:59  juengel
* Added GoalRecognizer, removed GoalSpecialist.
*
* Revision 1.9  2003/05/26 20:21:32  roefer
* widthOfFreePartOfGoal is a angle now
*
* Revision 1.8  2003/05/26 08:20:10  juengel
* Moved angleToFreePartOfGoal from specialPercept to obstaclesPercept.
*
* Revision 1.7  2003/05/07 16:07:50  roefer
* Major ball recognition bug fixed
*
* Revision 1.6  2003/05/05 14:53:29  dueffert
* warnings removed
*
* Revision 1.5  2003/05/04 21:48:46  roefer
* Filtering ball point outliers
*
* Revision 1.4  2003/05/04 17:31:52  roefer
* Flag and goal specialists added to GT2003IP
*
* Revision 1.3  2003/05/03 18:09:55  roefer
* Ball sampling only for small balls
*
* Revision 1.2  2003/05/01 23:33:56  roefer
* Anchors for flag specialist added
*
* Revision 1.1  2003/05/01 13:24:24  roefer
* Copied LinesImageProcessor2 to GT2003ImageProcessor
*
*
*/
