/**
* @file GT2003FlagSpecialist.cpp
*
* This file contains a class for Image Processing.
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
*/

#include "GT2003FlagSpecialist.h"

#include "Tools/FieldDimensions.h"
#include "Tools/Player.h"
#include "Tools/Math/Common.h"

GT2003FlagSpecialist::GT2003FlagSpecialist()
{
}

void GT2003FlagSpecialist::init(const Image& image)
{
  for(int i = 0; i < 6; i++)
  {
    numberOfBoundingBoxes[i] = 0;
  }
  INIT_DEBUG_IMAGE(imageProcessorFlags, image);
}

void GT2003FlagSpecialist::searchFlags
(
 const Image& image, 
 const ColorTable& colorTable,
 const CameraMatrix& cameraMatrix,
 colorClass color,
 bool pinkIsTop,
 const Geometry::Line horizonLine,
 int x, int y
 )
{
  bool gridPointIsCovered = false;
  
  int i, j;
  
  for(i = 0; i < 6; i++)
  {
    for(j = 0; j < numberOfBoundingBoxes[i]; j++)
    {
      if(x >= boundingBoxLeft[j][i] && x <= boundingBoxRight[j][i] &&
        y >= boundingBoxBottom[j][i] && y <= boundingBoxTop[j][i] )
      {
        gridPointIsCovered = true;
      }
    }// j
  }//i

  if(gridPointIsCovered) return;
  
  start.x = x; start.y = y;
  
  //find up
  findEndOfFlag(image, colorTable, start, horizonLine.direction, 
    up, north, color, pinkIsTop, pinkIsTop);
  //find right
  findEndOfFlag(image, colorTable, start, horizonLine.direction, 
    right, east, color, pinkIsTop, pinkIsTop);
  //find down
  findEndOfFlag(image, colorTable, start, horizonLine.direction, 
    down, south, color, pinkIsTop, pinkIsTop);
  //find left
  findEndOfFlag(image, colorTable, start, horizonLine.direction, 
    left, west, color, pinkIsTop, pinkIsTop);
  
  //go to center of pink
  start.x = (west.x + east.x) / 2;
  start.y = (west.y + east.y) / 2;
  //find north
  findEndOfFlag(image, colorTable, start, horizonLine.direction, 
    up, north, color, pinkIsTop, pinkIsTop);
  //find south
  findEndOfFlag(image, colorTable, start, horizonLine.direction, 
    down, south, color, pinkIsTop, pinkIsTop);
  
  
  double distance = sqrt((double)sqr(south.x-north.x) + sqr(south.y-north.y));
  int numberOfHorizontalScans = 5;
  //scan horizontal lines
  Vector2<double>step;
  step.x = (south.x - north.x) / (numberOfHorizontalScans + 1.0);
  step.y = (south.y - north.y) / (numberOfHorizontalScans + 1.0);
  
  Geometry::Line leftVerticalLine;
  leftVerticalLine.base = horizonLine.base;
  leftVerticalLine.direction.x = -horizonLine.direction.y;
  leftVerticalLine.direction.y = horizonLine.direction.x;
  
  double leftMin = 1000, rightMax = -1000;
  bool leftValid = true, rightValid = true;

  bool pointIsGood;
  bool startIsInTop = true;
  for(i = 1; i <= numberOfHorizontalScans; i++)
  {
    if(i==(numberOfHorizontalScans + 1)/2) 
    {
      i++; //no scan in the middle of the flag
      startIsInTop = false;
    }
    start.x = (int)(north.x + step.x * i);
    start.y = (int)(north.y + step.y * i); 
    //find right
    pointIsGood = findEndOfFlag(image, colorTable, start, horizonLine.direction, 
      right, east, color, pinkIsTop, startIsInTop);
    double rightDistance = 0.5 + Geometry::getDistanceToLine(leftVerticalLine, Vector2<double>(east.x,east.y));
    if(rightDistance > rightMax)
    {
      rightMax = rightDistance;
      if(!pointIsGood)
        rightValid = false;
      else
        rightValid = true;
    }
    
    //find left
    pointIsGood = findEndOfFlag(image, colorTable, start, horizonLine.direction, 
      left, west, color, pinkIsTop, startIsInTop);
    double leftDistance = -0.5 + Geometry::getDistanceToLine(leftVerticalLine, Vector2<double>(west.x,west.y));
    if(leftDistance < leftMin)
    {
      leftMin = leftDistance;
      if(!pointIsGood)
        leftValid = false;
      else
        leftValid = true;
    }
  }
  
  bool topValid = true, bottomValid = true;
  distance = sqrt((double)sqr(east.x-west.x) + sqr(east.y-west.y));
  int numberOfVerticalScans = 3;
  //scan vertical lines
  step.x = (east.x - west.x) / (numberOfVerticalScans + 1.0);
  step.y = (east.y - west.y) / (numberOfVerticalScans + 1.0);
  
  double topMax = -1000, bottomMin = 1000;
  
  for(i = 1; i <= numberOfVerticalScans; i++)
  {
    //    if(i==(numberOfHorizontalScans + 1)/2) i++;
    start.x = (int)(west.x + step.x * i);
    start.y = (int)(west.y + step.y * i); 
    //find top
    pointIsGood = findEndOfFlag(image, colorTable, start, horizonLine.direction, 
      up, north, color, pinkIsTop, pinkIsTop);
    double topDistance = 0.5 + Geometry::getDistanceToLine(horizonLine, Vector2<double>(north.x,north.y));
    if(topDistance > topMax)
    {
      topMax = topDistance;
      if(!pointIsGood)
        topValid = false;
      else
        topValid = true;
    }
    //find bottom
    pointIsGood = findEndOfFlag(image, colorTable, start, horizonLine.direction, 
      down, south, color, pinkIsTop, pinkIsTop);
    double bottomDistance = -0.5 + Geometry::getDistanceToLine(horizonLine, Vector2<double>(south.x,south.y));
    if(bottomDistance < bottomMin)
    {
      bottomMin = bottomDistance;
      if(!pointIsGood)
        bottomValid = false;
      else
        bottomValid = true;
    }
  }
  Flag::FlagType flagType = Flag::pinkAboveYellow;

  if(pinkIsTop)
  {
    switch(color)
    {
    case yellow:
      flagType = Flag::pinkAboveYellow;
      break;
    case green:
      flagType = Flag::pinkAboveGreen;
      break;
    case skyblue:
      flagType = Flag::pinkAboveSkyblue;
      break;
    }
  }
  else
  {
    switch(color)
    {
    case yellow:
      flagType = Flag::yellowAbovePink;
      break;
    case green:
      flagType = Flag::greenAbovePink;
      break;
    case skyblue:
      flagType = Flag::skyblueAbovePink;
      break;
    }
  }

  boundingBoxLeft[numberOfBoundingBoxes[flagType]][flagType] = leftMin;
  boundingBoxRight[numberOfBoundingBoxes[flagType]][flagType] = rightMax;
  boundingBoxTop[numberOfBoundingBoxes[flagType]][flagType] = topMax;
  boundingBoxBottom[numberOfBoundingBoxes[flagType]][flagType] = bottomMin;

  boundingBoxLeftValid[numberOfBoundingBoxes[flagType]][flagType] = leftValid;
  boundingBoxRightValid[numberOfBoundingBoxes[flagType]][flagType] = rightValid;
  boundingBoxTopValid[numberOfBoundingBoxes[flagType]][flagType] = topValid;
  boundingBoxBottomValid[numberOfBoundingBoxes[flagType]][flagType] = bottomValid;
  
  numberOfBoundingBoxes[flagType]++;
  if(numberOfBoundingBoxes[flagType] >= maxNumberOfBoundingBoxes)
  {
    numberOfBoundingBoxes[flagType] = maxNumberOfBoundingBoxes - 1;
  }
 }

bool GT2003FlagSpecialist::findEndOfFlag
(
 const Image& image,
 const ColorTable& colorTable,
 const Vector2<int> start,
 Vector2<double> horizonDirection,
 Direction directionToGo,
 Vector2<int>& destination,
 colorClass color,
 bool pinkIsTop,
 bool startIsInTop
 )
{
  int blackCounter = 0;
  colorClass currentColorClass;
  colorClass topColor = pink;
  colorClass bottomColor = pink;
  colorClass startColor;
  
  if(pinkIsTop) bottomColor = color; else topColor = color;
  if(startIsInTop) startColor = topColor; else startColor = bottomColor;
  
  destination = start;
  if(startColor == green) return false;

  Vector2<int> lastInsideFlag, lastDestination;
  lastInsideFlag = start;
  
  
  Vector2<double> direction;
  switch(directionToGo)
  {
  case up:
    direction.x = horizonDirection.y;
    direction.y = -horizonDirection.x;
    break;
  case right:
    direction = horizonDirection;
    break;
  case down:
    direction.x = -horizonDirection.y;
    direction.y = horizonDirection.x;
    break;
  case left:
    direction.x = -horizonDirection.x;
    direction.y = -horizonDirection.y;
    break;
  }
  enum {incX, decX, incY, decY} mode;
  if(direction.y < -fabs(direction.x)) mode = decY;
  else if(direction.y > fabs(direction.x)) mode = incY;
  else if(direction.x < -fabs(direction.y)) mode = decX;
  else mode = incX;
  
  
  Vector2<int> diff;
  
  bool goOn = true;
  while(goOn)
  {
    switch(mode)
    {
    case incX:
      diff.x++;
      diff.y = (int)(diff.x * direction.y / direction.x);
      break;
    case decX:
      diff.x--;
      diff.y = (int)(diff.x * direction.y / direction.x);
      break;
    case incY:
      diff.y++;
      diff.x = (int)(diff.y * direction.x / direction.y);
      break;
    case decY:
      diff.y--;
      diff.x = (int)(diff.y * direction.x / direction.y);
      break;
    }
    lastDestination = destination;
    destination = start + diff;
    
    DEBUG_IMAGE_SET_PIXEL_Y(imageProcessorFlags, destination.x, destination.y, 180)
    
      if(destination.x < 1 || destination.x >= image.cameraInfo.resolutionWidth - 1 ||
        destination.y < 1 || destination.y >= image.cameraInfo.resolutionHeight - 2)
    {
      goOn = false;
      destination = lastInsideFlag;
      return false;
    }
    else
    {
      currentColorClass = colorTable.getColorClass(
        image.image[destination.y][0][destination.x],
        image.image[destination.y][1][destination.x],
        image.image[destination.y][2][destination.x]);
      
      if(directionToGo == left || directionToGo == right)
      {
        if(currentColorClass == startColor)
          lastInsideFlag = destination;
        else blackCounter++;
      }
      
      else if(directionToGo == up)
      {
        if(currentColorClass == topColor)
        {
          lastInsideFlag = destination;
        }
        else if(currentColorClass != bottomColor) blackCounter++;
      }
      else // down
      {
        if(currentColorClass == bottomColor)
        {
          lastInsideFlag = destination;
        }
        else if(currentColorClass != topColor) blackCounter++;
      }

      if(blackCounter > 10)
      {
        goOn = false;
        destination = lastInsideFlag;
      }
    }// if inside image
  }//while goOn
  
  DEBUG_IMAGE_SET_PIXEL_DARK_GREEN(imageProcessorFlags, destination.x, destination.y)
  return true;
}

void GT2003FlagSpecialist::getFlagPercept
(
 const CameraMatrix& cameraMatrix, 
 const CameraInfo& cameraInfo, 
 const Geometry::Line horizonLine,
 LandmarksPercept& landmarksPercept
 )
{
  int flip = getPlayer().getTeamColor() == Player::blue ? -1 : 1;

  Vector2<double> verticalDirection;
  verticalDirection.x = -horizonLine.direction.y;
  verticalDirection.y = horizonLine.direction.x;

  //~ double factor = cameraInfo.resolutionWidth / 2 / tan(cameraInfo.openingAngleWidth / 2);
  double factor = cameraInfo.focalLength;
  for(int flagType = 0; flagType < 6; flagType++)
  {
    // find best bounding box
    for(int i = 0; i < numberOfBoundingBoxes[flagType]; i++)
    {
    }
    bestBoundingBox[flagType] = 0;

    if(numberOfBoundingBoxes[flagType] > 0)
    {
      
    Vector2<double> right, left, top, bottom;

    right = horizonLine.base + horizonLine.direction * boundingBoxRight[bestBoundingBox[flagType]][flagType];
    left = horizonLine.base + horizonLine.direction * boundingBoxLeft[bestBoundingBox[flagType]][flagType];

    top = horizonLine.base - verticalDirection * boundingBoxTop[bestBoundingBox[flagType]][flagType];
    bottom = horizonLine.base - verticalDirection * boundingBoxBottom[bestBoundingBox[flagType]][flagType];

    //Vector2<double> rightTop, leftTop, rightBottom, leftBottom;
    right = horizonLine.base + 
      horizonLine.direction * boundingBoxRight[bestBoundingBox[flagType]][flagType]
      - verticalDirection * 
      (boundingBoxTop[bestBoundingBox[flagType]][flagType] + 
       boundingBoxBottom[bestBoundingBox[flagType]][flagType] 
      ) / 2;

    left = horizonLine.base + 
      horizonLine.direction * boundingBoxLeft[bestBoundingBox[flagType]][flagType]
      - verticalDirection * 
      (boundingBoxTop[bestBoundingBox[flagType]][flagType] + 
       boundingBoxBottom[bestBoundingBox[flagType]][flagType] 
      ) / 2;

    top = horizonLine.base 
      - verticalDirection * boundingBoxTop[bestBoundingBox[flagType]][flagType]
      + horizonLine.direction * 
      (boundingBoxLeft[bestBoundingBox[flagType]][flagType] + 
       boundingBoxRight[bestBoundingBox[flagType]][flagType] 
      ) / 2;

    bottom = horizonLine.base 
      - verticalDirection * boundingBoxBottom[bestBoundingBox[flagType]][flagType]
      + horizonLine.direction * 
      (boundingBoxLeft[bestBoundingBox[flagType]][flagType] + 
       boundingBoxRight[bestBoundingBox[flagType]][flagType] 
      ) / 2;


    Vector3<double> vectorToLeft(factor,
                                                  cameraInfo.opticalCenter.x - left.x,
                                                  cameraInfo.opticalCenter.y - left.y);
    Vector3<double> vectorToRight(factor,
                                                  cameraInfo.opticalCenter.x - right.x,
                                                  cameraInfo.opticalCenter.y - right.y);
    Vector3<double> vectorToTop(factor,
                                                  cameraInfo.opticalCenter.x - top.x,
                                                  cameraInfo.opticalCenter.y - top.y);
    Vector3<double> vectorToBottom(factor,
                                                  cameraInfo.opticalCenter.x - bottom.x,
                                                  cameraInfo.opticalCenter.y - bottom.y);

    Vector3<double>
      vectorToLeftWorld = cameraMatrix.rotation * vectorToLeft,
    vectorToRightWorld = cameraMatrix.rotation * vectorToRight,
    vectorToTopWorld = cameraMatrix.rotation * vectorToTop,
    vectorToBottomWorld = cameraMatrix.rotation * vectorToBottom;
       
    double 
      leftAngle = atan2(vectorToLeftWorld.y,vectorToLeftWorld.x),
      rightAngle = atan2(vectorToRightWorld.y,vectorToRightWorld.x),
      topAngle = atan2(vectorToTopWorld.z,sqrt((double)sqr(vectorToTopWorld.x) + sqr(vectorToTopWorld.y)) ),
      bottomAngle = atan2(vectorToBottomWorld.z,sqrt((double)sqr(vectorToBottomWorld.x) + sqr(vectorToBottomWorld.y)) );


    Vector2<double>flagPosition;
    
    switch (flagType)
    {
    case Flag::pinkAboveYellow:
      flagPosition.x = xPosBackFlags * flip;
      flagPosition.y = yPosRightFlags * flip;
      break;
    case Flag::pinkAboveGreen:
      flagPosition.x = xPosHalfWayLine * flip;
      flagPosition.y = yPosRightFlags * flip;
      break;
    case Flag::pinkAboveSkyblue:
      flagPosition.x = xPosFrontFlags * flip;
      flagPosition.y = yPosRightFlags * flip;
      break;
    case Flag::yellowAbovePink:
      flagPosition.x = xPosBackFlags * flip;
      flagPosition.y = yPosLeftFlags * flip;
      break;
    case Flag::greenAbovePink:
      flagPosition.x = xPosHalfWayLine  * flip;
      flagPosition.y = yPosLeftFlags * flip;
      break;
    case Flag::skyblueAbovePink:
      flagPosition.x = xPosFrontFlags * flip;
      flagPosition.y = yPosLeftFlags * flip;
      break;
    }

    
    ConditionalBoundary boundary;

    boundary.addX(leftAngle,!boundingBoxLeftValid[bestBoundingBox[flagType]][flagType]);
    boundary.addX(rightAngle,!boundingBoxRightValid[bestBoundingBox[flagType]][flagType]);
    boundary.addY(topAngle,!boundingBoxTopValid[bestBoundingBox[flagType]][flagType]);
    boundary.addY(bottomAngle,!boundingBoxBottomValid[bestBoundingBox[flagType]][flagType]);
    landmarksPercept.addFlag((Flag::FlagType)flagType, flagPosition, boundary);
    } //if(numberOfBoundingBoxes[flagType]) > 0)
  }
  Vector2<double> cameraOffset(cameraMatrix.translation.x,
                                              cameraMatrix.translation.y);

  estimateOffsetForFlags(landmarksPercept, cameraOffset); 
  SEND_DEBUG_IMAGE(imageProcessorFlags);
}

void GT2003FlagSpecialist::estimateOffsetForFlags
(
 LandmarksPercept& landmarksPercept,
 const Vector2<double>& cameraOffset
 )
{
  for(int i = 0;i < landmarksPercept.numberOfFlags; ++i)
  {
    Flag& flag = landmarksPercept.flags[i];

    /** @todo improve, isOnBorder(flag.x.?) not checked */
    double distance;
    double direction = flag.x.getCenter();

    if(!flag.isOnBorder(flag.y.min) && !flag.isOnBorder(flag.y.max) && flag.type != Flag::greenAbovePink)
    {
      if(flag.y.min != flag.y.max)
      {
        distance = flagHeight / (tan(flag.y.max) - tan(flag.y.min)) + flagRadius;
        flag.distanceValidity = 0.8;
      }
      else
      {
        distance = 4500;
        flag.distanceValidity = 0.05;
      }
    }
    else
    {
      distance = flagRadius / sin(flag.x.getSize() / 2);
      if(!flag.isOnBorder(flag.x.min) && !flag.isOnBorder(flag.x.max)) // Flag touches no vertical border
        flag.distanceValidity = 0.7;
      else
        flag.distanceValidity = 0.2;
    }

    if(!flag.isOnBorder(flag.x.min) && !flag.isOnBorder(flag.x.max)) // Flag touches no vertical border
      flag.angleValidity = 0.8;
    else
      flag.angleValidity = 0.7;
    
    Pose2D p = Pose2D(cameraOffset) + Pose2D(direction) 
                       + Pose2D(Vector2<double>(distance,0));
//    flag.distance = p.translation.abs();
//    flag.angle = atan2(p.translation.y,p.translation.x);

      flag.distance = p.translation.abs();
      flag.angle = direction;
    
       
    if (flag.distance > 6000)
    {
      flag.distance = 6000;
      flag.distanceValidity=0; // flags far away are meassured very bad
    }
    else if (flag.distance > 3000) 
      flag.distanceValidity*=0.5; // flags medium far away are meassured medium bad
  }
}


/*
* Change log :
* 
* $Log: GT2003FlagSpecialist.cpp,v $
* Revision 1.5  2004/05/07 15:16:24  nistico
* All geometry calculations are making use of intrinsic functions.
* I updated most of the images processors to make correct use of this.
* Please test if there are any problems, because i'm going to remove the
* old code soon.
*
* Revision 1.4  2003/12/31 23:50:36  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.3  2003/12/15 11:47:04  juengel
* Introduced CameraInfo
*
* Revision 1.2  2003/10/31 11:46:06  juengel
* Replaced fixed image height and width by constant from RobotDimensions.h
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/09/26 11:38:51  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.3  2003/09/01 10:05:08  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.2  2003/07/29 12:44:56  juengel
* Removed getDistanceBySize
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.1  2003/05/04 17:31:52  roefer
* Flag and goal specialists added to GT2003IP
*
* Revision 1.14  2003/04/15 17:19:43  risler
* moved DDDGO2003 ImageProcessor to own module
* removed ContinuousBasicBehaviorTester
*
* Revision 1.12  2003/04/04 17:18:33  juengel
* Warnings removed.
*
* Revision 1.11  2003/03/07 11:17:58  juengel
* removed smoothed distance
*
* Revision 1.10  2003/03/06 11:40:27  dueffert
* optimized, missing case added implicitly, unused variable warning removed
*
* Revision 1.9  2003/02/19 14:59:55  roefer
* pColorTable -> colorTable
*
* Revision 1.8  2003/02/18 21:29:16  osterhues
* Changed all instances of ColorTable64 to new base class ColorTable
*
* Revision 1.7  2002/11/28 14:20:13  dueffert
* docu unified
*
* Revision 1.6  2002/11/12 23:00:47  dueffert
* started restore greenhills compatibility
*
* Revision 1.5  2002/09/24 14:19:59  juengel
* Changed calculation of flag.angle.
*
* Revision 1.4  2002/09/23 13:54:48  risler
* abs replaced by fabs/labs
*
* Revision 1.3  2002/09/22 18:40:51  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/19 23:38:53  juengel
* Changed debug image mechanisms.
*
* Revision 1.1  2002/09/10 15:36:15  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1  2002/09/07 13:36:57  loetzsch
* unified the vision modules into one module "ImageProcessor"
* - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
*   and PlayersPerceptor were are combined to the new solution
*   "BlobImageProcessor"
* - The GridImageProcessor and the SubPixelGradientCalculator became
*   a solution of "ImageProcessor"
*
* Revision 1.9  2002/09/03 16:01:41  juengel
* Some comments added.
*
* Revision 1.8  2002/08/22 16:20:25  mkunz
* validities corrected
*
* Revision 1.7  2002/08/21 18:42:22  mkunz
* validities adapted
*
* Revision 1.6  2002/07/23 13:33:39  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.5  2002/06/28 10:26:18  roefer
* OUTPUT is possible in constructors
*
* Revision 1.4  2002/06/22 06:31:10  Thomas Rfer
* greenAbovePink only used width for distance
*
* Revision 1.3  2002/06/08 16:23:27  Thomas Rfer
* Buffer overruns removed
*
* Revision 1.2  2002/06/01 10:14:43  juengel
* Improved recognition of flags close to the border of the image.
*
* Revision 1.1.1.1  2002/05/10 12:40:14  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.12  2002/05/06 17:53:31  juengel
* Removed memory access bug.
*
* Revision 1.11  2002/04/25 14:50:36  kallnik
* changed double/float to double
* added several #include GTMath
*
* PLEASE use double
*
* Revision 1.10  2002/04/16 16:55:45  dueffert
* no message
*
* Revision 1.4  2002/04/12 00:21:02  juengel
* Flags swapped.
*
* Revision 1.3  2002/04/11 15:58:06  juengel
* no message
*
* Revision 1.2  2002/04/11 12:02:58  juengel
* no message
*
* Revision 1.9  2002/04/08 19:49:54  juengel
* goals added.
*
* Revision 1.8  2002/04/06 11:47:32  roefer
* ConditionalBoundary corrected
*
* Revision 1.7  2002/04/05 18:30:47  juengel
* no message
*
* Revision 1.6  2002/04/05 13:42:03  juengel
* Step by step ...
*
* Revision 1.5  2002/04/04 18:43:58  juengel
* GT2003FlagSpecialist improved.
*
* Revision 1.4  2002/04/03 14:18:52  juengel
* camera matrix and odometry data added to streaming operator for images
*
* Revision 1.3  2002/04/03 13:25:41  risler
* changed doubles to double/Angle, tan to tan
*
* Revision 1.2  2002/04/03 12:50:20  juengel
* Flags and ball.
*
* Revision 1.1  2002/04/02 10:30:34  juengel
* GridImageProcessor enhanced.
*
*/
