/**
* @file DDD2004ImageProcessor.cpp
*
* Implementation of class DDD2004ImageProcessor
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#include "DDD2004ImageProcessor.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "Tools/FieldDimensions.h"
#include "Tools/RingBuffer.h"
#include "DDD2004ImageProcessorTools.h"

#define PLOT(p,c) COMPLEX_DRAWING(imageProcessor_general,plot(p,c))

const int Y = 0,
          U = cameraResolutionWidth_ERS7, /**< Relative offset of U component. */
          V = 2 * cameraResolutionWidth_ERS7; /**< Relative offset of V component. */

DDD2004ImageProcessor::DDD2004ImageProcessor(const ImageProcessorInterfaces& interfaces)
: ImageProcessor(interfaces),
manualCalibration(image, cameraMatrix, (ColorTable64&)colorTable, calibrationRequest),
flagSpecialist(colorCorrector),
goalRecognizer(image, cameraMatrix, colorTable, colorCorrector),
ballSpecialist(colorCorrector)

{
  yThreshold = 15;
  vThreshold = 8;
}

void DDD2004ImageProcessor::execute()
{
  INIT_DEBUG_IMAGE(imageProcessorPlayers, image);

  //~ xFactor = image.cameraInfo.resolutionWidth / tan(image.cameraInfo.openingAngleWidth / 2.0) / 2.0;
  //~ yFactor = image.cameraInfo.resolutionHeight / tan(image.cameraInfo.openingAngleHeight / 2.0) / 2.0;
  xFactor = yFactor = image.cameraInfo.focalLength;

  numberOfScannedPixels = 0;
  // reset everything
  landmarksPercept.reset(image.frameNumber);
  linesPercept.reset(image.frameNumber);
  ballPercept.reset(image.frameNumber);
  playersPercept.reset(image.frameNumber);
  obstaclesPercept.reset(image.frameNumber);
  flagSpecialist.init(image);

  // variations of the camera matrix for different object heights
  cmTricot = cameraMatrix;
  cmTricot.translation.z -= 100; // upper tricot border

  longestBallRun = 0;

  Vector3<double> vectorInDirectionOfViewCamera(1,0,0);
  Vector3<double> vectorInDirectionOfViewWorld;
  vectorInDirectionOfViewWorld = cameraMatrix.rotation * vectorInDirectionOfViewCamera;

  angleBetweenDirectionOfViewAndGround = 
    toDegrees(
    atan2(
    vectorInDirectionOfViewWorld.z,
    sqrt(sqr(vectorInDirectionOfViewWorld.x) + sqr(vectorInDirectionOfViewWorld.y))
    )
    );

//  OUTPUT(idText,text,"W:" << angleBetweenDirectionOfViewAndGround);

  horizon = Geometry::calculateHorizon(cameraMatrix, image.cameraInfo);

  LINE(imageProcessor_horizon,
    (int)(horizon.base.x - 120 * horizon.direction.x), (int)(horizon.base.y - 120 * horizon.direction.y), 
    (int)(horizon.base.x + 120 * horizon.direction.x), (int)(horizon.base.y + 120 * horizon.direction.y), 
    1, Drawings::ps_solid, Drawings::white );
  DEBUG_DRAWING_FINISHED(imageProcessor_horizon);
  
//  obstaclesNearPointOnField2.x = 0; obstaclesNearPointOnField2.y = 0;
//  obstaclesFarPointOnField2IsOnImageBorder = false;

//  obstaclesFarPointOnField1.x = 0; obstaclesFarPointOnField1.y = 0;
//  obstaclesFarPointOnField2.x = 0; obstaclesFarPointOnField2.y = 0;
  scanColumns();
  
/*  if(Geometry::distance(obstaclesFarPointOnField1, obstaclesFarPointOnField2) < 400)
  {
    obstaclesPercept.add(obstaclesNearPointOnField2, obstaclesFarPointOnField2, obstaclesFarPointOnField2IsOnImageBorder);
  }
*/
/*
  if(obstaclesPercept.numberOfPoints < 3)
  {
    obstaclesPercept.numberOfPoints = 0;
  }
*/
  scanRows();

  if (longestBallRun > 2)
    ballSpecialist.searchBall(image, colorTable, cameraMatrix,
      ballCandidate.x, ballCandidate.y, ballPercept);

  filterPercepts();

  landmarksPercept.cameraOffset = cameraMatrix.translation;
  flagSpecialist.getFlagPercept(cameraMatrix, image.cameraInfo, horizon, landmarksPercept);

  goalRecognizer.getGoalPercept(landmarksPercept, obstaclesPercept);

  INFO(printPixelUsage, idText, text, "used pixels: " << 100 * numberOfScannedPixels / (image.cameraInfo.resolutionWidth * image.cameraInfo.resolutionHeight) << " %");

  SEND_DEBUG_IMAGE(imageProcessorPlayers);
  DEBUG_DRAWING_FINISHED(imageProcessor_scanLines);
  DEBUG_DRAWING_FINISHED(imageProcessor_general);

  GENERATE_DEBUG_IMAGE(imageProcessorGeneral,
    Image correctedImage;
    colorCorrector.generateCorrectedImage(image, correctedImage);
    INIT_DEBUG_IMAGE(imageProcessorGeneral, correctedImage);)
  SEND_DEBUG_IMAGE(imageProcessorGeneral);

  GENERATE_DEBUG_IMAGE(segmentedImage1,
    Image correctedImage;
    colorCorrector.generateCorrectedImage(image, correctedImage);
    colorTable.generateColorClassImage(correctedImage, segmentedImage1ColorClassImage);  );
  SEND_DEBUG_COLOR_CLASS_IMAGE(segmentedImage1);

  GENERATE_DEBUG_IMAGE(imageProcessorBall,
    Image correctedImage;
  colorCorrector.generateCorrectedImage(image, correctedImage);
  for (int iPBix=0; iPBix<correctedImage.cameraInfo.resolutionWidth; iPBix++) {
		  for (int iPBiy=0; iPBiy<correctedImage.cameraInfo.resolutionHeight; iPBiy++) {
        correctedImage.image[iPBiy][0][iPBix]=ballSpecialist.getSimilarityToOrange(correctedImage.image[iPBiy][0][iPBix],correctedImage.image[iPBiy][1][iPBix],correctedImage.image[iPBiy][2][iPBix]);
        if (correctedImage.image[iPBiy][0][iPBix]==0)
        {
          correctedImage.image[iPBiy][1][iPBix]=127;
          correctedImage.image[iPBiy][2][iPBix]=127;
        }
        else
        {
          if (correctedImage.image[iPBiy][0][iPBix]>30)
            correctedImage.image[iPBiy][1][iPBix]=0;
          else
            correctedImage.image[iPBiy][1][iPBix]=255;
          if (correctedImage.image[iPBiy][0][iPBix]>60)
            correctedImage.image[iPBiy][2][iPBix]=0;
          else
            correctedImage.image[iPBiy][2][iPBix]=255;
        }
      }
  }
  INIT_DEBUG_IMAGE(imageProcessorBall, correctedImage);)
  SEND_DEBUG_IMAGE(imageProcessorBall);
/*
#ifdef _WIN32
#ifndef SIMROBOT

  for (int iPBix=0; iPBix<imageProcessorBallImage.cameraInfo.resolutionWidth; iPBix++) {
		for (int iPBiy=0; iPBiy<imageProcessorBallImage.cameraInfo.resolutionHeight; iPBiy++) {
			imageProcessorBallImage.image[iPBiy][0][iPBix]=ballSpecialist.getSimilarityToOrange(imageProcessorBallImage.image[iPBiy][0][iPBix],imageProcessorBallImage.image[iPBiy][1][iPBix],imageProcessorBallImage.image[iPBiy][2][iPBix]);
			imageProcessorBallImage.image[iPBiy][1][iPBix]=127;
			imageProcessorBallImage.image[iPBiy][2][iPBix]=127;
		}
	}
#endif
#endif
*/
}

void DDD2004ImageProcessor::scanColumns()
{
  const int scanLineSpacing = 3;

  Geometry::Line scanLine;
  Vector2<double> intersection;
  Vector2<int> topPoint,
               bottomPoint,
               topBorderPoint,
               bottomBorderPoint;
  double bottomBorderDist, topBorderDist;
  int i;
  bool noLines = false,
       skipLine;

  // Reset horizontal counters
  noRedCount = noBlueCount = 100;
  closestBottom = closestBottom = 40000;
  goalAtBorder = goalAtBorder = false;

  lastFlag = Vector2<int>(-100,0);
  for(i = 0; i < 6; ++i)
    flagCount[i] = 0;

  // scan lines are perpendicular to horizon
  scanLine.direction.x = -horizon.direction.y;
  scanLine.direction.y = horizon.direction.x;

  // calculate and scan lines
  for(int dir = -1; dir <= 1; dir += 2)
    for(i = 0; true; i++)
    {
      if (dir == 1 && i == 0) i = 1;
      scanLine.base.x = image.cameraInfo.resolutionWidth / 2 + horizon.direction.x * scanLineSpacing * dir * i;
      scanLine.base.y = image.cameraInfo.resolutionHeight / 2 + horizon.direction.y * scanLineSpacing * dir * i;

      //Does the scan line intersect with the image?
      if(!Geometry::getIntersectionPointsOfLineAndRectangle(
                                          Vector2<int>(0,0),
                                          Vector2<int>(image.cameraInfo.resolutionWidth - 1,image.cameraInfo.resolutionHeight - 1),
                                          scanLine, 
                                          topBorderPoint, bottomBorderPoint))
        break;

      Geometry::getIntersectionOfLines(horizon, scanLine, intersection);

      if (fabs(scanLine.direction.y) > 0.001)
      {
        topBorderDist = (topBorderPoint.y - intersection.y) / scanLine.direction.y;
        bottomBorderDist = (bottomBorderPoint.y - intersection.y) / scanLine.direction.y;
      } else {
        topBorderDist = (topBorderPoint.x - intersection.x) / scanLine.direction.x;
        bottomBorderDist = (bottomBorderPoint.x - intersection.x) / scanLine.direction.x;
      }

      skipLine = false;

      // upper boundary for scan lines
      if (topBorderDist > -50)
        topPoint = topBorderPoint;
      else if (bottomBorderDist < -50)
        skipLine = true;
      else
      {
        topPoint.x = (int)(intersection.x - scanLine.direction.x * 50);
        topPoint.y = (int)(intersection.y - scanLine.direction.y * 50);
        if (topPoint.x < 0) topPoint.x = 0;
        if (topPoint.x >= image.cameraInfo.resolutionWidth) topPoint.x = image.cameraInfo.resolutionWidth - 1;
        if (topPoint.y < 0) topPoint.y = 0;
        if (topPoint.y >= image.cameraInfo.resolutionHeight) topPoint.y = image.cameraInfo.resolutionHeight - 1;
      }
        
      // lower boundaries for scan lines
      switch (i & 3)
      {
        case 1:
        case 3:
          if (bottomBorderDist < 30)
            bottomPoint = bottomBorderPoint;
          else if (topBorderDist > 30)
            skipLine = true;
          else
          {
            bottomPoint.x = (int)(intersection.x + scanLine.direction.x * 30);
            bottomPoint.y = (int)(intersection.y + scanLine.direction.y * 30);
            if (bottomPoint.x < 0) bottomPoint.x = 0;
            if (bottomPoint.x >= image.cameraInfo.resolutionWidth) bottomPoint.x = image.cameraInfo.resolutionWidth - 1;
            if (bottomPoint.y < 0) bottomPoint.y = 0;
            if (bottomPoint.y >= image.cameraInfo.resolutionHeight) bottomPoint.y = image.cameraInfo.resolutionHeight - 1;
          }
          noLines = true;
          break;
        case 2:
          if (bottomBorderDist < 60)
            bottomPoint = bottomBorderPoint;
          else if (topBorderDist > 60)
            skipLine = true;
          else
          {
            bottomPoint.x = (int)(intersection.x + scanLine.direction.x * 60);
            bottomPoint.y = (int)(intersection.y + scanLine.direction.y * 60);
            if (bottomPoint.x < 0) bottomPoint.x = 0;
            if (bottomPoint.x >= image.cameraInfo.resolutionWidth) bottomPoint.x = image.cameraInfo.resolutionWidth - 1;
            if (bottomPoint.y < 0) bottomPoint.y = 0;
            if (bottomPoint.y >= image.cameraInfo.resolutionHeight) bottomPoint.y = image.cameraInfo.resolutionHeight - 1;
          }
          noLines = true;
          break;
        case 0:
        default:
          bottomPoint = bottomBorderPoint;
          noLines = false;
          break;
      }

      if (!skipLine)
        scan(topPoint, bottomPoint, true, noLines);
    }

  // finish clustering
  for(i = 0; i < 3; ++i)
  {
    clusterRobots(0,false,false);
  }
  clusterFlags(Flag::numberOfFlagTypes,Vector2<int>(1000,0));
}

void DDD2004ImageProcessor::scanRows()
{
  const int scanLineSpacing = 15,
            desiredNumberOfScanLines = 30;
  Geometry::Line scanLine = horizon;

  // start below horizon
  scanLine.base.x += -horizon.direction.y * 10;
  scanLine.base.y += horizon.direction.x * 10;

  for(int i = 1; i < desiredNumberOfScanLines; i++)
  {
    scanLine.base.x += -horizon.direction.y * scanLineSpacing;
    scanLine.base.y += horizon.direction.x * scanLineSpacing;
    
    Vector2<int> topPoint,
                 bottomPoint;
    //Does the scan line intersect with the image?
    if(Geometry::getIntersectionPointsOfLineAndRectangle(
                                        Vector2<int>(0,0),
                                        Vector2<int>(image.cameraInfo.resolutionWidth - 1,image.cameraInfo.resolutionHeight - 1),
                                        scanLine,
                                        topPoint, bottomPoint))
    {
      if(rand() > RAND_MAX / 2)
        scan(topPoint, bottomPoint, false, false);
      else
        scan(bottomPoint, topPoint, false, false);
    }
    else
      break;
  }
}

void DDD2004ImageProcessor::scan(const Vector2<int>& start, const Vector2<int>& end,
                                bool vertical, bool noLines)
{
  LinesPercept::LineType typeOfLinesPercept = LinesPercept::field;
  int pixelsSinceGoal = 0;
  int pixelsBetweenGoalAndObstacle = 0;
  
  Vector2<int> actual = start,
               diff = end - start,
               step(sgn(diff.x),sgn(diff.y)),
               size(abs(diff.x),abs(diff.y));
  bool isVertical = abs(diff.y) > abs(diff.x);
  int count,
      sum;

  LINE(imageProcessor_scanLines,
    start.x, start.y, end.x, end.y,
    1, Drawings::ps_solid, Drawings::gray );

  // init Bresenham
  if(isVertical)
  {
    count = size.y;
    sum = size.y / 2;
  }
  else
  {
    count = size.x;
    sum = size.x / 2;
  }

  if(count > 7)
  {
    int numberOfPoints[LinesPercept::numberOfLineTypes],
        i;
    for(i = 0; i < LinesPercept::numberOfLineTypes; ++i)
      numberOfPoints[i] = linesPercept.numberOfPoints[i];
    RingBuffer<const unsigned char*,7> pixelBuffer;
    RingBuffer<unsigned char, 7> yBuffer;
    RingBuffer<unsigned char, 7> vBuffer;
    RingBuffer<colorClass, 7> colorClassBuffer;

    //the image is scanned vertically
    for(i = 0; i < 6; ++i) // fill buffer
    {
      unsigned char y,u,v;
      pixelBuffer.add(&image.image[actual.y][0][actual.x]);
      if (actual.x < 0 || actual.x >= image.cameraInfo.resolutionWidth || actual.y < 0 || actual.y >= image.cameraInfo.resolutionHeight)
        y = u = v = 0;
      else
      {
        y = colorCorrector.correctY(actual.x, actual.y, image.image[actual.y][0][actual.x]);
        u = colorCorrector.correctU(actual.x, actual.y, image.image[actual.y][1][actual.x]);
        v = colorCorrector.correctV(actual.x, actual.y, image.image[actual.y][2][actual.x]);
      }
      yBuffer.add(y);
      vBuffer.add(v);
      colorClassBuffer.add(COLOR_CLASS(y, u, v));
      
      // Bresenham
      if(isVertical)
      {
        actual.y += step.y;
        sum += size.x;
        if(sum >= size.y)
        {
          sum -= size.y;
          actual.x += step.x;
        }
      }
      else        
      {
        actual.x += step.x;
        sum += size.y;
        if(sum >= size.x)
        {
          sum -= size.x;
          actual.y += step.y;
        }
      }
    }
    lineColor = Drawings::numberOfColors;

    int redCount = 0,
        blueCount = 0,
        greenCount = 0,
        noGreenCount = 100,
        pinkCount = 0,
        noPinkCount = 100,
        yellowCount = 0,
        noYellowCount = 100,
        skyblueCount = 0,
        noSkyblueCount = 100,
        orangeCount = 0,
        noOrangeCount = 100;
    const unsigned char* firstRed = 0,
                       * lastRed = 0,
                       * firstBlue = 0,
                       * lastBlue = 0,
                       * firstGreen = 0,
                       * firstOrange = 0,
                       * lastOrange = 0,
                       * firstPink = 0,
                       * lastPink = 0,
                       * pFirst = pixelBuffer[2],
                       * up = pFirst;
    enum{justUP, greenAndUP, down} state = justUP;
    bool borderFound = false; // there can be only one border point per scan line
    Flag::FlagType flagType = Flag::numberOfFlagTypes; // none

    // obstacles
    enum{noGroundFound, foundBeginOfGround, foundEndOfGround} groundState = noGroundFound;
    Vector2<int> firstGround(0,0);
    int numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;
    int numberOfPixelsSinceFirstOccurrenceOfNonGroundColor = 0;
    int numberOfPixelsSinceLastGround = 0;
    int numberOfGroundPixels = 0;
    int numberOfWhiteObstaclePixels = 0;
    int numberOfNotWhiteObstaclePixels = 0;
    bool beginOfGroundIsAtTheTopOfTheImage = false;

    for(; i <= count; ++i)
    {
      pixelsSinceGoal++;
      numberOfScannedPixels++;
      unsigned char y,u,v;
      pixelBuffer.add(&image.image[actual.y][0][actual.x]);
      if (actual.x < 0 || actual.x >= image.cameraInfo.resolutionWidth || actual.y < 0 || actual.y >= image.cameraInfo.resolutionHeight)
        y = u = v = 0;
      else
      {
        y = colorCorrector.correctY(actual.x, actual.y, image.image[actual.y][0][actual.x]);
        u = colorCorrector.correctU(actual.x, actual.y, image.image[actual.y][1][actual.x]);
        v = colorCorrector.correctV(actual.x, actual.y, image.image[actual.y][2][actual.x]);
      }
      yBuffer.add(y);
      vBuffer.add(v);
      colorClassBuffer.add(COLOR_CLASS(y, u, v));

      // line point state machine
      const unsigned char* p3 = pixelBuffer[3];
      //UP: y-component increases ////////////////////////////////////////
      if(yBuffer[3] > yBuffer[4] + yThreshold)
      {
        up = p3;
        if(colorClassBuffer[6] == green)
          state = greenAndUP;
        else 
          state = justUP;
      }
    
      //DOWN: y-component decreases //////////////////////////////////////
      else if(yBuffer[3] < 
              yBuffer[4] - yThreshold || 
              vBuffer[3] < 
              vBuffer[3] - vThreshold)
      {
        // is the pixel in the field ??
        if(state != down && // was an "up" pixel above?
           colorClassBuffer[0] == green) // is there green 3 pixels below ?
        {
          colorClass c = colorClassBuffer[6];
          if(!noLines || c == skyblue || c == yellow)
          {
            Vector2<int> coords = getCoords(p3),
                         pointOnField; //position on field, relative to robot

            if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
            {
              Vector2<int> upCoords = getCoords(up);
              int height = (coords - upCoords).abs();
              int distance = (int) sqrt(sqr(cameraMatrix.translation.z) + sqr(pointOnField.abs())),
                lineHeight = Geometry::getSizeByDistance(image.cameraInfo, 25, distance);

              //The bright region is assumed to be on the ground. If it is small enough, it is a field line.
              if(height < 3 * lineHeight && state == greenAndUP &&
                 (colorClassBuffer[5] == white ||
                  colorClassBuffer[4] == white))
              {
                linesPercept.add(LinesPercept::field,pointOnField);
                if(Geometry::calculatePointOnField(upCoords.x, upCoords.y, cameraMatrix, image.cameraInfo, pointOnField))
                  linesPercept.add(LinesPercept::field,pointOnField);
              }
              else if(height >= 3 && !borderFound)
              {
                switch(c)
                {
                  case skyblue:
                    if(vertical)
                    {
                      borderFound = true;
                      if(getPlayer().getTeamColor() == Player::red && !linesPercept.numberOfPoints[LinesPercept::skyblueGoal])
                        goalAtBorder = pointOnField.abs() < closestBottom;
                      closestBottom = 40000;
                      linesPercept.add(LinesPercept::skyblueGoal,pointOnField);
                      typeOfLinesPercept = LinesPercept::skyblueGoal;
                      pixelsSinceGoal = 0;

                      lastRed = lastBlue = 0;
                      redCount = blueCount = 0;
                    }
                    break;
                  case yellow:
                    if(vertical)
                    {
                      borderFound = true;
                      if(getPlayer().getTeamColor() == Player::blue && !linesPercept.numberOfPoints[LinesPercept::yellowGoal])
                        goalAtBorder = pointOnField.abs() < closestBottom;
                      closestBottom = 40000;
                      linesPercept.add(LinesPercept::yellowGoal,pointOnField);
                      typeOfLinesPercept = LinesPercept::yellowGoal;
                      pixelsSinceGoal = 0;

                      lastRed = lastBlue = 0;
                      redCount = blueCount = 0; 
                    }
                    break;
                  case white:
                    if(state != greenAndUP && height > lineHeight * 3 && (vertical || height > 30))
                    {
                      borderFound = true;
                      linesPercept.add(LinesPercept::border,pointOnField);
                      typeOfLinesPercept = LinesPercept::border;
                      lastRed = lastBlue = 0;
                      redCount = blueCount = 0; 
                    }
                    break;
                }
              }
            }
          }
          state = down;
        }
      }
      
      colorClass c3 = colorClassBuffer[3];
      ++noOrangeCount;
      if (c3 == orange)
      {
        if(noOrangeCount > 1)
        {
          if (orangeCount > longestBallRun)
          {
            longestBallRun = orangeCount;
            ballCandidate = (getCoords(firstOrange) + getCoords(lastOrange)) / 2;
          }
          orangeCount = 1;
          firstOrange = p3;
          lastOrange = p3;
        }
        else
        {
          ++orangeCount;
          if(orangeCount > 2) // ignore robot if ball is below, distance would be wrong
          {
            lastRed = lastBlue = 0;
            redCount = blueCount = 0; 
          }
          lastOrange = p3;
        }
        firstGreen = 0;
        noOrangeCount = 0;
      }
      if(vertical)
      {
        // in scanColumns, recognize player and ball points
        ++noGreenCount;
        ++noPinkCount;
        ++noYellowCount;
        ++noSkyblueCount;
        switch(c3)
        {
          case blue: // count blue, remember last
            if(blueCount == 3)
              firstBlue = pixelBuffer[6];
            ++blueCount;
            lastBlue = p3;
            firstGreen = 0;
            break;
          case gray: // drop green above gray (i.e. robot legs)
            if(firstGreen && noGreenCount < 8)
              firstGreen = 0;
            break;
          case green: // find first green below a robot
            if(!firstGreen)
            { 
              firstGreen = p3;
            }
            if(noGreenCount > 6)
              greenCount = 1;
            else
              ++greenCount;
            if(greenCount == 2 && noPinkCount < 5 && pinkCount >= 2)
              flagType = Flag::pinkAboveGreen;
            noGreenCount = 0;
            break;
          case red:
            if(orangeCount < 6 || noOrangeCount > 4 || redCount > orangeCount)
            { // count red, remember last
              if(redCount == 3)
                firstRed = pixelBuffer[6];
              ++redCount;
              lastRed = p3;
              firstGreen = 0;
              break;
            }
            // no break, red below orange is interpreted as orange
          case pink:
            if(noPinkCount > 6)
            {
              pinkCount = 1;
              firstPink = p3;
            }
            else
              ++pinkCount;
            if(pinkCount == 2)
            {
              if(noYellowCount < 5 && yellowCount >= 2)
                flagType = Flag::yellowAbovePink;
              else if(noSkyblueCount < 5 && skyblueCount >= 2)
                flagType = Flag::skyblueAbovePink;
              else if(noGreenCount < 5 && greenCount >= 2)
                flagType = Flag::greenAbovePink;
            }
            lastPink = p3;
            noPinkCount = 0;
            break;
          case yellow:
            if(noYellowCount > 6)
              yellowCount = 1;
            else
              ++yellowCount;
            if(yellowCount == 2 && noPinkCount < 5 && pinkCount >= 2)
              flagType = Flag::pinkAboveYellow;
            noYellowCount = 0;
            break;
          case skyblue:
            if(noSkyblueCount > 6)
              skyblueCount = 1;
            else
              ++skyblueCount;
            if(skyblueCount == 2 && noPinkCount < 5 && pinkCount >= 2)
              flagType = Flag::pinkAboveSkyblue;
            noSkyblueCount = 0;
            break;
          case white:
            switch(flagType)
            {
              /*
              // no more middle flags
              case Flag::pinkAboveGreen:
                if(noGreenCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              */
              case Flag::pinkAboveYellow:
                if(noYellowCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              case Flag::pinkAboveSkyblue:
                if(noSkyblueCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              /*
              // no more middle flags
              case Flag::greenAbovePink:
                if(noPinkCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              */
              case Flag::yellowAbovePink:
                if(noPinkCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              case Flag::skyblueAbovePink:
                if(noPinkCount < 5)
                  clusterFlags(flagType, (getCoords(lastPink) + getCoords(firstPink)) / 2);
                flagType = Flag::numberOfFlagTypes; // none
            }
            break;
          
        }//switch(c3)

        // obstacles state machine
        if(count > 90 && cameraMatrix.isValid)
        {
          DEBUG_IMAGE_SET_PIXEL_BLACK(imageProcessorPlayers, getCoords(pixelBuffer[0]).x, getCoords(pixelBuffer[0]).y);
          colorClass color = colorClassBuffer[0];
          if(groundState == noGroundFound)
          {
            DEBUG_IMAGE_SET_PIXEL_YELLOW(imageProcessorPlayers, getCoords(pixelBuffer[0]).x, getCoords(pixelBuffer[0]).y);
            if(color == green || color == orange)
            {
              if(numberOfPixelsSinceFirstOccurrenceOfGroundColor == 0)
              {
                firstGround = getCoords(pixelBuffer[0]);
              }
              numberOfPixelsSinceFirstOccurrenceOfGroundColor++;
              if(numberOfPixelsSinceFirstOccurrenceOfGroundColor > 3 || color == orange)
              {
                Vector2<int> coords = getCoords(pixelBuffer[0]);
                int lineHeight = Geometry::calculateLineSize(coords, cameraMatrix, image.cameraInfo);
                if(i < lineHeight * 4) beginOfGroundIsAtTheTopOfTheImage = true;
                groundState = foundBeginOfGround;
                pixelsBetweenGoalAndObstacle = pixelsSinceGoal;
                numberOfPixelsSinceFirstOccurrenceOfNonGroundColor = 0;
              }
            }
            else if (color != noColor)
            {
              numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;
              if(color == white) numberOfWhiteObstaclePixels++;
              else numberOfNotWhiteObstaclePixels++;
            }
          }
          else if(groundState == foundBeginOfGround)
          {
            DEBUG_IMAGE_SET_PIXEL_GREEN(imageProcessorPlayers, getCoords(pixelBuffer[0]).x, getCoords(pixelBuffer[0]).y);
            if(color != green && color != orange && color != noColor)
            {
              numberOfPixelsSinceFirstOccurrenceOfNonGroundColor++;
              if(numberOfPixelsSinceFirstOccurrenceOfNonGroundColor > 3)
              {
                groundState = foundEndOfGround;
                numberOfPixelsSinceLastGround = 0;
                numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;
              }
            }
            else if (color != noColor)
            {
              numberOfGroundPixels++;
              numberOfPixelsSinceFirstOccurrenceOfNonGroundColor = 0;
            }
          }
          else if(groundState == foundEndOfGround)
          {
            numberOfPixelsSinceLastGround++;
            DEBUG_IMAGE_SET_PIXEL_RED(imageProcessorPlayers, getCoords(pixelBuffer[0]).x, getCoords(pixelBuffer[0]).y);
            if(color == green || color == orange)
            {
              numberOfPixelsSinceFirstOccurrenceOfGroundColor++;
              if(numberOfPixelsSinceFirstOccurrenceOfGroundColor > 3 || color == orange)
              {
                numberOfPixelsSinceFirstOccurrenceOfNonGroundColor = 0;
                groundState = foundBeginOfGround;
//                pixelsBetweenGoalAndObstacle = pixelsSinceGoal;
                Vector2<int> coords = getCoords(pixelBuffer[0]),
                             pointOnField; //position on field, relative to robot
                int lineHeight = Geometry::calculateLineSize(coords, cameraMatrix, image.cameraInfo);
                if(numberOfPixelsSinceLastGround > lineHeight * 4)
                {
                  DEBUG_IMAGE_SET_PIXEL_PINK(imageProcessorPlayers, getCoords(pixelBuffer[0]).x, getCoords(pixelBuffer[0]).y);
                  firstGround = getCoords(pixelBuffer[0]);
                  numberOfGroundPixels = 0;
                  beginOfGroundIsAtTheTopOfTheImage = false;
                }
              }
            }
            else if (color != noColor)
            {
              numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;  
              if(color == white) 
                numberOfWhiteObstaclePixels++;
              else 
                numberOfNotWhiteObstaclePixels++;
            }
          }
        } // if(count > 100)

      }//if(vertical)
      
      PLOT(p3,ColorClasses::colorClassToDrawingsColor(c3));

      // Bresenham
      if(isVertical)
      {
        actual.y += step.y;
        sum += size.x;
        if(sum >= size.y)
        {
          sum -= size.y;
          actual.x += step.x;
        }
      }
      else        
      {
        actual.x += step.x;
        sum += size.y;
        if(sum >= size.x)
        {
          sum -= size.x;
          actual.y += step.y;
        }
      }
    }

    if (orangeCount > longestBallRun)
    {
      longestBallRun = orangeCount;
      ballCandidate = (getCoords(firstOrange) + getCoords(lastOrange)) / 2;
    }

    const unsigned char* pLast = pixelBuffer[3];
    PLOT(pLast,Drawings::numberOfColors);

    if(vertical)
    { // line scanning finished, analyze results (only in scanColumns)
      int goal = getPlayer().getTeamColor() == Player::red ? LinesPercept::skyblueGoal
                                                           : LinesPercept::yellowGoal;
      if(linesPercept.numberOfPoints[goal] == numberOfPoints[goal]) // no goal point found in this column
      {
        Vector2<int> coords = getCoords(pLast),
                     pointOnField;
        if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
        {
          int dist = pointOnField.abs();
          if(dist < closestBottom)
            closestBottom = dist;
        }
      }
              
      bool redFound = false,
           blueFound = false;

      // red robot point found?
      if(redCount > blueCount && (firstGreen && redCount > 2 || redCount > 20))
      { // drop border and goal points
        for(i = 1; i < LinesPercept::numberOfLineTypes; ++i)
          linesPercept.numberOfPoints[i] = numberOfPoints[i];
        if(redCount > 20)
        { // if robot is close, use upper boundary of tricot to calculate distance
          Vector2<int> coords = getCoords(firstRed),
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cmTricot, image.cameraInfo, pointOnField) &&
             pointOnField.abs() < 500)
          {
            DOT(imageProcessor_general, coords.x, coords.y, Drawings::red, Drawings::white);
            linesPercept.add(LinesPercept::redRobot,pointOnField);
            typeOfLinesPercept = LinesPercept::redRobot;
            redFound = true;
          }
        }
        if(!redFound)
        { // otherwise, use foot point
          Vector2<int> coords = getCoords(firstGreen),
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
          {
            Vector2<int> redCoords = getCoords(lastRed);
            if((redCoords - coords).abs() < Geometry::getSizeByDistance(image.cameraInfo, 100, pointOnField.abs()))
            {
              linesPercept.add(LinesPercept::redRobot,pointOnField);
              typeOfLinesPercept = LinesPercept::redRobot;
              redFound = true;
            }
          }
        }
      }
      // blue robot point found?
      else if(blueCount > redCount && (firstGreen && blueCount > 2 || blueCount > 10))
      { // drop border and goal points
        for(i = 1; i < LinesPercept::numberOfLineTypes; ++i)
          linesPercept.numberOfPoints[i] = numberOfPoints[i];
        if(blueCount > 10)
        { // if robot is close, use upper boundary of tricot to calculate distance
          Vector2<int> coords = getCoords(firstBlue),
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cmTricot, image.cameraInfo, pointOnField) &&
             pointOnField.abs() < 500)
          {
            DOT(imageProcessor_general, coords.x, coords.y, Drawings::pink, Drawings::white);
            linesPercept.add(LinesPercept::blueRobot,pointOnField);
            typeOfLinesPercept = LinesPercept::blueRobot;
            blueFound = true;
          }
        }
        if(!blueFound)
        { // otherwise, use foot point
          Vector2<int> coords = getCoords(firstGreen),
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
          {
            Vector2<int> blueCoords = getCoords(lastBlue);
            if((blueCoords - coords).abs() < Geometry::getSizeByDistance(image.cameraInfo, 100, pointOnField.abs()))
            {
              linesPercept.add(LinesPercept::blueRobot,pointOnField);
              typeOfLinesPercept = LinesPercept::blueRobot;
              blueFound = true;
            }
          }
        }
      }
      // cluster the robot points found
      clusterRobots(pLast, redFound, blueFound);

      // obstacles found?
      bool addObstaclesPoint = false;
      Vector2<int> newObstaclesNearPoint;
      Vector2<int> newObstaclesFarPoint;
      bool newObstaclesFarPointIsOnImageBorder = false;
      if(count > 90 && cameraMatrix.isValid &&
        (
         numberOfGroundPixels > numberOfWhiteObstaclePixels ||
         numberOfGroundPixels > 20 ||
         numberOfNotWhiteObstaclePixels > numberOfWhiteObstaclePixels * 2
         )
        )
      {
        Vector2<int> firstGroundOnField;
        bool firstGroundOnFieldIsOnField = 
          Geometry::calculatePointOnField(firstGround.x, firstGround.y, cameraMatrix, image.cameraInfo, firstGroundOnField);

        Vector2<int> imageBottom = getCoords(pixelBuffer[0]);
        Vector2<int> imageBottomOnField;
        bool imageBottomIsOnField = 
          Geometry::calculatePointOnField(imageBottom.x, imageBottom.y, cameraMatrix, image.cameraInfo, imageBottomOnField);

        if(groundState == foundBeginOfGround)
        {
          if(firstGroundOnFieldIsOnField)
          {
            addObstaclesPoint = true;
            newObstaclesNearPoint = imageBottomOnField;
            newObstaclesFarPoint = firstGroundOnField;
            newObstaclesFarPointIsOnImageBorder = beginOfGroundIsAtTheTopOfTheImage;
          }
        }
        else if(groundState == foundEndOfGround)
        {
          int lineHeight = Geometry::calculateLineSize(imageBottom, cameraMatrix, image.cameraInfo);
          if(numberOfPixelsSinceLastGround < lineHeight * 4)
          {
            if(firstGroundOnFieldIsOnField)
            {
              addObstaclesPoint = true;
              newObstaclesNearPoint = imageBottomOnField;
              newObstaclesFarPoint = firstGroundOnField;
              newObstaclesFarPointIsOnImageBorder = beginOfGroundIsAtTheTopOfTheImage;
            }
          }
          else if(imageBottomIsOnField)
          {
            addObstaclesPoint = true;
            newObstaclesNearPoint = imageBottomOnField;
            newObstaclesFarPoint = imageBottomOnField;
          }
        }
        else if(groundState == noGroundFound)
        {
          if(imageBottomIsOnField &&
            // the carpet often is not green under the robot
            imageBottomOnField.abs() > 150)
          {
            addObstaclesPoint = true;
            newObstaclesNearPoint = imageBottomOnField;
            newObstaclesFarPoint = imageBottomOnField;
          }
        }
      }// if(count > 90 && cameraMatrix.isValid)
      if(addObstaclesPoint)
      {
        if(
/*          (
          Geometry::distance(obstaclesFarPointOnField1, obstaclesFarPointOnField2) < 400 ||
          Geometry::distance(obstaclesFarPointOnField2, newObstaclesFarPoint) < 400
          ) && 
*/
          angleBetweenDirectionOfViewAndGround > -80.0 &&
          !(newObstaclesFarPoint.x == 0 && newObstaclesFarPoint.y == 0)
          )

        {
          ObstaclesPercept::ObstacleType obstacleType = ObstaclesPercept::unknown;
          switch(typeOfLinesPercept)
          {
          case LinesPercept::skyblueGoal: 
            if(pixelsBetweenGoalAndObstacle < 15) 
              obstacleType = ObstaclesPercept::goal; 
            break;
          case LinesPercept::yellowGoal: 
            if(pixelsBetweenGoalAndObstacle < 15) 
              obstacleType = ObstaclesPercept::goal; 
            break;
          case LinesPercept::blueRobot:
            if(getPlayer().getTeamColor() == Player::blue)
              obstacleType = ObstaclesPercept::teammate;
            else 
              obstacleType = ObstaclesPercept::opponent;
            break;
          case LinesPercept::redRobot: 
            if(getPlayer().getTeamColor() == Player::red) 
              obstacleType = ObstaclesPercept::teammate;
            else 
              obstacleType = ObstaclesPercept::opponent;
            break;
          }
          obstaclesPercept.add(
            newObstaclesNearPoint, 
            newObstaclesFarPoint,
            newObstaclesFarPointIsOnImageBorder,
            obstacleType
            );
        }

/*        obstaclesFarPointOnField1 = obstaclesFarPointOnField2;
        obstaclesFarPointOnField2 = newObstaclesFarPoint;
    
        obstaclesFarPointOnField2IsOnImageBorder = newObstaclesFarPointIsOnImageBorder;
        obstaclesNearPointOnField2 = newObstaclesNearPoint;
        */
      }
    }//if(vertical)
  }
}


void DDD2004ImageProcessor::clusterRobots(const unsigned char* bottomPoint,
                                         bool redFound, bool blueFound)
{
  Vector2<int> coords = getCoords(bottomPoint),
               pointOnField;
  int noRedCount2 = noRedCount;

  if(redFound)
  {
    lastRed = 
      linesPercept.points[LinesPercept::redRobot][linesPercept.numberOfPoints[LinesPercept::redRobot] - 1];
    if(noRedCount > 2)
      firstRed = closestRed = lastRed;
    else if(closestRed.abs() > lastRed.abs())
      closestRed = lastRed;
    noRedCount = 0;
  }
  // count number of columns without robot points, but ignore columns that are
  // not long enough to contain the robot. Drop single robot points of the other 
  // robot color.
  else if(noRedCount > 2 || !bottomPoint ||
          (Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField) &&
          closestRed.abs() > pointOnField.abs()))
  {
    if(++noRedCount == 3 && (firstRed != lastRed || noBlueCount > 4))
    {
      SinglePlayerPercept percept;
      percept.direction = (atan2((double)firstRed.y,(double)firstRed.x) + atan2((double)lastRed.y,(double)lastRed.x)) / 2;
      double distance = closestRed.abs();
      percept.offset.x = cos(percept.direction) * distance;
      percept.offset.y = sin(percept.direction) * distance;
      percept.validity = 1;
      playersPercept.addRedPlayer(percept);
    }
  }

  if(blueFound)
  {
    lastBlue = 
      linesPercept.points[LinesPercept::blueRobot][linesPercept.numberOfPoints[LinesPercept::blueRobot] - 1];
    if(noBlueCount > 2)
      firstBlue = closestBlue = lastBlue;
    else if(closestBlue.abs() > lastBlue.abs())
      closestBlue = lastBlue;
    noBlueCount = 0;
  }
  // count number of columns without robot points, but ignore columns that are
  // not long enough to contain the robot. Drop single robot points of the other 
  // robot color.
  else if(noBlueCount > 2 || !bottomPoint ||
          (Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField) &&
          closestBlue.abs() > pointOnField.abs()))
  {
    if(++noBlueCount == 3 && (firstBlue != lastBlue || noRedCount2 > 4))
    {
      SinglePlayerPercept percept;
      percept.direction = (atan2((double)firstBlue.y,(double)firstBlue.x) + atan2((double)lastBlue.y,(double)lastBlue.x)) / 2;
      double distance = closestBlue.abs();
      percept.offset.x = int(cos(percept.direction) * distance);
      percept.offset.y = int(sin(percept.direction) * distance);
      percept.validity = 1;
      playersPercept.addBluePlayer(percept);
    }
  }
}

void DDD2004ImageProcessor::clusterGoal()
{
  // add goal percepts
  if(linesPercept.numberOfPoints[LinesPercept::yellowGoal])
  {
    Vector2<int> pointOnField = linesPercept.points[LinesPercept::yellowGoal][linesPercept.numberOfPoints[LinesPercept::yellowGoal] / 2] * 2,
                 point;
    Geometry::calculatePointInImage(pointOnField, cameraMatrix, image.cameraInfo, point);
  }
  if(linesPercept.numberOfPoints[LinesPercept::skyblueGoal])
  {
    Vector2<int> pointOnField = linesPercept.points[LinesPercept::skyblueGoal][linesPercept.numberOfPoints[LinesPercept::skyblueGoal] / 2] * 2,
                 point;
    Geometry::calculatePointInImage(pointOnField, cameraMatrix, image.cameraInfo, point);
  }

  // calculate free part of goal
  int goal = getPlayer().getTeamColor() == Player::blue 
             ? LinesPercept::yellowGoal : LinesPercept::skyblueGoal;
  double maxWidth = 2 * ballRadius;
  Vector2<double> angles;
  Geometry::calculateAnglesForPoint(Vector2<int>(image.cameraInfo.resolutionWidth / 2, image.cameraInfo.resolutionHeight / 2),cameraMatrix, image.cameraInfo, angles);
  Pose2D camera(angles.x, Vector2<double>(cameraMatrix.translation.x,
                                          cameraMatrix.translation.y));
  if(linesPercept.numberOfPoints[goal])
  {
    bool goalAtBorder2 = linesPercept.points[goal][linesPercept.numberOfPoints[goal] - 1].abs() < closestBottom,
         isFirst = true;
    Vector2<double> first = (Pose2D(linesPercept.points[goal][0]) - camera).translation,
                    last = first;
    for(int i = 1; i < linesPercept.numberOfPoints[goal]; ++i)
    {
      Vector2<double> point = (Pose2D(linesPercept.points[goal][i]) - camera).translation;
      if(fabs(point.y - last.y) > 150)
      {
        if(isFirst && goalAtBorder)
        {
          Vector2<double> point2 = (Pose2D(linesPercept.points[goal][linesPercept.numberOfPoints[goal] - 1]) - camera).translation;
          first = point2 + (first - point2) * 700 / (first - point2).abs();
        }
        double width = fabs(last.y - first.y);
        if(width > maxWidth)
        {
          maxWidth = width;
          addGoal(camera, first, last);
        }
        first = point;
        isFirst = false;
      }
      last = point;
    }
    if(!isFirst && goalAtBorder2)
    {
      Vector2<double> point2 = (Pose2D(linesPercept.points[goal][0]) - camera).translation;
      last = point2 + (last - point2) * 700 / (last - point2).abs();
    }
    double width = fabs(last.y - first.y);
    if(width > maxWidth)
      addGoal(camera, first, last);
  }
}

void DDD2004ImageProcessor::addGoal(const Pose2D& camera, 
                                   const Vector2<double>& first, const Vector2<double>& last)
{
  bool firstIsClose = false,
       lastIsClose = false;
  for(int i = 0; i < playersPercept.numberOfRedPlayers + playersPercept.numberOfBluePlayers; ++i)
  {
    SinglePlayerPercept& p = i < playersPercept.numberOfRedPlayers 
                      ? playersPercept.redPlayers[i]
                      : playersPercept.bluePlayers[i - playersPercept.numberOfRedPlayers];
    Vector2<double> robot = (Pose2D(p.offset) - camera).translation;
    if(robot.x > 0 && fabs(robot.y - first.y) < 200)
      firstIsClose = true;
    if(robot.x > 0 && fabs(robot.y - last.y) < 200)
      lastIsClose = true;
  }

  Vector2<double> first2 = (camera + Pose2D(first)).translation,
                  last2 = (camera + Pose2D(last)).translation;
  double firstDir = atan2(first2.y, first2.x),
         lastDir = atan2(last2.y, last2.x),
         factor;
  if(firstIsClose == lastIsClose)
    factor = 0.5;
  else if(firstIsClose)
    factor = 0.7;
  else
    factor = 0.3;

  obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::opponentGoal] = true;
  obstaclesPercept.angleToFreePartOfGoal[ObstaclesPercept::opponentGoal] = firstDir + factor * (lastDir - firstDir);
  double distance = ((first2 + last2) / 2).abs();
  obstaclesPercept.distanceToFreePartOfGoal[ObstaclesPercept::opponentGoal] = distance;
  obstaclesPercept.widthOfFreePartOfGoal[ObstaclesPercept::opponentGoal] = distance ? 2 * atan((first - last).abs() / 2 / distance) : pi;
}

void DDD2004ImageProcessor::clusterFlags(Flag::FlagType flagType, const Vector2<int>& point)
{
  if((point - lastFlag).abs() > 9)
  {
    if(lastFlag.x >= 0)
    {
      Flag::FlagType bestType = Flag::numberOfFlagTypes;
      int maxCount = 0;
      bool ambiguous = true;
      for(int i = 0; i < 6; ++i)
        if(flagCount[i] > maxCount)
        {
          bestType = Flag::FlagType(i);
          maxCount = flagCount[i];
          ambiguous = false;
        }
        else if(flagCount[i] == maxCount)
          ambiguous = true;
      if(!ambiguous)
      {
        Vector2<int> center = (firstFlag + lastFlag) / 2;
        static const colorClass otherColor[6] = {yellow, green, skyblue, yellow, green, skyblue};
        flagSpecialist.searchFlags(
          image, colorTable, cameraMatrix, otherColor[bestType], 
          bestType <= Flag::pinkAboveSkyblue,
          horizon, center.x, center.y);
      }
    }
    for(int i = 0; i < 6; ++i)
      flagCount[i] = 0;
    firstFlag = point;
  }
  lastFlag = point;
  if(flagType != Flag::numberOfFlagTypes)
    ++flagCount[flagType];
}

void DDD2004ImageProcessor::filterPercepts()
{
  // Close robots produce far ghost robots. Remove them.
  int i;
  for(i = 0; i < playersPercept.numberOfRedPlayers + playersPercept.numberOfBluePlayers; ++i)
  {
    SinglePlayerPercept& pi = i < playersPercept.numberOfRedPlayers 
                      ? playersPercept.redPlayers[i]
                      : playersPercept.bluePlayers[i - playersPercept.numberOfRedPlayers];
    double iDist = pi.offset.abs(),
           iAngle = atan2(pi.offset.y, pi.offset.x),
           ignoreAngle = atan2(150, iDist);
    for(int j = 0; j < playersPercept.numberOfRedPlayers + playersPercept.numberOfBluePlayers; ++j)
      if(i != j)
      {
        SinglePlayerPercept& pj = j < playersPercept.numberOfRedPlayers 
                            ? playersPercept.redPlayers[j]
                            : playersPercept.bluePlayers[j - playersPercept.numberOfRedPlayers];
        if(iDist < pj.offset.abs() &&
           fabs(atan2(pj.offset.y, pj.offset.x) - iAngle) < ignoreAngle)
        {
          if(j < playersPercept.numberOfRedPlayers)
            pj = playersPercept.redPlayers[--playersPercept.numberOfRedPlayers];
          else
            pj = playersPercept.bluePlayers[--playersPercept.numberOfBluePlayers];
          if(i > j)
          {
            i = j; // check pi again
            break;
          }
          else
            --j; // check pj again
        }
      }
  }

  // If there are too few line points of a certain type, remove them all, they may be misreadings
  for(i = 0; i < LinesPercept::numberOfLineTypes; ++i)
    if(linesPercept.numberOfPoints[i] < 3)
      linesPercept.numberOfPoints[i] = 0;

  // if many more field line points than border points have been found,
  // it is very likely that the border points also result from field lines
  // So remove them.
  if(linesPercept.numberOfPoints[LinesPercept::field] > 
     3 * linesPercept.numberOfPoints[LinesPercept::border])
    linesPercept.numberOfPoints[LinesPercept::border] = 0;
}

Vector2<int> DDD2004ImageProcessor::getCoords(const unsigned char* p) const
{
  int diff = p - &image.image[0][0][0];
  return Vector2<int>(diff % cameraResolutionWidth_ERS7,diff / (cameraResolutionWidth_ERS7 * 6));
}

void DDD2004ImageProcessor::plot(const unsigned char* p,Drawings::Color color)
{
  if(lineColor == Drawings::numberOfColors)
  {
    last = p;
    lineColor = color;
  }
  else if(color != lineColor)
  {
    Vector2<int> p1 = getCoords(last),
                 p2 = getCoords(p);
    LINE(imageProcessor_general,p1.x,p1.y,p2.x,p2.y,0,Drawings::ps_solid,lineColor);
    last = p;
    lineColor = color;
  }
}

bool DDD2004ImageProcessor::handleMessage(InMessage& message)
{
  switch(message.getMessageID())
  {
    case idLinesImageProcessorParameters:
      GenericDebugData d;
      message.bin >> d;
      yThreshold = (int)d.data[1];
      vThreshold = (int)d.data[2];
      execute();
      return true;
  }
  return false;
}

/*
* Change log :
* 
* $Log: DDD2004ImageProcessor.cpp,v $
* Revision 1.26  2004/05/07 15:16:24  nistico
* All geometry calculations are making use of intrinsic functions.
* I updated most of the images processors to make correct use of this.
* Please test if there are any problems, because i'm going to remove the
* old code soon.
*
* Revision 1.25  2004/04/07 12:28:57  risler
* ddd checkin after go04 - first part
*
* Revision 1.3  2004/03/29 15:20:36  Marc
* DDD2004 Headcontrol installed
*
* Revision 1.2  2004/03/29 09:54:54  risler
* obstacle lines without ground pixels reenabled
*
* Revision 1.24  2004/03/28 22:27:15  risler
* softened obstacle detection
*
* Revision 1.23  2004/03/28 17:07:29  risler
* ball specialist trigger threshold increased (since specialist now can reject balls)
*
* Revision 1.22  2004/03/28 13:44:06  mkunz
* more colors in imageProcessorBall image
*
* Revision 1.21  2004/03/27 21:35:54  risler
* no more scans beyond image border
*
* Revision 1.20  2004/03/27 16:58:32  petters
* removed trigger of middle flags
*
* Revision 1.19  2004/03/27 12:25:45  risler
* changed the imageProcessorBall debug image one more time, hope everyone is happy with it now
*
* Revision 1.18  2004/03/27 11:11:38  thomas
* readded ifdefs
*
* Revision 1.17  2004/03/27 11:07:21  thomas
* fix compile-error
*
* Revision 1.16  2004/03/27 10:24:07  roefer
* Warnings removed
*
* Revision 1.15  2004/03/26 15:38:54  loetzsch
* bug fix (did not compile)
*
* Revision 1.14  2004/03/26 14:20:00  risler
* ball specialist triggering adjusted
*
* Revision 1.13  2004/03/26 14:02:39  brunn
* added getSimilarityToOrange
*
* Revision 1.12  2004/03/26 13:27:09  brunn
* added imageProcessorBall DebugImage
*
* Revision 1.11  2004/03/26 11:14:14  risler
* bugfix: reenabled color correction, working this time
*
* Revision 1.10  2004/03/26 11:01:20  risler
* bugfix: reenabled color correction
*
* Revision 1.9  2004/03/26 10:49:35  risler
* bugfix: reenabled color correction
*
* Revision 1.8  2004/03/26 09:16:35  risler
* increased ballSpecialist trigger threshold
*
* Revision 1.7  2004/03/26 09:12:42  risler
* added DDD2004BallSpecialist
* moved ball related stuff from ImageProcessor to BallSpecialist
*
* Revision 1.6  2004/03/24 19:26:13  risler
* added DDD2004GoalRecognizer
*
* Revision 1.5  2004/03/24 18:40:59  risler
* improved scan line selection
*
* Revision 1.4  2004/03/24 12:56:51  risler
* column scan up to image border
*
* Revision 1.3  2004/03/18 14:47:14  uhrig
* Added Levenberg-Marquardt algorithm
*
* Revision 1.2  2004/03/17 15:31:16  risler
* added DDD2004ColorCorrector
*
* Revision 1.1  2004/03/16 14:16:06  risler
* cloned GT2003ImageProcessor to DDD2004ImageProcessor
*
* Revision 1.19  2004/03/12 17:22:26  nistico
* - Fixed a problem in CT32K_LAYOUT with localization
*
* Revision 1.18  2004/03/08 01:38:59  roefer
* Interfaces should be const
*
* Revision 1.17  2004/03/04 17:51:17  nistico
* -Bug with MSH2004ColorCorrection fixed
*
* Revision 1.16  2004/03/03 17:18:41  nistico
* -Introduced fast ColorTable32K access through conditional compilation
* For Dortmund: uncomment (define) CT32K_LAYOUT in GT2003ImageProcessor.h
* For other universities: leave undefined :-)
*
* Revision 1.15  2004/03/02 00:26:21  roefer
* Warnings removed
*
* Revision 1.14  2004/03/01 11:56:44  juengel
* Recognition of obstacle types added.
*
* Revision 1.13  2003/12/31 23:50:36  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.12  2003/12/30 20:12:02  roefer
* Image size is now 208 x 160. Smaller images are placed in the upper left corner
*
* Revision 1.11  2003/12/26 13:15:29  roefer
* calculation of V corrected
*
* Revision 1.10  2003/12/16 13:44:19  roefer
* cameraInfo used for offset calculation
*
* Revision 1.9  2003/12/15 11:47:04  juengel
* Introduced CameraInfo
*
* Revision 1.8  2003/12/01 16:26:12  juengel
* Added calculation of "pixel usage" goal pixels are not counted yet.
*
* Revision 1.7  2003/11/15 17:09:02  juengel
* changed BallPercept
*
* Revision 1.6  2003/11/12 16:19:35  goehring
* frameNumber added to percepts
*
* Revision 1.5  2003/11/05 16:43:59  juengel
* Added drawing imageProcessor_ball2
*
* Revision 1.4  2003/11/05 16:38:48  goehring
* FrameNumber added
*
* Revision 1.3  2003/10/31 11:46:06  juengel
* Replaced fixed image height and width by constant from RobotDimensions.h
*
* Revision 1.2  2003/10/12 20:20:15  juengel
* ColorTable is not const anymore.
* Added manual calibration.
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.10  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.9  2003/09/26 11:38:51  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.8  2003/09/01 10:05:08  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.7  2003/08/30 10:16:42  juengel
* DebugDrawings clean-up 1
*
* Revision 1.6  2003/08/18 12:07:49  juengel
* Moved method calculateLineSize to class Geometry.
*
* Revision 1.5  2003/07/29 12:48:35  juengel
* Moved calculateHorizon, getDistanceBySize and getSizeByDistance to Geometry.
*
* Revision 1.4  2003/07/21 13:42:21  juengel
* Moved ColorTableAuto and GoalRecognizer to ImageProcessorTools.
*
* Revision 1.3  2003/07/03 14:01:41  roefer
* Ball recognition at lower image border improved
*
* Revision 1.2  2003/07/03 10:11:16  roefer
* Ball position clipped close to body
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.21  2003/06/28 08:08:06  juengel
* Moved horizon one pixel to the left.
*
* Revision 1.20  2003/06/27 11:18:12  juengel
* Distance to a ball above the horizon is calculated by size.
*
* Revision 1.19  2003/06/26 13:22:39  mkunz
* extended the shorter scanlines
*
* Revision 1.18  2003/06/25 18:44:13  juengel
* Added nearPoints, farPoints[maxNumberOfPoints] and farPointIsOnImageBorder to ObstaclesPercept.
*
* Revision 1.17  2003/06/24 08:16:32  dueffert
* calculation optimized
*
* Revision 1.16  2003/06/22 17:09:08  juengel
* Added filter for ObstaclesPercept.
*
* Revision 1.15  2003/06/21 12:51:52  juengel
* ObstaclesPercept is calculated when cameraMatrix.isValid only.
*
* Revision 1.14  2003/06/20 21:12:45  roefer
* If camera matrix is invalid, ball distance is calculated from size
*
* Revision 1.13  2003/06/18 13:59:18  juengel
* Improved perception of obstacles.
*
* Revision 1.12  2003/06/17 19:43:31  juengel
* Orange and green are "ground color" for ObstaclesPercept.
*
* Revision 1.11  2003/06/13 14:56:09  juengel
* Points in ObstaclesPercept are filled.
*
* Revision 1.10  2003/06/12 16:57:59  juengel
* Added GoalRecognizer, removed GoalSpecialist.
*
* Revision 1.9  2003/05/26 20:21:32  roefer
* widthOfFreePartOfGoal is a angle now
*
* Revision 1.8  2003/05/26 08:20:10  juengel
* Moved angleToFreePartOfGoal from specialPercept to obstaclesPercept.
*
* Revision 1.7  2003/05/07 16:07:50  roefer
* Major ball recognition bug fixed
*
* Revision 1.6  2003/05/05 14:53:29  dueffert
* warnings removed
*
* Revision 1.5  2003/05/04 21:48:46  roefer
* Filtering ball point outliers
*
* Revision 1.4  2003/05/04 17:31:52  roefer
* Flag and goal specialists added to GT2003IP
*
* Revision 1.3  2003/05/03 18:09:55  roefer
* Ball sampling only for small balls
*
* Revision 1.2  2003/05/01 23:33:56  roefer
* Anchors for flag specialist added
*
* Revision 1.1  2003/05/01 13:24:24  roefer
* Copied LinesImageProcessor2 to GT2003ImageProcessor
*
*
*/
