/**
* @file DDD2004GoalRecognizer.h
*
* Definition of class DDD2004GoalRecognizer
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#ifndef __DDD2004GoalRecognizer_h_
#define __DDD2004GoalRecognizer_h_

#include "DDD2004ColorCorrector.h"

#include "Representations/Perception/Image.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/ColorTable.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/LandmarksPercept.h"


#include "Tools/Math/Geometry.h"
#include "Tools/Debugging/DebugImages.h"

/**
* @class DDD2004GoalRecognizer
*
* The goal recognizer recognizes the goals in the image.
*
* The recognizer scans horizontal lines in the image left to right.
* Indications for a goal are grouped and checked vertical.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/ 

class DDD2004GoalRecognizer //: public ImageProcessorInterfaces
{
public:
  DDD2004GoalRecognizer(
    const Image& image,
    const CameraMatrix& cameraMatrix,
    const ColorTable& colorTable,
    const DDD2004ColorCorrector& colorCorrector
    );

  
  ~DDD2004GoalRecognizer();

  void getGoalPercept
  (
   LandmarksPercept& landmarksPercept,
   ObstaclesPercept& obstaclesPercept,
   int distanceAboveHorizon = 40,
   int distanceBelowHorizon = -25,
   int numberOfScanLines = 10
  );

  struct ColoredPartsCheck
  {
    Vector2<int> firstPoint;
    Vector2<int> lastPoint;
    int rangeOfColor;
    int numberOfColoredPixels;
    
    enum{maxNumberOfParts = 40};
    
    int numberOfLargeParts;
    int sizeOfLargePart[maxNumberOfParts];
    Vector2<int> largePartBegin[maxNumberOfParts];
    Vector2<int> largePartEnd[maxNumberOfParts];
    Vector2<double> largePartBeginAngles[maxNumberOfParts];
    Vector2<double> largePartEndAngles[maxNumberOfParts];
    bool largePartBeginIsOnBorder[maxNumberOfParts];
    bool largePartEndIsOnBorder[maxNumberOfParts];
    
    ColoredPartsCheck() 
    {
      numberOfLargeParts = 0;
    }
    
    bool determineLargePart(int size, bool beginIsOnBorder, bool endIsOnBorder, CameraMatrix cameraMatrix, CameraInfo cameraInfo)
    {
      bool foundLargePart = false;
//      if(numberOfColoredPixels > 3)
      {
        if(rangeOfColor > size)
        {
          sizeOfLargePart[numberOfLargeParts] = rangeOfColor;
          largePartBegin[numberOfLargeParts].x = firstPoint.x;
          largePartBegin[numberOfLargeParts].y = firstPoint.y;
          largePartEnd[numberOfLargeParts].x = lastPoint.x;
          largePartEnd[numberOfLargeParts].y = lastPoint.y;
          largePartBeginIsOnBorder[numberOfLargeParts] = beginIsOnBorder;
          largePartEndIsOnBorder[numberOfLargeParts] = endIsOnBorder;
          
          Vector2<double> minAngles, maxAngles;
          Geometry::calculateAnglesForPoint(largePartBegin[numberOfLargeParts], cameraMatrix, cameraInfo, largePartBeginAngles[numberOfLargeParts]);
          Geometry::calculateAnglesForPoint(largePartEnd[numberOfLargeParts], cameraMatrix, cameraInfo, largePartEndAngles[numberOfLargeParts]);
          
          numberOfLargeParts++;
          foundLargePart = true;
          LINE(imageProcessor_flagsAndGoals, firstPoint.x/2, firstPoint.y/2, lastPoint.x/2, lastPoint.y/2,
            2, Drawings::ps_solid, Drawings::pink);
        }
        
        if(numberOfLargeParts >= maxNumberOfParts)
        {
          numberOfLargeParts = maxNumberOfParts-1;
        }
      }
      return foundLargePart;
    }
  };

private:
  
  /** Calculates the scan lines near the horizon for the goals */
  void calculateScanLinesParallelToHorizon(
	const CameraInfo& bwCameraInfo,
    int distanceAboveHorizon,
    int distanceBelowHorizon,
    int numberOfScanLines
    );
  

void calculatePointByAngles
(
 const Vector2<double>& angles,
 const CameraMatrix& cameraMatrix, 
 const CameraInfo& cameraInfo, 
 Vector2<int>& point
 );

  /** Scans horizontal for goals */
  void scanHorizontalForGoals
  (
   const CameraInfo& bwCameraInfo,
   LandmarksPercept& landmarksPercept,
   ObstaclesPercept& obstaclesPercept,
   int maxNumberOfFalsePixels
  );

  /** Calculates the vertical scan lines for the goals */
  void calculateVerticalGoalScanLines
  (
   const CameraInfo& bwCameraInfo,
   LandmarksPercept& landmarksPercept,
   ObstaclesPercept& obstaclesPercept
  );

  /** Scans vertical for goals */
  void scanLinesForGoals
  (
    const CameraInfo& bwCameraInfo,
    LandmarksPercept& landmarksPercept,
    ObstaclesPercept& obstaclesPercept
  );

  DECLARE_DEBUG_IMAGE(imageProcessorGoals);

  /** A reference to the image that is scanned for a goal */
  const Image& image;

  /** A reference to the camera matrix that describes position and rotation of the camera when the image was aquired */
  const CameraMatrix& cameraMatrix;

  /** A reference to the color table */
  const ColorTable& colorTable;

  /** A reference to the color corrector */
  const DDD2004ColorCorrector& colorCorrector;

  /** The color class of the opponent goal */
  colorClass colorOfOpponentGoal;

  /** The color class of the own goal */
  colorClass colorOfOwnGoal;

  /** A representation of the horizon */
  Geometry::Line horizonLine, verticalLine;

  /** The number of horizontal scan lines */
  int numberOfHorizontalScanLines;

  enum{maxNumberOfHorizontalScanLines = 32};
  enum{maxNumberOfGoalScanLines = 255};

  /** representation of the left points of the horizontal scan lines */
  Vector2<int> leftPoint[maxNumberOfHorizontalScanLines];
  
  /** representation of the right points of the horizontal scan lines */
  Vector2<int> rightPoint[maxNumberOfHorizontalScanLines];

  /** the number of indications for goals */
  int numberOfGoalIndications;

  /** left point of an indications for a goal */
  Vector2<int> goalIndicationLeft[maxNumberOfGoalScanLines];
  
  /** left point of an indications for a goal */
  Vector2<int> goalIndicationCenter[maxNumberOfGoalScanLines];
  
  /** right point of an indication for a goal */
  Vector2<int> goalIndicationRight[maxNumberOfGoalScanLines];

  /** true if the left end of the goal indication is on the border of the image */
  bool leftOfGoalIndicationIsOnBorder[maxNumberOfGoalScanLines];
  
  /** true if the right end of the goal indication is on the border of the image */
  bool rightOfGoalIndicationIsOnBorder[maxNumberOfGoalScanLines];

  /** the color class of the goal indication */
  colorClass colorOfGoalIndication[maxNumberOfGoalScanLines];

  /** The number of vertical scan lines where a goal is searched*/
  int numberOfGoalScanLines;

  /** representation of the top points of the goal scan lines */
  Vector2<int> topGoalPoint[maxNumberOfGoalScanLines];
  
  /** representation of the bottom points of the goal scan lines */
  Vector2<int> bottomGoalPoint[maxNumberOfGoalScanLines];
  
  bool scanLineToCheckBestAngle[maxNumberOfGoalScanLines];
  
  /** */
  int indexOfGoalIndication[maxNumberOfGoalScanLines];
  
  /** */
  colorClass colorOfGoalScanLine[maxNumberOfGoalScanLines];

};

#endif

/*
* Change log :
* 
* $Log: DDD2004GoalRecognizer.h,v $
* Revision 1.3  2004/04/07 12:28:57  risler
* ddd checkin after go04 - first part
*
* Revision 1.5  2004/03/31 10:56:53  risler
* bugfix
*
* Revision 1.4  2004/03/30 20:18:31  park
* works now with BlackWhite Image
*
* Revision 1.3  2004/03/30 15:38:45  park
* just fewer horizontal lines
*
* Revision 1.2  2004/03/29 14:22:38  Schmitt
* New parameter maxNumberOfFalsePixels in scanHorizontal, some simplifications
*
* Revision 1.1.1.1  2004/03/29 08:28:47  Administrator
* initial transfer from tamara
*
* Revision 1.2  2004/03/26 09:09:18  risler
* GoalRecognizer uses corrected colors
*
* Revision 1.1  2004/03/24 19:26:13  risler
* added DDD2004GoalRecognizer
*
* Revision 1.4  2003/12/15 11:47:02  juengel
* Introduced CameraInfo
*
* Revision 1.3  2003/11/26 11:43:00  juengel
* new scan line calculation method added
*
* Revision 1.2  2003/11/07 11:24:59  juengel
* ColoredPartsCheck is public now.
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.5  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.4  2003/09/01 10:16:17  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.3  2003/08/30 10:19:53  juengel
* DebugDrawings clean-up 1
*
* Revision 1.2  2003/07/29 12:46:28  juengel
* Moved calculateHorizon to Geometry
*
* Revision 1.1  2003/07/21 13:40:24  juengel
* Moved ColorTableAuto and GoalRecognizer to ImageProcessorTools.
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.1  2003/06/12 16:49:50  juengel
* Added GoalRecognizer.
*
*
*/
