/**
* @file DDD2004GoalRecognizer.cpp
*
* Implementation of class DDD2004GoalRecognizer
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#include "DDD2004GoalRecognizer.h"
#include "DDD2004ImageProcessorTools.h"
#include "Tools/Player.h"
#include "Tools/FieldDimensions.h"
#include "Tools/RangeArray.h"
#include "Representations/Perception/ColorTable64.h"

#include "../ImageProcessorTools/ColorTableReferenceColor.h"

DDD2004GoalRecognizer::DDD2004GoalRecognizer
(
 const Image& image, 
 const CameraMatrix& cameraMatrix,
 const ColorTable& colorTable,
 const DDD2004ColorCorrector& colorCorrector
 )
 :
image(image),
cameraMatrix(cameraMatrix),
colorTable(colorTable),
colorCorrector(colorCorrector)
{
}


DDD2004GoalRecognizer::~DDD2004GoalRecognizer()
{
}

void DDD2004GoalRecognizer::getGoalPercept
(
 LandmarksPercept& landmarksPercept,
 ObstaclesPercept& obstaclesPercept,
 int distanceAboveHorizon,
 int distanceBelowHorizon,
 int numberOfScanLines
 )
{
	if(getPlayer().getTeamColor() == Player::blue) 
	{
		colorOfOpponentGoal = yellow;
		colorOfOwnGoal = skyblue;
	}
	else 
	{
		colorOfOpponentGoal = skyblue;
		colorOfOwnGoal = yellow;
	}
	
	CameraInfo bwCameraInfo = image.cameraInfo;
	bwCameraInfo.resolutionHeight*=2;
	bwCameraInfo.resolutionWidth*=2;
	bwCameraInfo.focalLength*=2;
  bwCameraInfo.focalLengthInv/=2;
	bwCameraInfo.opticalCenter.x*=2;
	bwCameraInfo.opticalCenter.y*=2;
	
	
	INIT_DEBUG_IMAGE(imageProcessorGoals, image);
	
	//horizonLine = Geometry::calculateHorizon(cameraMatrix, image.cameraInfo);
	horizonLine = Geometry::calculateHorizon(cameraMatrix, bwCameraInfo);
	verticalLine.base = horizonLine.base;
	verticalLine.direction.x = - horizonLine.direction.y;
	verticalLine.direction.y = horizonLine.direction.x;
	
	calculateScanLinesParallelToHorizon(bwCameraInfo, distanceAboveHorizon, distanceBelowHorizon, numberOfScanLines);
	
	//scanHorizontalForGoals(landmarksPercept, obstaclesPercept);
	scanHorizontalForGoals(bwCameraInfo, landmarksPercept, obstaclesPercept, 10);
	calculateVerticalGoalScanLines(bwCameraInfo, landmarksPercept, obstaclesPercept);
	scanLinesForGoals(bwCameraInfo, landmarksPercept, obstaclesPercept);
	
	SEND_DEBUG_IMAGE(imageProcessorGoals);
}

void DDD2004GoalRecognizer::calculateScanLinesParallelToHorizon
(
 const CameraInfo& bwCameraInfo,
 int distanceAboveHorizon,
 int distanceBelowHorizon,
 int numberOfScanLines
 )
{
	
	numberOfHorizontalScanLines = 0;
	
	Geometry::Line scanLine;
	scanLine.direction = horizonLine.direction;
	
	int i;
	
	Vector2<int> bottomLeft(0,0);
	//Vector2<int> topRight(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3);
	Vector2<int> topRight(bwCameraInfo.resolutionWidth - 1, bwCameraInfo.resolutionHeight - 3);
	
	for(i = 0; i < numberOfScanLines; i++)
	{
		scanLine.base = 
			horizonLine.base + 
			verticalLine.direction * -(distanceBelowHorizon + (distanceAboveHorizon - distanceBelowHorizon) * i / (numberOfScanLines - 1));
		//      verticalLine.direction * distanceBelowHorizon;// - (-distanceAboveHorizon + distanceBelowHorizon) * i / numberOfScanLines);
		
		if(Geometry::getIntersectionPointsOfLineAndRectangle(
			bottomLeft, topRight, scanLine,
			leftPoint[numberOfHorizontalScanLines], 
			rightPoint[numberOfHorizontalScanLines])
			)
		{
			numberOfHorizontalScanLines++;
		}
	}
}

void DDD2004GoalRecognizer::scanHorizontalForGoals
(
 const CameraInfo& bwCameraInfo,
 LandmarksPercept& landmarksPercept,
 ObstaclesPercept& obstaclesPercept,
 int maxNumberOfFalsePixels
 )
{
	enum {yellowGoal = 1, skyblueGoal = 0, numberOfGoalColors = 2};
	int colorToCheck;
	colorClass colorClassToCheck;
	
	RangeArray<double> goalClusters[numberOfGoalColors];
	numberOfGoalIndications = 0;
	
	// variables needed to determine free part of goal
	RangeArray<double> goalClustersForFreePart[numberOfGoalColors];
	int numberOfLinesWithLargeParts[numberOfGoalColors] = {0, 0};
	
	int i;
	for(i = 0; i < numberOfHorizontalScanLines; i++)
	{
		Geometry::PixeledLine line(leftPoint[i], rightPoint[i]);
		// state machine
		// states
		enum {noGoalColorFound, foundBeginOfGoalColor, foundEndOfGoalColor}
		goalColorState[numberOfGoalColors] = {noGoalColorFound, noGoalColorFound};
		// counter
		int numberOfPixelsSinceLastOccurrenceOfGoalColor[numberOfGoalColors];
		int numberOfPixelsSinceFirstOccurrenceOfGoalColor[numberOfGoalColors] = {0,0};
		
		bool colorStartsAtBeginOfImage[numberOfGoalColors] = {false, false};
		ColoredPartsCheck goalCheck[numberOfGoalColors];
		
		int x = 0, y = 0;
		colorClass color;
		
		if(line.getNumberOfPixels() > 6)
		{
			for(int z = 2; z < line.getNumberOfPixels(); z++)
			{
				x = line.getPixelX(z);
				y = line.getPixelY(z);
				
				//color = CORRECTED_COLOR_CLASS(x,y,image.image[y][0][x], image.image[y][1][x], image.image[y][2][x]);
				color = CORRECTED_COLOR_CLASS(
					x / 2,y / 2,
					image.getHighResY(x,y),
					image.image[y / 2][1][x / 2],
					image.image[y / 2][2][x / 2]);
				
				for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
				{
					if(colorToCheck == yellowGoal) colorClassToCheck = yellow;
					else colorClassToCheck = skyblue;
					
					if(goalColorState[colorToCheck] == noGoalColorFound)
					{
						//DEBUG_IMAGE_SET_PIXEL_BLACK(imageProcessorGoals, x, y + colorToCheck);
						DEBUG_IMAGE_SET_PIXEL_BLACK(imageProcessorGoals, x/2, y/2 + colorToCheck);
						if(color == colorClassToCheck)
						{
							goalCheck[colorToCheck].numberOfColoredPixels = 0;
							goalCheck[colorToCheck].rangeOfColor = 0;
							if(numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] == 0)
							{
								goalCheck[colorToCheck].firstPoint.x = x;
								goalCheck[colorToCheck].firstPoint.y = y;
							}
							numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck]++;
							if(numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] > 4)
							{
								goalColorState[colorToCheck] = foundBeginOfGoalColor;
								//added				
								goalCheck[colorToCheck].lastPoint.x = x;
								goalCheck[colorToCheck].lastPoint.y = y;
								
								if(z < 5) 
								{
									//DOT(imageProcessor_general, x, y, Drawings::green, Drawings::green);
									DOT(imageProcessor_general, x/2, y/2, Drawings::green, Drawings::green);
									colorStartsAtBeginOfImage[colorToCheck] = true;
								}
							}
						}
						else
						{
							numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] = 0;
						}
					}
					else if(goalColorState[colorToCheck] == foundBeginOfGoalColor)
					{
						//DEBUG_IMAGE_SET_PIXEL_GRAY(imageProcessorGoals, x, y + colorToCheck);
						DEBUG_IMAGE_SET_PIXEL_GRAY(imageProcessorGoals, x/2, y/2 + colorToCheck);
						
						// added		
						goalCheck[colorToCheck].lastPoint.x = x;
						goalCheck[colorToCheck].lastPoint.y = y;
						
						if(color == colorClassToCheck) goalCheck[colorToCheck].numberOfColoredPixels++;
						else
						{
							goalCheck[colorToCheck].lastPoint.x = x;
							goalCheck[colorToCheck].lastPoint.y = y;
							goalColorState[colorToCheck] = foundEndOfGoalColor;
							numberOfPixelsSinceLastOccurrenceOfGoalColor[colorToCheck] = 0;
							numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] = 0;
						}
						goalCheck[colorToCheck].rangeOfColor++;
					}
					else if(goalColorState[colorToCheck] == foundEndOfGoalColor)
					{
						//DEBUG_IMAGE_SET_PIXEL_WHITE(imageProcessorGoals, x, y + colorToCheck);
						DEBUG_IMAGE_SET_PIXEL_WHITE(imageProcessorGoals, x/2, y/2 + colorToCheck);
						numberOfPixelsSinceLastOccurrenceOfGoalColor[colorToCheck]++;
						if(color == colorClassToCheck)
						{
							numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck]++;
							if(numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] > 4)
							{
								goalColorState[colorToCheck] = foundBeginOfGoalColor;
							}
						}
						else if(
							numberOfPixelsSinceLastOccurrenceOfGoalColor[colorToCheck] > maxNumberOfFalsePixels
							||
							color == white
							)
						{
							goalColorState[colorToCheck] = noGoalColorFound;
							//goalCheck[colorToCheck].determineLargePart(10, colorStartsAtBeginOfImage[colorToCheck], false, cameraMatrix, image.cameraInfo);
							goalCheck[colorToCheck].determineLargePart(10, colorStartsAtBeginOfImage[colorToCheck], false, cameraMatrix, bwCameraInfo);
							//DOT(imageProcessor_general, x, y, Drawings::green, Drawings::white);
							DOT(imageProcessor_general, x/2, y/2, Drawings::green, Drawings::white);
							colorStartsAtBeginOfImage[colorToCheck] = false;
						}
						else
						{
							numberOfPixelsSinceFirstOccurrenceOfGoalColor[colorToCheck] = 0;  
						}
						goalCheck[colorToCheck].rangeOfColor++;
					}
				}//for(int colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
      }//for(int z = 2; z < line.getNumberOfPixels(); z++)
	  
      for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
      {
		  
		  if(goalColorState[colorToCheck] == foundBeginOfGoalColor || goalColorState[colorToCheck] == foundEndOfGoalColor)
		  {
			  if(goalColorState[colorToCheck] == foundBeginOfGoalColor)
			  {
				  goalCheck[colorToCheck].lastPoint.x = x;
				  goalCheck[colorToCheck].lastPoint.y = y;
			  }
		  }//if(goalColorState == foundBeginOfGoalColor || goalColorState == foundEndOfGoalColor)
		  
		  if (goalColorState[colorToCheck] != noGoalColorFound) 
			  //goalCheck[colorToCheck].determineLargePart(10, colorStartsAtBeginOfImage[colorToCheck], true, cameraMatrix, image.cameraInfo);
			  goalCheck[colorToCheck].determineLargePart(10, colorStartsAtBeginOfImage[colorToCheck], true, cameraMatrix, bwCameraInfo);
		  
      }//for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
    }//if(line.getNumberOfPixels() > 3)
	
    for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
    {
		// Add all large parts found on that line to the structure that determines clusters.
		if(goalCheck[colorToCheck].numberOfLargeParts > 0) numberOfLinesWithLargeParts[colorToCheck]++;
		int index;
		for(index = 0; index < goalCheck[colorToCheck].numberOfLargeParts; index++)
		{
			goalClusters[colorToCheck].add(
				Range<double>(goalCheck[colorToCheck].largePartEndAngles[index].x, goalCheck[colorToCheck].largePartBeginAngles[index].x), 
				goalCheck[colorToCheck].largePartBeginIsOnBorder[index],
				goalCheck[colorToCheck].largePartEndIsOnBorder[index]
				);
			if(numberOfLinesWithLargeParts[colorToCheck] > 2 && numberOfLinesWithLargeParts[colorToCheck] < 5)
			{
				goalClustersForFreePart[colorToCheck].add(
					Range<double>(goalCheck[colorToCheck].largePartEndAngles[index].x, goalCheck[colorToCheck].largePartBeginAngles[index].x), 
					goalCheck[colorToCheck].largePartBeginIsOnBorder[index],
					goalCheck[colorToCheck].largePartEndIsOnBorder[index]
					);
			}
		}
    }
  }// for(int i = 0; i < numberOfHorizontalScanLines; i++)
  
  for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
  {
	  if(colorToCheck == yellowGoal) colorClassToCheck = yellow;
	  else colorClassToCheck = skyblue;
	  
	  // Add one goal indication for each cluster
	  int index;
	  for(index = 0; index < goalClusters[colorToCheck].getNumberOfClusters(); index++)
	  {
		  Range<double> currentRange = goalClusters[colorToCheck].getCluster(index);
		  Vector2<int> leftPoint, rightPoint;
		  Geometry::calculatePointByAngles(Vector2<double>(currentRange.max,0.02 * index), cameraMatrix, bwCameraInfo, leftPoint);
		  Geometry::calculatePointByAngles(Vector2<double>(currentRange.min,0.02 * index), cameraMatrix, bwCameraInfo, rightPoint);
		  //Geometry::calculatePointByAngles(Vector2<double>(currentRange.max,0.02 * index), cameraMatrix, image.cameraInfo, leftPoint);
		  //Geometry::calculatePointByAngles(Vector2<double>(currentRange.min,0.02 * index), cameraMatrix, image.cameraInfo, rightPoint);
		  LINE(imageProcessor_flagsAndGoals, leftPoint.x/2, leftPoint.y/2, rightPoint.x/2, rightPoint.y/2, 2, Drawings::ps_solid, Drawings::blue);
		  goalIndicationLeft[numberOfGoalIndications]   = leftPoint;
		  goalIndicationRight[numberOfGoalIndications]  = rightPoint;
		  goalIndicationCenter[numberOfGoalIndications] = (leftPoint + rightPoint) / 2;
		  leftOfGoalIndicationIsOnBorder[numberOfGoalIndications] = goalClusters[colorToCheck].isLeftOnBorder(index);
		  rightOfGoalIndicationIsOnBorder[numberOfGoalIndications] = goalClusters[colorToCheck].isRightOnBorder(index);
		  colorOfGoalIndication[numberOfGoalIndications] = colorClassToCheck;
		  numberOfGoalIndications++;
	  }
	  
	  // Determine the free part of the goal
	  for(index = 0; index < goalClustersForFreePart[colorToCheck].getNumberOfClusters(); index++)
	  {
		  Range<double> currentRange = goalClustersForFreePart[colorToCheck].getCluster(index);
		  Vector2<int> leftPoint, rightPoint;
		  Geometry::calculatePointByAngles(Vector2<double>(currentRange.min,-0.1 + 0.02 * index), cameraMatrix, bwCameraInfo, leftPoint);
		  Geometry::calculatePointByAngles(Vector2<double>(currentRange.max,-0.1 + 0.02 * index), cameraMatrix, bwCameraInfo, rightPoint);
		  //Geometry::calculatePointByAngles(Vector2<double>(currentRange.min,-0.1 + 0.02 * index), cameraMatrix, image.cameraInfo, leftPoint);
		  //Geometry::calculatePointByAngles(Vector2<double>(currentRange.max,-0.1 + 0.02 * index), cameraMatrix, image.cameraInfo, rightPoint);
		  LINE(imageProcessor_flagsAndGoals, leftPoint.x/2, leftPoint.y/2, rightPoint.x/2, rightPoint.y/2, 2, Drawings::ps_solid, Drawings::green);
	  }
	  
	  double center, angle;
	  bool determinedFreePartOfGoal = false;
	  if(goalClustersForFreePart[colorToCheck].getNumberOfClusters() == 1)
	  {
		  bool leftOfPartIsOnBorder = goalClustersForFreePart[colorToCheck].isLeftOnBorder(0);
		  bool rightOfPartIsOnBorder = goalClustersForFreePart[colorToCheck].isRightOnBorder(0);
		  Range<double> freePartRange = goalClustersForFreePart[colorToCheck].getCluster(0);
		  //if(!leftOfPartIsOnBorder && !rightOfPartIsOnBorder || freePartRange.getSize() > 0.7 * image.cameraInfo.openingAngleWidth)
		  if(!leftOfPartIsOnBorder && !rightOfPartIsOnBorder || freePartRange.getSize() > 0.7 * bwCameraInfo.openingAngleWidth)
		  {
			  determinedFreePartOfGoal = true;
			  center = (freePartRange.min + freePartRange.max) / 2.0;
			  angle = freePartRange.max - freePartRange.min;
		  }
	  }
	  else if(goalClustersForFreePart[colorToCheck].getNumberOfClusters() == 2)
	  {
		  Range<double> freePartRange[2];
		  freePartRange[0] = goalClustersForFreePart[colorToCheck].getCluster(0);
		  freePartRange[1] = goalClustersForFreePart[colorToCheck].getCluster(1);
		  
		  bool freePartIsSeenCompletely[2];
		  
		  freePartIsSeenCompletely[0] =
			  !goalClustersForFreePart[colorToCheck].isLeftOnBorder(0) && 
			  !goalClustersForFreePart[colorToCheck].isRightOnBorder(0);
		  
		  freePartIsSeenCompletely[1] =
			  !goalClustersForFreePart[colorToCheck].isLeftOnBorder(1) && 
			  !goalClustersForFreePart[colorToCheck].isRightOnBorder(1);
		  
		  int largePart, smallPart;
		  if(freePartRange[0].getSize() < freePartRange[1].getSize()) 
		  {
			  largePart = 1;
			  smallPart = 0;
		  }
		  else 
		  {
			  largePart = 0;
			  smallPart = 1;
		  }
		  
		  if(
			  freePartIsSeenCompletely[largePart] && freePartIsSeenCompletely[smallPart] ||
			  !freePartIsSeenCompletely[largePart] && freePartIsSeenCompletely[smallPart]
			  )
		  {
			  determinedFreePartOfGoal = true;
			  center = (freePartRange[largePart].min + freePartRange[largePart].max) / 2.0;
			  angle = freePartRange[largePart].max - freePartRange[largePart].min;
		  }
	  }
	  
	  if(determinedFreePartOfGoal)
	  {
		  if(colorClassToCheck == colorOfOpponentGoal)
		  {
			  obstaclesPercept.angleToFreePartOfGoal[ObstaclesPercept::opponentGoal] = center;
			  obstaclesPercept.widthOfFreePartOfGoal[ObstaclesPercept::opponentGoal] = angle;
			  obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::opponentGoal] = true;
		  }
		  else
		  {
			  obstaclesPercept.angleToFreePartOfGoal[ObstaclesPercept::ownGoal] = center;
			  obstaclesPercept.widthOfFreePartOfGoal[ObstaclesPercept::ownGoal] = angle;
			  obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::ownGoal] = true;
		  }
	  }
	  
  }//for(colorToCheck = 0; colorToCheck < numberOfGoalColors; colorToCheck++)
}//void GridImageProcessor2::scanHorizontalForFlagsAndGoals()

void DDD2004GoalRecognizer::calculateVerticalGoalScanLines
(
 const CameraInfo& bwCameraInfo,
 LandmarksPercept& landmarksPercept,
 ObstaclesPercept& obstaclesPercept
 )
{
	numberOfGoalScanLines = 0;
	int i;
	Vector2<double> anglesForGoalScanLine;
	for(i = 0; i < numberOfGoalIndications; i++)
	{ 
		// Line through center of indication
		//Geometry::calculateAnglesForPoint(goalIndicationCenter[i], cameraMatrix, image.cameraInfo, anglesForGoalScanLine);
		Geometry::calculateAnglesForPoint(goalIndicationCenter[i], cameraMatrix, bwCameraInfo, anglesForGoalScanLine);
		anglesForGoalScanLine.y = -1.0;
		//Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		anglesForGoalScanLine.y =  0.3;
		//Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, topGoalPoint[numberOfGoalScanLines]);
		Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, topGoalPoint[numberOfGoalScanLines]);
		
		//if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
		if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(bwCameraInfo.resolutionWidth - 1, bwCameraInfo.resolutionHeight - 3),
			topGoalPoint[numberOfGoalScanLines], bottomGoalPoint[numberOfGoalScanLines]))
		{
			colorOfGoalScanLine[numberOfGoalScanLines] = colorOfGoalIndication[i];
			indexOfGoalIndication[numberOfGoalScanLines] = i;
			scanLineToCheckBestAngle[numberOfGoalScanLines] = false;
			numberOfGoalScanLines++;
		}
		
		// Line at the left side
		Geometry::calculateAnglesForPoint((goalIndicationCenter[i]+goalIndicationLeft[i])/2, cameraMatrix, bwCameraInfo, anglesForGoalScanLine);
		//Geometry::calculateAnglesForPoint(((goalIndicationCenter[i]-goalIndicationLeft[i])/3)*2 + goalIndicationCenter[i], cameraMatrix, image.cameraInfo, anglesForGoalScanLine);
		//Geometry::calculateAnglesForPoint(((goalIndicationCenter[i]-goalIndicationLeft[i])/3)*2 + goalIndicationCenter[i], cameraMatrix, bwCameraInfo, anglesForGoalScanLine);
		anglesForGoalScanLine.y = -1.0;
		//Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		anglesForGoalScanLine.y =  0.3;
		//Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, topGoalPoint[numberOfGoalScanLines]);
		Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, topGoalPoint[numberOfGoalScanLines]);
		
		//if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
		if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(bwCameraInfo.resolutionWidth - 1, bwCameraInfo.resolutionHeight - 3),
			topGoalPoint[numberOfGoalScanLines], bottomGoalPoint[numberOfGoalScanLines]))
		{
			colorOfGoalScanLine[numberOfGoalScanLines] = colorOfGoalIndication[i];
			indexOfGoalIndication[numberOfGoalScanLines] = i;
			scanLineToCheckBestAngle[numberOfGoalScanLines] = false;
			numberOfGoalScanLines++;
		}
		
		// Line between left and center side
		//Geometry::calculateAnglesForPoint(((goalIndicationCenter[i]-goalIndicationLeft[i])/3) + goalIndicationCenter[i], cameraMatrix, image.cameraInfo, anglesForGoalScanLine);
		Geometry::calculateAnglesForPoint(((goalIndicationCenter[i]-goalIndicationLeft[i])/3) + goalIndicationCenter[i], cameraMatrix, bwCameraInfo, anglesForGoalScanLine);
		anglesForGoalScanLine.y = -1.0;
		//Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		anglesForGoalScanLine.y =  0.3;
		//Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, topGoalPoint[numberOfGoalScanLines]);
		Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, topGoalPoint[numberOfGoalScanLines]);
		
		//if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
		if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(bwCameraInfo.resolutionWidth - 1, bwCameraInfo.resolutionHeight - 3),
			topGoalPoint[numberOfGoalScanLines], bottomGoalPoint[numberOfGoalScanLines]))
		{
			colorOfGoalScanLine[numberOfGoalScanLines] = colorOfGoalIndication[i];
			indexOfGoalIndication[numberOfGoalScanLines] = i;
			scanLineToCheckBestAngle[numberOfGoalScanLines] = false;
			numberOfGoalScanLines++;
		}
		
		// Line between center and right side
		//Geometry::calculateAnglesForPoint(goalIndicationCenter[i] - (goalIndicationRight[i] - goalIndicationCenter[i])/3, cameraMatrix, image.cameraInfo, anglesForGoalScanLine);
		Geometry::calculateAnglesForPoint(goalIndicationCenter[i] - (goalIndicationRight[i] - goalIndicationCenter[i])/3, cameraMatrix, bwCameraInfo, anglesForGoalScanLine);
		anglesForGoalScanLine.y = -1.0;
		//Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		anglesForGoalScanLine.y =  0.3;
		//Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, topGoalPoint[numberOfGoalScanLines]);
		Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, topGoalPoint[numberOfGoalScanLines]);
		
		//if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
		if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(bwCameraInfo.resolutionWidth - 1, bwCameraInfo.resolutionHeight - 3),
			topGoalPoint[numberOfGoalScanLines], bottomGoalPoint[numberOfGoalScanLines]))
		{
			colorOfGoalScanLine[numberOfGoalScanLines] = colorOfGoalIndication[i];
			indexOfGoalIndication[numberOfGoalScanLines] = i;
			scanLineToCheckBestAngle[numberOfGoalScanLines] = false;
			numberOfGoalScanLines++;
		}
		
		
		// Line at the right side
		//Geometry::calculateAnglesForPoint((goalIndicationCenter[i] + goalIndicationRight[i])/2, cameraMatrix, bwCameraInfo, anglesForGoalScanLine);
		//Geometry::calculateAnglesForPoint(goalIndicationCenter[i] - (((goalIndicationRight[i] - goalIndicationCenter[i])/3)*2), cameraMatrix, image.cameraInfo, anglesForGoalScanLine);
		Geometry::calculateAnglesForPoint(goalIndicationCenter[i] - (((goalIndicationRight[i] - goalIndicationCenter[i])/3)*2), cameraMatrix, bwCameraInfo, anglesForGoalScanLine);
		anglesForGoalScanLine.y = -1.0;
		//Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		anglesForGoalScanLine.y =  0.3;
		//Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, topGoalPoint[numberOfGoalScanLines]);
		Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, topGoalPoint[numberOfGoalScanLines]);
		
		//if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
		if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(bwCameraInfo.resolutionWidth - 1, bwCameraInfo.resolutionHeight - 3),
			topGoalPoint[numberOfGoalScanLines], bottomGoalPoint[numberOfGoalScanLines]))
		{
			colorOfGoalScanLine[numberOfGoalScanLines] = colorOfGoalIndication[i];
			indexOfGoalIndication[numberOfGoalScanLines] = i;
			scanLineToCheckBestAngle[numberOfGoalScanLines] = false;
			numberOfGoalScanLines++;
		}
		
  }
  
  // line for the free part of the goal
  for(i = 0; i < 2; i++)
  {
	  if(obstaclesPercept.angleToFreePartOfGoalWasDetermined[i] == true)
	  {
		  anglesForGoalScanLine.x = obstaclesPercept.angleToFreePartOfGoal[i];
		  anglesForGoalScanLine.y = -1.0;
		  //Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		  Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, bottomGoalPoint[numberOfGoalScanLines]);
		  anglesForGoalScanLine.y =  0.3;
		  //Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, image.cameraInfo, topGoalPoint[numberOfGoalScanLines]);
		  Geometry::calculatePointByAngles(anglesForGoalScanLine, cameraMatrix, bwCameraInfo, topGoalPoint[numberOfGoalScanLines]);
		  
		  //if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
		  if(Geometry::clipLineWithRectangleCohenSutherland(Vector2<int>(0,0), Vector2<int>(bwCameraInfo.resolutionWidth - 1, bwCameraInfo.resolutionHeight - 3),
			  topGoalPoint[numberOfGoalScanLines], bottomGoalPoint[numberOfGoalScanLines]))
		  {
			  if(i==1) colorOfGoalScanLine[numberOfGoalScanLines] = colorOfOpponentGoal;
			  else colorOfGoalScanLine[numberOfGoalScanLines] = colorOfOwnGoal;
			  scanLineToCheckBestAngle[numberOfGoalScanLines] = true;
			  numberOfGoalScanLines++;
		  }
	  }
  }
}// calculateVerticalGoalScanLines()

void DDD2004GoalRecognizer::scanLinesForGoals
(
 const CameraInfo& bwCameraInfo,
 LandmarksPercept& landmarksPercept,
 ObstaclesPercept& obstaclesPercept
 )
{
	ConditionalBoundary yellowGoalBoundary, skyblueGoalBoundary;
	ConditionalBoundary boundaryOfIndication[maxNumberOfGoalScanLines];
	bool useBoundaryOfIndication[maxNumberOfGoalScanLines];
	int i;
	for(i = 0; i < numberOfGoalIndications; i++) useBoundaryOfIndication[i] = true;
	
	bool foundYellowGoal = false;
	bool foundSkyblueGoal = false;
	bool pixelHasGoalColor;
	
	for(i = 0; i < numberOfGoalScanLines; i++)
	{
		Geometry::PixeledLine line(topGoalPoint[i], bottomGoalPoint[i]);
		// state machine
		// states
		enum { noGoalColorFound, 
			foundBeginOfGoalColor, 
			foundEndOfGoalColor,
			determinedGoal
		} goalColorState = noGoalColorFound;
		// counter
		int numberOfPixelsSinceLastOccurrenceOfGoalColor = 0;
		int numberOfPixelsSinceFirstOccurrenceOfGoalColor = 0;
		int whiteAndPinkCounter = 0;
		
		bool colorStartsAtBeginOfImage = false;
		
		ColoredPartsCheck goalCheck;
		int x = 0, y = 0;
		colorClass color;
		
		if(line.getNumberOfPixels() > 6)
		{
			bool scanLineIsPartOfGoal = true;
			for(int z = 2; z < line.getNumberOfPixels(); z++) 
			{
				pixelHasGoalColor = false;
				x = line.getPixelX(z);
				y = line.getPixelY(z);
				if(scanLineToCheckBestAngle[i]){
					DEBUG_IMAGE_SET_PIXEL_GRAY(imageProcessorGoals, x/2, y/2);
				}
				else { 
					DEBUG_IMAGE_SET_PIXEL_WHITE(imageProcessorGoals, x/2, y/2);			
				}
				
				//color = CORRECTED_COLOR_CLASS(x,y,image.image[y][0][x], image.image[y][1][x], image.image[y][2][x]);
				color = CORRECTED_COLOR_CLASS(
					x / 2,y / 2,
					image.getHighResY(x,y),
					image.image[y / 2][1][x / 2],
					image.image[y / 2][2][x / 2]);
				
				if(colorOfGoalScanLine[i] == skyblue && color == skyblue) pixelHasGoalColor = true;
				if(colorOfGoalScanLine[i] == yellow && color == yellow) pixelHasGoalColor = true;
				
				if(goalColorState == noGoalColorFound)
				{
					if(pixelHasGoalColor)
					{
						DEBUG_IMAGE_SET_PIXEL_BLACK(imageProcessorGoals, x/2, y/2);
						goalCheck.numberOfColoredPixels = 0;
						goalCheck.rangeOfColor = 0;
						if(numberOfPixelsSinceFirstOccurrenceOfGoalColor == 0)
						{
							goalCheck.firstPoint.x = x;
							goalCheck.firstPoint.y = y;
						}
						numberOfPixelsSinceFirstOccurrenceOfGoalColor++;
						if(numberOfPixelsSinceFirstOccurrenceOfGoalColor > 4)
						{
							goalColorState = foundBeginOfGoalColor;
							if(z < 5) colorStartsAtBeginOfImage = true;
						}
					}
					/*
					else if(color == pink) 
					{
						DEBUG_IMAGE_SET_PIXEL_PINK(imageProcessorGoals, x/2, y/2);
						pinkCounter++;
						if (pinkCounter > 5)
						{
							goalColorState = determinedGoal;
							scanLineIsPartOfGoal = false;
							DOT(imageProcessor_general, x/2, y/2, Drawings::blue, Drawings::blue);
							whiteCounter = 0;
							pinkCounter = 0;
							colorStartsAtBeginOfImage = false;
						}
					}*/
					else
					{
						numberOfPixelsSinceFirstOccurrenceOfGoalColor = 0;
					}
				}
				else if(goalColorState == foundBeginOfGoalColor)
				{
					DEBUG_IMAGE_SET_PIXEL_GREEN(imageProcessorGoals, x/2, y/2);
					if(pixelHasGoalColor) goalCheck.numberOfColoredPixels++;
					else
					{
						goalCheck.lastPoint.x = x;
						goalCheck.lastPoint.y = y;
						goalColorState = foundEndOfGoalColor;
						numberOfPixelsSinceLastOccurrenceOfGoalColor = 0;
						numberOfPixelsSinceFirstOccurrenceOfGoalColor = 0;
					}
					goalCheck.rangeOfColor++;
				}
				else if(goalColorState == foundEndOfGoalColor)
				{
					numberOfPixelsSinceLastOccurrenceOfGoalColor++;
					DEBUG_IMAGE_SET_PIXEL_RED(imageProcessorGoals, x/2, y/2);
					if(pixelHasGoalColor)
					{
						numberOfPixelsSinceFirstOccurrenceOfGoalColor++;
						if(numberOfPixelsSinceFirstOccurrenceOfGoalColor > 4)
						{
							goalColorState = foundBeginOfGoalColor;
						}
					}
					else if(numberOfPixelsSinceLastOccurrenceOfGoalColor > 40)
					{
						DEBUG_IMAGE_SET_PIXEL_BLUE(imageProcessorGoals, x/2, y/2);
						//            goalColorState = noGoalColorFound;
						goalColorState = determinedGoal;
						//if(goalCheck.determineLargePart(7, colorStartsAtBeginOfImage, false, cameraMatrix, image.cameraInfo))
						if(goalCheck.determineLargePart(14, colorStartsAtBeginOfImage, false, cameraMatrix, bwCameraInfo))
						{
							if(whiteAndPinkCounter >10)
							{
								scanLineIsPartOfGoal = false;
								DOT(imageProcessor_general, x/2, y/2, Drawings::blue, Drawings::blue);
							}
						}
						whiteAndPinkCounter = 0;
						//pinkCounter = 0;
						colorStartsAtBeginOfImage = false;
					}
					//else if(color == white || color == pink) 
					else if(color == pink || color == white) 
					{
						DEBUG_IMAGE_SET_PIXEL_PINK(imageProcessorGoals, x/2, y/2);
						whiteAndPinkCounter++;			
					}
					/*
					else if(color==white)
					{
						DEBUG_IMAGE_SET_PIXEL_PINK(imageProcessorGoals, x/2, y/2);
						whiteCounter++;
					}*/
					else
					{
						numberOfPixelsSinceFirstOccurrenceOfGoalColor = 0;  
					}
					goalCheck.rangeOfColor++;
				}
      }//for(int z = 0; z < line.getNumberOfPixels(); z++) 
      if(goalColorState == foundBeginOfGoalColor || goalColorState == foundEndOfGoalColor)
      {
		  if(goalColorState == foundBeginOfGoalColor)
		  {
			  goalCheck.lastPoint.x = x;
			  goalCheck.lastPoint.y = y;
		  }
		  //goalCheck.determineLargePart(7, colorStartsAtBeginOfImage, true, cameraMatrix,image.cameraInfo);
		  goalCheck.determineLargePart(7, colorStartsAtBeginOfImage, true, cameraMatrix,bwCameraInfo);
      }//if(goalColorState == foundBeginOfGoalColor || goalColorState == foundEndOfGoalColor)
      
      if(!scanLineIsPartOfGoal && !scanLineToCheckBestAngle[i]) 
		  useBoundaryOfIndication[indexOfGoalIndication[i]] = false;
	  
      if(goalCheck.numberOfLargeParts > 0)
      {
		  Vector2<double> topAngles, bottomAngles, leftAngles, rightAngles;
		  //Geometry::calculateAnglesForPoint(goalCheck.largePartBegin[0], cameraMatrix, image.cameraInfo, topAngles);
		  //Geometry::calculateAnglesForPoint(goalCheck.largePartEnd[goalCheck.numberOfLargeParts - 1], cameraMatrix, image.cameraInfo, bottomAngles);
		  Geometry::calculateAnglesForPoint(goalCheck.largePartBegin[0], cameraMatrix, bwCameraInfo, topAngles);
		  Geometry::calculateAnglesForPoint(goalCheck.largePartEnd[goalCheck.numberOfLargeParts - 1], cameraMatrix, bwCameraInfo, bottomAngles);
		  
		  if(scanLineToCheckBestAngle[i])
		  {
			  if(bottomAngles.y > 0 || !scanLineIsPartOfGoal) 
			  {
				  if(colorOfOpponentGoal == colorOfGoalScanLine[i])
					  obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::opponentGoal] = false;
				  else
					  obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::ownGoal] = false;
			  }
			  else
			  {
				  //calculate distance
          int distance = Geometry::getDistanceBySize(
            bwCameraInfo,
					  goalHeight, 
					  (goalCheck.largePartBegin[0] - goalCheck.largePartEnd[goalCheck.numberOfLargeParts - 1]).abs()
					  );
				  //~ int distance = Geometry::getDistanceBySize(
					  //~ goalHeight, 
					  //~ (goalCheck.largePartBegin[0] - goalCheck.largePartEnd[goalCheck.numberOfLargeParts - 1]).abs(), 
					  //~ bwCameraInfo.resolutionWidth, bwCameraInfo.openingAngleWidth);
				  
				  if(colorOfOpponentGoal == colorOfGoalScanLine[i])
					  obstaclesPercept.distanceToFreePartOfGoal[ObstaclesPercept::opponentGoal] = distance;
				  else
					  obstaclesPercept.distanceToFreePartOfGoal[ObstaclesPercept::ownGoal] = distance;
			  }
		  }
		  else
		  {
			  //Geometry::calculateAnglesForPoint(goalIndicationLeft[indexOfGoalIndication[i]], cameraMatrix, image.cameraInfo, leftAngles);
			  //Geometry::calculateAnglesForPoint(goalIndicationRight[indexOfGoalIndication[i]], cameraMatrix, image.cameraInfo, rightAngles);
			  Geometry::calculateAnglesForPoint(goalIndicationLeft[indexOfGoalIndication[i]], cameraMatrix, bwCameraInfo, leftAngles);
			  Geometry::calculateAnglesForPoint(goalIndicationRight[indexOfGoalIndication[i]], cameraMatrix, bwCameraInfo, rightAngles);
			  
			  // if(bottomAngles.y < 0)
			  //if(topAngles.y > 0)
			  {
				  boundaryOfIndication[indexOfGoalIndication[i]].addY(topAngles.y, goalCheck.largePartBeginIsOnBorder[0]);
				  boundaryOfIndication[indexOfGoalIndication[i]].addY(bottomAngles.y, goalCheck.largePartEndIsOnBorder[goalCheck.numberOfLargeParts - 1]);
				  boundaryOfIndication[indexOfGoalIndication[i]].addX(leftAngles.x, leftOfGoalIndicationIsOnBorder[indexOfGoalIndication[i]]);
				  boundaryOfIndication[indexOfGoalIndication[i]].addX(rightAngles.x, rightOfGoalIndicationIsOnBorder[indexOfGoalIndication[i]]);
			  }//
		  }//!scanLineToCheckBestAngle[i]
      }//if(goalCheck.numberOfLargeParts > 0)
      else
      {
		  if(scanLineToCheckBestAngle[i])
		  {
			  if(colorOfOpponentGoal == colorOfGoalScanLine[i])
				  obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::opponentGoal] = false;
			  else
				  obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::ownGoal] = false;
		  }
      }
    }//if(line.getNumberOfPixels() > 3)
  }//for(int i = 0; i < numberOfGoalScanLines; i++)
  
  
  Vector2<int> corner[4];
  int penWidth[4] = {3,3,3,3};
  
  for(i = 0; i < numberOfGoalIndications; i++)
  {
	  Drawings::Color color;
	  color = Drawings::dark_gray;
	  
	  if(useBoundaryOfIndication[i])
	  {
		  color = Drawings::light_gray;
		  if(colorOfGoalIndication[i] == yellow) 
		  {
			  yellowGoalBoundary.add(boundaryOfIndication[i]);
			  foundYellowGoal = true;
		  }
		  if(colorOfGoalIndication[i] == skyblue) 
		  {
			  skyblueGoalBoundary.add(boundaryOfIndication[i]);
			  foundSkyblueGoal = true;
		  }
	  }
	  
	  (boundaryOfIndication[i].isOnBorder(boundaryOfIndication[i].y.max))? penWidth[0] = 1 : penWidth[0] = 3;
	  (boundaryOfIndication[i].isOnBorder(boundaryOfIndication[i].x.min))? penWidth[1] = 1 : penWidth[1] = 3;
	  (boundaryOfIndication[i].isOnBorder(boundaryOfIndication[i].y.min))? penWidth[2] = 1 : penWidth[2] = 3;
	  (boundaryOfIndication[i].isOnBorder(boundaryOfIndication[i].x.max))? penWidth[3] = 1 : penWidth[3] = 3;
	  
	  Geometry::calculatePointByAngles(
		  Vector2<double>(boundaryOfIndication[i].x.max,boundaryOfIndication[i].y.max),
		  //cameraMatrix, image.cameraInfo, 
		  cameraMatrix, bwCameraInfo, 
		  corner[0]
		  );
	  
	  Geometry::calculatePointByAngles(
		  Vector2<double>(boundaryOfIndication[i].x.min,boundaryOfIndication[i].y.max),
		  //cameraMatrix, image.cameraInfo, 
		  cameraMatrix, bwCameraInfo, 
		  corner[1]
		  );
	  
	  Geometry::calculatePointByAngles(
		  Vector2<double>(boundaryOfIndication[i].x.min,boundaryOfIndication[i].y.min),
		  //cameraMatrix, image.cameraInfo, 
		  cameraMatrix, bwCameraInfo, 
		  corner[2]
		  );
	  
	  Geometry::calculatePointByAngles(
		  Vector2<double>(boundaryOfIndication[i].x.max,boundaryOfIndication[i].y.min),
		  //cameraMatrix, image.cameraInfo, 
		  cameraMatrix, bwCameraInfo, 
		  corner[3]
		  );
	  
	  LINE(imageProcessor_flagsAndGoals,
		  (int)corner[0].x, (int)corner[0].y, 
		  (int)corner[1].x, (int)corner[1].y,
		  penWidth[0], Drawings::ps_solid, color);
	  LINE(imageProcessor_flagsAndGoals,
		  (int)corner[1].x, (int)corner[1].y, 
		  (int)corner[2].x, (int)corner[2].y,
		  penWidth[1], Drawings::ps_solid, color);
	  LINE(imageProcessor_flagsAndGoals,
		  (int)corner[2].x, (int)corner[2].y, 
		  (int)corner[3].x, (int)corner[3].y,
		  penWidth[2], Drawings::ps_solid, color);
	  LINE(imageProcessor_flagsAndGoals,
		  (int)corner[3].x, (int)corner[3].y, 
		  (int)corner[0].x, (int)corner[0].y,
		  penWidth[3], Drawings::ps_solid, color);
  }
  
  bool ownTeamColorIsBlue = getPlayer().getTeamColor() == Player::blue;
  if(
	  foundYellowGoal && //yellowGoalBoundary.y.min < fromDegrees(0) &&	  
	  (
	  yellowGoalBoundary.x.getSize() > 1.5 * yellowGoalBoundary.y.getSize() ||
      yellowGoalBoundary.isOnBorder(yellowGoalBoundary.x.min) ||
      yellowGoalBoundary.isOnBorder(yellowGoalBoundary.x.max)
	  )
	  )
  {
	  landmarksPercept.addGoal(yellow, ownTeamColorIsBlue, yellowGoalBoundary);
	  Vector2<double> cameraOffset(cameraMatrix.translation.x, cameraMatrix.translation.y);
	  landmarksPercept.estimateOffsetForGoals(cameraOffset); 
  }
  if(
	  foundSkyblueGoal && //skyblueGoalBoundary.y.min < fromDegrees(0) &&	  
	  (
	  skyblueGoalBoundary.x.getSize() > 1.5 * skyblueGoalBoundary.y.getSize() ||
      skyblueGoalBoundary.isOnBorder(skyblueGoalBoundary.x.min) ||
      skyblueGoalBoundary.isOnBorder(skyblueGoalBoundary.x.max)
	  )
	  )
  {
	  landmarksPercept.addGoal(skyblue, ownTeamColorIsBlue, skyblueGoalBoundary);
	  Vector2<double> cameraOffset(cameraMatrix.translation.x, cameraMatrix.translation.y);
	  landmarksPercept.estimateOffsetForGoals(cameraOffset); 
  }
}//scanLinesForGoals()

 /*
 * Change log :
 * 
 * $Log: DDD2004GoalRecognizer.cpp,v $
 * Revision 1.4  2004/05/07 15:16:24  nistico
 * All geometry calculations are making use of intrinsic functions.
 * I updated most of the images processors to make correct use of this.
 * Please test if there are any problems, because i'm going to remove the
 * old code soon.
 *
 * Revision 1.3  2004/04/07 12:28:57  risler
 * ddd checkin after go04 - first part
 *
 * Revision 1.12  2004/04/06 13:19:36  risler
 * cleaned up and improved high resolution image support
 *
 * Revision 1.11  2004/04/01 10:22:48  park
 * better ignoring of false goals
 *
 * Revision 1.10  2004/03/31 13:51:20  park
 * another bugfix
 *
 * Revision 1.9  2004/03/31 12:04:08  park
 * improved isolation of landmarks, some minor changes in conditions for goal percept
 *
 * Revision 1.8  2004/03/30 21:26:57  park
 * added optical center and focal length info to bwCameraInfo, for use
 * in INTRINSIC methods
 *
 * Revision 1.7  2004/03/30 20:19:42  park
 * removed debug messages again
 *
 * Revision 1.6  2004/03/30 20:18:38  park
 * works now with BlackWhite Image
 *
 * Revision 1.5  2004/03/30 15:49:45  park
 * removed debug messages
 *
 * Revision 1.4  2004/03/30 15:39:15  park
 * minor changes in conditions for goal-percept
 *
 * Revision 1.3  2004/03/29 17:59:56  park
 * added more vertical lines
 *
 * Revision 1.2  2004/03/29 14:22:34  Schmitt
 * New parameter maxNumberOfFalsePixels in scanHorizontal, some simplifications
 *
 * Revision 1.1.1.1  2004/03/29 08:28:47  Administrator
 * initial transfer from tamara
 *
 * Revision 1.2  2004/03/26 09:09:18  risler
 * GoalRecognizer uses corrected colors
 *
 * Revision 1.1  2004/03/24 19:26:13  risler
 * added DDD2004GoalRecognizer
 *
 * Revision 1.6  2004/02/03 15:29:00  nistico
 * GoalRecognizer is now ColorTable agnostic, no relevant adverse performance impact found
 *
 * Revision 1.5  2003/12/31 23:50:37  roefer
 * Removed inclusion of RobotDimensions.h because it is not used
 *
 * Revision 1.4  2003/12/15 11:47:00  juengel
 * Introduced CameraInfo
 *
 * Revision 1.3  2003/11/26 11:43:00  juengel
 * new scan line calculation method added
 *
 * Revision 1.2  2003/10/23 07:24:20  juengel
 * Renamed ColorTableAuto to ColorTableReferenceColor.
 *
 * Revision 1.1  2003/10/06 14:10:14  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.8  2003/09/26 15:27:49  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.7  2003/09/26 11:38:51  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.6  2003/09/01 10:16:17  juengel
 * DebugDrawings clean-up 2
 * DebugImages clean-up
 * MessageIDs clean-up
 * Stopwatch clean-up
 *
 * Revision 1.5  2003/08/30 10:19:53  juengel
 * DebugDrawings clean-up 1
 *
 * Revision 1.4  2003/08/29 13:15:05  juengel
 * auto calibrating version uses level 2 colors.
 *
 * Revision 1.3  2003/08/18 12:06:59  juengel
 * Changed usage of numberOfPixelsSinceLastOccurrenceOfGoalColor.
 *
 * Revision 1.2  2003/07/29 12:46:28  juengel
 * Moved calculateHorizon to Geometry
 *
 * Revision 1.1  2003/07/21 13:40:24  juengel
 * Moved ColorTableReferenceColor and GoalRecognizer to ImageProcessorTools.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.2  2003/06/24 08:15:12  dueffert
 * calculation optimized
 *
 * Revision 1.1  2003/06/12 16:49:50  juengel
 * Added GoalRecognizer.
 *
 *
 */
